// -*-Mode: C++;-*-
//
// Style sheet implementation in StyleMgr database
//
// $Id: StyleMgrStyleImpl.cpp,v 1.3 2011/04/17 06:16:17 rishitani Exp $

#include <common.h>

#include "StyleMgr.hpp"
#include "StyleSet.hpp"
#include "StyleSheet.hpp"

#include <qlib/FileStream.hpp>
#include <qlib/LDOM2Stream.hpp>

using namespace qsys;
using qlib::LDom2Node;

namespace {
bool splitKeyName(const LString &keyname, LString &subkey, LString &trailkey)
{
  int dpos = keyname.indexOf(".");

  if (dpos<0) {
    subkey = keyname;
    trailkey = "";
    return false;
  }
  else {
    subkey = keyname.substr(0, dpos);
    trailkey = keyname.substr(dpos+1);
    return true;
  }
}
}

/*
LDom2Node *StyleMgr::getStyleNode2(const LString &dotname,
                                   qlib::uid_t ctxt)
{
  LString style_name;
  LString prop_names;
  if (!splitKeyName(dotname, style_name, prop_names)) {
    MB_DPRINTLN("StyleMgr.getStyleNode2> invalid key name: %s",dotname.c_str());
    return NULL;
  }

  return getStyleNode2(style_name, prop_names, ctxt);
}
*/

LDom2Node *StyleMgr::getStyleNode2(const LString &stylename,
                                   const LString &prop_names,
                                   qlib::uid_t ctxt)
{
  return getStyleNode2(LString(), stylename, prop_names, ctxt, false);
}

LDom2Node *StyleMgr::getStyleNode2(const LString &aSetID,
                                   const LString &stylename,
                                   const LString &prop_names,
                                   qlib::uid_t ctxt,
                                   bool bCreate)
{
  StyleList *pSList = getCreateStyleList(ctxt);
  LDom2Node *pStyle=NULL, *pNode=NULL;
  MB_ASSERT(pSList!=NULL);

  // Search style set list from top to bottom
  BOOST_FOREACH (StyleSet *pStySet, *pSList) {
    const LString &setid = pStySet->getName();
    
    // Empty aSetID matches all style sets
    if (!aSetID.isEmpty() && !aSetID.equals(setid))
      continue;
    
    pStyle = pStySet->getData(makeStyleKey(stylename));
    if (pStyle==NULL) {
      if (!bCreate) {
        // The styleset pStySet does not contain style named "stylename".
        // --> search the next style sheet
        //MB_DPRINTLN("Style %s is not found in stylesheet %s",
        //stylename.c_str(), setid.c_str());
        continue;
      }
      else {
        // create a new empty style node named stylename
        pStyle = new LDom2Node();
        pStyle->setTagName("style");
        pStySet->putData(makeStyleKey(stylename), pStyle);
        MB_DPRINTLN("New style %s is created in sheet %s",
                    stylename.c_str(), setid.c_str());
      }
    }

    pNode = findStyleNodeByName(pStyle, prop_names, bCreate);
    if (pNode==NULL) {
      // the style pStyle does not contain style definition for "prop_names"
      // --> search next style set
      continue;
    }

    // style definition node for "stylename,prop_names" is found
    //MB_DPRINTLN("StyleNode for %s:%s is found in stylesheet %s",
    //stylename.c_str(), prop_names.c_str(), pStySet->getName().c_str());
    return pNode;
  }


  // fall-back to the global context search
  if (ctxt!=qlib::invalid_uid)
    return getStyleNode2(aSetID, stylename, prop_names, qlib::invalid_uid, bCreate);

  // not found!!
  return NULL;
}

/// Find style's node from the style root node pSty
LDom2Node *StyleMgr::findStyleNodeByName(LDom2Node *pSty, const LString &keyname, bool bCreate)
{
  LString subkey;
  LString trailkey;

  splitKeyName(keyname, subkey, trailkey);
  
  // MB_DPRINTLN("find %s -> %s, %s", keyname.c_str(), subkey.c_str(), trailkey.c_str());

  LDom2Node *pChNode = pSty->findChild(subkey);
  if (pChNode==NULL) {
    // style node is not found.
    if (!bCreate) {
      //MB_DPRINTLN("subkey %s is not found.", subkey.c_str());
      return NULL;
    }
    else {
      pChNode = new LDom2Node();
      pChNode->setTagName(subkey);
      pSty->appendChild(pChNode);
      MB_DPRINTLN("New Style Node created for %s", pChNode->getTagName().c_str());
    }
  }
  
  if (trailkey.isEmpty())
    // pChNode is style (leaf) node for keyname.
    return pChNode;

  // recursively search in the child nodes
  return findStyleNodeByName(pChNode, trailkey, bCreate);
}

//////////
// value

LString StyleMgr::getStyleValue(qlib::uid_t ctxt,
                                const LString &setid,
                                const LString &dotname)
{
  LString style_name;
  LString prop_names;
  if (!splitKeyName(dotname, style_name, prop_names)) {
    LString msg = LString::format("StyleMgr.getStyleNode2> invalid key name: %s",dotname.c_str());
    MB_THROW(qlib::RuntimeException, msg);
    return LString();
  }

  LDom2Node *pNode = getStyleNode2(setid, style_name, prop_names, ctxt, false);
  if (pNode==NULL)
    return LString();
  return pNode->getValue();
}
    
void StyleMgr::setStyleValue(qlib::uid_t ctxt,
                             const LString &setid,
                             const LString &dotname,
                             const LString &value)
{
  LString style_name;
  LString prop_names;
  if (!splitKeyName(dotname, style_name, prop_names)) {
    LString msg = LString::format("StyleMgr.getStyleNode2> invalid key name: %s",dotname.c_str());
    MB_THROW(qlib::RuntimeException, msg);
    return;
  }

  LDom2Node *pNode = getStyleNode2(setid, style_name, prop_names, ctxt, true);
  if (pNode==NULL) {
    MB_ASSERT(false);
    return;
  }
  pNode->setValue(value);

  // set to the pending event list
  m_pendEvts.insert(PendEventSet::value_type(ctxt, style_name));
}


//////////////////////////////////////////////////////////////////////////////
// Style manupilation methods for UI

StyleSet *StyleMgr::createStyleSet(const LString &id, qlib::uid_t ctxt)
{
  if (hasStyleSet(id, ctxt)) return NULL;
  
  StyleList *pSL = getCreateStyleList(ctxt);
  MB_ASSERT(pSL!=NULL);

  StyleSet *pSet = new StyleSet;
  pSet->setContextID(ctxt);
  pSet->setSource("");
  pSet->setName(id);
  pSL->push_front(pSet);

  return pSet;
}

bool StyleMgr::hasStyleSet(const LString &id, qlib::uid_t ctxt)
{
  StyleList *pSL = getCreateStyleList(ctxt);
  MB_ASSERT(pSL!=NULL);

  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    if (pSet->getName().equals(id))
      return true;
  }

  return false;
}
    
void StyleMgr::saveStyleSetToFile(const LString &setid, qlib::uid_t ctxt, const LString &path)
{
  StyleList *pSL = getCreateStyleList(ctxt);
  MB_ASSERT(pSL!=NULL);

  StyleSet *pStyleSet = NULL;
  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    if (pSet->getName().equals(setid)) {
      pStyleSet = pSet;
      break;
    }
  }
  if (pStyleSet==NULL) return;

  qlib::FileOutStream fos;
  try {
    fos.open(path);
  }
  catch (qlib::LException &e) {
    LOG_DPRINT("SaveStyle> cannot open file %s\n",path.c_str());
    LOG_DPRINT("SaveStyle>   (reason: %s)\n", e.getMsg().c_str());
    return;
  }
    
  try {
    qlib::LDom2OutStream oos(fos);
    qlib::LDom2Tree tree("styles");
    tree.top()->appendStrAttr("id", setid);

    StyleSet::data_iterator iter = pStyleSet->dataBegin();
    StyleSet::data_iterator eiter = pStyleSet->dataEnd();
    for (; iter!=eiter; ++iter) {
      LString style_id = iter->first.c_str();
      style_id = style_id.substr(0, style_id.length()-6);
      MB_DPRINTLN("write style node: %s.%s", setid.c_str(), style_id.c_str());
      LDom2Node *pNode = new LDom2Node(*iter->second);
      pNode->appendStrAttr("id", style_id);
      tree.top()->appendChild(pNode);
    }

    oos.write(&tree);
    oos.close();
    fos.close();

    //tree.detach();
    //delete pIdNode;
  }
  catch (qlib::LException &e) {
    LOG_DPRINT("LoadStyle> cannot read file %s\n",path.c_str());
    LOG_DPRINT("LoadStyle>   (reason: %s)\n", e.getMsg().c_str());
    return;
  }

  return;
}

void StyleMgr::firePendingEvents()
{
  BOOST_FOREACH (const PendEventSet::value_type &elem, m_pendEvts) {
    fireEventImpl(elem.first, elem.second);
  }

  clearPendingEvents();
}

void StyleMgr::fireEventImpl(qlib::uid_t uid, const LString &setname)
{
  if (m_pLsnrs->isEmpty())
    return;
  
  int i, nsize = m_pLsnrs->getSize();
  std::vector<StyleEventListener *> cblist(nsize);
    
  if (m_pLsnrs->isLocked()) {
    MB_ASSERT(false);
    return;
  }

  {
    // make copy of listener list
    qlib::EventCastLocker lock(m_pLsnrs);
    StyleEventCaster::const_iterator iter = m_pLsnrs->begin();
    StyleEventCaster::const_iterator iend = m_pLsnrs->end();
    for (i=0 ; iter!=iend; iter++, i++)
      cblist[i] = iter->second;
  }

  StyleEvent ev;
  
  for (i=0; i<nsize; ++i) {
    StyleEventListener *pLsnr = cblist[i];
    qlib::uid_t nCtxtID = pLsnr->getStyleCtxtID();

    if (uid!=0 && uid!=nCtxtID)
      continue;

    if (!setname.isEmpty()) {
      StyleSheet *pSheet = pLsnr->getStyleSheet();
      if (!pSheet->contains(setname))
        continue;
    }

    pLsnr->styleChanged(ev);
  }
}

void StyleMgr::clearPendingEvents()
{
  m_pendEvts.clear();
}

StyleEvent::~StyleEvent()
{
}

qlib::LCloneableObject *StyleEvent::clone() const
{
  return new StyleEvent(*this);
}

