// -*-Mode: C++;-*-
//
// Style/color database
//

#ifndef QSYS_STYLE_MGR_HPP_INCLUDED
#define QSYS_STYLE_MGR_HPP_INCLUDED

#include <qsys/qsys.hpp>

#include <qlib/SingletonBase.hpp>
#include <qlib/LScrObjects.hpp>
#include <qlib/mcutils.hpp>
#include <qlib/MapTable.hpp>
#include <qlib/EventCaster.hpp>

#include <gfx/gfx.hpp>
#include <gfx/Material.hpp>

#include "StyleSupports.hpp"

#define STYLEMGR_DB_DELIM ">"

namespace qlib {
  class PrintStream;
  class InStream;
  class OutStream;
  class LDom2Node;
}

namespace qsys {

  class StyleSet;
  class StyleList;
  class StyleEvent;
  class StyleEventCaster;

  using qlib::LString;
  using qlib::LDom2Node;
  using gfx::ColorPtr;
  using gfx::Material;

  ///
  /// Style database (singleton, scriptable)
  ///
  class QSYS_API StyleMgr : public qlib::LSingletonScrObject,
  public qlib::SingletonBase<StyleMgr>
  {
    MC_SCRIPTABLE;

  private:
    typedef qlib::SingletonBase<StyleMgr> super_t;
    
    typedef std::map<qlib::uid_t, StyleList *> data2_t;

    /// Mapping from Scene ID to StyleList
    data2_t m_data2;

    /// Current context (==scene) ID
    // qlib::uid_t m_nCurCtxtID;
    std::list<qlib::uid_t> m_curCtxtStack;

    /// Global context
    StyleList *m_pGlob;

    ////////////////////////////////////////////
    
    LString getStrImpl(const LString &key, qlib::uid_t nScopeID);

    ////////////////////////////////////////////
  public:
    
    StyleMgr();
    virtual ~StyleMgr();
    
    //////////////////////////////
    // context management

    /// set current context ID (usually context ID is scene ID)
    void setContextID(qlib::uid_t nid) {
      *(m_curCtxtStack.begin()) = nid;
      // m_nCurCtxtID = nid;
    }

    void pushContextID(qlib::uid_t nid) {
      m_curCtxtStack.push_front(nid);
    }

    void popContextID() {
      m_curCtxtStack.pop_front();
    }

    /// get current context ID
    qlib::uid_t getContextID() const {
      return *(m_curCtxtStack.begin());
      //return m_nCurCtxtID;
    }

    /// destroy all stylesheets related to context ID nid (usually context ID is scene ID)
    void destroyContext(qlib::uid_t nid);

    //////////////////////////////

    /// Get style list by context ID.
    /// If style list doesn't exist, new style list will be created
    StyleList *getCreateStyleList(qlib::uid_t nid);

    StyleSet *getCreateStyleSet(const LString &path);

    //////////////////////////////
    // basic access operations

    ColorPtr getColor(const LString &key);
    ColorPtr getColor(const LString &key, qlib::uid_t nScopeID);

    //bool putColor(const LString &key, const LString &value);

    Material *getMaterial(const LString &mat_id, qlib::uid_t nScopeID);

    /// Get renderer-type-dependent string material definition
    LString getMaterial(const LString &mat_id, const LString &rend_type);

    /// Get material definition for internal (OpenGL) renderer
    ///  (nType is type enum defined in Material.hpp)
    double getMaterial(const LString &mat_id, int nType);

    LString getConfig(const LString &key, const LString &rend_type);

    LString getStrData(const LString &cat, const LString &key, qlib::uid_t nScopeID);

    ////////////////////////////////////////////////////////////
    // Style manipulations (impl is in StyleMgrStyleImpl.cpp)
    
    static inline LString makeStyleKey(const LString &arg)
    {
      return arg + STYLEMGR_DB_DELIM + "style";
    }

    StyleSet *createStyleSet(const LString &id, qlib::uid_t ctxt);
    
    bool createStyleSetScr(const LString &id, qlib::uid_t ctxt) {
      return (createStyleSet(id, ctxt) != NULL);
    }
    
    bool hasStyleSet(const LString &id, qlib::uid_t ctxt);

    /// Get style node by name in dot notation
    LDom2Node *getStyleNode2(const LString &stylename,
                             const LString &prop_names,
                             qlib::uid_t ctxt);

    LDom2Node *getStyleNode2(const LString &set_id,
                             const LString &stylename,
                             const LString &prop_names,
                             qlib::uid_t ctxt,
                             bool bCreate);

    LString getStyleValue(qlib::uid_t ctxt, const LString &setid, const LString &dotname);
    
    void setStyleValue(qlib::uid_t ctxt, const LString &setid, const LString &dotname,
                       const LString &value);

    void saveStyleSetToFile(const LString &id, qlib::uid_t ctxt, const LString &fname);

  private:
    /// Search style nodes recursively to get a node with name, keyname
    LDom2Node *findStyleNodeByName(LDom2Node *pSty, const LString &keyname, bool bCreate);

    void fireEventImpl(qlib::uid_t uid, const LString &setname);

  private:
    ////////////////////////////////////////////////////////////
    // Style event management
    StyleEventCaster *m_pLsnrs;

    typedef std::set<std::pair<qlib::uid_t, LString> > PendEventSet;

    // pending events
    PendEventSet m_pendEvts;

  public:

    void addListener(StyleEventListener *pLsnr);

    void removeListener(StyleEventListener *pLsnr);

    void fireEvent(StyleEvent &evt);

    void clearPendingEvents();
    void firePendingEvents();

    ////////////////////////////////////////////////////////////
    // script specific interface

    bool scrLoadFile(const LString &path, qlib::uid_t scope);

    // get string definitions (for GUI)
    LString getStrDataDefsJSON(const LString &cat, qlib::uid_t nScopeID);

    // get color definitions (for GUI)
    LString getColorDefsJSON(qlib::uid_t nScopeID);

    // compile and create new color object (for GUI)
    ColorPtr compileColor(const LString &rep, qlib::uid_t nScopeID);
    
    LString getStyleNamesJSON(qlib::uid_t nSceneID);

    //////////
    // Initializer/finalizer

    static bool init();
    
    static void fini();
  };

  /////////////////////////

  class StyleEventCaster
       : public qlib::LEventCaster<StyleEvent, StyleEventListener>
  {
    virtual void execute(StyleEvent &ev, StyleEventListener *p)
    {
      p->styleChanged(ev);
    }
  };

}

#endif
