// -*-Mode: C++;-*-
//
// Scene exporter
//
// $Id: SceneXMLReader.cpp,v 1.14 2011/04/10 10:48:09 rishitani Exp $

#include <common.h>

#include "SceneXMLReader.hpp"
#include "StreamManager.hpp"
#include "SceneEvent.hpp"
#include "ObjReader.hpp"
#include "style/AutoStyleCtxt.hpp"

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
namespace fs = boost::filesystem;

#include <qlib/LDOM2Stream.hpp>
#include <qlib/FileStream.hpp>

using namespace qsys;

SceneXMLReader::SceneXMLReader()
{
}

SceneXMLReader::~SceneXMLReader()
{
}

int SceneXMLReader::getCatID() const
{
  return IOH_CAT_SCEREADER;
}

/// attach to and lock the target object
void SceneXMLReader::attach(ScenePtr pScene)
{
  // TO DO: lock scene
  m_pClient = pScene;
}
    
/// detach from the target object
ScenePtr SceneXMLReader::detach()
{
  // TO DO: unlock scene
  ScenePtr p = m_pClient;
  m_pClient = ScenePtr();
  return p;
}

/// Get name of the writer
const char *SceneXMLReader::getName() const
{
  //return "qsc_xmlreader";
  return "qsc_xml";
}

/// Get file-type description
const char *SceneXMLReader::getTypeDescr() const
{
  return "CueMol Scene (*.qsc)";
}

/// Get file extension
const char *SceneXMLReader::getFileExt() const
{
  return "*.qsc";
}

//////////////////////////////////////////////////

void SceneXMLReader::read()
{
  // convert to absolute path name (based on the cwd)
  fs::path curpath = fs::current_path();
  LString localfile = qlib::makeAbsolutePath(getPath(), curpath.file_string());
  /*
  fs::path inpath(getPath());
  if (!inpath.is_complete()) {
    inpath = fs::complete(inpath, fs::current_path());
  }
  LString localfile(inpath.file_string());
   */

  //gfx::StyleMgr *pSM = gfx::StyleMgr::getInstance();
  //pSM->pushContextID(m_pClient->getUID());
  // Enter the context
  AutoStyleCtxt style_ctxt(m_pClient->getUID());
  
  //
  // Setup streams
  //
  qlib::FileInStream fis;
  fis.open(localfile);
  qlib::LDom2InStream ois(fis);

  //
  // Construct data structure from the file
  //
  qlib::LDom2Tree tree;
  ois.read(tree);
  // tree.top()->dump();

  m_pClient->setSource(localfile);
  m_pClient->setSourceType(getName());

  // perform deserialization to the client scene (m_pClient)
  tree.deserialize(&m_pClient);
  // fs::path base_path(m_pClient->getBasePath());

  //
  // Gather the source info & load external data
  //
  Scene::ObjIter oi = m_pClient->beginObj();
  Scene::ObjIter oend = m_pClient->endObj();

  std::map<LString, ObjectPtr> chunkmap;
  std::map<LString, LString> typemap;
  
  for (; oi!=oend; ++oi) {
    ObjectPtr robj = oi->second;
    LString src = robj->getSource();
    LString altsrc = robj->getAltSource();
    LString srctype = robj->getSourceType();
    if (src.isEmpty())
      continue;
    if (srctype.isEmpty()) {
      // ERROR!! (TO DO: handling)
      LOG_DPRINTLN("SceneXML> src %s: srctype is not defined. (ignored)", src.c_str());
      continue;
    }
    if (src.startsWith("datachunk:") && src.length()==15) {
      // Prepare for reading from data chunk of the stream
      MB_DPRINTLN("Data chunk found %s, %s ",
                   src.c_str(), srctype.c_str());
      chunkmap.insert(std::pair<LString, ObjectPtr>(src, robj));
      typemap.insert(std::pair<LString, LString>(src, srctype));
      continue;
    }

    // Load from external source

    // perform conversion & check source path
    LString abs_path = m_pClient->resolveBasePath(src);
    if (!qlib::isFileReadable(abs_path)) {
      // check the alternative source info
      if (altsrc.isEmpty()) {
        LString msg = LString::format("Fatal error, cannot open file: \"%s\"",
                                      abs_path.c_str());
        LOG_DPRINTLN("SceneXML> %s", msg.c_str());
        MB_THROW(qlib::IOException, msg);
        continue;
      }
      else {
        abs_path = m_pClient->resolveBasePath(altsrc);
        if (!qlib::isFileReadable(abs_path)) {
          LString msg = LString::format("Fatal error, cannot open file: \"%s\"",
                                        abs_path.c_str());
          LOG_DPRINTLN("SceneXML> %s", msg.c_str());
          MB_THROW(qlib::IOException, msg);
          continue;
        }
      }
    }

    if (!readObjectFrom(robj, abs_path)) {
      LString msg = LString::format("Fatal error: load file src=\"%s\", srctype=%s failed!!",
                                    src.c_str(), srctype.c_str());
      LOG_DPRINTLN("SceneXML> %s", msg.c_str());
      MB_THROW(qlib::IOException, msg);
      continue;
    }
  }
  
  //
  // Read embedded object from data chunks
  //
  for (;;) {
    LString chunkid = ois.getNextDataChunkID();
    if (chunkid.isEmpty())
      break;
    ObjectPtr robj;
    {
      std::map<LString, ObjectPtr>::const_iterator ci = chunkmap.find(chunkid);
      if (ci==chunkmap.end()) {
        LString msg = LString::format("Chunk (ID: \"%s\") is not found", chunkid.c_str());
        LOG_DPRINTLN("SceneXMLRead> %s", msg.c_str());
        MB_THROW(qlib::IOException, msg);
        break;
      }
      robj = ci->second;
    }
    LString srctype;
    {
      std::map<LString, LString>::const_iterator ci = typemap.find(chunkid);
      if (ci==typemap.end()) {
        LString msg = LString::format("Chunk type (ID: \"%s\") is not found", chunkid.c_str());
        LOG_DPRINTLN("SceneXMLRead> %s", msg.c_str());
        MB_THROW(qlib::IOException, msg);
        break;
      }
      srctype = ci->second;
    }

    qlib::InStream *pin = ois.getNextChunkStream();
    
    if (!readObjectFrom(robj, *pin)) {
      LString msg = LString::format("Fatal error in read embedded object src %s: srctype %s", chunkid.c_str(), srctype.c_str());
      LOG_DPRINTLN("SceneXMLRead> %s", msg.c_str());
      MB_THROW(qlib::IOException, msg);
    }
    ois.closeChunkStream(pin);
  }

  ois.close();

  m_pClient->setUpdateFlag();

  // pSM->popContextID();

  //////////
  // fire the scene-loaded event
  {
    SceneEvent ev;
    ev.setTarget(m_pClient->getUID());
    ev.setType(SceneEvent::SCE_SCENE_ONLOADED);
    m_pClient->fireSceneEvent(ev);
  }
}

bool SceneXMLReader::readObjectFrom(ObjectPtr obj, const LString &url)
{
  // MB_DPRINTLN("StreamManager.readObjectFrom(%s,%s) called", url.c_str(), ftype.c_str());

  // Construct a stream from the requested url
  qlib::FileInStream fis;
  fis.open(url);

  bool res = readObjectFrom(obj, fis);
  fis.close();
  if (!res)
    return false;

  obj->setSource(url);
  // obj->setSourceType(ftype);
  return true;
}

bool SceneXMLReader::readObjectFrom(ObjectPtr obj, qlib::InStream &ins)
{
  const LString &ftype = obj->getSourceType();
  StreamManager *pSM = StreamManager::getInstance();
  
  // Create the requested reader obj
  ObjReader *pRdr = pSM->createReaderPtr(ftype);
  if (pRdr==NULL) {
    LString msg = LString::format("ObjReader for type \"%s\" is not found", ftype.c_str());
    LOG_DPRINTLN("SceneXMLRead> %s", msg.c_str());
    MB_THROW(qlib::IOException, msg);
    return false;
  }
    
  qlib::LDom2Node *pROpts = obj->getReaderOpts();
  if (pROpts!=NULL) {
    pRdr->readFrom2(pROpts);
  }

  bool res;
  try {
    pRdr->attach(obj);
    res = pRdr->read(ins);
    pRdr->detach();
  }
  catch (...) {
    delete pRdr;
    throw;
  }

  // END
  delete pRdr;

  return true;
}

//////////////////////////////////////////////////

SceneXMLWriter::SceneXMLWriter()
{
  //m_bForceEmbedAll = true;
  m_bForceEmbedAll = false;
}

SceneXMLWriter::~SceneXMLWriter()
{
}

int SceneXMLWriter::getCatID() const
{
  return IOH_CAT_SCEWRITER;
}

/// attach to and lock the target object
void SceneXMLWriter::attach(ScenePtr pScene)
{
  // TO DO: lock scene
  m_pClient = pScene;
}
    
/// detach from the target object
ScenePtr SceneXMLWriter::detach()
{
  // TO DO: unlock scene
  ScenePtr p = m_pClient;
  m_pClient = ScenePtr();
  return p;
}

/// Get name of the writer
const char *SceneXMLWriter::getName() const
{
  //return "qsc_xmlwriter";
  return "qsc_xml";
}

/// Get file-type description
const char *SceneXMLWriter::getTypeDescr() const
{
  return "CueMol Scene (*.qsc)";
}

/// Get file extension
const char *SceneXMLWriter::getFileExt() const
{
  return "*.qsc";
}

void SceneXMLWriter::write()
{
  LString src, srctype;

  // Update the SourcePath/BasePath
  {
    // Convert to absolute path name (based on the cwd)
    fs::path curpath = fs::current_path();
    LString localfile = qlib::makeAbsolutePath(getPath(), curpath.file_string());
    /*
    fs::path outpath(getPath());
    if (!outpath.is_complete()) {
      outpath = fs::complete(outpath, fs::current_path());
    }
     */

    m_pClient->setSource(localfile);
    m_pClient->setSourceType(getName());
  }
  
  //
  // Setup streams
  //
  qlib::FileOutStream fos;
  fos.open(getPath());

  qlib::LDom2OutStream oos(fos);

  Scene::ObjIter oiter = m_pClient->beginObj();
  Scene::ObjIter oiter_end = m_pClient->endObj();
  for (; oiter!=oiter_end; ++oiter) {
    ObjectPtr obj = oiter->second;
    bool bEmbed = m_bForceEmbedAll;

    // Determine whether the object should be embedded or not.
    if (!bEmbed) {
      src = obj->getSource();
      srctype = obj->getSourceType();
      if (src.isEmpty() || srctype.isEmpty()) {
        // invalid source info --> embed it
        bEmbed = true;
      }
      else if (src.startsWith("datachunk:")) {
        // already embedded datasource will be anyway embedded.
        bEmbed = true;
      }
      else if (obj->getModifiedFlag()) {
        // modified since the obj loaded from file
        bEmbed = true;
      }
    }

    if (bEmbed) {
      qlib::LDataChunk *pDC = dynamic_cast<qlib::LDataChunk *>(obj.get());
      if (pDC!=NULL) {
        src = oos.prepareDataChunk(obj->getUID());
        srctype = pDC->getDataChunkReaderName();
        obj->setSource(src);
        obj->setSourceType(srctype);
      }
    }

    // ATTN: Path conversion is performed in object classes
    /*else {
      // convert to the relative path against BasePath
      src = obj->getSource();
      LString relpath = qlib::makeRelativePath(src, m_pClient->getBasePath());
      obj->setAltSource(relpath);
    }*/
  }

  // build LDOM2 tree
  qlib::LDom2Tree tree("scene");
  tree.serialize(m_pClient.get(), false);

  // write both tree and datachunks
  oos.write(&tree);
  oos.close();

  // End of writing
  fos.close();

}

