// -*-Mode: C++;-*-
//
//  Povray display context implementation
//
//  $Id: PovDisplayContext.cpp,v 1.17 2011/04/11 11:37:29 rishitani Exp $

#include <common.h>

#include "PovDisplayContext.hpp"
#include "PovIntData.hpp"
#include "SceneManager.hpp"
#include <qlib/PrintStream.hpp>
#include <qlib/Utils.hpp>
#include <gfx/SolidColor.hpp>
#include "style/StyleMgr.hpp"

using namespace qsys;

using qlib::PrintStream;
using qlib::Matrix4D;
using qlib::Matrix3D;

PovDisplayContext::PovDisplayContext()
     : m_pIntData(NULL)
{
  // m_fUseTexBlend = false;
  m_fPerspective = true;
  m_bUnitary = true;
  m_pPovOut = NULL;
  m_pIncOut = NULL;
  m_nDetail = 3;

  m_dUniTol = 1e-3;
}

PovDisplayContext::~PovDisplayContext()
{
}

////////////////////////////////////////////////////////////
// generic implementation

void PovDisplayContext::vertex(double x, double y, double z)
{
  vertex(Vector4D(x,y,z));
}

void PovDisplayContext::vertex(const Vector4D &aV)
{
  Vector4D v(aV);
  xform_vec(v);

  switch (m_nDrawMode) {
  default:
  case POV_NONE:
    MB_DPRINTLN("POVWriter> vertex command ignored.");
    break;

  case POV_POLYGON:
    MB_DPRINTLN("POVWriter> polygon is not supported (vertex command ignored.)");
    break;

  case POV_LINES:
    if (!m_fPrevPosValid) {
      m_prevPos = v;
      m_fPrevPosValid = true;
      break;
    }
    else {
      drawLine(v, m_prevPos);
      m_fPrevPosValid = false;
    }
    break;

    //////////////////////////////////////////////////////
  case POV_LINESTRIP:
    if (!m_fPrevPosValid) {
      m_prevPos = v;
      m_fPrevPosValid = true;
      break;
    }
    else {
      drawLine(v, m_prevPos);
      m_prevPos = v;
    }
    break;

    //////////////////////////////////////////////////////
  case POV_TRIGS:
    m_pIntData->meshVertex(v, m_norm, m_pColor);
    break;

    //////////////////////////////////////////////////////
  case POV_TRIGSTRIP:
    m_pIntData->meshVertex(v, m_norm, m_pColor);
    break;

    //////////////////////////////////////////////////////
  case POV_TRIGFAN:
    m_pIntData->meshVertex(v, m_norm, m_pColor);
    break;
  }

}

void PovDisplayContext::normal(double x, double y, double z)
{
  normal(Vector4D(x,y,z));
}

void PovDisplayContext::normal(const Vector4D &av)
{
  Vector4D v(av);
  xform_norm(v);

  const double len = v.length();
  if (len<F_EPS4) {
    LOG_DPRINTLN("PovDisp> Normal vector <%f,%f,%f> is too small.", v.x(), v.y(), v.z());
    m_norm = Vector4D(1.0,0.0,0.0);
    return;
  }
  m_norm = v.scale(1.0/len);
}

void PovDisplayContext::color(const gfx::ColorPtr &c)
{
  m_pColor = c;
}

////////////////////////////////////////////////

void PovDisplayContext::pushMatrix()
{
  if (m_matstack.size()<=0) {
    Matrix4D m;
    m_matstack.push_front(m);
    return;
  }
  const Matrix4D &top = m_matstack.front();
  m_matstack.push_front(top);
}

void PovDisplayContext::popMatrix()
{
  if (m_matstack.size()<=1) {
    LString msg("POVWriter> FATAL ERROR: cannot popMatrix()!!");
    LOG_DPRINTLN(msg);
    MB_THROW(qlib::RuntimeException, msg);
    return;
  }
  m_matstack.pop_front();
}

void PovDisplayContext::checkUnitary()
{
  const Matrix4D &top = m_matstack.front();
  Matrix3D t = top.getMatrix3D();
  Matrix3D t2 = t;
  t2.transpose();
  t2.matprod(t);
  if (t2.isIdent())
    m_bUnitary = true;
  else
    m_bUnitary = false;
}  

void PovDisplayContext::multMatrix(const Matrix4D &mat)
{
  Matrix4D top = m_matstack.front();
  top.matprod(mat);
  m_matstack.front() = top;

  // check unitarity
  checkUnitary();
}

void PovDisplayContext::loadMatrix(const Matrix4D &mat)
{
  m_matstack.front() = mat;

  // check unitarity
  checkUnitary();
}

void PovDisplayContext::setLineWidth(double lw)
{
  m_linew = lw;
}

void PovDisplayContext::setPointSize(double size)
{
}

void PovDisplayContext::setLineStipple(unsigned short pattern)
{
}

void PovDisplayContext::setPolygonMode(int id)
{
}

void PovDisplayContext::startPoints()
{
  m_nDrawMode = POV_POINTS;
}

void PovDisplayContext::startPolygon()
{
  m_nDrawMode = POV_POLYGON;
}

void PovDisplayContext::startLines()
{
  if (m_nDrawMode!=POV_NONE) {
    MB_THROW(qlib::RuntimeException, "PovDisplayContext: Unexpected condition");
    return;
  }
  m_nDrawMode = POV_LINES;
  //startUnion();
}

void PovDisplayContext::startLineStrip()
{
  if (m_nDrawMode!=POV_NONE) {
    MB_THROW(qlib::RuntimeException, "PovDisplayContext: Unexpected condition");
    return;
  }
  m_nDrawMode = POV_LINESTRIP;
  //startUnion();
}

void PovDisplayContext::startTriangles()
{
  if (m_nDrawMode!=POV_NONE) {
    MB_THROW(qlib::RuntimeException, "PovDisplayContext: Unexpected condition");
    return;
  }
  m_nDrawMode = POV_TRIGS;
  m_pIntData->meshStart();
}

void PovDisplayContext::startTriangleStrip()
{
  if (m_nDrawMode!=POV_NONE) {
    MB_THROW(qlib::RuntimeException, "PovDisplayContext: Unexpected condition");
    return;
  }
  m_nDrawMode = POV_TRIGSTRIP;
  m_pIntData->meshStart();
}

void PovDisplayContext::startTriangleFan()
{
  m_nDrawMode = POV_TRIGFAN;
  m_pIntData->meshStart();
}

void PovDisplayContext::startQuadStrip()
{
  m_nDrawMode = POV_QUADS;
}

void PovDisplayContext::startQuads()
{
  m_nDrawMode = POV_QUADSTRIP;
}

void PovDisplayContext::end()
{
  switch (m_nDrawMode) {
  case POV_LINES:
  case POV_LINESTRIP:
    m_fPrevPosValid = false;
    //endUnion();
    break;

  case POV_TRIGS:
    m_pIntData->meshEndTrigs();
    break;

  case POV_TRIGFAN:
    m_pIntData->meshEndFan();
    break;

  case POV_TRIGSTRIP:
    m_pIntData->meshEndTrigStrip();
    break;
  }
  m_nDrawMode = POV_NONE;
}

void PovDisplayContext::sphere(double r, const Vector4D &vec)
{
  if (m_nDrawMode!=POV_NONE) {
    MB_THROW(qlib::RuntimeException, "PovDisplayContext: Unexpected condition");
    return;
  }

  Vector4D v(vec);
  xform_vec(v);
  m_pIntData->sphere(v, r, m_nDetail);

}

void PovDisplayContext::sphere()
{
  if (m_nDrawMode!=POV_NONE) {
    MB_THROW(qlib::RuntimeException, "PovDisplayContext: Unexpected condition");
    return;
  }

  Vector4D v(0, 0, 0);
  xform_vec(v);
  
  const Matrix4D &mtop = m_matstack.front();
  if (mtop.isIdentAffine(F_EPS4))
    m_pIntData->sphere(v, 1.0, m_nDetail);
  else {
    LOG_DPRINTLN("ERROR, sphere(): unsupported operation!!");
    m_pIntData->sphere(v, 1.0, m_nDetail);
  }

}

void PovDisplayContext::cone(double r1, double r2,
                             const Vector4D &pos1, const Vector4D &pos2,
                             bool bCap)
{
  if (m_nDrawMode!=POV_NONE) {
    MB_THROW(qlib::RuntimeException, "PovDisplayContext: Unexpected condition");
    return;
  }

  if (pos1.equals(pos2))
    return;

  const Matrix4D &xm = m_matstack.front();
  Matrix3D xm3 = xm.getMatrix3D(), test;
  bool bUnitary = true;
  if (!xm3.isIdent()) {
    test = xm3.transpose() * xm3;
    bUnitary = test.isIdent(m_dUniTol);
  }
  
  //if (m_bUnitary) {
  if (bUnitary) {
    Vector4D p1 = pos1;
    Vector4D p2 = pos2;
    xform_vec(p1);
    xform_vec(p2);
    m_pIntData->cylinder(p1, p2, r1, r2, m_nDetail, NULL);
  }
  else {
    m_pIntData->cylinder(pos1, pos2, r1, r2, m_nDetail, &xm);
  }

}

void PovDisplayContext::drawMesh(const gfx::Mesh &mesh)
{
  m_pIntData->mesh(m_matstack.front(), mesh);
}

/// Draw a single line segment from v1 to v2 to the output
/// v1 and v2 should be transformed by matrix stack
void PovDisplayContext::drawLine(const Vector4D &v1, const Vector4D &v2)
{
  if (v1.equals(v2))
    return;
  m_pIntData->line(v1, v2, m_linew);
}

void PovDisplayContext::setDetail(int n)
{
  m_nDetail = n;
}

int PovDisplayContext::getDetail() const
{
  return m_nDetail;
}

void PovDisplayContext::setLighting(bool f/*=true*/)
{
}

//////////////////////////////

bool PovDisplayContext::setCurrent()
{
  return true;
}

bool PovDisplayContext::isCurrent() const
{
  return true;
}

void PovDisplayContext::startSection(const LString &name)
{
  // start of rendering
  if (m_pIntData!=NULL) {
    MB_THROW(qlib::RuntimeException, "Unexpected condition");
    return ;
  }
  m_pIntData = new PovIntData(this);
  if (m_bUseClipZ)
    m_pIntData->m_dClipZ = m_dSlabDepth/2.0;
  else
    m_pIntData->m_dClipZ = -1.0;
  m_pIntData->start(m_pPovOut, m_pIncOut, name);
}

void PovDisplayContext::endSection()
{
  // end of rendering
  if (m_pIntData==NULL) {
    MB_THROW(qlib::RuntimeException, "Unexpected condition");
    return ;
  }
  m_pIntData->end();
  delete m_pIntData;
  m_pIntData = NULL;
}

//////////////////////////////

void PovDisplayContext::init(qlib::OutStream *pPovOut, qlib::OutStream *pIncOut)
{
  m_matstack.erase(m_matstack.begin(), m_matstack.end());

  pushMatrix();
  loadIdent();
  m_linew = 1.0;
  m_pColor = gfx::SolidColor::createRGB(1,1,1);
  m_nDrawMode = POV_NONE;
  m_fPrevPosValid = false;
  m_nTriIndex = 0;
  m_dZoom = 100;
  m_dViewDist = 100;
  m_dSlabDepth = 100;

  if (m_pIntData!=NULL)
    delete m_pIntData;
  m_pIntData = NULL;

  m_pPovOut = pPovOut;
  m_pIncOut = pIncOut;
}

void PovDisplayContext::startRender()
{
  writeHeader();
}

void PovDisplayContext::endRender()
{
  if (m_pPovOut!=NULL) {
    writeTailer();
    m_pPovOut->close();
  }
  if (m_pIncOut!=NULL) {
    m_pIncOut->close();
  }
}

void PovDisplayContext::writeHeader()
{
  MB_DPRINTLN("povwh: zoom=%f", m_dZoom);
  MB_DPRINTLN("povwh: dist=%f", m_dViewDist);

  double fovy = qlib::toDegree(::atan((m_dZoom/2.0)/m_dViewDist))*2.0;

  SceneManager *pmod = SceneManager::getInstance();
  LString ver = LString::format("Version %d.%d.%d.%d (build %s)",
                                pmod->getMajorVer(),pmod->getMinorVer(),
                                pmod->getRevision(),pmod->getBuildNo(),
                                pmod->getBuildID().c_str());

  PrintStream ps(*m_pPovOut);
  PrintStream ips(*m_pIncOut);

  Vector4D bgcolor;
  if (!m_bgcolor.isnull()) {
    bgcolor.x() = m_bgcolor->fr();
    bgcolor.y() = m_bgcolor->fg();
    bgcolor.z() = m_bgcolor->fb();
  }
  
  ps.println("/*");
  ps.formatln("  POV-Ray output from CueMol %s", ver.c_str());
  ps.format(" */\n");
  ps.format("\n");

  ps.format("#version 3.5;\n");

  StyleMgr *pSM = StyleMgr::getInstance();
  LString preamble = pSM->getConfig("preamble", "pov").trim(" \r\t\n");
  if (!preamble.isEmpty())
    ps.println(preamble);

  ps.format("\n");
  ps.format("background {color rgb <%f,%f,%f>}\n", bgcolor.x(), bgcolor.y(), bgcolor.z());
  ps.format("\n");
  ps.format("#declare _distance = %f;\n", m_dViewDist);

  ps.format("\n");
  ps.format("// _stereo ... 0:none, 1:for right eye, -1:for left eye\n");
  ps.format("// _perspective ... 0:orthogonal projection, 1:perspective projection\n");
  ps.format("// _iod ... inter-ocullar distance\n");
  ps.format("\n");
  ps.format("#ifndef (_perspective)\n");
  ps.format("  #declare _perspective = %d;\n", m_fPerspective);
  ps.format("#end\n");
  ps.format("#ifndef (_stereo)\n");
  ps.format("  #declare _stereo = 0;\n");
  ps.format("#end\n");
  ps.format("#ifndef (_iod)\n");
  ps.format("  #declare _iod = 0.03;\n");
  ps.format("#end\n");

  ps.format("\n");
  ps.format("camera {\n");
  ps.format(" #if (_perspective)\n");
  ps.format(" perspective\n");
  ps.format(" direction <0,0,-1>\n");
  ps.format(" up <0,1,0>\n");
  ps.format(" right <1,0,0>\n");
  ps.format(" angle %f\n", fovy);
  ps.format(" location <_stereo*_distance*_iod,0,_distance>\n");
  ps.format(" look_at <0,0,0>\n");
  ps.format(" #else\n");
  ps.format(" orthographic\n");
  ps.format(" direction <0,0,-1>\n");
  ps.format(" up <0, %f, 0>\n", m_dZoom);
  ps.format(" right <%f, 0, 0>\n", m_dZoom);
  ps.format(" location <_stereo*_distance*_iod,0,_distance>\n");
  ps.format(" look_at <0,0,0>\n");
  ps.format(" #end\n");
  ps.format("}\n");
  ps.format("\n");
  ps.format("light_source {<_stereo*_distance*_iod,0,_distance> color rgb 1}\n");
  ps.format("light_source {<-1,1,1>*10000 color rgb 0.5 shadowless}\n");
  ps.format("\n");

  ps.format("#ifndef (_no_fog)\n");
  ps.format("fog {\n");
  ps.format("  distance %f/5\n", m_dSlabDepth);
  ps.format("  color rgbf <%f,%f,%f,0>\n", bgcolor.x(), bgcolor.y(), bgcolor.z());
  ps.format("  fog_type 2\n");
  ps.format("  fog_offset 0\n");
  ps.format("  fog_alt 1.0e-10\n");
  ps.format("  up <0,0,1>\n");
  ps.format("}\n");
  ps.format("#end\n");
  ps.format("\n");

  ps.format("\n");
  ps.format("/////////////////////////////////////////////\n");

  ips.format("/*\n");
  ips.format("  POV-Ray output from CueMol (%s)\n", ver.c_str());
  ips.format(" */\n");
  ips.format("\n");
  ips.format("\n");

  ips.format("union {\n");
}

void PovDisplayContext::writeTailer()
{
  PrintStream ps(*m_pPovOut);
  PrintStream ips(*m_pIncOut);

  ps.format("\n");
  ps.format("//////////////////////////////////////////////\n");
  ps.format("\n");
  ps.format("#declare _scene = #include \"%s\"\n", m_incFileName.c_str());
  ps.format("\n");
  ps.format("object{\n");
  ps.format("  _scene\n");
  /*
  // clipping plane (slow!!)
  ps.format("#ifdef (_clipping_plane)\n");
  ps.format("bounded_by {\n");
  ps.format("  plane {z, %f}\n", m_dSlabDepth/2.0);
  ps.format("}\n");
  ps.format("clipped_by { bounded_by }\n");
  ps.format("#end\n");
   */
  ps.format("}\n");
  ps.format("\n");

  ips.println("} // union");
  ips.println("");
}

