
///////////////

namespace qsys {

  class PipeStreamImpl : public qlib::detail::IOImpl
  {
  private:
    std::deque<char> m_data;
    bool m_feof;

    boost::mutex m_mu;
    boost::condition m_cond;

  public:
    PipeStreamImpl() : m_feof(false) {}

    /** check if input is available. */
    virtual bool ready();
    
    /** read one byte */
    virtual int read();
    
    /** read into mem block */
    virtual int read(char *buf, int off, int len);
    
    /**
       Try to skip n bytes.
       @return the actual number of bytes skipped
    */
    virtual int skip(int n);
    
    /** close the stream */
    virtual void i_close();
    
    /** get source URI of this stream */
    virtual LString getSrcURI() const;

    ////////////////////
    
    /** write out mem block */
    virtual int write(const char *buf, int off, int len);
    
    /** write one byte */
    virtual void write(int b);
    
    /** flush output stream */
    virtual void flush();
    
    /** close the stream */
    virtual void o_close();
    
    /** get destination URI of this stream */
    virtual LString getDestURI() const;
    
  };

  class PipeInStream : public qlib::InStreamAdaptor
  {
  private:
    qlib::sp<PipeStreamImpl> m_pimpl;
    
  public:
    void setImpl(qlib::sp<PipeStreamImpl> pimpl) {
      m_pimpl = pimpl;
    }
    
    virtual qlib::InStream::impl_type getImpl() const {
      return m_pimpl;
    }
  };

  class IOThread
  {
  public:
    boost::thread *m_pthr;

    ObjReader *m_pRdr;
    ObjectPtr m_pObj;

    qlib::sp<PipeStreamImpl> m_pipeImpl;

    PipeInStream m_in;
    // PipeOutStream m_out;

    //////////

    IOThread() : m_pthr(NULL), m_pRdr(NULL) {}

    ~IOThread() {
      if (m_pthr) delete m_pthr;
      if (m_pRdr) delete m_pRdr;
    }

    //////////

    void run() {
      MB_DPRINTLN("*** Thread %p started ***", m_pthr);

      m_pipeImpl = qlib::sp<PipeStreamImpl>(new PipeStreamImpl);
      m_in.setImpl(m_pipeImpl);
      ObjectPtr pret = m_pRdr->load(m_in);
      m_pObj = pret;

      MB_DPRINTLN("*** Thread %p end ***", m_pthr);
    }

    void kick() {
      m_pthr = new boost::thread(boost::bind(&IOThread::run, this));
    }

    void supplyData(const char *pbuf, int nlen) {
      MB_DPRINTLN("supply data (%d)", nlen);
      m_pipeImpl->write(pbuf, 0, nlen);
    }

    void notifyEos() {
      MB_DPRINTLN("notify EOS");
      m_pipeImpl->o_close();
    }

    void waitTermination() {
      MB_DPRINTLN("wait termination ...");
      if (m_pthr)
	m_pthr->join();
      delete m_pthr;
      m_pthr = NULL;
      MB_DPRINTLN("wait termination OK.");
    }

  };


///////////////

/** check if input is available. */
bool PipeStreamImpl::ready()
{
  boost::mutex::scoped_lock lk(m_mu);
  if (m_data.size()>0)
    return true;
  // data buffer is empty
  if (m_feof)
    return false; // channel was closed

  // still connected
  return true; 
}
    
/** read one byte */
int PipeStreamImpl::read()
{
  boost::mutex::scoped_lock lk(m_mu);
  
  while (!m_feof && m_data.size()<=0)
    m_cond.wait(lk);

  if (m_feof && m_data.size()<=0)
    return -1;

  /*

    if (m_data.size()<=0) {
    MB_THROW(qlib::IOException, "");
    return 0;
    }

  */
  
  int rval = m_data.back();
  m_data.pop_back();

  MB_DPRINTLN("thread read 1 ok");
  return rval;
}

/** read into mem block */
int PipeStreamImpl::read(char *buf, int off, int len)
{
  boost::mutex::scoped_lock lk(m_mu);
  
  while (!m_feof && m_data.size()<=0)
    m_cond.wait(lk);

  if (m_feof && m_data.size()<=0)
    return -1;
  
  int i;
  for (i=0; i<len; ++i) {
    if (m_data.size()<=0)
      break;
    buf[off+i] = m_data.back();
    m_data.pop_back();
  }

  MB_DPRINTLN("thread read (%d) ok", i);
  return i;
}

/**
   Try to skip n bytes.
   @return the actual number of bytes skipped
*/
int PipeStreamImpl::skip(int len)
{
  boost::mutex::scoped_lock lk(m_mu);
  
  while (!m_feof && m_data.size()<=0)
    m_cond.wait(lk);

  if (m_feof && m_data.size()<=0)
    return -1;
  
  int i;
  for (i=0; i<len; ++i) {
    if (m_data.size()<=0)
      return i;
    read();
  }

  return i;
}

/** close the stream */
void PipeStreamImpl::i_close()
{
}

/** get source URI of this stream */
LString PipeStreamImpl::getSrcURI() const
{
  return LString();
}

////////////////////

/** write out mem block */
int PipeStreamImpl::write(const char *buf, int off, int len)
{
  boost::mutex::scoped_lock lk(m_mu);
  
  for (int i=0; i<len; ++i) {
    m_data.push_front(buf[i+off]);
  }
  
  m_cond.notify_all();
  return len;
}

/** write one byte */
void PipeStreamImpl::write(int b)
{
  boost::mutex::scoped_lock lk(m_mu);
  m_data.push_front((char)b);
  m_cond.notify_all();
}

/** flush output stream */
void PipeStreamImpl::flush()
{
}

/** close the stream */
void PipeStreamImpl::o_close()
{
  boost::mutex::scoped_lock lk(m_mu);
  m_feof = true;
  m_cond.notify_all();
}

/** get destination URI of this stream */
LString PipeStreamImpl::getDestURI() const
{
  return LString();
}

}
