//
// Vector class with N dimension value
//

#ifndef __QLIB_VECTOR_ND_HPP__
#define __QLIB_VECTOR_ND_HPP__

#include "qlib.hpp"
#include "Utils.hpp"

namespace qlib {

  template <int _N_DIM, typename _ValueType = double>
  class VectorND
  {
  public:
    static const int dimension = _N_DIM;
    static const int _N_ELEM = _N_DIM*1;

  private:
    _ValueType m_value[_N_ELEM];

  public:
    /////////////////
    // constructors

    /** default constructor */
    VectorND()
    {
      zero();
    }

    /** constructor without initialization */
    explicit
    VectorND(int, detail::no_init_tag)
    {
    }

    /** copy constructor */
    VectorND(const VectorND &arg)
    {
      for (int i=0; i<_N_ELEM; ++i)
	m_value[i] = arg.m_value[i];
    }


  public:

    // FORTRAN-type (1-base index) access methods
    inline _ValueType ai(int i) const { return m_value[i-1]; }
    inline _ValueType &ai(int i) { return m_value[i-1]; }

    /////////////////////////
    // unary operator

    /** assignment operator */
    const VectorND &operator=(const VectorND &arg) {
      if(&arg!=this)
	for (int i=0; i<_N_ELEM; ++i)
	  m_value[i] = arg.m_value[i];
      return *this;
    }

    /** += operator */
    const VectorND &operator+=(const VectorND &arg) {
      for (int i=0; i<_N_ELEM; ++i)
	m_value[i] += arg.m_value[i];
      return *this;
    }

    /** -= operator */
    const VectorND &operator-=(const VectorND &arg) {
      for (int i=0; i<_N_ELEM; ++i)
	m_value[i] -= arg.m_value[i];
      return *this;
    }

    /** *= operator (scaling) */
    const VectorND &operator*=(double arg)
    {
      for (int i=0; i<_N_ELEM; ++i)
	m_value[i] *= arg;
      return *this;
    }

    /** /= operator (scaling) */
    const VectorND &operator/=(double arg)
    {
      for (int i=0; i<_N_ELEM; ++i)
	m_value[i] /= arg;
      return *this;
    }

    /** - operator */
    VectorND operator-() const
    {
      VectorND retval(0, detail::no_init_tag());
      for (int i=0; i<_N_ELEM; ++i)
	retval.m_value[i] = - m_value[i];
      return retval;
    }

    /** + operator */
    VectorND operator+()
    {
      return *this;
    }

    //////////////////////////////////////////
    // methods

    bool equals(const VectorND &arg, _ValueType dtol = _ValueType(F_EPS8)) const
    {
      for (int i=0; i<_N_ELEM; ++i) {
        if (! (qlib::abs<_ValueType>(m_value[i] - arg.m_value[i])<dtol) )
	  return false;
      }
      return true;
    }

    bool isZero(_ValueType dtol = _ValueType(F_EPS8)) const
    {
      for (int i=0; i<_N_ELEM; ++i) {
        if (! (qlib::abs<_ValueType>(m_value[i])<dtol) )
	  return false;
      }
      return true;
    }

    // square of vector length
    _ValueType sqlen() const
    {
      return dot(*this);
    }

    _ValueType length() const {
      return _ValueType(::sqrt(sqlen()));
    }

    VectorND scale(_ValueType arg) const
    {
      VectorND ret(0, detail::no_init_tag());
      for (int i=0; i<_N_ELEM; ++i)
	ret.m_value[i] = m_value[i]*arg;
      return ret;
    }

    VectorND divide(_ValueType arg) const
    {
      VectorND ret(0, detail::no_init_tag());
      for (int i=0; i<_N_ELEM; ++i)
	ret.m_value[i] = m_value[i]/arg;
      return ret;
    }

    VectorND add(const VectorND &arg) const {
      VectorND retval(0, detail::no_init_tag());
      for (int i=0; i<_N_ELEM; ++i) {
	retval.m_value[i] = this->m_value[i] + arg.m_value[i];
      }
      return retval;
    }

    VectorND sub(const VectorND &arg) const {
      VectorND retval(0, detail::no_init_tag());
      for (int i=0; i<_N_ELEM; ++i) {
	retval.m_value[i] = this->m_value[i] - arg.m_value[i];
      }
      return retval;
    }

    VectorND normalize() const {
      return divide(length());
    }

    /** inner product */
    double dot(const VectorND &arg) const
    {
      double ret=0.0;
      for (int i=0; i<_N_ELEM; ++i)
	ret += m_value[i]*arg.m_value[i];
      return ret;
    }

    // clear this vector
    void zero() {
      for (int i=0; i<_N_ELEM; ++i)
        m_value[i] = _ValueType(0);
    }

  };

/*
  ///////////////////////////////////////////////
  // Definitions of non-member binary operators

  template <int _N_DIM, typename _ValueType>
  inline VectorND<_N_DIM,_ValueType> operator+(const VectorND<_N_DIM,_ValueType> &p1,const VectorND<_N_DIM,_ValueType> &p2)
  {
    return p1.add(p2);
  }

  template <int _N_DIM, typename _ValueType>
  inline VectorND<_N_DIM,_ValueType> operator-(const VectorND<_N_DIM,_ValueType> &p1,const VectorND<_N_DIM,_ValueType> &p2)
  {
    return p1.sub(p2);
  }

  template <int _N_DIM, typename _ValueType>
  inline bool operator==(const VectorND<_N_DIM,_ValueType> &p1,const VectorND<_N_DIM,_ValueType> &p2)
  {
    return p1.equals(p2);
  }
*/
  
} // namespace qlib

#endif

