//
// Vector class with 4 dimension real value
//

#ifndef __QLIB_VECTOR_4D_HPP__
#define __QLIB_VECTOR_4D_HPP__

#include "qlib.hpp"
#include "VectorND.hpp"

namespace qlib {

  class QLIB_API Vector4D : public VectorND<4, LReal>
  {
  public:
    typedef VectorND<4, LReal> super_t;
    
  public:
    /////////////////
    // constructors

    /** default constructor */
    Vector4D()
    {
      super_t::zero();
    }

    /** constructor without initialization */
    explicit
    Vector4D(int, detail::no_init_tag)
    {
    }

    /** copy constructor */
    Vector4D(const Vector4D &arg)
         : super_t(arg)
    {
    }

    /** Implicit conversion */
    Vector4D(const super_t &arg)
         : super_t(arg)
    {
    }

    Vector4D(double ax,double ay ,double az, double aw)
    {
      super_t::ai(1) = ax;
      super_t::ai(2) = ay;
      super_t::ai(3) = az;
      super_t::ai(4) = aw;
    }

    Vector4D(double ax,double ay ,double az)
    {
      super_t::ai(1) = ax;
      super_t::ai(2) = ay;
      super_t::ai(3) = az;
      super_t::ai(4) = 0.0;
    }

  public:

    inline double x() const { return super_t::ai(1); }
    inline double y() const { return super_t::ai(2); }
    inline double z() const { return super_t::ai(3); }
    inline double w() const { return super_t::ai(4); }

    inline double &x() { return super_t::ai(1); }
    inline double &y() { return super_t::ai(2); }
    inline double &z() { return super_t::ai(3); }
    inline double &w() { return super_t::ai(4); }

    //////////////////////////////////////////
    // methods

    void set(double ax, double ay, double az) {
      ai(1) = ax;
      ai(2) = ay;
      ai(3) = az;
    }

    void set(double ax, double ay, double az, double aw) {
      set(ax, ay, az);
      ai(4) = aw;
    }

    /** outer product */
    Vector4D cross(const Vector4D &arg) const
    {
      return Vector4D(y()*arg.z() - z()*arg.y(),
                      z()*arg.x() - x()*arg.z(),
                      x()*arg.y() - y()*arg.x());
    }

    ///////////////////////////

    /// Calc Angle between two vectors, this and v2
    double angle(const Vector4D &v2) const
    {
      return ::acos( dot(v2)/(length()*v2.length()) );
    }

    /// Calc Angle between two vectors, v1, and v2
    inline static double angle(const Vector4D &v1, const Vector4D &v2)
    {
      return ::acos( v1.dot(v2)/(v1.length()*v2.length()) );
    }

    /// calc torsion angle (throw exception in singlarity case)
    static double torsion(const Vector4D &veci,
                          const Vector4D &vecj,
                          const Vector4D &veck,
                          const Vector4D &vecl);

    LString toString() const;
    static bool fromStringS(const LString &src, Vector4D &result);


  };

  ///////////////////////////////////////////////
  // Definitions of non-member binary operators

  inline Vector4D operator+(const Vector4D &p1,const Vector4D &p2)
  {
    return p1.add(p2);
  }

  inline Vector4D operator-(const Vector4D &p1,const Vector4D &p2)
  {
    return p1.sub(p2);
  }

  inline bool operator==(const Vector4D &p1,const Vector4D &p2)
  {
    return p1.equals(p2);
  }

} // namespace qlib

#endif

