// -*-Mode: C++;-*-
//
// RangeSet : set of integer ranges
//
// $Id: RangeSet.hpp,v 1.6 2011/04/16 14:32:28 rishitani Exp $

#ifndef RANGE_SET_H__
#define RANGE_SET_H__

#include "qlib.hpp"

namespace qlib {

  namespace detail {

    ///
    /// Unit range data structure (nstart<= x < nend)
    ///
    template <typename _Type>
    struct Range {
    public:
      typedef _Type value_type;
      
      _Type nstart;
      _Type nend;
      
      Range(_Type s, _Type e) : nstart(s), nend(e) { MB_ASSERT(s<e); }

      int compare(const Range &t) const {
	MB_ASSERT(nstart<=nend);
	if (nend < t.nstart)
	  return -1;
	if (t.nend < nstart)
	  return 1;
	return 0;
      }

      void merge(const Range &t) {
	if (t.nstart<nstart)
	  nstart = t.nstart;
	if (nend<t.nend)
	  nend = t.nend;
	//MB_ASSERT(nstart<=nend);
	MB_ASSERT(!(nend<nstart));
      }

      void remove(const Range &nr) {
	if (nr.nend<nend)
	  nstart = nr.nend;
	if (nstart<nr.nstart)
	  nend = nr.nstart;
	//MB_ASSERT(nstart<=nend);
	MB_ASSERT(!(nend<nstart));
      }

      bool equals(const Range &t) const {
	//MB_ASSERT(nstart<=nend);
	MB_ASSERT(!nend<nstart);
	return (nstart == t.nstart &&
		t.nend == nend);
      }

      bool includes(const Range &t) const {
	//MB_ASSERT(nstart<=nend);
	MB_ASSERT(!nend<nstart);
	return (nstart < t.nstart &&
		t.nend < nend);
      }
      
      bool contains(const Range &t) const {
	//MB_ASSERT(nstart<=nend);
	MB_ASSERT( !(nend<nstart) );
	//return ( nstart<=t.nstart && t.nend<=nend );
	return ( !(t.nstart<nstart) && !(nend<t.nend) );
      }

      bool contains(const _Type &val) const {
	//MB_ASSERT(nstart<=nend);
	MB_ASSERT(!(nend<nstart));
	//return ( nstart<=val && val<nend );
	return ( !(val<nstart) && val<nend );
      }

      bool empty() const {
	//MB_ASSERT(nstart<=nend);
	MB_ASSERT(!(nend<nstart));
	return (nstart==nend);
      }
    };

  }

  template <typename _Type>
  class RangeSet
  {

  public:
  private:
    typedef detail::Range<_Type> elem_type;
    typedef std::list<elem_type> data_t;

    /// list of the unit ranges
    data_t m_data;

  public:
    typedef typename data_t::const_iterator const_iterator;

  public:
    RangeSet() {}

    RangeSet(const RangeSet &src) : m_data(src.m_data) {}

    ~RangeSet() {}

    const_iterator begin() const { return m_data.begin(); }
    const_iterator end() const { return m_data.end(); }

    /////////////////////////////////////////////////
    // add/remove selection

    void clear() {
      m_data.erase(m_data.begin(), m_data.end());
    }

    bool isEmpty() const {
      return m_data.empty();
    }

    /// Append new unit range
    void append(_Type nstart, _Type nend)
    {
      MB_ASSERT(checkConsistency());

      elem_type nr(nstart, nend);
      if (m_data.empty()) {
        m_data.push_back(nr);
        return;
      }

      typename data_t::iterator iter2;
      typename data_t::iterator iter = m_data.begin();
      for (; iter!=m_data.end(); ) {
        elem_type &tg = *iter;
        int res = nr.compare(tg);
        if (res==-1) {
          // nr is smaller than tg
          m_data.insert(iter, nr);
          return;
        }
        if (res==0) {
          // nr overwraps with tg
          //   incorporate tg into nr and remove tg from m_data
          nr.merge(tg);
          iter2 = iter;
          ++iter;
          m_data.erase(iter2);
          continue;
        }
        ++iter;
      }
      
      // nr is larger than any elements in m_data
      m_data.insert(iter, nr);
    }

    void remove(_Type nstart, _Type nend)
    {
      MB_ASSERT(checkConsistency());

      if (m_data.empty())
        return;

      elem_type nr(nstart, nend);

      typename data_t::iterator iter2;
      typename data_t::iterator iter = m_data.begin();
      for (; iter!=m_data.end(); ) {
        elem_type &tg = *iter;
        int res = nr.compare(tg);
        if (res==-1) {
          // nr is smaller than tg
          break;
        }
        if (res==0) {
          // nr overwraps with tg
          if (nr.includes(tg) || nr.equals(tg)) {
            // nr includes or equals tg --> remove tg
            iter2 = iter;
            ++iter;
            m_data.erase(iter2);
            continue;
          }
          else if (tg.includes(nr)) {
            // nr is included in tg --> split tg into 2 elems
            elem_type newr(tg.nstart, nr.nstart);
            m_data.insert(iter, newr);
            tg.nstart = nr.nend;
          }
          else {
            tg.remove(nr);
          }
        }
        ++iter;
      }

      MB_ASSERT(checkConsistency());
    }

    //RangeSet negate() const {
    //}

    void merge(const RangeSet &r) {
      const data_t &rdat = r.m_data;
      typename data_t::const_iterator iter = rdat.begin();
      for (; iter!=rdat.end(); ++iter)
	append(iter->nstart, iter->nend);
    }

    void remove(const RangeSet &r) {
      const data_t &rdat = r.m_data;
      typename data_t::const_iterator iter = rdat.begin();
      for (; iter!=rdat.end(); ++iter)
	remove(iter->nstart, iter->nend);
    }

    bool contains(_Type nstart, _Type nend) const {
      typename data_t::const_iterator iter = m_data.begin();
      elem_type nr(nstart, nend);
      for (; iter!=m_data.end(); ++iter)
	if (iter->contains(nr))
	  return true;
      return false;
    }

    bool contains(_Type val) const {
      typename data_t::const_iterator iter = m_data.begin();
      for (; iter!=m_data.end(); ++iter)
	if (iter->contains(val))
	  return true;
      return false;
    }

    bool checkConsistency() const
    {
      if (m_data.size()<=1) return true;
      typename data_t::const_iterator iter = m_data.begin();
      typename data_t::const_iterator iter2 = m_data.begin();
      ++iter2;
      for (; iter2!=m_data.end(); ++iter, ++iter2) {
        if (!(iter->nend < iter2->nstart))
          return false;
        if (iter->empty())
          return false;
      }

      if (iter->empty())
        return false;
      
      return true;
    }

  }; // class RangeSet<_Type>

  inline LString rangeToString(const RangeSet<int> &range)
  {
    LString rval;
    RangeSet<int>::const_iterator ebegin = range.begin();
    RangeSet<int>::const_iterator eend = range.end();
    RangeSet<int>::const_iterator eiter = ebegin;
    for (; eiter!=eend; ++eiter) {
      int nstart = eiter->nstart, nend = eiter->nend;
      if (eiter!=ebegin)
        rval += ",";
      
      if (nstart==nend-1)
        rval += LString::format("%d",nstart);
      else
        rval += LString::format("%d:%d",nstart,nend-1);
    }

    return rval;
  }


} // namespace qlib

#endif // RANGE_SET_H__


#if 0
    ///
    /// STL-style iterator for RangeSet
    ///
    class elem_iterator
    {
      friend class RangeSet<_Type>;

    private:
      data_t::const_iterator m_iter;
      const data_t *m_pdata;
      _Type m_n;

    public:
      elem_iterator() : m_pdata(NULL) {}

      elem_iterator(data_t::const_iterator ii, const data_t *pdata)
           : m_iter(ii), m_pdata(pdata)
      {
        if (m_iter==m_pdata->end())
          m_n = _Type(0);
        else
	  m_n = m_iter->nstart;
      }

      elem_iterator(const elem_iterator& __x)
	: m_iter(__x.m_iter), m_pdata(__x.m_pdata), m_n(__x.m_n)
      {
      }

      ///////
      
      _Type operator*() const
      {
	MB_ASSERT(m_pdata!=NULL);
	MB_ASSERT(m_iter!=m_pdata->end());
	return m_n;
      }

      elem_iterator& operator++()
      {
	MB_ASSERT(m_pdata!=NULL);
	MB_ASSERT(m_iter!=m_pdata->end());
	++m_n;
	if (m_iter->nend<=m_n) {
	  ++m_iter;
	  if (m_iter==m_pdata->end())
	    m_n = 0;
	  else
	    m_n = m_iter->nstart;
	}
	return *this;
      }

      elem_iterator operator++(int)
      {
	// save old value for returning
	elem_iterator __tmp = *this;
	elem_iterator::operator++();
	return __tmp;
      }

      elem_iterator& operator--()
      {
	MB_ASSERT(m_pdata!=NULL);

	if (m_iter==m_pdata->end()) {
	  --m_iter;
	  m_n = m_iter->nend-1;
	  MB_ASSERT(m_iter->nstart<=m_n);
	  return *this;
	}
	
	--m_n;
	if (m_n<m_iter->nstart) {
	  MB_ASSERT(m_iter!=m_pdata->begin());
	  --m_iter;
	  m_n = m_iter->nend-1;
	}
	return *this;
      }

      elem_iterator
      operator--(int)
      {
	elem_iterator __tmp = *this;
	elem_iterator::operator--();
	return __tmp;
      }

      bool operator==(const elem_iterator& __x) const
      {
	MB_ASSERT(m_pdata!=NULL && __x.m_pdata!=NULL);
	if (m_iter!=__x.m_iter)
	  return false;
	return (m_n==__x.m_n);
      }

      bool operator!=(const elem_iterator& __x) const
      {
	return ! (operator==(__x));
      }
    };

    /*
    /////////////////////////////////////////////////
    // range travarse

    elem_iterator elem_begin() const { return elem_iterator(m_data.begin(), &m_data); }
    elem_iterator elem_end() const { return elem_iterator(m_data.end(), &m_data); }

    const_iterator begin() const { return m_data.begin(); }
    const_iterator end() const { return m_data.end(); }
     */
    

#endif

