//
// Quaternion object
//


#ifndef L_QUAT_HPP__
#define L_QUAT_HPP__

#include "qlib.hpp"

#include "Vector4D.hpp"
#include "Matrix4D.hpp"

// #include "LSerial.hpp"

namespace qlib {

  class QLIB_API LQuat
  {
      
  public:
    /** representation of the quaternion */
    Vector4D m_data;

  public:
    // constructors

    // default constructor
    LQuat()
      : m_data()
    {
    }

    // copy constructor
    LQuat(const LQuat &arg)
      : m_data(arg.m_data)
    {
    }


    LQuat(double a, double x, double y, double z)
      : m_data(x, y, z, a)
    {
    }

    LQuat(const Vector4D &axis, double phi)
    {
      m_data = axis.scale(::sin(phi));
      m_data.ai(4) = ::cos(phi);
    }

    /** convert from the old vector-form representation */
    LQuat(const Vector4D &vrep)
      : m_data(vrep)
      //: V(vrep.x, vrep.y, vrep.z), a(vrep.w)
    {
    }

  public:

    //////////////////////////////////////////
    // C++ operators

    // = operator
    const LQuat &operator=(const LQuat &arg)
    {
      if(&arg!=this)
	m_data = arg.m_data;
      return *this;
    }

    // += operator
    const LQuat &operator+=(const LQuat &arg)
    {
      m_data += arg.m_data;
      return *this;
    }
    // -= operator
    const LQuat &operator-=(const LQuat &arg)
    {
      m_data -= arg.m_data;
      return *this;
    }

    // *= operator (scaling)
    const LQuat &operator*=(double arg)
    {
      scaleSelf(arg);
      return *this;
    }

    // /= operator (scaling)
    const LQuat &operator/=(double arg)
    {
      divideSelf(arg);
      return *this;
    }

    // - operator
    LQuat operator-() const
    {
      return LQuat(-m_data);
    }

    // + operator
    LQuat operator+()
    {
      return *this;
    }

    // *= operator
    const LQuat &operator*=(const LQuat &p2)
    {
      mulSelf(p2);
      return *this;
    }

    //////////////////////////////////////////
    // Access methods

    inline double Vx() const { return m_data.x(); }
    inline double Vy() const { return m_data.y(); }
    inline double Vz() const { return m_data.z(); }
    inline double a() const { return m_data.w(); }

    inline double &Vx() { return m_data.x(); }
    inline double &Vy() { return m_data.y(); }
    inline double &Vz() { return m_data.z(); }
    inline double &a()  { return m_data.w(); }

    //////////////////////////////////////////
    // binary operator

    bool equals(const LQuat &arg) const
    {
      return m_data.equals(arg.m_data);
    }

    // square of vector length
    double sqlen() const
    {
      return m_data.sqlen();
    }

    LQuat scale(double arg) const
    {
      LQuat retval(m_data);
      retval *= arg;
      return retval;
    }

    LQuat divide(double arg) const
    {
      LQuat retval(m_data);
      retval /= arg;
      return retval;
    }

    void scaleSelf(double arg)
    {
      m_data *= arg;
    }

    void divideSelf(double arg)
    {
      m_data /= arg;
    }

    LQuat normalize() const
    {
      LQuat retval(m_data);
      retval /= ::sqrt(sqlen());
      return retval;
    }

    void normalizeSelf()
    {
      m_data /= ::sqrt(sqlen());
    }

    void normalizeSelf(double dtol)
    {
      double sql = sqlen();
      if (qlib::abs<double>(sql-1.0)>dtol) {
        m_data /= ::sqrt(sql);
      }
    }

    void mulSelf(const LQuat &p2)
    {
      const LQuat &p1 = *this;

      const double tmpa = p1.m_data.w()*p2.m_data.w() -
	(p1.m_data.x()*p2.m_data.x() + p1.m_data.y()*p2.m_data.y() + p1.m_data.z()*p2.m_data.z());

      m_data.x() = p1.m_data.x()*p2.m_data.w() + p2.m_data.x()*p1.m_data.w() +
	p1.m_data.y()*p2.m_data.z() - p1.m_data.z()*p2.m_data.y();
      
      m_data.y() = p1.m_data.y()*p2.m_data.w() + p2.m_data.y()*p1.m_data.w() +
	p1.m_data.z()*p2.m_data.x() - p1.m_data.x()*p2.m_data.z();

      m_data.z() = p1.m_data.z()*p2.m_data.w() + p2.m_data.z()*p1.m_data.w() +
	p1.m_data.x()*p2.m_data.y() - p1.m_data.y()*p2.m_data.x();

      m_data.w() = tmpa;
    }

    LQuat mul(const LQuat &p2) const {
      LQuat ret(*this);
      ret.mulSelf(p2);
      return ret;
    }

    LQuat inv() const
    {
      LQuat retval;

      const double scl = -1.0/sqlen();
      retval.m_data.w() = m_data.w()/sqlen();
      retval.m_data.x() = m_data.x() * scl;
      retval.m_data.y() = m_data.y() * scl;
      retval.m_data.z() = m_data.z() * scl;

      return retval;
    }

    LQuat conj() const
    {
      LQuat retval;

      retval.m_data.w() = m_data.w();
      retval.m_data.x() = -m_data.x();
      retval.m_data.y() = -m_data.y();
      retval.m_data.z() = -m_data.z();

      return retval;
    }

    void fromEuler(double alpha, double beta, double gamma)
    {
      double cr, cp, cy, sr, sp, sy, cpcy, spsy;

      //If we are in Degree mode, convert to Radians
      alpha = qlib::toRadian(alpha);
      beta = qlib::toRadian(beta);
      gamma = qlib::toRadian(gamma);

      //Calculate trig identities
      //Formerly roll, pitch, yaw
      cr = cos(alpha/2);
      cp = cos(beta/2);
      cy = cos(gamma/2);
      sr = sin(alpha/2);
      sp = sin(beta/2);
      sy = sin(gamma/2);

      cpcy = cp * cy;
      spsy = sp * sy;

      m_data.w() = cr * cpcy + sr * spsy;
      m_data.x() = sr * cpcy - cr * spsy;
      m_data.y() = cr * sp * cy + sr * cp * sy;
      m_data.z() = cr * cp * sy - sr * sp * cy;
    }

    static LQuat makeFromEuler(double alpha, double beta, double gamma)
    {
      LQuat q;
      q.fromEuler(alpha, beta, gamma);
      return q;
    }

    void toEuler(double &roll, double &pitch, double &yaw) const
    {
      double tanyaw, sinpitch, tanroll;

      tanyaw = 2.0*(m_data.x()*m_data.y() + m_data.w()*m_data.z()) /
	(m_data.w()*m_data.w() + m_data.x()*m_data.x() - m_data.y()*m_data.y() - m_data.z()*m_data.z());
      sinpitch = -2.0*(m_data.x()*m_data.z()-m_data.w()*m_data.y());
      tanroll = 2.0*(m_data.w()*m_data.x()+m_data.y()*m_data.z()) /
	(m_data.w()*m_data.w() - m_data.x()*m_data.x() - m_data.y()*m_data.y() + m_data.z()*m_data.z());

      roll = atan(tanroll);
      pitch = asin(sinpitch);
      yaw = atan(tanyaw);
    }

    Matrix4D toRotMatrix() const {
      Matrix4D rmat;
      rmat.aij(1,1) = 1.0 - 2.0 * (m_data.y() * m_data.y() + m_data.z() * m_data.z());
      rmat.aij(1,2) = 2.0 * (m_data.x() * m_data.y() + m_data.z() * m_data.w());
      rmat.aij(1,3) = 2.0 * (m_data.z() * m_data.x() - m_data.y() * m_data.w());
      rmat.aij(1,4) = 0.0;

      rmat.aij(2,1) = 2.0 * (m_data.x() * m_data.y() - m_data.z() * m_data.w());
      rmat.aij(2,2) = 1.0 - 2.0 * (m_data.z() * m_data.z() + m_data.x() * m_data.x());
      rmat.aij(2,3) = 2.0 * (m_data.y() * m_data.z() + m_data.x() * m_data.w());
      rmat.aij(2,4) = 0.0;

      rmat.aij(3,1) = 2.0 * (m_data.z() * m_data.x() + m_data.y() * m_data.w());
      rmat.aij(3,2) = 2.0 * (m_data.y() * m_data.z() - m_data.x() * m_data.w());
      rmat.aij(3,3) = 1.0 - 2.0 * (m_data.y() * m_data.y() + m_data.x() * m_data.x());
      rmat.aij(3,4) = 0.0;

      rmat.aij(4,1) = 0.0;
      rmat.aij(4,2) = 0.0;
      rmat.aij(4,3) = 0.0;
      rmat.aij(4,4) = 1.0;
      return rmat;
    }

#if 0
    /** convert to the old-style vector representation (used in the QS) */
    Vector4D toVector3DRep() const {
      return Vector3D(m_data.x(), m_data.y(), m_data.z(), a);
    }
#endif

  };


  // Definitions of non-member binary operator functions

  inline LQuat operator+(const LQuat &p1,const LQuat &p2)
  {
    LQuat retval(p1);
    retval += p2;
    return retval;
  }

  inline LQuat operator-(const LQuat &p1,const LQuat &p2)
  {
    LQuat retval(p1);
    retval -= p2;
    return retval;
  }

  inline LQuat operator*(const LQuat &p1,const LQuat &p2)
  {
    return p1.mul(p2);
  }

  inline bool operator==(const LQuat &p1,const LQuat &p2)
  {
    return p1.equals(p2);
  }

}

#endif // QUATERNION_H__
