// -*-Mode: C++;-*-
//
//  molecular surface renderer
//
// $Id: MolSurfRenderer.cpp,v 1.12 2011/04/02 07:57:34 rishitani Exp $

#include <common.h>

#include "MolSurfRenderer.hpp"

#include <gfx/Mesh.hpp>
#include <gfx/DisplayContext.hpp>
#include <gfx/GradientColor.hpp>
#include <qsys/ScalarObject.hpp>
#include <qsys/Scene.hpp>
#include <modules/molstr/AtomPosMap.hpp>
#include <modules/molstr/MolAtom.hpp>
#include <modules/molstr/MolCoord.hpp>

//#include <gfx/DisplayList.hpp>
//#include <mbsys/QsysModule.hpp>
//#include <molstr/MolCoord.hpp>

using namespace surface;
using molstr::MolAtomPtr;
using molstr::MolCoord;

/// default constructor
MolSurfRenderer::MolSurfRenderer()
{
  m_pSurf = NULL;
  m_pAmap = NULL;
}

/** destructor */
MolSurfRenderer::~MolSurfRenderer()
{
  if (m_pAmap!=NULL) delete m_pAmap;
}

///////////////////////////////////////////

const char *MolSurfRenderer::getTypeName() const
{
  return "molsurf";
}

LString MolSurfRenderer::toString() const
{
  return LString("molsurf");
}

bool MolSurfRenderer::isCompatibleObj(qsys::ObjectPtr pobj) const
{
  MolSurfObj *ptest = dynamic_cast<MolSurfObj *>(pobj.get());
  return ptest!=NULL;
}

///////////////////////////////////////////////

bool MolSurfRenderer::getColorMol(float x, float y, float z, ColorPtr &rcol)
{
  //double par;

  if (m_pMol.isnull())
    return false;
  if (m_pAmap==NULL)
    return false;

  Vector4D pos(x, y, z);
  int aid = m_pAmap->searchNearestAtom(pos);
  if (aid<0) {
    MB_DPRINTLN("nearest atom is not found at (%f,%f,%f)", x, y, z);
    return false;
  }

  MolAtomPtr pa = m_pMol->getAtom(aid);
  rcol = molstr::ColSchmHolder::getColor(pa);

  return true;
  //pdl->color(col);
}

bool MolSurfRenderer::getColorSca(float x, float y, float z, ColorPtr &rcol)
{
  if (m_pScaObj==NULL)
    return false;

  Vector4D pos(x, y, z);
  double par = m_pScaObj->getValueAt(pos);

  if (par<m_dParLow) {
    rcol = m_colLow;
  }
  else if (par>m_dParHigh) {
    rcol = m_colHigh;
  }
  else if (par>m_dParMid) {
    // high<-->mid
    double ratio;
    if (qlib::Util::isNear(m_dParHigh, m_dParMid))
      ratio = 1.0;
    else
      ratio = (par-m_dParMid)/(m_dParHigh-m_dParMid);

    rcol = ColorPtr(new gfx::GradientColor(m_colHigh, m_colMid, ratio));
    // rcol = LColor(m_colHigh, m_colMid, ratio);
  }
  else {
    // mid<-->low
    double ratio;
    if (qlib::Util::isNear(m_dParMid, m_dParLow))
      ratio = 1.0;
    else
      ratio = (par-m_dParLow)/(m_dParMid-m_dParLow);

    rcol = ColorPtr(new gfx::GradientColor(m_colMid, m_colLow, ratio));
    // rcol = LColor(m_colMid, m_colLow, ratio);
  }

  return true;
}

void MolSurfRenderer::render(DisplayContext *pdl)
{
  MolSurfObj *pSurf = dynamic_cast<MolSurfObj *>(getClientObj().get());
  if (pSurf==NULL) {
    LOG_DPRINTLN("MolSurfRenerer> ERROR: client is null or not molsurf");
    return;
  }

  if (m_bDotSurf) {
    pdl->setLighting(false);
    pdl->setPolygonMode(gfx::DisplayContext::POLY_POINT);
  }
  else {
    pdl->setLighting(true);
    
    if (m_bWireFrame)
      pdl->setPolygonMode(gfx::DisplayContext::POLY_LINE);
    else
      pdl->setPolygonMode(gfx::DisplayContext::POLY_FILL);
  }
  
  if (!m_bCullFace)
    pdl->setCullFace(false);
  // pdl->sphere(10.0, getCenter());

  gfx::Mesh mesh;
  mesh.setDefaultAlpha(getDefaultAlpha());

  ColorPtr col;
  int i, j, nfsiz = pSurf->getFaceSize();
  int nvsiz = pSurf->getVertSize();
  
  mesh.init(nvsiz, nfsiz);

  mesh.color(getDefaultColor());
  if (m_nMode==SFREND_SIMPLE) {
  }
  else if (m_nMode==SFREND_SCAPOT) {
    // ELEPOT mode --> resolve target name
    qsys::ObjectPtr pobj;
    m_pScaObj = NULL;
    if (!m_sTgtElePot.isEmpty()) {
      pobj = ensureNotNull(getScene())->getObjectByName(m_sTgtElePot);
      m_pScaObj = dynamic_cast<qsys::ScalarObject*>(pobj.get());
    }
    if (m_pScaObj==NULL) {
      // try "target" property (for old version compat)
      if (!m_sTgtObj.isEmpty()) {
        pobj = ensureNotNull(getScene())->getObjectByName(m_sTgtObj);
        m_pScaObj = dynamic_cast<qsys::ScalarObject*>(pobj.get());
      }
    }
    
    if (m_pScaObj==NULL) {
      LOG_DPRINTLN("MolSurfRend> \"%s\" is not a scalar object.", m_sTgtElePot.c_str());
    }
  }
  else if (m_nMode==SFREND_MOLFANC) {
    // MOLFANC mode --> resolve target name
    if (!m_sTgtObj.isEmpty()) {
      qsys::ObjectPtr pobj = ensureNotNull(getScene())->getObjectByName(m_sTgtObj);
      m_pMol = MolCoordPtr(pobj, qlib::no_throw_tag());

      if (!m_pMol.isnull()) {
        // TO DO: re-generate atom-map only when Mol is changed.
        // (this impl always updates atommap when the renderer is invalidated.)
        makeAtomPosMap();
        
        // initialize the coloring scheme (with the target mol, but not this)
        molstr::ColoringSchemePtr pCS = getColSchm();
        if (!pCS.isnull())
          pCS->init(m_pMol, this);
      }
    }
    
    if (m_pMol.isnull()) {
      LOG_DPRINTLN("MolSurfRend> object \"%s\" is not found.", m_sTgtObj.c_str());
    }
  }
  

  // setup verteces
  for (i=0; i<nvsiz; i++) {
    MSVert v = pSurf->getVertAt(i);
    if (m_nMode==SFREND_SCAPOT) {
      bool res;
      if (m_bRampAbove) {
        const float fx = v.x + v.nx * 1.4f;
        const float fy = v.y + v.ny * 1.4f;
        const float fz = v.z + v.nz * 1.4f;
        res = getColorSca(fx, fy, fz, col);
      }
      else {
        res = getColorSca(v.x, v.y, v.z, col);
      }
      if (res) {
        mesh.color(col);
      }
    }
    else if (m_nMode>=SFREND_MOLSIMP) {
      if (getColorMol(v.x, v.y, v.z, col)) {
        mesh.color(col);
      }
    }
#ifdef USE_VERT_TYPE_ID
    else {
      switch (v.ntype) {
      case MolSurf::FTID_DBG1:
        mesh.color(LColor(0.0f, 0.0f, 1.0f));
        break;
      case MolSurf::FTID_DBG2:
        mesh.color(LColor(0.0f, 1.0f, 0.0f));
        break;
      case MolSurf::FTID_DBG3:
        mesh.color(LColor(0.0f, 1.0f, 1.0f));
        break;
      case MolSurf::FTID_DBG4:
        mesh.color(LColor(1.0f, 0.0f, 0.0f));
        break;
      case MolSurf::FTID_DBG5:
        mesh.color(LColor(1.0f, 0.0f, 1.0f));
        break;
      case MolSurf::FTID_DBG6:
        mesh.color(LColor(1.0f, 1.0f, 0.0f));
        break;
      case MolSurf::FTID_DBG7:
        mesh.color(LColor(1.0f, 1.0f, 1.0f));
        break;
      }
    }
#endif
    
    mesh.normal(v.n3d());
    mesh.setVertex(i, v.v3d());
  }

  // setup faces
  for ( i=0; i<nfsiz; i++) {
    int id[3];
    id[0] = (pSurf->getFaceAt(i)).id1;
    id[1] = (pSurf->getFaceAt(i)).id2;
    id[2] = (pSurf->getFaceAt(i)).id3;

    if (id[0]<0||id[0]>nvsiz||
        id[1]<0||id[1]>nvsiz||
        id[2]<0||id[2]>nvsiz) {
      //LOG_DPRINTLN("MSurfRend> invalid face index %d,%d,%d", id[0], id[1], id[2]);
      //break;
      continue;
    }

    mesh.setFace(i, id[0], id[1], id[2]);
  }

  // do it!!
  pdl->drawMesh(mesh);

  if (m_bWireFrame)
    pdl->setPolygonMode(gfx::DisplayContext::POLY_FILL);

  if (!m_bCullFace)
    pdl->setCullFace(true);

  pdl->setLighting(false);


  if (m_pAmap!=NULL) {
    delete m_pAmap;
    m_pAmap=NULL;
  }
  m_pMol = MolCoordPtr();
  m_pScaObj = NULL;

    /*
  if (m_bWireFrame) {
  pdl->color_3f(1.0, 0.0, 0.0);
  if (pSurf->m_dbg.size()>0) {
    for (i=0; i<pSurf->m_dbg.size(); ++i) {
      const Vector3D &v = pSurf->m_dbg[i].first;
      const LString &str = pSurf->m_dbg[i].second;
      pdl->drawString(v.x, v.y, v.z, str);
      //pdl->sphere(0.01, v);
    }
  }
  }
     */
}



Vector4D MolSurfRenderer::getCenter() const
{
  Vector4D pos;
  int i, n=0;

  MolSurfObj *pSurf = dynamic_cast<MolSurfObj *>(getClientObj().get());
  if (pSurf==NULL) {
    LOG_DPRINTLN("MolSurfRenerer> ERROR: client is null or not molsurf");
    return Vector4D();
  }
  int nvert = pSurf->getVertSize();
  int nskip = qlib::max<int>(1, nvert/1000-1);

  for (i=0; i<nvert; i+=nskip) {
    pos.x() += (pSurf->getVertAt(i)).x;
    pos.y() += (pSurf->getVertAt(i)).y;
    pos.z() += (pSurf->getVertAt(i)).z;
    n++;
  }

  if (n==0) {
    // TO DO: throw exception
    LOG_DPRINT("Renderer> cannot determine the center for ");
    LOG_DPRINTLN("%s:%s",
                 (pSurf->getName()).c_str(),
                 getName().c_str());
    return Vector4D();
  }

  pos = pos.scale(1.0/double(n));
  return pos;
}

void MolSurfRenderer::propChanged(qlib::LPropEvent &ev)
{
  if (ev.getName().equals("coloring")||
      ev.getParentName().equals("coloring")||
      ev.getParentName().startsWith("coloring.")) {
    invalidateDisplayCache();
  }
  else if (ev.getName().equals("target")||
           ev.getName().equals("defaultcolor")||
           ev.getName().equals("colormode")) {
    invalidateDisplayCache();
  }
  else if (ev.getName().equals("cullface")||
           ev.getName().equals("wireframe")||
           ev.getName().equals("dot")) {
    invalidateDisplayCache();
  }

  super_t::propChanged(ev);

}

void MolSurfRenderer::makeAtomPosMap()
{
  if (!m_pMol.isnull()) {
    if (m_pAmap!=NULL) delete m_pAmap;
    m_pAmap = new AtomPosMap();
    m_pAmap->setTarget(m_pMol);
    m_pAmap->setSpacing(3.5);
    m_pAmap->generate(m_pMolSel);
  }
}
