// -*-Mode: C++;-*-
//
//  Tube section class
//

#ifndef TUBE_SECTION_HPP_INCLUDED
#define TUBE_SECTION_HPP_INCLUDED

#include <qlib/Vector4D.hpp>
#include <qlib/LScrObjects.hpp>
#include <qlib/LScrSmartPtr.hpp>
#include <qlib/mcutils.hpp>

#include <qsys/style/StyleSupports.hpp>

namespace molvis {

using qlib::Vector4D;

class TubeSection : public qsys::StyleScrObject //qlib::LSimpleCopyScrObject
{
  MC_SCRIPTABLE;
  MC_CLONEABLE;

private:

  ////////////////////////////////
  // properties

  /** tube width */
  //MCINFO: double m_lw => width
  double m_lw;

  /** tuber ratio */
  //MCINFO: double m_tuber => tuber
  double m_tuber;

  /** alpha ratio (valid only for type SQUARE) */
  //MCINFO: double m_alpha => sharp
  double m_alpha;

  /** section type */
  //MCINFO: int m_nSectType => type
  int m_nSectType;

  /** detailness in section direction */
  //MCINFO: int m_nSectDetail => detail
  int m_nSectDetail;

  ////////////////////////////////
  // internal data
  
  /// section of the tube
  Vector4D *m_pSectTab;

  /// size of m_pSectTab
  int m_nSectTabSz;

public:

  /// section type ID
  enum {
    TS_ELLIPTICAL=0,
    TS_SQUARE=1,
    TS_RECT=2
  };

public:

  TubeSection();
  ~TubeSection();

  /////////////////////////////////////
  // Setters/Getters

  void setWidth(double d) {
    m_lw = d;
    //setupSectionTable();
    invalidate();
  }
  double getWidth() const { return m_lw; }

  void setTuber(double d) {
    m_tuber = d;
    invalidate();
    //setupSectionTable();
  }
  double getTuber() const { return m_tuber; }

  void setSharp(double d) {
    m_alpha = d;
    if (m_nSectType==TS_SQUARE)
      invalidate();
      //setupSectionTable();
  }
  double getSharp() const { return m_alpha; }

  void setType(int d) {
    m_nSectType = d;
    invalidate();
    //setupSectionTable();
  }
  int getType() const { return m_nSectType; }

  void setDetail(int d) {
    m_nSectDetail = d;
    invalidate();
    //setupSectionTable();
  }
  int getDetail() const { return m_nSectDetail; }

  void setupSectionTable();

  /////////////////////////////////////
  // data access

  /// returns true, if current data is valid.
  bool isValid() const { return m_pSectTab!=NULL; }

  /// Clear the current section table data
  void invalidate();

  /// get the size of section coordinates
  int getSize() const { return m_nSectTabSz; }

  ///
  /// Get the section vector of j th coordinate
  /// @param j Index number for the section table.
  ///        j will be truncated to be modulus of m_nSectTabSz.
  /// @param e1 basis vector along X direction.
  /// @param e2 basis vector along Y direction.
  ///
  Vector4D getVec(int j, const Vector4D &e1, const Vector4D &e2) const {
    j = j%m_nSectTabSz;
    Vector4D g = e1.scale(m_pSectTab[j].x()) + e2.scale(m_pSectTab[j].y());
    return g;
  }

  ///
  /// Get the section normal vector of j th coordinate
  /// @param j Index number for the section table.
  ///        j will be truncated to be modulus of m_nSectTabSz.
  /// @param e1 basis vector along X direction.
  /// @param e2 basis vector along Y direction.
  /// 
  Vector4D getNormVec(int j, const Vector4D &e1, const Vector4D &e2) const {
    j = j%m_nSectTabSz;
    Vector4D g = e1.scale(m_pSectTab[j].z()) + e2.scale(m_pSectTab[j].w());
    return g;
  }

  Vector4D getSectTab(int j) const { return m_pSectTab[j%m_nSectTabSz]; }

private:
  void setupEllipticalSection();
  void setupSquareSection();
  void setupRectSection();
};


typedef qlib::LScrSp<TubeSection> TubeSectionPtr;

}

#endif

