// -*-Mode: C++;-*-
//
//  Tube section class
//
//  $Id: TubeSection.cpp,v 1.4 2011/01/02 13:11:02 rishitani Exp $

#include <common.h>

#include "TubeSection.hpp"
#include <qlib/Vector4D.hpp>

using namespace molvis;
//using namespace molstr;

TubeSection::TubeSection()
{
  m_pSectTab = NULL;

  //m_nSectType = TS_ELLIPTICAL;
  //m_lw = 0.35;
  //m_tuber = 1.0;
  //m_alpha = 0.4;
  //m_nSectDetail = 16;

  resetAllProps();
}

TubeSection::~TubeSection()
{
  if (m_pSectTab!=NULL)
    delete [] m_pSectTab;
}

void TubeSection::setupEllipticalSection()
{
  int i;

  m_nSectTabSz = m_nSectDetail;
  m_pSectTab = new Vector4D[m_nSectTabSz];
  
  double dth = M_PI/double(m_nSectTabSz);
  
  for (i=0; i<m_nSectTabSz; i++) {
    const double th = double(i)*2.0*M_PI/double(m_nSectTabSz);
    const double si = ::sin(th+dth);
    const double co = ::cos(th+dth);
    const double nlen = ::sqrt(co*co*m_tuber*m_tuber + si*si);
    m_pSectTab[i].x() = co*m_lw;
    m_pSectTab[i].y() = si*m_lw*m_tuber;
    m_pSectTab[i].z() = co*m_tuber/nlen;
    m_pSectTab[i].w() = si/nlen;
  }
}

void TubeSection::setupSquareSection()
{
  double Rx, Ax, Ay;
  if (m_tuber>1.0) {
    Rx = m_lw*(1-m_alpha);
    Ax = m_lw*m_alpha;
    Ay = m_lw*m_tuber - Rx;
  }
  else {
    Rx = m_lw*m_tuber*(1-m_alpha);
    Ax = m_lw - Rx;
    Ay = m_lw*m_tuber*m_alpha;
  }

  const double len = (Ax+Ay)*2.0 + Rx*M_PI*2.0;
  const double ddel = len/double(m_nSectDetail);
  const int Nx = qlib::max(1, int(::floor(Ax*2.0/ddel)));
  const int Ny = qlib::max(1, int(::floor(Ay*2.0/ddel)));
  const int Nr = qlib::max(1, int(::floor(Rx*M_PI*0.5/ddel))) * 2;
  m_nSectTabSz = Nx*2 + Ny*2 + Nr*4;
  m_pSectTab = new Vector4D[m_nSectTabSz];

  int i,j;
  double th, si, co;


  for (j=0, i=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr));
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = Ax + co*Rx;
    m_pSectTab[i].y() = Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Nx; i++, j++) {
    const double d = 2.0*Ax * double(j)/double(Nx);
    m_pSectTab[i].x() = Ax - d;
    m_pSectTab[i].y() = Ay + Rx;
    m_pSectTab[i].z() = 0.0;
    m_pSectTab[i].w() = 1.0;
  }
  for (j=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr)) + M_PI*0.5;
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = - Ax + co*Rx;
    m_pSectTab[i].y() = Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Ny; i++, j++) {
    const double d = 2.0*Ay * double(j)/double(Ny);
    m_pSectTab[i].x() = -Ax - Rx;
    m_pSectTab[i].y() = Ay - d;
    m_pSectTab[i].z() = -1.0;
    m_pSectTab[i].w() =  0.0;
  }
  for (j=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr)) + M_PI;
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = - Ax + co*Rx;
    m_pSectTab[i].y() = - Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Nx; i++, j++) {
    const double d = 2.0*Ax * double(j)/double(Nx);
    m_pSectTab[i].x() = -Ax + d;
    m_pSectTab[i].y() = -Ay - Rx;
    m_pSectTab[i].z() =  0.0;
    m_pSectTab[i].w() = -1.0;
  }
  for (j=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr)) + M_PI*1.5;
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = Ax + co*Rx;
    m_pSectTab[i].y() = - Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Ny; i++, j++) {
    const double d = 2.0*Ay * double(j)/double(Ny);
    m_pSectTab[i].x() =  Ax + Rx;
    m_pSectTab[i].y() = -Ay + d;
    m_pSectTab[i].z() =  1.0;
    m_pSectTab[i].w() =  0.0;
  }
}

void TubeSection::setupRectSection()
{
  const double Ax = m_lw;
  const double Ay = m_lw*m_tuber;
  const double Wx = Ax*2.0;
  const double Wy = Ay*2.0;

  const double delta = (2.0*Wx+2.0*Wy)/double(m_nSectDetail);
  MB_DPRINTLN("TubSec> delta = %f", delta);
  const int Nx = qlib::max(1, int(::floor(Wx/delta)));
  const int Ny = qlib::max(1, int(::floor(Wy/delta)));
  MB_DPRINTLN("TubSec> Wx(%f)/del = Nx = %d", Wx, Nx);
  MB_DPRINTLN("TubSec> Wy(%f)/del = Ny = %d", Wy, Ny);
  m_nSectTabSz = 8 + (Nx-1)*2 + (Ny-1)*2;
  m_pSectTab = new Vector4D[m_nSectTabSz];

  // const double Rx = m_lw*(1-m_alpha);
  int i = 0, j;

  m_pSectTab[i] = Vector4D(Ax, Ay, 1.0, 0.0);
  ++i;
  m_pSectTab[i] = Vector4D(Ax, Ay, 0.0, 1.0);
  ++i;

  for (j=0; j<Nx-1; ++j, ++i) {
    m_pSectTab[i] = Vector4D(Ax - (Wx*(j+1.0))/double(Nx), Ay, 0.0, 1.0);
  }

  m_pSectTab[i] = Vector4D(-Ax, Ay, 0.0, 1.0);
  ++i;
  m_pSectTab[i] = Vector4D(-Ax, Ay, -1.0, 0.0);
  ++i;

  for (j=0; j<Ny-1; ++j, ++i) {
    //m_pSectTab[i] = Vector4D(-Ax, 0, -1.0, 0.0);
    m_pSectTab[i] = Vector4D(-Ax, Ay - (Wy*(j+1.0))/double(Ny), -1.0, 0.0);
  }

  m_pSectTab[i] = Vector4D(-Ax, -Ay, -1.0, 0.0);
  ++i;
  m_pSectTab[i] = Vector4D(-Ax, -Ay, 0.0, -1.0);
  ++i;

  for (j=0; j<Nx-1; ++j, ++i) {
    //m_pSectTab[i] = Vector4D(0, -Ay, 0.0, -1.0);
    m_pSectTab[i] = Vector4D(-Ax + (Wx*(j+1.0))/double(Nx), -Ay, 0.0, -1.0);
  }

  m_pSectTab[i] = Vector4D(Ax, -Ay, 0.0, -1.0);
  ++i;
  m_pSectTab[i] = Vector4D(Ax, -Ay, 1.0, 0.0);
  ++i;

  for (j=0; j<Ny-1; ++j, ++i) {
    //m_pSectTab[i] = Vector4D(Ax, 0, 1.0, 0.0);
    m_pSectTab[i] = Vector4D(Ax, -Ay + (Wy*(j+1.0))/double(Ny), 1.0, 0.0);
  }
}

void TubeSection::setupSectionTable()
{
  if (m_pSectTab!=NULL)
    delete [] m_pSectTab;

  switch (m_nSectType) {
  default:
  case TS_ELLIPTICAL: {
    setupEllipticalSection();
    break;
  }
    
  case TS_SQUARE: {
    setupSquareSection();
    break;
  }

  case TS_RECT: {
    setupRectSection();
    break;
  }
    
  } // switch

  return;
}

void TubeSection::invalidate()
{
  if (m_pSectTab!=NULL)
    delete [] m_pSectTab;
  m_pSectTab = NULL;
}

#if 0
//////////////////////////////////////////////////////////////////
// property handlers

bool TubeSection::getPropInt(const char *propname, int &value)
{
  if (LChar::equals(propname, "sectdetail")) {
    value = m_nSectDetail;
    return true;
  }
  else if (LChar::equals(propname, "secttype")) {
    value = m_nSectType;
    return true;
  }

  return false;
}
bool TubeSection::setPropInt(const char *propname, int value)
{
  if (LChar::equals(propname, "sectdetail")) {
    if (m_nSectDetail != value) {
      m_nSectDetail = value;
      setupSectionTable();
      // invalidateCache();
    }
    return true;
  }
  else if (LChar::equals(propname, "secttype")) {
    if (m_nSectType != value) {
      if (value==TS_ELLIPTICAL ||
          value==TS_SQUARE ||
          value==TS_RECT)
        m_nSectType = value;
      else
        return false;
      setupSectionTable();
      // invalidateCache();
    }
    return true;
  }

  return false;
}

////////////////////////////////////////////////////////////////////
// real value property

bool TubeSection::getPropReal(const char *propname, double &value)
{
  if (LChar::equals(propname, "width")) {
    value = m_lw;
    return true;
  }
  else if (LChar::equals(propname, "tuber")) {
    value = m_tuber;
    return true;
  }
  else if (LChar::equals(propname, "sharp")) {
    value = m_alpha;
    return true;
  }

  return false;
}

bool TubeSection::setPropReal(const char *propname, double value)
{
  if (LChar::equals(propname, "width")) {
    m_lw = value;
    setupSectionTable();
    // invalidateCache();
    return true;
  }
  else if (LChar::equals(propname, "tuber")) {
    m_tuber = value;
    setupSectionTable();
    // invalidateCache();
    return true;
  }
  else if (LChar::equals(propname, "sharp")) {
    m_alpha = value;
    // if section mode is not SQUARE, dont update figures
    if (m_nSectType==TS_SQUARE) {
      setupSectionTable();
      // invalidateCache();
    }
    return true;
  }

  return false;
}

#endif

