// -*-Mode: C++;-*-
//
//  Backbone spline-trace renderer class
//
//  $Id: SplineRenderer.cpp,v 1.16 2010/12/30 17:49:41 rishitani Exp $

#include <common.h>
#include "molvis.hpp"

#include "SplineRenderer.hpp"
#include "TubeSection.hpp"

#include <modules/molstr/MolCoord.hpp>
#include <modules/molstr/MolChain.hpp>
#include <modules/molstr/MolResidue.hpp>

#include <gfx/GradientColor.hpp>
//#include <qsys/ObjectEvent.hpp>

//#include <molstr/MolEvent.hpp>
//#include <molstr/ResiToppar.hpp>
//#include <molstr/ResiLink.hpp>

using namespace molvis;
using namespace molstr;

SplineRenderer::SplineRenderer()
{
  m_scs.setParent(this);
  m_scs.setSmooth(0.0);
}

SplineRenderer::~SplineRenderer()
{
}

const char *SplineRenderer::getTypeName() const
{
  return "spline";
}

/*
void SplineRenderer::targetChanged(MbObjEvent &ev)
{
  if (ev.instanceOf<MolAtomsMovedEvent>()) {
    // atoms are moved: we must remake the spline coefficients!
    m_scs.cleanup();
  }
  MainChainRenderer::targetChanged(ev);
}
*/

void SplineRenderer::setPivAtomName(const LString &aname)
{
  m_scs.cleanup();
  super_t::setPivAtomName(aname);
}

void SplineRenderer::beginRend(DisplayContext *pdl)
{
  //
  // Generate (or regenerate) the spline coeff set object, (if required)
  //
  if (!m_scs.isValid()) {
    MolCoordPtr pmol = getClientMol();
    if (!m_scs.generate(pmol)) {
      LOG_DPRINTLN("SplineRenderer> Fatal error: cannot create spline interpolator.");
    }
  }
  pdl->setLineWidth(m_dLineWidth);
}

void SplineRenderer::endRend(DisplayContext *pdl)
{
  pdl->setLineWidth(1.0);
}

//////////

void SplineRenderer::beginSegment(DisplayContext *pdl, MolResiduePtr pRes)
{
  m_pStartRes = pRes;
}

void SplineRenderer::rendResid(DisplayContext *pdl, MolResiduePtr pRes)
{
}

void SplineRenderer::endSegment(DisplayContext *pdl, MolResiduePtr pEndRes)
{
  //
  // get coeff obj for the start residue
  //
  SplineCoeff *pCoeff = m_scs.searchCoeff(m_pStartRes);

  if (pCoeff==NULL) {
    LOG_DPRINTLN("SplineRenderer> fatal error at endSegment(coeff not found for %s.%s:%s.%s)",
                 m_pStartRes->getParentChain()->getName().c_str(),
                 m_pStartRes->getIndex().toString().c_str(),
                 pEndRes->getParentChain()->getName().c_str(),
                 pEndRes->getIndex().toString().c_str());
    LOG_DPRINTLN("SplineRenderer> rendering aborted.");
    return;
  }

  //
  // calculate f-parameter for the start and end residues
  //

  double fdum1, fcent, fstart, fend;

  if (!pCoeff->getParamRange(m_pStartRes, fstart, fdum1, fcent)) {
    LOG_DPRINTLN("SplineRenderer> Fatal Error: get param for start residue failed.");
    return;
  }
  if (!qlib::isNear4(fstart, fcent)) {
    // fstart!=fcent --> start from the halfway
    // TO DO: configurable
    fstart = fcent;
  }

  if (!pCoeff->getParamRange(pEndRes, fdum1, fend, fcent)) {
    LOG_DPRINTLN("SplineRenderer> Fatal Error: get param for start residue failed.");
    return;
  }
  if (!qlib::isNear4(fend, fcent)) {
    // fend!=fcent --> end with the halfway
    // TO DO: configurable
    fend = fcent;
  }

  renderSpline(pdl, pCoeff, m_pStartRes, fstart, pEndRes, fend);
}

void SplineRenderer::renderSpline(DisplayContext *pdl, SplineCoeff *pCoeff,
                                MolResiduePtr pStartRes, double fstart,
                                MolResiduePtr pEndRes, double fend)
{
  //
  // calculate num of drawing points ndelta
  //
  int ndelta = (int) ::floor( (fend-fstart)*m_nAxialDetail );
  if (ndelta<=0) {
    // degenerated (single point)
    // TO DO: impl
    return;
  }

  pdl->setLighting(false);
  ColorPtr pPrevCol;

  pdl->startLineStrip();
  int i;
  for (i=0; i<=ndelta; i++) {
    double par = fstart + double(i)/double(m_nAxialDetail);

    ColorPtr pCol = calcColor(par, pCoeff);

    Vector4D f1, vpt;
    Vector4D bnorm;
    pCoeff->interpNormal(par, &bnorm);
    pCoeff->interpAxis(par, &f1, &vpt);

    Vector4D e12 = (bnorm - f1);
    Vector4D e11 = ( e12.cross(vpt) ).normalize();

    if (!isSmoothColor() && i!=0) {
      pdl->color(pPrevCol);
      pdl->vertex(f1);
    }
    pdl->color(pCol);
    pdl->vertex(f1);
    if (i%m_nAxialDetail==0) {
      pdl->vertex(f1+e11.scale(0.5));
      pdl->vertex(f1);
      pdl->vertex(f1+e12);
      pdl->vertex(f1);
    }
    pPrevCol = pCol;
  }
  pdl->end();
  
  pdl->setLighting(false);

}

//virtual
void SplineRenderer::setAxialDetail(int nlev)
{
  m_nAxialDetail = nlev;
  invalidateSplineCoeffs();
}

void SplineRenderer::propChanged(qlib::LPropEvent &ev)
{
  if (ev.getParentName().equals("coloring")||
      ev.getParentName().startsWith("coloring.")) {
    invalidateDisplayCache();
  }

  super_t::propChanged(ev);
}

void SplineRenderer::objectChanged(qsys::ObjectEvent &ev)
{
  if (ev.getType()==qsys::ObjectEvent::OBE_CHANGED) {
    invalidateSplineCoeffs();
    invalidateDisplayCache();
    return;
  }
  
  super_t::objectChanged(ev);
}

void SplineRenderer::invalidateSplineCoeffs()
{
  m_scs.cleanup();
  invalidateDisplayCache();
}

ColorPtr SplineRenderer::calcColor(double par, SplineCoeff *pCoeff)
{
  int nprev = int(::floor(par));
  int nnext = int(::ceil(par));
  double rho = par - double(nprev);

  if (!isSmoothColor()) {
    if (rho>0.5)
      // next
      return ColSchmHolder::getColor(pCoeff->getAtom(nnext));
    else
      // prev
      return ColSchmHolder::getColor(pCoeff->getAtom(nprev));
  }

  // liner interporation of color (nnext, nprev)
  ColorPtr pCol1 = ColSchmHolder::getColor(pCoeff->getAtom(nnext));
  ColorPtr pCol2 = ColSchmHolder::getColor(pCoeff->getAtom(nprev));

  if (pCol1->equals(*pCol2.get()))
    return pCol1;

  ColorPtr pGradCol = ColorPtr(new gfx::GradientColor(pCol1, pCol2, rho));
  return pGradCol;
}

/////////////////////////////////////////////////////////////
// Tube cap rendering routine

void SplineRenderer::makeCap(DisplayContext *pdl,
                           bool fStart, int nType,
                           TubeSection *pTs,
                           const Vector4D &f, const Vector4D &vpt,
                           const Vector4D &e1, const Vector4D &e2)
{
  switch (nType) {
  case 2:
  default:
    // no cap (transparent)
    break;

  case 1:
    // Flat cap
    makeFlatCap(pdl, fStart, pTs, f, vpt, e1, e2);
    break;

  case 0:
    // Spherical cap
    makeSpherCap(pdl, fStart, pTs, f, vpt, e1, e2);
    break;
  }
}

void SplineRenderer::makeFlatCap(DisplayContext *pdl,
                               bool fStart,
                               TubeSection *pTs,
                               const Vector4D &f, const Vector4D &vpt,
                               const Vector4D &e1, const Vector4D &e2)
{
  int j;

  pdl->startTriangleFan();
  // pdl->color(col);
  if (fStart)
    pdl->normal(-vpt);
  else
    pdl->normal(vpt);
  
  pdl->vertex(f);
  // Vector4D prev_g2;

  if (!fStart) {
    for (j=0; j<=pTs->getSize(); j++) {
      Vector4D g2 = pTs->getVec(j, e1, e2);
      pdl->vertex(f+g2);
    }
  }
  else {
    for (j=pTs->getSize(); j>=0; j--) {
      Vector4D g2 = pTs->getVec(j, e1, e2);
      pdl->vertex(f+g2);
    }
  }
  pdl->end();
}

void SplineRenderer::makeSpherCap(DisplayContext *pdl,
                                bool fStart,
                                TubeSection *pTs,
                                const Vector4D &f, const Vector4D &vpt,
                                const Vector4D &e1, const Vector4D &e2)
{
  int i,j;

  const int detail = 5;
  double sign, e1len = pTs->getVec(0, e1, e2).length();
  sign = (fStart) ? -1.0 : 1.0;
  
  Vector4D v = vpt.scale(1.0/vpt.length());
  v = v.scale(sign*e1len/double(detail));
  
  double gpar2, t2;
  Vector4D f2, e21, e22;
  
  //  int stab_sz =
  for (i=0; i<=detail; i++) {
    double t = double(i)/double(detail);
    double gpar = ::sqrt(1.0-t*t);
    //double dgp = -t/gpar;
    Vector4D f1 = f+v.scale(double(i));
    Vector4D e11 = e1.scale(gpar);
    Vector4D e12 = e2.scale(gpar);
    
    if (i==0) {
      t2 = t;
      gpar2 = gpar;
      f2 = f1;
      e21 = e11;
      e22 = e12;
      continue;
    }
    
    // render tube body
    pdl->startTriangleStrip();
    for (j=0; j<=pTs->getSize(); j++) {
      Vector4D stab = pTs->getSectTab(j);
      
      Vector4D g1 = e11.scale(stab.x()) + e12.scale(stab.y());
      Vector4D dg1 = e11.scale(stab.z()) + e12.scale(stab.w());
      Vector4D dgp1 = e11.scale(-stab.w()) + e12.scale(stab.z());
      if (i==detail)
        dg1 = v;
      else
        dg1 = dg1.scale(e1len*gpar) - (dgp1.cross(g1)).scale(t*sign);
      
      Vector4D g2 = e21.scale(stab.x()) + e22.scale(stab.y());
      Vector4D dg2 = e21.scale(stab.z()) + e22.scale(stab.w());
      Vector4D dgp2 = e21.scale(-stab.w()) + e22.scale(stab.z());
      dg2 = dg2.scale(e1len*gpar2) - (dgp2.cross(g2)).scale(t2*sign);
      
      // pdl->color(col);
      if (fStart) {
        pdl->normal(dg2);
        pdl->vertex(f2+g2);
        pdl->normal(dg1);
        pdl->vertex(f1+g1);
      }
      else {
        pdl->normal(dg1);
        pdl->vertex(f1+g1);
        pdl->normal(dg2);
        pdl->vertex(f2+g2);
        }
    }
    
    pdl->end();
    t2 = t;
    gpar2 = gpar;
      f2 = f1;
    e21 = e11;
    e22 = e12;
    
  }
}



