//
// Biomolecule analysis manager singleton class
//
// $Id: MolAnlManager.cpp,v 1.6 2011/03/05 17:09:02 rishitani Exp $
//

#include <common.h>
#include "MolAnlManager.hpp"

#include <qlib/Vector4D.hpp>
#include <qlib/Matrix4D.hpp>
#include <modules/molstr/MolAtom.hpp>
#include <modules/molstr/AtomIterator.hpp>
#include <modules/molstr/SelCommand.hpp>

#include "mmdb/mmdb_manager.h"
#include "ssmlib/ssm_align.h"

SINGLETON_BASE_IMPL(molanl::MolAnlManager);

///////////////

using namespace molanl;
using namespace molstr;
using qlib::Vector4D;
using qlib::Matrix4D;

MolAnlManager::MolAnlManager()
{
  MB_DPRINTLN("MolAnlManager(%p) created", this);
}

MolAnlManager::~MolAnlManager()
{
  MB_DPRINTLN("MolAnlManager(%p) destructed", this);
}

///////////////


namespace {
void copySelected(CMMDBManager *pMol1, MolCoordPtr pmol, SelectionPtr psel)
{
  int index;
  int serNum;
  LString atomName;
  LString resName;
  LString chainID;
  int seqNum;
  InsCode insCode;
  LString altLoc = "";
  const char *segID = "";
  LString element;
  Vector4D pos;

  index = 1;
  serNum = 0;
  seqNum = 0;

  AtomIterator iter(pmol, psel);
  for (iter.first(); iter.hasMore(); iter.next()) {
    MolAtomPtr pAtom = iter.get();

    chainID = pAtom->getChainName();
    //segID;

    resName = pAtom->getResName();
    ResidIndex rind = pAtom->getResIndex();
    insCode[0] = rind.second;
    insCode[1] = '\0';
    seqNum = rind.first;

    atomName = pAtom->getName();
    if (atomName.equals("CA")) atomName = " CA ";
    if (atomName.equals("N")) atomName = " N  ";
    if (atomName.equals("C")) atomName = " C  ";
    if (atomName.equals("O")) atomName = " O  ";
    //altLoc
    element = pAtom->getElementName();

    pos = pAtom->getPos();

    int rc = pMol1->PutAtom(index, serNum, atomName, resName, chainID, seqNum, insCode, altLoc, segID, element);
    PCAtom pMMDBAtom = pMol1->GetAtomI(index);
    if (pMMDBAtom==NULL) {
      LOG_DPRINTLN("MMDB: copySelected failed!!");
      MB_THROW(qlib::RuntimeException, "Coordinate conversion for MMDB is failed!!");
      return;
    }
    ++index;
    pMMDBAtom->SetCoordinates(pos.x(), pos.y(), pos.z(), pAtom->getOcc(), pAtom->getBfac());
  }

  pMol1->FinishStructEdit();
}
}

void MolAnlManager::superposeSSM1(MolCoordPtr pmol_ref, SelectionPtr psel_ref,
                                 MolCoordPtr pmol_mov, SelectionPtr psel_mov)
{
  // mol1: reference molecule
  CMMDBManager *pMol1 = new CMMDBManager;
  copySelected(pMol1, pmol_ref, psel_ref);

  // mol2: moving molecule
  CMMDBManager *pMol2 = new CMMDBManager;
  copySelected(pMol2, pmol_mov, psel_mov);

  int precision = SSMP_Normal;
  int connectivity = CSSC_Flexible;

  CSSMAlign *pAln = new CSSMAlign();
  int rc = pAln->Align(pMol2, pMol1, precision, connectivity);
  if (rc!=0) {
    LString msg = LString::format("SSM-superpose is failed (error code=%d)", rc);
    MB_THROW(qlib::RuntimeException, msg);
    return;
  }

  Matrix4D xfmat;
  int i,j;
  for (i=0;i<4;i++)
    for (j=0;j<4;j++)
      xfmat.aij(i+1,j+1) = pAln->TMatrix[i][j];
  
  LOG_DPRINTLN("=== SSM superpose result ===");
  LOG_DPRINTLN(" RMSD: %f angstrom", pAln->rmsd);
  LOG_DPRINTLN(" Nalgn: %d", pAln->nalgn);
  LOG_DPRINTLN(" Ngaps: %d", pAln->ngaps);
  //LOG_DPRINTLN(" Nres 1: %d 2: %d", pAln->nres1, pAln->nres2);
  //LOG_DPRINTLN(" Nsel 1: %d 2: %d", pAln->nsel1, pAln->nsel2);
  LOG_DPRINTLN("======");

  delete pAln;
  delete pMol1;
  delete pMol2;

  pmol_mov->xformByMat(xfmat);
  pmol_mov->fireAtomsMoved();
}

void MolAnlManager::superposeSSM2(qlib::uid_t mol_ref, const LString &sel_ref,
                                  qlib::uid_t mol_mov, const LString &sel_mov)
{
  MolCoordPtr pMolRef = MolCoord::getMolByID(mol_ref);
  MolCoordPtr pMolMov = MolCoord::getMolByID(mol_mov);

  SelectionPtr pSelRef = SelectionPtr(new SelCommand(sel_ref));
  SelectionPtr pSelMov = SelectionPtr(new SelCommand(sel_mov));

  superposeSSM1(pMolRef, pSelRef, pMolMov, pSelMov);
}



