//
//
// $Id: Interp.cpp,v 1.17 2010/12/31 12:01:26 rishitani Exp $

#include <common.h>
#include <qlib/qlib.hpp>

#ifdef USE_INTERNAL_JS
#  include <js/src/jsapi.h>
#else
#  include <jsapi.h>
#endif

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>

#include "jsbr.hpp"
#include "Interp.hpp"
#include "ObjWrapper.hpp"
#include "AutoJSRequest.hpp"

namespace fs = boost::filesystem;

using namespace jsbr;
using qlib::LString;

static JSClass global_class = {
  "CueMol2 global class", 0,
  JS_PropertyStub, JS_PropertyStub,
  JS_PropertyStub, JS_PropertyStub,
  JS_EnumerateStub,
  JS_ResolveStub,
  JS_ConvertStub,
  JS_FinalizeStub,
  NULL, NULL,
  NULL,
  NULL, NULL, NULL
};

static void
errorReporterFunc(JSContext *cx, const char *message, JSErrorReport *report)
{
  if (!report) {
    LOG_DPRINTLN("%s", message);
    return;
  }

  LString prefix;
  if (report->lineno) { 
    prefix = LString::format("%s %u:",
			     (report->filename)?(report->filename):"(unknown)",
			     report->lineno);
  }
  else if (report->filename) {
    prefix = LString::format("%s:",
			     report->filename);
  }
  
  if (JSREPORT_IS_WARNING(report->flags)) {
    prefix += LString::format("%swarning: ",
			      JSREPORT_IS_STRICT(report->flags) ? "strict " : "");
  }
  
  LOG_DPRINTLN("%s %s", prefix.c_str(), message);
  return;
}

////////////////////////////////////////////////////////////

Interp::~Interp()
{
  // cleanup related callback obj to this context
  ObjWrapper::invalidateCallbackObj(this);

  JSContext *pcx = (JSContext *)m_pdata;
  JS_DestroyContext(pcx);
}

bool Interp::init(qlib::LScriptable *pGlob)
{
  JSContext *pcx = (JSContext *)m_pdata;

  AutoJSRequest ar(pcx);

  JS_SetErrorReporter(pcx, errorReporterFunc);

  JSObject *pglob, *ppar;
  if (pGlob!=NULL)
    ppar = ObjWrapper::makeWrapper(pcx, pGlob);
  else
    ppar = NULL;

  //pglob = JS_NewObject(pcx, &global_class, NULL, ppar);
  pglob = JS_NewObject(pcx, &global_class, NULL, NULL);
  
  if (!pglob)
    return false;

  JSBool res = JS_InitStandardClasses(pcx, pglob);
  if (!res) return false;

  res = JS_DefineProperty(pcx, pglob, "scene", OBJECT_TO_JSVAL(ppar),
                          NULL, NULL, JSPROP_ENUMERATE|JSPROP_PERMANENT|JSPROP_READONLY);
  // res = JS_SetParent(pcx, pglob, ppar);
  if (!res) return false;

  m_pGlob = pglob;

#if (JS_VERSION>180)
#define CREATE_OBJECT_FUN ObjWrapper::createObject18
#define GET_SERVICE_FUN ObjWrapper::getService18
#define PRINT_LOG_FUN ObjWrapper::printLog18
#else
#define CREATE_OBJECT_FUN ObjWrapper::createObject
#define GET_SERVICE_FUN ObjWrapper::getService
#define PRINT_LOG_FUN ObjWrapper::printLog
#endif


  // Define "createObject" method
  JSFunction *pfun = JS_DefineFunction(pcx, pglob, "newObj",
                                       CREATE_OBJECT_FUN,
                                       1, 0);
  if (pfun==NULL) {
    LOG_DPRINTLN("JS_deffun failed!!");
  }

  // Define "getService" method
  pfun = JS_DefineFunction(pcx, pglob, "getService",
                           GET_SERVICE_FUN, 1, 0);
  if (pfun==NULL) {
    LOG_DPRINTLN("JS_deffun failed!!");
  }

  // Define "print" method
  pfun = JS_DefineFunction(pcx, pglob, "print",
                           PRINT_LOG_FUN, 1, 0);
  if (pfun==NULL) {
    LOG_DPRINTLN("JS_deffun failed!!");
  }

  // Define "exec" method
  pfun = JS_DefineFunction(pcx, pglob, "exec",
                           ObjWrapper::execFile, 1, 0);
  if (pfun==NULL) {
    LOG_DPRINTLN("JS_deffun failed!!");
  }

  // Define "callGC" method
  pfun = JS_DefineFunction(pcx, pglob, "callGC",
                           ObjWrapper::callGC, 0, 0);
  if (pfun==NULL) {
    LOG_DPRINTLN("JS_deffun failed!!");
  }

  // set private data
  JS_SetContextPrivate(pcx, this);

  return true;
}

void Interp::eval(const qlib::LString &scr)
{
  JSContext *pcx = (JSContext *)m_pdata;
  JSObject *pglob = (JSObject *)m_pGlob;
  
  const char *script = scr.c_str();
  const char *filename = "(none)";
  int lineno = 0;
  jsval rval;
  JSString *str;
  JSBool ok;

  ok = JS_EvaluateScript(pcx, pglob, script, strlen(script),
			 filename, lineno, &rval);

  /*
  if (ok) {
    str = JS_ValueToString(cx, rval);
    printf("script result: %s\n", JS_GetStringBytes(str));
  }
  else {
    printf("err\n");
    }*/

}

bool Interp::execFile(const qlib::LString &fname)
{
  JSContext *pcx = (JSContext *)m_pdata;
  JSObject *pglob = (JSObject *)m_pGlob;

  JSBool res = ObjWrapper::execFileImpl(pcx, pglob, fname, true);

  return res;
}

LString Interp::resolvePath(const LString &fname) const
{
  fs::path inpath(fname);
  if (inpath.is_complete())
    return inpath.file_string();

  qlib::MapTable<LString>::const_iterator iter = m_pathTab.begin();
  qlib::MapTable<LString>::const_iterator eiter = m_pathTab.end();
  for (; iter!=eiter; ++iter) {
    const LString &path = iter->second;
    fs::path base_path(path);
    fs::path test_path = fs::complete(inpath, base_path);
    if (is_regular_file(test_path))
      return test_path.file_string();
  }

  return fname;
}

