/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.smtp.store;

import java.io.*;
import java.util.*;

import javax.activation.DataHandler;
import javax.mail.*;
import javax.mail.internet.*;

import jp.co.connectone.eai.smtp.user.SmtpAccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.client.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;


public class SmtpMailStoreImpl implements IMailStore
{
	private String serverAddress;

	public String getServerAddress()
	{
		return serverAddress;
	}

	public void setServerAddress(String serverAddress)
	{
		this.serverAddress = serverAddress;
	}

	public static final IStoreID storeID = new SimpleStoreID(SmtpMailStoreImpl.class.getName());
	public static final String storeName = "Smtp Mail Store";
	
	public static SmtpMailStoreImpl getInstance()
	{
		return new SmtpMailStoreImpl();// implement resource_pool here if needed.
	}
	
	public IServiceInfo getServiceInfo() throws Exception
	{
		return new SmtpServiceInfo(storeID, storeName);
	}

	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		SmtpServiceInfo info = new SmtpServiceInfo(storeID, storeName);
		String serverAddress = (String) h.get("server");
		if (serverAddress != null) info.setServerAddress(serverAddress);
		return info;
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	public String getName() throws Exception
	{
		return storeName;
	}

	public void deleteMail(IAccountData arg0, IObjectIndex arg1) throws ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support deleteMail.");
	}

	public IMailDTO[] getHeaders(IAccountData arg0, ISearchDestination arg1) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getHeaders.");
	}

	public IMailDTO[] getHeadersByDate(IAccountData arg0, ISearchDestination arg1, Date arg2) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getHeadersByDate.");
	}

	public IMailDTO getMail(IAccountData arg0, ISearchDestination arg1, IObjectIndex arg2) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getMail.");
	}

	public FolderMetadata[] getMailFolders(IAccountData arg0, ISearchDestination arg1) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getMailFolders.");
	}

	public IMailDTO[] getMailsByDate(IAccountData arg0, ISearchDestination arg1, Date arg2) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getMailsByDate.");
	}

	public IMailDTO[] getSentMails(IAccountData arg0) throws ServerDown, HandleException
	{
		throw new HandleException(storeName+" does not support getSentMails.");
	}

	public IMailDTO[] getSentMailsByDate(IAccountData arg0, Date arg1) throws ServerDown, HandleException
	{
		throw new HandleException(storeName+" does not support getSentMailsByDate.");
	}

	public IObjectIndex[] getUIDLs(IAccountData arg0, ISearchDestination arg1) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getUIDLs.");
	}

	public IObjectIndex[] getUIDLsByDate(IAccountData arg0, ISearchDestination arg1, Date arg2) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getUIDLsByDate.");
	}

	public IObjectIndex sendMail(IAccountData acc, IMailDTO dto) throws NoSuchRights, ServerDown, IncorrectData, HandleException
	{
		if (!(acc instanceof SmtpAccountData)) {
			throw new IncorrectData("account data type must be SmtpAccountData");
		}
		setServerAddress(acc.getServiceInfo().getServerAddress());
		MimeMessage msg = populateMail(dto);

		try {
			Transport.send(msg);
		}
		catch(Exception e) {
			e.printStackTrace();
			throw new ServerDown("got error (" + e.getMessage() + ") from transport, assume server doesn't accept this machine as mail sender");
		}
		return null;//smtp cannot return oid
	}

	public void delete(IAccountData arg0, ISearchDestination arg1, IObjectIndex arg2) throws Exception
	{
		throw new HandleException(storeName+" does not support delete.");
	}

	public IRecordObject[] getAllDatas(IAccountData arg0, ISearchDestination arg1) throws Exception
	{
		throw new HandleException(storeName+" does not support getAllDatas.");
	}

	public IRecordObject getFolderIndexFromString(String arg0) throws IncorrectData
	{
		throw new IncorrectData(storeName+" does not support getFolderIndexFromString.");
	}

	public FolderMetadata[] getFolderList(IAccountData arg0, ISearchDestination arg1) throws Exception
	{
		throw new HandleException(storeName+" does not support getFolderList.");
	}

	public ISearchDestination getPresetDestination(IAccountData arg0, int arg2) throws IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support getPresetDestination.");
	}

	public IRecordObject read(IAccountData arg0, ISearchDestination arg1, IObjectIndex arg2) throws Exception
	{
		throw new HandleException(storeName+" does not support read.");
	}

	public IRecordObject[] search(IAccountData arg0, ISearchFormula arg1) throws Exception
	{
		throw new HandleException(storeName+" does not support search.");
	}

	public IRecordObject[] searchByDate(IAccountData arg0, ISearchDestination arg1, Date arg2) throws Exception
	{
		throw new HandleException(storeName+" does not support searchByDate.");
	}

	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject ro) throws Exception
	{
		if (!(ro instanceof BasicMailDTO)) {
			throw new IncorrectData("record object type must be BasicMailDTO");
		}
		BasicMailDTO dto = (BasicMailDTO)ro;
		return sendMail(acc, dto);
	}
	
	private Address populateAddress(MailAddress src)
	{
		InternetAddress add = null;
		if (src==null) return add;

		try {
			String address=src.getAddressPart();
			String name = src.getNamePart();
			String raw = src.getRawAddress();
			if (address.equalsIgnoreCase(raw)) {
				add = new InternetAddress(address,name,"ISO-2022-JP");
			}
			else {
				add = new InternetAddress(raw);
			}
		}
		catch(UnsupportedEncodingException ue) {
			ue.printStackTrace();
		}
		catch (AddressException ae) {
			ae.printStackTrace();
		}
		
		return add;
	}

	private MimeMessage populateMail(IMailDTO dto ) throws NoSuchRights,HandleException,IncorrectData
	{
		MimeMessage mail = getMessageInstance();
		HeaderDTO header = dto.getHeader();
		try {
			mail.setSentDate(new Date());
			mail.setSubject(header.getHeaderSubject(),"ISO-2022-JP");
			mail.setFrom(populateAddress(header.getHeaderFrom()));
			if (dto.getHeader().getHeaderReplyTo()!=null) {
				MailAddress replyTo = dto.getHeader().getHeaderReplyTo();
Log.trace("replyTo:"+replyTo);
				Address[] replyAddress = new Address[1];
				replyAddress[0] = populateAddress(replyTo);
				mail.setReplyTo(replyAddress);
			}
			MailAddress[] toAddress = header.getHeaderTo();
			if (toAddress != null) {
				if (toAddress.length>0) {
					Address[] to = new Address[toAddress.length];
					for (int i=0;i<to.length;i++) {
						to[i] = populateAddress(toAddress[i]);
					}
					mail.setRecipients(Message.RecipientType.TO,to);
				}
			}
			MailAddress[] ccAddress = header.getHeaderCC();
			if (ccAddress != null) {
				if (ccAddress.length>0) {
					Address[] cc = new Address[ccAddress.length];
					for (int i=0;i<cc.length;i++) {
						cc[i] = populateAddress(ccAddress[i]);
					}
					mail.setRecipients(Message.RecipientType.CC,cc);
				}
			}
			MailAddress[] bccAddress = header.getHeaderBCC();
			if (bccAddress != null) {
				if (bccAddress.length>0) {
					Address[] bcc = new Address[bccAddress.length];
					for (int i=0;i<bcc.length;i++) {
						bcc[i] = populateAddress(bccAddress[i]);
					}
					mail.setRecipients(Message.RecipientType.BCC,bcc);
				}
			}
			if (dto.getNumberOfAttachments()>0) {
Log.debug("having attachments");
				MimeMultipart part1 = new MimeMultipart();
				MimeBodyPart body1 = new MimeBodyPart();
				
				body1.setText(dto.getBody(),"ISO-2022-JP");
				part1.addBodyPart(body1);
				
				for (int i=0;i<dto.getNumberOfAttachments();i++) {
					AttachmentDTO att = dto.getAttachment(i);

					MimeBodyPart attach = new MimeBodyPart();
					attach.setDataHandler(new DataHandler(new AttachmentDataSource(att)));
					String attName = att.getFileName();
					try {
						attName = MimeUtility.encodeText(attName,"ISO-2022-JP","B");
					}
					catch (Exception e) {
						Log.error("Exception on encoding Japanese Filename",e);
						e.printStackTrace();
					}
Log.trace("attName="+attName);
					attach.setFileName(attName);
					part1.addBodyPart(attach);
				}
				
				mail.setContent(part1);
			}
			else {
				try {
					mail.setText(dto.getBody());
				}
				catch (Exception e) {
					Log.error("Exception on encoding Japanese Body",e);
					e.printStackTrace();
				}
				mail.setHeader("Content-transfer-encoding","base64");
			}
			mail.saveChanges();
		}
		catch(MessagingException e) {
			e.printStackTrace();
		}
		return mail;
	}
	
	protected MimeMessage getMessageInstance()
	{
		Properties env = new Properties();
		env.put("mail.smtp.host",getServerAddress());
		Session session = Session.getInstance(env);
		MimeMessage msg = new MimeMessage(session);
		
		return msg;
	}
}


