/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.pop3.socket;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.util.*;
import java.util.regex.*;

import jp.co.connectone.eai.pop3.exception.ServerCommunicationException;
import jp.co.connectone.eai.pop3.socket.Pop3SocketCommand.CMD;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;

public class Pop3ProtocolSocketImpl extends BaseSocketImpl
{
	private String OK_Pattern="^\\+OK.*$";
	private Pattern STAT_Pattern=Pattern.compile("^\\+OK *([0-9]*) *([0-9]*) *$");
	private Pattern UIDL_Pattern=Pattern.compile("^\\+OK *([0-9]*) *(.*) *$");

	@Override
	protected boolean isTerminate(String line)
	{
		if (".".equals(line)) return true;
		return false;
	}
	
	protected Pop3ProtocolSocketImpl()
	{
		defaultPort=110;
	}

	public static Pop3ProtocolSocketImpl getInstance() throws HandleException
	{
		return new Pop3ProtocolSocketImpl();
	}

	public boolean doUser(String userId) throws HandleException
	{
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.USER,userId));
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on USER command transaction.",e);
		}
		return isOk(rc);
	}

	public boolean doPass(String passwd) throws HandleException
	{
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.PASS,passwd));
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on PASS command transaction.",e);
		}
		return isOk(rc);
	}

	public int[] doStat() throws HandleException
	{
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.STAT));
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on STAT command transaction.",e);
		}
		if (!isOk(rc)) {
			return null;
		}
		try {
	        return parseStatReturn(rc);
		}
		catch (NumberFormatException e) {
			throw new HandleException("STAT command reply("+rc+") is number format error.", e);
		}
	}

	private int[] parseStatReturn(String line) throws NumberFormatException
	{
		Matcher m;
		int[] rc = null;
		m = STAT_Pattern.matcher(line);
		if (m.find()) {
			rc = new int[2];
			String number = m.group(1);
			String bytes = m.group(2);
			Log.debug("number=" + number + ",bytes=" + bytes);
			rc[0] = Integer.parseInt(number);
			rc[1] = Integer.parseInt(bytes);
		}
		return rc;
	}

	public String doUidl(int index) throws HandleException
	{
		index += 1;// for pop index starts from 1.
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.UIDL,""+index));
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on UIDL command transaction.",e);
		}
		if (!isOk(rc)) {
			return null;
		}
		return parseUidlReturn(rc);
	}
	
	public String parseUidlReturn(String line)
	{
		if (line==null) return null;
		Matcher m;
		String rc=null;
		m = UIDL_Pattern.matcher(line);
		if (m.find()) {
			rc = m.group(2);
		}
		return rc;
	}

	public String[] doUidl() throws HandleException
	{
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.UIDL),true);
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on UIDL command transaction.",e);
		}
		String[] uidls = rc.split("\n");
		if (!isOk(uidls[0])) {
			return null;
		}
		
		Vector<String> v = new Vector<String>();
		for (int i=1;i<uidls.length;i++) {
			v.add(uidls[i].split(" ")[1]);
		}
		
		return v.toArray(new String[0]);
	}

	public String doTop(int index) throws HandleException
	{
		index += 1;// for pop index starts from 1.
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.TOP,""+index,""+0),true);
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on TOP command transaction.",e);
		}
		return parseTopRetrLines(rc);
	}

	public String doRetr(int index) throws HandleException
	{
		index += 1;// for pop index starts from 1.
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.RETR,""+index),true);
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on RETR command transaction.",e);
		}
		return parseTopRetrLines(rc);
	}

	public boolean doDelete(int index) throws HandleException
	{
		index += 1;// for pop index starts from 1.
		String rc="";
		try {
			rc = super.transaction(new Pop3SocketCommand(CMD.DELE,""+index),false);
		}
		catch (IOException e) {
			if (e instanceof HandleException) throw (HandleException)e;
			throw new ServerCommunicationException("error on DELETE command transaction.",e);
		}
		return isOk(rc);
	}
	
	private String parseTopRetrLines(String rc)
	{
		if (rc==null) return null;
		if (!rc.startsWith("+OK")) return null;
		return rc.substring(rc.indexOf("\n")+1);
	}

	public boolean doQuit() throws HandleException
	{
		try {
			super.transaction(new Pop3SocketCommand(CMD.QUIT));
		}
		catch (IOException e) {
			Log.error("error on QUIT command transaction.",e);
		}
		return true;
	}

	protected boolean isOk(String line)
	{
		if (line==null) return false;
		boolean rc = line.matches(OK_Pattern);
		if (!rc) {
			Log.trace("pop3 returns error:"+line);
		}
		return rc;
	}

	@Override
	public void connect(String server1,String server2, int port) throws HandleException
	{
		super.connect(server1, server2, port);
		String line = null;
		try {
			line = reader.readLine();
		}
		catch (SocketTimeoutException e) {
			Log.debug("Socket.reading, connection timeout.", e);
			throw new ServerCommunicationException("error on connect timeout.", e);
		}
		catch (IOException e) {
			Log.debug("Socket.reading, connect error.", e);
			throw new ServerCommunicationException("error on connect.", e);
		}
		if (line == null) {// Is EOF
			Log.debug("Socket.reading, returns null on connect.");
			throw new ServerCommunicationException("POP connection is not alive.");
		}
		if (!isOk(line)) {
			Log.debug("pop3 returns error on connect. " + line);
			throw new ServerCommunicationException("pop3 returns error on connect. " + line);
		}
		Log.trace("pop3:" + line);
	}

	@Override
	public void connect(String server1,String server2) throws HandleException
	{
		connect(server1,server2,defaultPort);
	}
}
