/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.pop3.socket;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

import jp.co.connectone.eai.pop3.store.*;
import jp.co.connectone.eai.pop3.user.Pop3AccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;

public class Pop3MailStoreImpl extends SocketBasedPop3BaseImpl implements IMailStore
{
	public static final IStoreID storeID = new SimpleStoreID(Pop3MailStoreImpl.class.getName());
	public static final String storeName = "Pop3 Mail Store";
	private static final SimpleDateFormat sf = new SimpleDateFormat("yyyy/MM/dd 00:00:00 Z");
	private static final SimpleDateFormat ef = new SimpleDateFormat("yyyy/MM/dd 23:59:59 Z");
	private static final SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss Z");

    private enum MODE {GREATER,LESSER};

    private int binarySearch(Vector<UidlElement> uidlList, HashMap<String, IMailDTO> topList, Date d,MODE mode) throws HandleException{
        int l = 0;
        int r = uidlList.size()-1;
        int m;
        int val = -1;

        while(l<=r){
            m = (l+r) / 2;

            Date from = getDateFromHeaders(uidlList, topList, m);

            int comp = from.compareTo(d);

            if(comp < 0){
                l = m + 1;
                val = m;
                Log.trace("");
                if (mode==MODE.GREATER) val = val + 1;
            }
            else if(comp > 0){
                r = m - 1;
                val = m;
                Log.trace("");
                if (mode==MODE.LESSER) val = val - 1;
            }
            else{
                Log.trace("=");
                val = m;
                // Linear search.
                // There can be duplication of time.
                // Calculate the approximate value of the minimum difference.
                if (mode == MODE.GREATER) {
                    m = m - 1;
                    while (l <= m) {
                        from = getDateFromHeaders(uidlList, topList, m);
                        comp = from.compareTo(d);
                        if (comp != 0) {
                            break;
                        }
                        val = m;
                        m = m - 1;
                    }
                } else if (mode == MODE.LESSER) {
                    m = m + 1;
                    while (r >= m) {
                        from = getDateFromHeaders(uidlList, topList, m);
                        comp = from.compareTo(d);
                        if (comp != 0) {
                            break;
                        }
                        val = m;
                        m = m + 1;
                    }
                }
                break;
            }
        }
        return val;
    }
    
    private Date getDateFromHeaders(Vector<UidlElement> uidlList, HashMap<String, IMailDTO> topList, int index) throws HandleException
	{
    	UidlElement uidl = uidlList.get(index);
    	IMailDTO header = topList.get(uidl.getUidl());
    	if (header==null) {
    		String topStr = getTopByUidlIndexWithoutOpen(uidlList,index);
    		header = Pop3Translator.TopString2MailDTO(topStr, uidl.getUidl());
    		topList.put(uidl.getUidl(), header);
    	}
    	return header.getHeader().getHeaderSendDate();
	}

	private boolean checkDataExist(int start,int end,int size)
    {
    	if (start<0) return false;
    	if (end<0) return false;
    	if (start>size) return false;
    	if (start>end) return false;
    	return true;
    }
	
	public static Pop3MailStoreImpl getInstance()
	{
		return new Pop3MailStoreImpl();// implement resource_pool here if needed.
	}
	
	public IServiceInfo getServiceInfo() throws Exception
	{
		return new Pop3ServiceInfo(storeID, storeName);
	}

	public void deleteMail(IConnectorSession cacheObj,IAccountData acc, IObjectIndex oidx) throws ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		Vector<UidlElement> uidlList = getCurrentUidlList(cacheObj);
		Pop3AccountData pacc = (Pop3AccountData)acc;
        setAccount(pacc.getUserID(), pacc.getPassword(), pacc.getServiceInfo().getServerAddress(), pacc.getServiceInfo().getServerAddress2(), pacc.getServiceInfo().getPort());

		Object oid = oidx.getIndex();

		Log.debug("deleteMail::oid="+oid);
		super.delete(uidlList,oid.toString());
	}
	
	private HashMap<String, IMailDTO> getCurrentTopList(IConnectorSession cacheObj)
	{
		if (cacheObj==null) return null;
		return cacheObj.getTopList();
	}

	private Vector<UidlElement> getCurrentUidlList(IConnectorSession cacheObj)
	{
		if (cacheObj==null) return null;
		return cacheObj.getUidlList();
	}

	public IMailDTO[] getHeaders(IConnectorSession cacheObj,IAccountData acc, ISearchDestination dest) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		Vector<UidlElement> uidlList = getCurrentUidlList(cacheObj);
		IMailDTO[] headers = null;
		if (!(acc instanceof Pop3AccountData)) {
			throw new IncorrectData("account data type must be Pop3AccountData");
		}
		Pop3AccountData pacc = (Pop3AccountData)acc;
        
		setUserID(pacc.getUserID());
		setPassword(pacc.getPassword());
		setServerAddress(acc.getServiceInfo().getServerAddress());
        Vector<IMailDTO> ar = getList(uidlList);
        headers = (IMailDTO[])ar.toArray(headers);
		
        return headers;
	}

	private Vector<IMailDTO> getList(Vector<UidlElement> uidlList)
	{
		// TODO ꂽ\bhEX^u
		return null;
	}

	public IMailDTO[] getHeadersByDate(IConnectorSession cacheObj,IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		Pop3AccountData pacc = (Pop3AccountData)acc;
		setAccount(pacc.getUserID(), pacc.getPassword(), pacc.getServiceInfo().getServerAddress(), pacc.getServiceInfo().getServerAddress2(), pacc.getServiceInfo().getPort());
		Vector<UidlElement> uidlList = getCurrentUidlList(cacheObj);
		try {
			open();
			uidlList = checkUidlListNotEmptyWithoutOpen(uidlList);
			cacheObj.setUidlList(uidlList);
			HashMap<String, IMailDTO> topList = getCurrentTopList(cacheObj);
			if (topList==null) {
				topList = new HashMap<String, IMailDTO>();
			}
			IMailDTO[] headers=null;
			String sStr = sf.format(date);
			String eStr = ef.format(date);

			Date start = null;
			Date end = null;
			try {
				start = df.parse(sStr);
				end = df.parse(eStr);
			}
			catch (ParseException e) {
				close();
				throw new HandleException("can't create datetime-range.");
			}
			Log.debug("From "+start+" TO "+end);

			int sidx = binarySearch(uidlList, topList, start, MODE.GREATER);
			int eidx = binarySearch(uidlList, topList, end, MODE.LESSER);
			eidx+=1;
			Log.debug("sidx="+sidx+",eidx="+eidx);
			if (checkDataExist(sidx, eidx, uidlList.size())==false) {
				close();
				return new BasicMailDTO[0];
			}
			topList = fillTopListByRange(sidx, eidx, uidlList, topList);
			close();
			if (cacheObj!=null) {
				cacheObj.setTopList(topList);
			}
			headers = populateHeaderList(sidx, eidx, uidlList,topList);
			return headers;
		} 
		catch (HandleException e) {
			close();
			throw e;
		}
	}

	public IMailDTO[] getHeadersByDateRange(IConnectorSession cacheObj,IAccountData acc, ISearchDestination dest, Date date,int sidx,int eidx) throws HandleException
	{
		Pop3AccountData pacc = (Pop3AccountData)acc;
		setAccount(pacc.getUserID(), pacc.getPassword(), pacc.getServiceInfo().getServerAddress(), pacc.getServiceInfo().getServerAddress2(), pacc.getServiceInfo().getPort());
		Vector<UidlElement> uidlList = getCurrentUidlList(cacheObj);
		try {
			open();
			uidlList = checkUidlListNotEmptyWithoutOpen(uidlList);
			cacheObj.setUidlList(uidlList);
			HashMap<String, IMailDTO> topList = getCurrentTopList(cacheObj);
			if (topList==null) {
				topList = new HashMap<String, IMailDTO>();
			}
			IMailDTO[] headers=null;
			String sStr = sf.format(date);
			String eStr = ef.format(date);

			Date start = null;
			Date end = null;
			try {
				start = df.parse(sStr);
				end = df.parse(eStr);
			} catch (ParseException e) {
				close();
				throw new HandleException("can't create datetime-range.");
			}
			Log.debug("From "+start+" TO "+end);

			int dsidx = binarySearch(uidlList, topList, start, MODE.GREATER);
			int deidx = binarySearch(uidlList, topList, end, MODE.LESSER);
			eidx+=1;
			Log.debug("dsidx="+dsidx+",deidx="+deidx);
			if (checkDataExist(dsidx, deidx, uidlList.size())==false) {
				close();
				return new BasicMailDTO[0];
			}
			int aeidx = deidx - sidx;
			if (aeidx<dsidx) {
				close();
				return new BasicMailDTO[0];
			}
			int asidx = deidx - eidx;
			if (asidx<dsidx) {
				asidx=dsidx;
			}
			Log.debug("asidx="+asidx+",aeidx="+aeidx);
			topList = fillTopListByRange(asidx, aeidx, uidlList, topList);
			close();
			if (cacheObj!=null) {
				cacheObj.setTopList(topList);
			}
			headers = populateHeaderList(asidx, aeidx, uidlList,topList);
			return headers;
		} 
		catch (HandleException e) {
			close();
			throw e;
		}
	}

	private IMailDTO[] populateHeaderList(int start,int end,Vector<UidlElement> uidlList, HashMap<String, IMailDTO> topList)
	{
		Vector<IMailDTO> rc = new Vector<IMailDTO>();
		end = end + 1;
		if (end>=uidlList.size()) {
			end = uidlList.size();
		}
		for (int i=start;i<end;i++) {
			IMailDTO dto = topList.get(uidlList.get(i).getUidl());
			rc.add(0,dto);
		}
		return rc.toArray(new IMailDTO[0]);
	}

	private Vector<IMailDTO> getListByDate(Vector<UidlElement> uidlList, Date date)
	{
		// TODO ꂽ\bhEX^u
		return null;
	}

	public IMailDTO getMail(IConnectorSession cacheObj,IAccountData acc, ISearchDestination dest, IObjectIndex oidx) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		Vector<UidlElement> uidlList = getCurrentUidlList(cacheObj);
		if (oidx == null) {
			throw new IncorrectData("message-id is null. cannot retrieve data.");
		}
		if (!(acc instanceof Pop3AccountData)) {
			throw new IncorrectData("account data type must be Pop3AccountData");
		}
		Pop3AccountData pacc = (Pop3AccountData)acc;
		setAccount(pacc.getUserID(), pacc.getPassword(), pacc.getServiceInfo().getServerAddress(), pacc.getServiceInfo().getServerAddress2(), pacc.getServiceInfo().getPort());

		Object oid = oidx.getIndex();

		Log.debug("getMail::oid="+oid);

		IMailDTO mail  = getMail(uidlList,oid.toString());

		return mail;
	}
	
	protected IMailDTO getMail(Vector<UidlElement> uidlList,String uidl) throws HandleException
	{
		String retr = getRetrByUidl(uidlList,uidl);
		
		return Pop3Translator.RetrString2MailDTO(retr, uidl);
	}

	public FolderMetadata[] getMailFolders(IAccountData arg0, ISearchDestination arg1) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		FolderMetadata folders[] = new FolderMetadata[1];
		folders[0] = populateFolder(new Pop3FolderIndex());
		return folders;
	}

	public IMailDTO[] getMailsByDate(IConnectorSession cacheObj,IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		Vector<UidlElement> uidlList = getCurrentUidlList(cacheObj);
		IMailDTO[] headers = null;
		if (!(acc instanceof Pop3AccountData)) {
			throw new IncorrectData("account data type must be Pop3AccountData");
		}
		Pop3AccountData pacc = (Pop3AccountData)acc;

		setUserID(pacc.getUserID());
		setPassword(pacc.getPassword());
		setServerAddress(acc.getServiceInfo().getServerAddress());
		Vector<IMailDTO> ar = getListByDate(uidlList,date);
        headers = (IMailDTO[])ar.toArray(headers);

        return headers;
	}

	public IMailDTO[] getSentMails(IAccountData arg0) throws ServerDown, HandleException
	{
		throw new HandleException(storeName+" does not support getSentMails.");
	}

	public IMailDTO[] getSentMailsByDate(IAccountData arg0, Date arg1) throws ServerDown, HandleException
	{
		throw new HandleException(storeName+" does not support getSentMailsByDate.");
	}
	
	public Vector<UidlElement> getInitialUidls(IAccountData acc) throws HandleException
	{
		Pop3AccountData pacc = (Pop3AccountData)acc;
		setAccount(pacc.getUserID(), pacc.getPassword(), pacc.getServiceInfo().getServerAddress(), pacc.getServiceInfo().getServerAddress2(), pacc.getServiceInfo().getPort());
		return super.getUIDLList();
	}

	public IObjectIndex[] getUIDLs(IAccountData acc, ISearchDestination dest) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		if (!(acc instanceof Pop3AccountData)) {
			throw new IncorrectData("account data type must be Pop3AccountData");
		}
		Pop3AccountData pacc = (Pop3AccountData)acc;
		setAccount(pacc.getUserID(), pacc.getPassword(), pacc.getServiceInfo().getServerAddress(), pacc.getServiceInfo().getServerAddress2(), pacc.getServiceInfo().getPort());
		Vector<UidlElement> uidlList = super.getUIDLList();
		ArrayList<Pop3ObjectIndex> uidlarray = new ArrayList<Pop3ObjectIndex>();
		for (UidlElement uidl : uidlList) {
			uidlarray.add(new Pop3ObjectIndex(uidl.getUidl()));
		}
		Pop3ObjectIndex[] uidls = new Pop3ObjectIndex[1];
		uidls = uidlarray.toArray(uidls);
		
		return uidls;
	}

	public IObjectIndex[] getUIDLsByDate(IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		//TODO : replace
		IMailDTO[] headers = getHeadersByDate(acc, dest, date);
		if (headers==null) return new Pop3ObjectIndex[0];
		ArrayList<Pop3ObjectIndex> uidlarray = new ArrayList<Pop3ObjectIndex>();
		for (int i=0;i<headers.length;i++) {
			String uidl = headers[i].getHeader().getHeaderUIDL();
			if (uidl==null) {
				Log.debug("skip message "+headers[i].getHeader().getHeaderSubject()+" because of no Message-ID.");
			}
			else {
				uidlarray.add(new Pop3ObjectIndex(uidl));
			}
		}
		Pop3ObjectIndex[] uidls = new Pop3ObjectIndex[uidlarray.size()];
		uidls = uidlarray.toArray(uidls);

		return uidls;
	}

	public IObjectIndex sendMail(IAccountData arg0, IMailDTO arg1) throws NoSuchRights, ServerDown, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support sendMail.");
	}

	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oidx) throws Exception
	{
		deleteMail(acc, oidx);
	}

	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getHeaders(acc,dest);
	}

	public IRecordObject getFolderIndexFromString(String arg0) throws IncorrectData
	{
		return (IRecordObject)(new Pop3FolderIndex());
	}

	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getMailFolders(acc,dest);
	}

	public String getName() throws Exception
	{
		return storeName;
	}

	public ISearchDestination getPresetDestination(IAccountData arg0, int arg2) throws IncorrectData, HandleException
	{
		BasicSearchDestination dst = new BasicSearchDestination(null,new Pop3FolderIndex());
		return dst;
	}

	@SuppressWarnings("unchecked")
	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		Pop3ServiceInfo info = new Pop3ServiceInfo(storeID, storeName);
		String serverAddress = (String) h.get("server");
		if (serverAddress != null) info.setServerAddress(serverAddress);
		return info;
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oidx) throws Exception
	{
		return getMail(acc, dest, oidx);
	}

	public IRecordObject[] search(IAccountData arg0, ISearchFormula arg1) throws Exception
	{
		throw new HandleException(storeName+" does not support search.");
	}

	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date date) throws Exception
	{
		return getHeadersByDate(acc,dest,date);
	}

	public IObjectIndex write(IAccountData arg0, ISearchDestination arg1, IRecordObject arg2) throws Exception
	{
		throw new HandleException(storeName+" does not support write.");
	}

	protected FolderMetadata populateFolder(IFolderIndex idx)
	{
		FolderMetadata folder = new FolderMetadata();
		folder.setOid(idx);
		HashMap<String,Object> h = new HashMap<String,Object>();
		h.put("oid", idx);
		folder.setFieldSet(h);
		folder.setFolderName(idx.getIndex().toString());
		return folder;
	}

	@Override
	public void deleteMail(IAccountData acc, IObjectIndex oid) throws ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		this.deleteMail(null,acc,oid);
	}

	@Override
	public IMailDTO[] getHeaders(IAccountData acc, ISearchDestination dest) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		return this.getHeaders(null, acc, dest);
	}

	@Override
	public IMailDTO[] getHeadersByDate(IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		return this.getHeadersByDate(null, acc, dest, date);
	}

	@Override
	public IMailDTO getMail(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		return this.getMail(null,acc, dest, oid);
	}

	@Override
	public IMailDTO[] getMailsByDate(IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		return getMailsByDate(null, acc, dest, date);
	}

	
	protected HashMap<String, IMailDTO> fillTopListByRange(int start,int end,Vector<UidlElement> uidlList,HashMap<String,IMailDTO> topList) throws HandleException
	{
		//open mode only.
		if (topList==null) {
			topList = new HashMap<String, IMailDTO>();
		}
		if (start>=uidlList.size()) {
			return topList;
		}
		if (end<start) {
			return topList;
		}
		end = end + 1;//List.subList returns element[start] to element[end-1]
		if (end>=uidlList.size()) {
			end = uidlList.size();
		}
		List<UidlElement> sub = uidlList.subList(start, end);
		
		for (UidlElement uidl : sub) {
			IMailDTO top = topList.get(uidl.getUidl());
			if (top!=null) continue;
			String topStr = getTopByUidlWithoutOpen(uidlList, uidl.getUidl());
			topList.put(uidl.getUidl(), Pop3Translator.TopString2MailDTO(topStr, uidl.getUidl()));
		}
		
		return topList;
	}
	
	public HashMap<String, IMailDTO> fillTopListByRangeR(int start,int end,Vector<UidlElement> uidlList,HashMap<String,IMailDTO> topList) throws HandleException
	{
		checkUidlListNotEmpty(uidlList);

		int s = reverseIndex(end, uidlList.size());
		int e = reverseIndex(start, uidlList.size());
		
		return fillTopListByRange(s,e,uidlList,topList);
	}
}
