/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.pop3.socket;

import java.io.*;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.SocketTimeoutException;
import java.util.Properties;
import jp.co.connectone.common.PropertyHandlerBaseImpl;
import jp.co.connectone.eai.pop3.exception.ServerCommunicationException;
import jp.co.connectone.exception.HandleException;
import jp.co.connectone.exception.IncorrectData;
import jp.co.connectone.exception.ServerDown;
import jp.co.connectone.log.Log;

public abstract class BaseSocketImpl
{
	protected Socket clientSocket=null;
	protected BufferedReader reader=null;
	protected BufferedWriter writer=null;
	protected int defaultPort=0;
	protected int connection_timeout = 3*1000;
	protected int transaction_timeout = 30*1000;
	private Properties prop=null;

	public static BaseSocketImpl getInstance() throws HandleException
	{
		throw new HandleException("don't call BaseSocketImpl.getInstance() directly. use sub-class.");
	}

	public void connect(String server1,String server2) throws HandleException
	{
		connect(server1,server2,defaultPort);
	}

	public void connect(String server1,String server2, int port) throws HandleException
	{
		if (server1==null) {
			throw new IncorrectData("server must not be null.");
		}
		if (port<=0) {
			throw new IncorrectData("port number must be greater than 0.");
		}
		if (prop==null) {
			try {
				prop = PropertyHandlerBaseImpl.getInstance().getProperties("jp.co.connectone.eai.pop3.socket.BaseSocketImpl");
				connection_timeout = Integer.parseInt(prop.getProperty("connection_timeout_sec","3")) * 1000;
				transaction_timeout = Integer.parseInt(prop.getProperty("connection_timeout_sec","30")) * 1000;
			}
			catch (Exception e) {
				Log.error("",e);
			}
		}

		try {
			clientSocket = new Socket();
			SocketAddress addr = new InetSocketAddress(server1, port);
			clientSocket.connect(addr , connection_timeout);
			clientSocket.setSoTimeout(transaction_timeout);
		    reader=new BufferedReader(new InputStreamReader(clientSocket.getInputStream()));
		    writer=new BufferedWriter(new OutputStreamWriter(clientSocket.getOutputStream()));
		}
		catch (IOException ioe) {
			Log.debug(server1+" connection failed.("+ioe.getMessage()+") try server2...", ioe);
			if (server2!=null && !"".equals(server2)) {
				try {
					clientSocket = new Socket();
					SocketAddress addr = new InetSocketAddress(server2, port);
					clientSocket.connect(addr , connection_timeout);
					clientSocket.setSoTimeout(transaction_timeout);
				    reader=new BufferedReader(new InputStreamReader(clientSocket.getInputStream()));
				    writer=new BufferedWriter(new OutputStreamWriter(clientSocket.getOutputStream()));
				}
				catch (IOException ioe2) {
					Log.error(server2+" connection failed.("+ioe2.getMessage()+")", ioe2);
					disconnect();
					throw new ServerDown(server2+" connection failed.("+ioe2.getMessage()+")");
				}
			} else {
				disconnect();
				throw new ServerDown(server1+" connection failed.("+ioe.getMessage()+")");
			}
		}

	}

	public void disconnect()
	{
		if (clientSocket==null) return;
		try {
			if (clientSocket.isClosed() == false) {
				clientSocket.close();// this also closes Input/OutputStream.
			}
		}
		catch (IOException e) {
			Log.error("error on Socket close.", e);
		}
		finally {
			clientSocket = null;
			reader = null;
			writer = null;
		}
	}

	public boolean isClosed()
	{
		return (clientSocket == null || clientSocket.isClosed());
	}

	protected String transaction(ISocketCommand cmd) throws IOException
	{
		return transaction(cmd, false);
	}

	protected String transaction(ISocketCommand cmd, boolean isMultiLine) throws IOException
	{
		String cmdStr = cmd.getCmdWithParams() + "\r\n";
		Log.trace("c1:" + cmdStr);
		try {
			writer.write(cmdStr);
			writer.flush();
		}
		catch (SocketTimeoutException e) {
			Log.debug("Socket.writing, connection timeout.", e);
			throw e;
		}
		catch (IOException e) {
			Log.debug("error on Socket.writing.", e);
			throw e;
		}
		try {
			String line = reader.readLine();
			if (line == null) {// Is EOF
				Log.debug("readLine returns null.");
				throw new ServerCommunicationException("POP connection is not alive.");
			}
			if (line.startsWith("+OK")) {
				if (isMultiLine) {
					Log.trace("multi line.");
					line = getMultiLine(line);
				}
				else {
					Log.trace("single line.");
				}
			}
			Log.trace("pop3:" + line);
			return line;
		}
		catch (SocketTimeoutException e) {
			Log.debug("Socket.reading, connection timeout.", e);
			throw e;
		}
		catch (IOException e) {
			Log.debug("error on Socket.reading.", e);
			throw e;
		}
	}

	protected abstract boolean isTerminate(String line);

	private String getMultiLine(String line) throws IOException
	{
		for (;;) {
			String addline = reader.readLine();
			if (addline == null) {// Is EOF
				Log.debug("readLine returns null.");
				throw new ServerCommunicationException("POP connection is not alive.");
			}
			if (isTerminate(addline)) break;
			line = line + "\n" + addline;
		}
		return line;
	}

	@Override
	protected void finalize() throws Throwable
	{
		disconnect();
	}
}
