/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.exchews.store;

import java.util.*;

import javax.net.ssl.HttpsURLConnection;
import javax.xml.bind.JAXBElement;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Holder;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.handler.MessageContext;

import com.microsoft.schemas.exchange.services._2006.messages.*;
import com.microsoft.schemas.exchange.services._2006.types.*;

import jp.co.connectone.eai.exchews.ews.*;
import jp.co.connectone.eai.exchews.ssl.NoSecurityHostnameVerifier;
import jp.co.connectone.eai.exchews.ssl.NoSecuritySSLSocketFactory;
import jp.co.connectone.eai.exchews.user.ExchewsAccountData;
import jp.co.connectone.eai.exchews.util.ExchEWSTranslator;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.AttachmentDTO;
import jp.co.connectone.user.*;

public abstract class EAIExchewsBase
{
	protected static final com.microsoft.schemas.exchange.services._2006.types.ObjectFactory typesObjectFactory = new com.microsoft.schemas.exchange.services._2006.types.ObjectFactory();

	protected abstract UnindexedFieldURIType getDateField();
	protected boolean includeMimeContent=true;
	protected static final DistinguishedFolderIdNameType rootElement = DistinguishedFolderIdNameType.MSGFOLDERROOT;
	protected static final DistinguishedFolderIdNameType inboxElement = DistinguishedFolderIdNameType.INBOX;
	protected static final DistinguishedFolderIdNameType shceduleElement = DistinguishedFolderIdNameType.CALENDAR;
	protected static final DistinguishedFolderIdNameType addressbookElement = DistinguishedFolderIdNameType.CONTACTS;
	protected static final DistinguishedFolderIdNameType deletedItemsElement = DistinguishedFolderIdNameType.DELETEDITEMS;
	protected static final DistinguishedFolderIdNameType draftElement = DistinguishedFolderIdNameType.DRAFTS;
	protected static final DistinguishedFolderIdNameType memoElement = DistinguishedFolderIdNameType.NOTES;
	protected static final DistinguishedFolderIdNameType sentboxElement = DistinguishedFolderIdNameType.SENTITEMS;
	protected static final DistinguishedFolderIdNameType taskElement = DistinguishedFolderIdNameType.TASKS;
	protected static final DistinguishedFolderIdNameType jumkElement = DistinguishedFolderIdNameType.JUNKEMAIL;

	protected static final String HTTP_PROTOCOL_PREFIX = "http://";
	protected static final String SSL_PROTOCOL_PREFIX = "https://";
	protected static final String DEFAULT_PROTOCOL_PREFIX = SSL_PROTOCOL_PREFIX;
	protected static final RequestServerVersion version;
	protected static final TimeZoneContextType tzct;

	protected String protocolPrefix = DEFAULT_PROTOCOL_PREFIX;
	protected ServerVersionInfo serverVersion = new ServerVersionInfo();
	protected Holder<ServerVersionInfo> serverVersionHolder = new Holder<ServerVersionInfo>(serverVersion);
	protected CalendarItemCreateOrDeleteOperationType sendMailPolicy=null;

	protected IObjectIndex idToSee=null;

	static {
		version = new RequestServerVersion();
		version.setVersion(ExchangeVersionType.EXCHANGE_2007_SP_1);
        HttpsURLConnection.setDefaultSSLSocketFactory(new NoSecuritySSLSocketFactory());
        HttpsURLConnection.setDefaultHostnameVerifier(new NoSecurityHostnameVerifier());
		tzct = new TimeZoneContextType();
		TimeZoneDefinitionType tz = new TimeZoneDefinitionType();
		tz.setId("Tokyo Standard Time");
		tzct.setTimeZoneDefinition(tz);
	}

	protected void setEndPointUri(ExchewsServiceInfo si) throws ServerDown, HandleException
	{
		this.endPointUri = si.getEndPointUri();
	}

	protected boolean sendSavedMsg(ItemIdType id,DistinguishedFolderIdType folder) throws HandleException, ServerDown, DataNotFound
	{
		boolean rc = false;
		SendItemType siSendItem = new SendItemType();
		NonEmptyArrayOfBaseItemIdsType ids = new NonEmptyArrayOfBaseItemIdsType();
		ids.getItemIdOrOccurrenceItemIdOrRecurringMasterItemId().add(id);
		siSendItem.setItemIds(ids );
		TargetFolderIdType target = new TargetFolderIdType();
		target.setDistinguishedFolderId(folder);
		siSendItem.setSavedItemFolderId(target);
		siSendItem.setSaveItemToFolder(true);

		Holder<SendItemResponseType> responseHolder = new Holder<SendItemResponseType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.sendItem(siSendItem, version, responseHolder, serverVersionHolder);

			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			SendItemResponseType siRes = responseHolder.value;
			responses = siRes.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for ( JAXBElement<? extends ResponseMessageType> b : responses) {
				ResponseMessageType response = b.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.warn("Send Item Response Error: " + response.getMessageText());
					throw new IncorrectData(response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.warn("Send Item Response Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("Send Item SUCCESS.");
					rc = true;
				}

			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (IncorrectData icd) {
			throw icd;
		}
		catch (Exception e) {
			throw new RuntimeException("", e);
		}

		return rc;
	}

	protected List<? extends ItemType> searchByDate(ISearchDestination dest, List<String> itemList, SearchConditionCollection conds, Date date, String itemType) throws NullPointerException, ServerDown, HandleException
	{
		NonEmptyArrayOfPathsToElementType props = ExchEWSTranslator.createAdditionalProperties(UnindexedFieldURIType.ITEM_ITEM_CLASS);
		ItemResponseShapeType irst = new ItemResponseShapeType();
		irst.setAdditionalProperties(props);
		irst.setBaseShape(DefaultShapeNamesType.ALL_PROPERTIES);

		EWSSearchConditionCollection col = (EWSSearchConditionCollection)getDateSearchCondition(getDateField(),date);
		SearchExpressionType at = col.generateSearchExpression();

		// find the items
		FindItemType fit = new FindItemType();
		fit.setItemShape(irst);
		NonEmptyArrayOfBaseFolderIdsType folderIds = getEWSParamFolderIDs(dest);
		fit.setParentFolderIds(folderIds);
		fit.setTraversal(ItemQueryTraversalType.SHALLOW);

		if ("IPM.Appointment".equals(itemType)) {
			CalendarViewType calView = ExchEWSTranslator.createCalendarViewType(date);
			fit.setCalendarView(calView);
		}
		else {
			RestrictionType rt = ExchEWSTranslator.createRestriction(at);
			fit.setRestriction(rt);
		}

		FindItemResponseType findItemResponse = new FindItemResponseType();
		Holder<FindItemResponseType> responseHolder = new Holder<FindItemResponseType>(findItemResponse);
		List<ItemType> messages = new ArrayList<ItemType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.findItem(fit, version, tzct, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			responses = responseHolder.value.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for (JAXBElement<? extends ResponseMessageType> jaxResponse : responses) {
				ResponseMessageType response = jaxResponse.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					processResponseErrorCode(response);
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("findItem SUCCESS.");
					FindItemResponseMessageType findResponse = (FindItemResponseMessageType) response;
					for (ItemType item : findResponse.getRootFolder().getItems().getItemOrMessageOrCalendarItem()) {
						messages.add(item);
					}
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (NoSuchRights ne) {
			throw ne;
		}
		catch (Exception e) {
			throw new RuntimeException("", e);
		}
		return messages;
	}

	private void processResponseErrorCode(ResponseMessageType response) throws ServerDown, NoSuchRights, HandleException
	{
		String code = response.getResponseCode();
		String message = response.getMessageText();
		Log.trace("ResponseCode:"+response);
		if (code.indexOf("ErrorFolderNotFound")>=0 ||
				code.indexOf("ErrorMissingEmailAddress")>=0 ||
				code.indexOf("ErrorNonExistentMailbox")>=0 ||
				code.indexOf("ErrorAccessDenied")>=0) {
			throw new NoSuchRights(code);
		}
		throw new HandleException("Exchange Server returns Error: " + message + "(" + code + ")");
	}

	private void processWebServiceException(WebServiceException wse) throws ServerDown, NoSuchRights
	{
		String msg = wse.getMessage();
		Log.trace("WebServiceException:"+msg);
		msg=wse.getLocalizedMessage();
		Log.trace(msg);
		Log.trace("class:"+wse.getClass());
		if (msg.indexOf("Authentication failure")>=0 || (msg.indexOf("401")>=0 && msg.indexOf("Unauthorized")>=0) || msg.indexOf("ErrorFolderNotFound")>=0) {
			throw new NoSuchRights(msg);
		}
		else if (msg.indexOf("timed out")>=0) {
			throw new ServerDown(msg);
		}
		else {
			Throwable errCause = wse;
			for (int i = 0; i < 4; i++) {
				errCause = errCause.getCause();
				if (errCause==null)break;

				msg = errCause.getMessage();
				Log.trace("WebServiceException:"+msg);
				msg=errCause.getLocalizedMessage();
				Log.trace(msg);
				Log.trace("class:"+errCause.getClass());

				if (msg.indexOf("Authentication failure")>=0 || (msg.indexOf("401")>=0 && msg.indexOf("Unauthorized")>=0) || msg.indexOf("ErrorFolderNotFound")>=0) {
					throw new NoSuchRights(msg);
				}
				else if (msg.indexOf("timed out")>=0) {
					throw new ServerDown(msg);
				}
			}
			Log.error("error on processing EWS request", wse);
			throw wse;
		}
	}
	protected List<? extends ItemType> search(ISearchDestination dest, List<String> itemList, SearchConditionCollection conds) throws NullPointerException, ServerDown, HandleException
	{
		// find the items
		FindItemType fit = new FindItemType();

		NonEmptyArrayOfPathsToElementType props = ExchEWSTranslator.createAdditionalProperties(UnindexedFieldURIType.ITEM_ITEM_CLASS);
		ItemResponseShapeType irst = new ItemResponseShapeType();
		irst.setAdditionalProperties(props);
		irst.setBaseShape(DefaultShapeNamesType.ALL_PROPERTIES);

		EWSSearchConditionCollection col = (EWSSearchConditionCollection)conds;
		SearchExpressionType at = col.generateSearchExpression();

		if (at != null) {
			RestrictionType rt = ExchEWSTranslator.createRestriction(at);
			fit.setRestriction(rt);
		}

		fit.setItemShape(irst);
		NonEmptyArrayOfBaseFolderIdsType folderIds = getEWSParamFolderIDs(dest);
		fit.setParentFolderIds(folderIds);
		fit.setTraversal(ItemQueryTraversalType.SHALLOW);

		FindItemResponseType findItemResponse = new FindItemResponseType();
		Holder<FindItemResponseType> responseHolder = new Holder<FindItemResponseType>(findItemResponse);
		List<ItemType> messages = new ArrayList<ItemType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.findItem(fit, version, tzct, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			responses = responseHolder.value.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for (JAXBElement<? extends ResponseMessageType> jaxResponse : responses) {
				ResponseMessageType response = jaxResponse.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("findItem SUCCESS.");
					FindItemResponseMessageType findResponse = (FindItemResponseMessageType) response;
					for (ItemType item : findResponse.getRootFolder().getItems().getItemOrMessageOrCalendarItem()) {
						messages.add(item);
					}
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (Exception e) {
			throw new RuntimeException("", e);
		}
		return messages;
	}

	private void setEWSAuthParams(ExchangeServicePortType proxy)
	{
		BindingProvider bp = (BindingProvider) proxy;
		Map<String, Object> map = bp.getRequestContext();
		map.put(BindingProvider.USERNAME_PROPERTY, userID+"@"+domain);
		map.put(BindingProvider.PASSWORD_PROPERTY, password);
		map.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, endPointUri);
		map.put(MessageContext.HTTP_REQUEST_HEADERS,Collections.singletonMap("X-connectone-Version",Collections.singletonList("3.0-RC")));
	}

	private NonEmptyArrayOfBaseFolderIdsType getEWSParamFolderIDs(ISearchDestination dest)
	{
		BaseFolderIdType id = (BaseFolderIdType) dest.getFolder().getIndex();
		if (idToSee!=null) {
			if (id instanceof DistinguishedFolderIdType) {
				EmailAddressType addr=new EmailAddressType();
				addr.setEmailAddress(idToSee.getIndex().toString());
				((DistinguishedFolderIdType)id).setMailbox(addr);
			}
			idToSee=null;
		}
		NonEmptyArrayOfBaseFolderIdsType folderIds = ExchEWSTranslator.createFolderIds(id);
		return folderIds;
	}

	private List<BaseFolderType> getFirstChildlen(BaseFolderIdType parentFolder) throws ServerDown, NoSuchRights
	{
//		System.setProperty("javax.xml.transform.TransformerFactory","myTransformerFactory");
//		TransformerFactory trfactory = TransformerFactory.newInstance();
//		Transformer tr = null;
//		try {
//			tr = trfactory.newTransformer();
//		}
//		catch (TransformerConfigurationException e1) {
//			e1.printStackTrace();
//		}

		FindFolderType finder = new FindFolderType();

		FolderResponseShapeType folderShape = new FolderResponseShapeType();
		folderShape.setBaseShape(DefaultShapeNamesType.ALL_PROPERTIES);

		finder.setFolderShape(folderShape);
		finder.setTraversal(FolderQueryTraversalType.SHALLOW);

		IndexedPageViewType index = ExchEWSTranslator.createEWSParamPaging(0,128);

		finder.setIndexedPageFolderView(index);

		NonEmptyArrayOfBaseFolderIdsType folderIds = ExchEWSTranslator.createFolderIds(parentFolder);
		finder.setParentFolderIds(folderIds);

		FindFolderResponseType findFolderResponse = new FindFolderResponseType();
		Holder<FindFolderResponseType> responseHolder = new Holder<FindFolderResponseType>(findFolderResponse);

		ExchangeServicePortType proxy = null;
		List<BaseFolderType> mailFolders = new ArrayList<BaseFolderType>();
		List<JAXBElement<? extends ResponseMessageType>> responses = null;

		try {
			ExchangeServices service = new ExchangeServices();
			proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.findFolder(finder, version, tzct, responseHolder, serverVersionHolder);

			responses = responseHolder.value.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for (JAXBElement<? extends ResponseMessageType> jaxResponse : responses) {
				ResponseMessageType response = jaxResponse.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.warn("Get Child Folders Response Error: " + response.getMessageText());
					processResponseErrorCode(response);
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.warn("Get Child Folders Response Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("findFolder SUCCESS.");
					FindFolderResponseMessageType findResponse = (FindFolderResponseMessageType) response;
					List<BaseFolderType> allFolders = findResponse.getRootFolder().getFolders().getFolderOrCalendarFolderOrContactsFolder();
					for (BaseFolderType folder : allFolders) {
						Log.debug("folder name:" + folder.getDisplayName());
						Log.debug("number of child folders" + folder.getChildFolderCount());
						mailFolders.add(folder);
					}
				}
			}

		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (NoSuchRights ne) {
			throw ne;
		}
		catch (Exception e) {
			Log.error("Exception on getFirstChildlen.", e);
			throw new RuntimeException("Exception Performing getFirstChildlen", e);
		}
		return mailFolders;
	}

	protected List<BaseFolderType> getAllSubFolders(BaseFolderIdType parentFolder,int layer) throws ServerDown, HandleException
	{
		List<BaseFolderType> folders = new ArrayList<BaseFolderType>();
		List<BaseFolderType> thisFolder = null;
		Log.debug("getAllSubFolders:enter " + parentFolder);

		String layerStr = "-";
		for (int i = 0; i < layer; i++) {
			layerStr = layerStr + "-";
		}

		try {
			thisFolder = getFirstChildlen(parentFolder);
		}
		catch (HandleException he) {
			throw he;
		}
		catch (Exception e) {
			Log.debug("getAllSubFolders:could not get subfolder of " + parentFolder);
			return thisFolder;
		}
		for (BaseFolderType folder : thisFolder) {
			folder.setDisplayName(layerStr+folder.getDisplayName());
			folders.add(folder);
			Log.debug("name:" + folder.getDisplayName());
			if (folder.getChildFolderCount().intValue() > 0) {
				folders.addAll(getAllSubFolders(folder.getFolderId(),layer+1));
			}
		}

		return folders;
	}

	protected List<? extends ItemType> getListByDate(ISearchDestination dest, List<String> itemList, Date date) throws NullPointerException, ServerDown, HandleException
	{
		if (itemList == null) return null;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (endPointUri == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		NonEmptyArrayOfPathsToElementType props = ExchEWSTranslator.createAdditionalProperties(UnindexedFieldURIType.ITEM_ITEM_CLASS);
		ItemResponseShapeType irst = new ItemResponseShapeType();
		irst.setAdditionalProperties(props);
		irst.setBaseShape(DefaultShapeNamesType.ALL_PROPERTIES);
		irst.setIncludeMimeContent(this.includeMimeContent);

		EWSSearchConditionCollection col = (EWSSearchConditionCollection)getDateSearchCondition(getDateField(),date);
		SearchExpressionType at = col.generateSearchExpression();

		RestrictionType rt = ExchEWSTranslator.createRestriction(at);

		// find the items
		FindItemType fit = new FindItemType();
		fit.setItemShape(irst);
		NonEmptyArrayOfBaseFolderIdsType folderIds = getEWSParamFolderIDs(dest);
		fit.setParentFolderIds(folderIds);
		fit.setTraversal(ItemQueryTraversalType.SHALLOW);
		fit.setRestriction(rt);

		FindItemResponseType findItemResponse = new FindItemResponseType();
		Holder<FindItemResponseType> responseHolder = new Holder<FindItemResponseType>(findItemResponse);
		List<ItemType> messages = new ArrayList<ItemType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.findItem(fit, version, tzct, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			responses = responseHolder.value.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for (JAXBElement<? extends ResponseMessageType> jaxResponse : responses) {
				ResponseMessageType response = jaxResponse.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("findItem SUCCESS.");
					FindItemResponseMessageType findResponse = (FindItemResponseMessageType) response;
					for (ItemType item : findResponse.getRootFolder().getItems().getItemOrMessageOrCalendarItem()) {
						messages.add(item);
					}
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (Exception e) {
			Log.error("Exception on getListByDate.", e);
			throw new RuntimeException("Exception Performing getListByDate", e);
		}
		return messages;
	}

	protected List<? extends ItemType> getList(ISearchDestination dest, List<String> itemList) throws NullPointerException, ServerDown, HandleException
	{
		if (itemList == null) return null;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (endPointUri == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		NonEmptyArrayOfPathsToElementType props = ExchEWSTranslator.createAdditionalProperties(UnindexedFieldURIType.ITEM_ITEM_CLASS);
		ItemResponseShapeType irst = new ItemResponseShapeType();
		irst.setAdditionalProperties(props);
		irst.setBaseShape(DefaultShapeNamesType.ALL_PROPERTIES);

		// find the items
		FindItemType fit = new FindItemType();
		fit.setItemShape(irst);
		NonEmptyArrayOfBaseFolderIdsType folderIds = getEWSParamFolderIDs(dest);
		fit.setParentFolderIds(folderIds);
		fit.setTraversal(ItemQueryTraversalType.SHALLOW);

		FindItemResponseType findItemResponse = new FindItemResponseType();
		Holder<FindItemResponseType> responseHolder = new Holder<FindItemResponseType>(findItemResponse);
		List<ItemType> messages = new ArrayList<ItemType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.findItem(fit, version, tzct, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			responses = responseHolder.value.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for (JAXBElement<? extends ResponseMessageType> jaxResponse : responses) {
				ResponseMessageType response = jaxResponse.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					processResponseErrorCode(response);
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("findItem SUCCESS.");
					FindItemResponseMessageType findResponse = (FindItemResponseMessageType) response;
					for (ItemType item : findResponse.getRootFolder().getItems().getItemOrMessageOrCalendarItem()) {
						messages.add(item);
					}
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (NoSuchRights ne) {
			throw ne;
		}
		catch (Exception e) {
			Log.error("Exception on getList.", e);
			throw new RuntimeException("Exception Performing getList", e);
		}
		return messages;
	}

	protected List<AttachmentType> getAttachments(ItemType msg) throws ServerDown, NoSuchRights
	{
		if (msg==null) return null;
		List<AttachmentType> rc = new ArrayList<AttachmentType>();
		if (msg.isHasAttachments()) {
			if (msg.getAttachments()==null) {
				// in case findItem Response
				AttachmentType dummy = new FileAttachmentType();
				dummy.setName("dummy attachment");
				rc.add(dummy);
				return rc;
			}
		}

		GetAttachmentType req = new GetAttachmentType();
		req.setAttachmentIds(ExchEWSTranslator.createAttachIdList(msg.getAttachments().getItemAttachmentOrFileAttachment()));

		AttachmentResponseShapeType arst = new AttachmentResponseShapeType();
		arst.setIncludeMimeContent(this.includeMimeContent);
		req.setAttachmentShape(arst);


		Holder<GetAttachmentResponseType> responseHolder = new Holder<GetAttachmentResponseType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.getAttachment(req, version, tzct, responseHolder, serverVersionHolder);

			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			GetAttachmentResponseType atRes = responseHolder.value;
			responses = atRes.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for (JAXBElement<? extends ResponseMessageType> jaxResponse : responses) {
				ResponseMessageType response = jaxResponse.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("getAttachment SUCCESS.");
					AttachmentInfoResponseMessageType attResponse = (AttachmentInfoResponseMessageType) response;
					rc.addAll(attResponse.getAttachments().getItemAttachmentOrFileAttachment());
				}
			}

		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (Exception e) {
			Log.error("Exception on getAttachments.", e);
			throw new RuntimeException("Exception Performing getAttachments", e);
		}


		return rc;
	}

	protected ItemIdType deleteItem(ItemIdType iit) throws ServerDown, NoSuchRights, IncorrectData
	{
		DeleteItemType dit = new DeleteItemType();
		NonEmptyArrayOfBaseItemIdsType value = new NonEmptyArrayOfBaseItemIdsType();
		value.getItemIdOrOccurrenceItemIdOrRecurringMasterItemId().add(iit);
		dit.setItemIds(value);
		dit.setDeleteType(DisposalType.MOVE_TO_DELETED_ITEMS);
		dit.setAffectedTaskOccurrences(AffectedTaskOccurrencesType.ALL_OCCURRENCES);

		dit.setSendMeetingCancellations(sendMailPolicy);

		Holder<DeleteItemResponseType> responseHolder = new Holder<DeleteItemResponseType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.deleteItem(dit, version, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			DeleteItemResponseType diRes = responseHolder.value;
			responses = diRes.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for ( JAXBElement<? extends ResponseMessageType> b : responses) {
				ResponseMessageType response = b.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.warn("Delete Item Response Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.warn("Delete Item Response Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("deleteItem SUCCESS.");
					return iit;
				}

			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (IncorrectData icd) {
			throw icd;
		}
		catch (Exception e) {
			Log.error("Exception on deleteItem.", e);
			throw new RuntimeException("Exception Performing deleteItem", e);
		}
		return null;

	}

	protected void updateItem(ItemIdType id,UpdateItemMap itemMap) throws ServerDown, NoSuchRights
	{
		UpdateItemType uit = new UpdateItemType();
		uit.setConflictResolution(ConflictResolutionType.AUTO_RESOLVE);
		uit.setMessageDisposition(MessageDispositionType.SAVE_ONLY);
//		uit.setSendMeetingInvitationsOrCancellations(CalendarItemUpdateOperationType.SEND_TO_NONE);
//		uit.setSavedItemFolderId(getTargetFolder());
		uit.setItemChanges(buildItemChamgeArray(id,itemMap));
		try {
			setAdditionalRequestProperties(uit);
		}
		catch (HandleException e) {
			Log.error("failed to set Additional Request Properties.", e);
		}

		Holder<UpdateItemResponseType> responseHolder = new Holder<UpdateItemResponseType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.updateItem(uit, version, tzct, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			UpdateItemResponseType uiRes = responseHolder.value;
			responses = uiRes.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for ( JAXBElement<? extends ResponseMessageType> b : responses) {
				ResponseMessageType response = b.getValue();
				if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("updateItem SUCCESS.");
					ItemInfoResponseMessageType mes = (ItemInfoResponseMessageType)response;
					List<ItemType> items = mes.getItems().getItemOrMessageOrCalendarItem();
					Log.trace(items.get(0).toString());
				}
				else if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (Exception e) {
			Log.error("Exception on updateItem.", e);
			throw new RuntimeException("Exception Performing updateItem", e);
		}
		return ;
	}

	protected abstract TargetFolderIdType getTargetFolder();

	protected NonEmptyArrayOfItemChangesType buildItemChamgeArray(ItemIdType id,UpdateItemMap fields)
	{
		NonEmptyArrayOfItemChangesType chgs = new NonEmptyArrayOfItemChangesType();
		List<ItemChangeType> icl = chgs.getItemChange();
		ItemChangeType ict = new ItemChangeType();
		ict.setItemId(id);

		NonEmptyArrayOfItemChangeDescriptionsType upd = new NonEmptyArrayOfItemChangeDescriptionsType();
		List<ItemChangeDescriptionType> list = upd.getAppendToItemFieldOrSetItemFieldOrDeleteItemField();

		for (UnindexedFieldURIType uri : fields.keySet()) {
			ItemType item = fields.get(uri);
			list.add(generateSetItemField(uri,item));
		}

		ict.setUpdates(upd);

		icl.add(ict);

		return chgs;
	}

	protected SetItemFieldType generateSetItemField(UnindexedFieldURIType ftype,ItemType msg)
	{
		PathToUnindexedFieldType ufut = new PathToUnindexedFieldType();
		ufut.setFieldURI(ftype);
		JAXBElement<? extends BasePathToElementType> elem = typesObjectFactory.createFieldURI(ufut);

		SetItemFieldType sift = new SetItemFieldType();
		sift.setPath(elem);
		sift.setItem(msg);
		return sift;
	}

	protected void setAdditionalRequestProperties(BaseRequestType item) throws HandleException
	{
		// do nothing
	}

	protected void addAttach(ItemIdType itemId, AttachmentDTO[] src) throws ServerDown, NoSuchRights
	{
		NonEmptyArrayOfAttachmentsType atts = ExchEWSTranslator.createArrayOfAttachmentsType(src);
		CreateAttachmentType cat = new CreateAttachmentType();
		cat.setAttachments(atts);
		cat.setParentItemId(itemId);

		Holder<CreateAttachmentResponseType> responseHolder = new Holder<CreateAttachmentResponseType>();
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.createAttachment(cat, version, tzct, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			CreateAttachmentResponseType caRes = responseHolder.value;
			responses = caRes.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for ( JAXBElement<? extends ResponseMessageType> b : responses) {
				AttachmentInfoResponseMessageType response = (AttachmentInfoResponseMessageType) b.getValue();
				if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("createAttachment SUCCESS.");

					String key = response.getAttachments().getItemAttachmentOrFileAttachment().get(0).getAttachmentId().getRootItemChangeKey();
					itemId.setChangeKey(key);
				}
				else if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (Exception e) {
			Log.error("Exception on createAttachment.", e);
			throw new RuntimeException("Exception Performing addAttach", e);
		}
	}

	protected ItemType createItem(DistinguishedFolderIdNameType dist, ItemType msg) throws ServerDown, NoSuchRights
	{
		CreateItemType cit = new CreateItemType();
		cit.setMessageDisposition(MessageDispositionType.SAVE_ONLY);

		TargetFolderIdType target = new TargetFolderIdType();
		DistinguishedFolderIdType id = new DistinguishedFolderIdType();
		id.setId(dist);
		target.setDistinguishedFolderId(id);
		cit.setSavedItemFolderId(target);
		NonEmptyArrayOfAllItemsType ar = new NonEmptyArrayOfAllItemsType();
		ar.getItemOrMessageOrCalendarItem().add(msg);
		cit.setItems(ar);
		try {
			setAdditionalRequestProperties(cit);
		}
		catch (HandleException e) {
			Log.error("failed to set Additional Request Properties.", e);
		}


		Holder<CreateItemResponseType> responseHolder = new Holder<CreateItemResponseType>();
		ItemType rc = null;
		try {
			ExchangeServices service = new ExchangeServices();
			ExchangeServicePortType proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.createItem(cit, version, tzct, responseHolder, serverVersionHolder);
			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			CreateItemResponseType ciRes = responseHolder.value;
			responses = ciRes.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();

			for ( JAXBElement<? extends ResponseMessageType> b : responses) {
				ResponseMessageType response = b.getValue();
				if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.debug("createItem SUCCESS.");
					ItemInfoResponseMessageType mes = (ItemInfoResponseMessageType)response;
					List<ItemType> items = mes.getItems().getItemOrMessageOrCalendarItem();
					rc = items.get(0);
				}
				else if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.debug("Exchange Server returns Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.debug("Exchange Server returns Warning: " + response.getMessageText());
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (Exception e) {
			Log.error("Exception on createItem.", e);
			throw new RuntimeException("Exception Performing createItem", e);
		}
		return rc;
	}

	protected ItemType getItem(ISearchDestination dest, ItemIdType iit) throws HandleException
	{
		ItemType rc = null;
        GetItemType git = new GetItemType();
		git.setItemIds(ExchEWSTranslator.createItemIdList(iit));

		ItemResponseShapeType irst = new ItemResponseShapeType();
		irst.setBaseShape(DefaultShapeNamesType.ALL_PROPERTIES);
		irst.setIncludeMimeContent(this.includeMimeContent);
		irst.setBodyType(BodyTypeResponseType.TEXT);
        git.setItemShape(irst);

		Holder<GetItemResponseType> responseHolder = new Holder<GetItemResponseType>();
		ExchangeServicePortType proxy = null;
		try {
			ExchangeServices service = new ExchangeServices();
			proxy = service.getExchangeServicePort();
			setEWSAuthParams(proxy);

			proxy.getItem(git, version, tzct, responseHolder, serverVersionHolder);

			List<JAXBElement<? extends ResponseMessageType>> responses = null;
			responses = responseHolder.value.getResponseMessages().getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage();
			for (JAXBElement<? extends ResponseMessageType> jaxResponse : responses) {
				ResponseMessageType response = jaxResponse.getValue();
				if (response.getResponseClass().equals(ResponseClassType.ERROR)) {
					Log.warn("Get Item Response Error: " + response.getMessageText());
					throw new HandleException("Exchange Server returns Error: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.WARNING)) {
					Log.warn("Get Item Response Warning: " + response.getMessageText());
				}
				else if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
					Log.info("getItem SUCCESS.");
					ItemInfoResponseMessageType res = (ItemInfoResponseMessageType)response;
					List<ItemType> items = res.getItems().getItemOrMessageOrCalendarItem();
					rc = items.get(0);
				}
			}
		}
		catch (WebServiceException wse) {
			processWebServiceException(wse);
		}
		catch (Exception e) {
			Log.error("Exception on getItem.", e);
			throw new RuntimeException("Exception Performing getItem", e);
		}

		return rc;
	}

	/*---- getter/setter --------*/
	protected String endPointUri;

	protected String path;

	protected String userID;

	protected String password;

	protected String domain;

	/**
	 * @return domain
	 */
	public String getDomain()
	{
		return domain;
	}

	/**
	 * @param domain
	 *            domain
	 */
	protected void setDomain(String domain)
	{
		this.domain = domain;
	}

	/**
	 * @return password
	 */
	public String getPassword()
	{
		return password;
	}

	/**
	 * @param password
	 *            password
	 */
	protected void setPassword(String password)
	{
		this.password = password;
	}

	/**
	 * @return path
	 */
	public String getPath()
	{
		return path;
	}

	/**
	 * @param path
	 *            path
	 */
	protected void setPath(String path)
	{
		this.path = path;
	}

	/**
	 * @return serverAddress
	 */
	public String getServerAddress()
	{
		return endPointUri;
	}

	/**
	 * @return userID
	 */
	public String getUserID()
	{
		return userID;
	}

	/**
	 * @param userID
	 *            userID
	 */
	protected void setUserID(String userID)
	{
		this.userID = userID;
	}

	protected void setAccountData(IAccountData acc) throws HandleException
	{
		ExchewsServiceInfo si = (ExchewsServiceInfo) acc.getServiceInfo();
		ExchewsAccountData eacc = (ExchewsAccountData) acc;
		setDomain(eacc.getDomain());
		setUserID(eacc.getUserID());
		setPassword(eacc.getPassword());
		setEndPointUri(si);
	}

	protected int delete(IObjectIndex oid) throws NullPointerException, ServerDown, NoSuchRights, IncorrectData
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (endPointUri == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		int code = 0;

		deleteItem((ItemIdType)oid.getIndex());

		return code;
	}

	public IRecordObject getFolderIndexFromString(String folderName) throws IncorrectData
	{
		BasicRecordObject rc = new BasicRecordObject();
		rc.setOid(new ExchewsFolderIndex(folderName));
		HashMap<String,Object> h = new HashMap<String,Object>();
		h.put("IFolderIndex", rc.getOid());
		try {
			rc.setFieldSet(h);
		}
		catch (IllegalAccessException e) {
			Log.error("getFolderIndexFromString:error on setting IFolderIndex", e);
			throw new IncorrectData("getFolderIndexFromString:error on setting IFolderIndex");
		}
		return rc;
	}

	protected FolderMetadata populateFolder(IFolderIndex idx)
	{
		FolderMetadata folder = new FolderMetadata();
		folder.setOid(idx);
		HashMap<String, Object> h = new HashMap<String, Object>();
		h.put("oid", idx);
		folder.setFieldSet(h);
		folder.setFolderName(idx.getIndex().toString());
		return folder;
	}

	protected ArrayList<FolderMetadata> populateFolderList(int layer, String root, Vector<?> src) throws HandleException
	{
		if (src == null) return null;
		if (src.size() == 0) return null;
		Enumeration<?> em = src.elements();
		ArrayList<FolderMetadata> dst = new ArrayList<FolderMetadata>();

		String path = null;
		String sub = null;
		String layerStr = "";
		for (int i = 0; i < layer; i++) {
			layerStr = layerStr + "-";
		}
		while (em.hasMoreElements()) {
			Object o = em.nextElement();
			try {
				Class<?> c = o.getClass();
				if (c.equals(Class.forName("java.lang.String"))) {
					Log.debug("raw data of foldername:" + (String) o);
					FolderMetadata folder = new FolderMetadata();
					path = (String) o;
					sub = path.substring(path.indexOf(root) + root.length());
					Log.debug("substring:" + sub);
					IFolderIndex oid = new ExchewsFolderIndex(path);
					folder.setOid(oid);
					Log.debug("oid:" + (String) oid.getIndex());
					HashMap<String,Object> h = new HashMap<String,Object>();
					h.put("oid", oid);
					folder.setFieldSet(h);
					Log.debug("layerStr:" + layerStr);
					folder.setFolderName(layerStr + sub.replaceAll("/", ""));
					Log.debug("folderName:" + folder.getFolderName());
					dst.add(folder);
				}
				else if (c.equals(Class.forName("java.util.Vector"))) {
					Vector<?> v = (Vector<?>) o;
					dst.addAll(populateFolderList(layer + 1, path, v));
				}
			}
			catch (ClassNotFoundException e) {
				throw new HandleException(e.getMessage());
			}
		}

		return dst;
	}

	public IServiceInfo getServiceInfo(IStoreID storeID, String storeName, IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		ExchewsServiceInfo info = new ExchewsServiceInfo(storeID, storeName);
		String serverAddress = (String) h.get("server");
		if (serverAddress != null) info.setServerAddress(serverAddress);
		String tmp = (String) h.get("serverAddress");
		if (tmp != null) info.setServerAddress(tmp);
		tmp = (String) h.get("protocolPrefix");
		if (tmp != null) info.setProtocolPrefix(tmp);
		return info;
	}

	public SearchConditionCollection getDateSearchCondition(UnindexedFieldURIType dateField,Date date) throws HandleException
	{
		EWS_AND_SearchConditionCollection c = new EWS_AND_SearchConditionCollection();
		EWSSearchCondition cond = new EWSSearchCondition(dateField, EWSSearchType.TYPE_DATE_0000, date, EWSSearchOperator.OP_BIG_EQUAL);
		c.add(cond);
		cond = new EWSSearchCondition(dateField, EWSSearchType.TYPE_DATE_2359, date, EWSSearchOperator.OP_SMALL_EQUAL);
		c.add(cond);

		return c;
	}
}
