/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.exch2k.store;

import java.util.*;

import jp.co.connectone.eai.exch2k.dav.DavSearchCondition;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;
import jp.co.connectone.user.PasswordBasedAccountDataImpl;

public class Exch2kTaskStoreImpl extends EAIExch2kBase implements ITaskStore
{
	private static final String DEFAULT_EXCHANGE_TASK_PATH_ELEMENT="d";
	private IFolderIndex taskElement = new Exch2kFolderIndex(DEFAULT_EXCHANGE_TASK_PATH_ELEMENT);
	protected static final String elmentName=DEFAULT_EXCHANGE_TASK_PATH_ELEMENT;

	public static final IStoreID storeID = new SimpleStoreID(Exch2kTaskStoreImpl.class.getName());
	public static final String storeName = "Exchange2K task store";
	public String getName() throws Exception
	{
		return storeName;
	}
	
	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		return super.getServiceInfo(storeID, storeName, serviceData);
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new Exch2kServiceInfo(storeID,storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}


	public ITaskDTO[] getAllTasks(IAccountData acc) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		ITaskDTO[] l_scheduleDtoArray = null;

		Vector<String> v = makeItemList();
		setAccountData((PasswordBasedAccountDataImpl)acc);
        Vector<HashMap<String,Object>> col = super.getList(taskElement,v);
        int len = col.size();
        l_scheduleDtoArray = new ITaskDTO[len];
        for (int i=0;i<len;i++) {
        	l_scheduleDtoArray[i] = populateTaskDTO(col.elementAt(i));
        }


		return l_scheduleDtoArray;
	}
	
	public ITaskDTO[] getTasksByDate(IAccountData acc, Date dateObj) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		ITaskDTO[] l_scheduleDtoArray = null;

		Vector<String> v = makeItemList();
		setAccountData((PasswordBasedAccountDataImpl)acc);
        Vector<HashMap<String,Object>> col = super.getListByDate(taskElement,v,dateObj,"urn:schemas:httpmail:datereceived");
        int len = col.size();
        l_scheduleDtoArray = new ITaskDTO[len];
        for (int i=0;i<len;i++) {
        	l_scheduleDtoArray[i] = populateTaskDTO(col.elementAt(i));
        }


		return l_scheduleDtoArray;
	}

	/**
	 * @return
	 */
	private Vector<String> makeItemList()
	{
		Vector<String> v = new Vector<String>();
		v.add("DAV:id");
		v.add("DAV:href");
		v.add("urn:schemas:httpmail:subject");
		v.add("urn:schemas:httpmail:textdescription");
		v.add("urn:schemas:httpmail:sendername");
		v.add("urn:schemas:httpmail:datereceived");
		v.add("urn:schemas:httpmail:importance");
		return v;
	}
	
	public ITaskDTO getTaskByOID(IAccountData acc, IObjectIndex oid) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, HandleException
	{
Log.debug("getTaskByOID::oid="+oid);
		setAccountData((PasswordBasedAccountDataImpl)acc);
		HashMap<String,Object> task = super.getProps((String)oid.getIndex(),makeItemList());
		return populateTaskDTO(task);
	}

	public IObjectIndex createNewTask(IAccountData acc, ITaskDTO ITaskDTO) throws IncorrectData, NoSuchRights, DataNotFound, IncorrectStore, ServerDown, HandleException
	{
		IObjectIndex oid = null;

		Hashtable<String,String> h = populateHashtable(ITaskDTO);
		setAccountData((PasswordBasedAccountDataImpl)acc);

		String url = generateBaseURL() + elmentName + "/" + new Date().getTime() +".EML";
		oid = super.saveData(escapeURL(url),h);

		return oid;
	}

	public IObjectIndex updateTask(IAccountData acc, ITaskDTO objITaskDTO) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, SearchConditionFailed, UpdateFailed,HandleException
	{
		IObjectIndex oid = null;
		Hashtable<String,String> h = populateHashtable(objITaskDTO);
		setAccountData((PasswordBasedAccountDataImpl)acc);

		String url = (String)objITaskDTO.getOid().getIndex();
		oid = super.saveData(url,h);

		return oid;

	}

	public void deleteTask(IAccountData acc, IObjectIndex oid) throws IncorrectData, DeleteFailed, NoSuchRights, ServerDown,DataNotFound, IncorrectStore, SearchConditionFailed,HandleException
	{
		setAccountData((PasswordBasedAccountDataImpl)acc);
		super.delete(oid);
	}


	private ITaskDTO populateTaskDTO(HashMap<String,Object> itemList) throws ServerDown, DataNotFound
	{
		ITaskDTO l_ITaskDTO = new BasicTaskDTO();
		l_ITaskDTO.setTid((String)itemList.get("id"));
		l_ITaskDTO.setOid(new Exch2kObjectIndex(itemList.get("href")));
		l_ITaskDTO.setSubject((String)itemList.get("subject"));
		l_ITaskDTO.setBody((String)itemList.get("textdescription"));
		l_ITaskDTO.setSenderName((String)itemList.get("sendername"));
		l_ITaskDTO.setDate(davDateStr2JavaDate((String)itemList.get("datereceived")));
		try {l_ITaskDTO.setPriority(Integer.parseInt((String)itemList.get("importance")));}catch(NumberFormatException e){};
		
		return l_ITaskDTO;
	}


	private Hashtable<String,String> populateHashtable(ITaskDTO task)
	{
		Hashtable<String,String> h = new Hashtable<String,String>();
Log.debug("task="+task);
		
		if (task.getSubject()!=null) h.put("urn:schemas:httpmail:subject",task.getSubject());
		if (task.getBody()!=null) h.put("urn:schemas:httpmail:textdescription",task.getBody());
		if (task.getSenderName()!=null) h.put("urn:schemas:httpmail:sendername",task.getSenderName());
		
		return h;
	}
	protected DavSearchCondition convertCondition(SearchCondition src) throws HandleException
	{
		throw new HandleException("Not yet implemented");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#delete(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		deleteTask(acc,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getAllDatas(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getAllTasks(acc);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getFolderList(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		FolderMetadata[]  l_strFolderlist = new FolderMetadata[0]; 		   	 /* convert Mail Folders in Array of String[] */
		// Folder String format should be "root/parent/child/..." format.
		
		setAccountData((PasswordBasedAccountDataImpl)acc);
		IFolderIndex path = dest.getFolder();
		
		if (path==null) path=taskElement;
		Vector<Object> lists = super.getAllSubFolders((String)path.getIndex());
		ArrayList<FolderMetadata> arrFolderlist = populateFolderList(0,(String)path.getIndex(),lists);
		l_strFolderlist = (FolderMetadata[])arrFolderlist.toArray(l_strFolderlist);
		for (int i=0;i<l_strFolderlist.length;i++) {
			if (l_strFolderlist[i].getFolderName().length()==0) {
				l_strFolderlist[i].setFolderName((String)path.getIndex());
			}
		}
		return l_strFolderlist;
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#read(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return getTaskByOID(acc,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#search(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.ISearchFormula)
	 */
	public IRecordObject[] search(IAccountData acc, ISearchFormula col) throws Exception
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("search is not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#searchByDate(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, java.util.Date)
	 */
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date date) throws Exception
	{
		return getTasksByDate(acc,date);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#write(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IRecordObject)
	 */
	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject data) throws Exception
	{
		IObjectIndex rc = null;
		if (data.isNew()) {
			rc = createNewTask(acc,(ITaskDTO)data);
		}
		else {
			rc = updateTask(acc,(ITaskDTO)data);
		}
		return rc;
	}


	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getPresetDestination(int)
	 */
	public ISearchDestination getPresetDestination(IAccountData acc,int type) throws IncorrectData, HandleException
	{
		super.setAccountData((PasswordBasedAccountDataImpl)acc);
		IFolderIndex folder = null;
		IDatabaseIndex db = null;
		switch (type) {
		case ITaskStore.DEST_TYPE_DEFAULT_TASK_FOLDER:
			folder = (IFolderIndex)populateFolder(taskElement).getOid();
			break;
		default:
			throw new IncorrectData("illegal folderType");
		}
		return (ISearchDestination)new BasicSearchDestination(db,folder);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.eai.exch2k.store.EAIExch2kBase#getElmentName()
	 */
	protected String getElmentName()
	{
		return DEFAULT_EXCHANGE_TASK_PATH_ELEMENT;
	}

}

