/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.notes.store;

import java.lang.reflect.Method;
import java.util.*;

import jp.co.connectone.eai.notes.store.IRecordElement;
import jp.co.connectone.eai.notes.user.NotesAccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.service.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.db.*;
import jp.co.connectone.user.IAccountData;
import lotus.domino.*;
import jp.co.connectone.log.Log;

public class NotesDBStoreImpl extends EAINotesBase implements IDBStore
{
	public static final IStoreID storeID = new SimpleStoreID(NotesDBStoreImpl.class.getName());
	public static final String storeName = NotesConstants.DBSTORE_NAME;
	public String getName() throws Exception
	{
		return storeName;
	}

	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		NotesServiceInfo info = new NotesServiceInfo(storeID,storeName);
		String serverAddress = (String)h.get("server");
		String database = (String)h.get("database");
		String serverAddress2 = (String)h.get("server2");
		if (serverAddress!=null) {
			info.setServerAddress(serverAddress);
			if (serverAddress2!=null) info.setServerAddress2(serverAddress2);
			info.setDatabase(database);
		}
		return info;
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new NotesServiceInfo(storeID,storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		super.initSession((NotesAccountData)acc,dest.getDatabase());
		IRecordObject rc = null;
		Document doc = super.getDocument(oid);
		try {
			rc = new NotesDBFieldSet(doc.getUniversalID(),super.populateHashMap(doc));			
		}
		catch(NullPointerException l_nullExObj) {
			throw new HandleException(l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new HandleException(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		finally {
			try {
				if(doc !=null) {
					doc.recycle();
				}
			}
			catch(NotesException l_excepObj) {
				throw new ServerDown(l_excepObj.id + l_excepObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}
		
		return rc;
	}

	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject objRecord) 
		throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		IObjectIndex rc = null;
		super.initSession((NotesAccountData)acc,dest.getDatabase());
		Document doc = null;
		if (objRecord.isNew()) {
Log.debug("new record");
			doc = super.createDocument();
			//TODO: create doc with IRecordObject
		}
		else {
Log.debug("overwrite");
			doc = super.getDocument(objRecord.getOid());
			if (objRecord instanceof NotesDBFieldSet) {
				updateFieldsFromNotesDBFieldSet(doc,objRecord);
			}
			else {
				//TODO: update with custom IRecordObject
			}
		}
		
		try {
			doc.save();
			rc = new NotesObjectIndex(doc.getUniversalID()); 
		}
		catch(NotesException notesEx){
			throw new HandleException(notesEx.id + notesEx.text);
		}
		finally{
			try	{
				if(doc !=null) {
					doc.recycle();
				}
			}
			catch(NotesException l_notesExp){
				new ServerDown(l_notesExp.id + l_notesExp.text);
			}
			finally{
				super.recycleObjects();
			}
		}
		return rc;
	}

	public FolderMetadata[] getFolders(IAccountData acc, ISearchDestination dest)	 throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		FolderMetadata[]  l_strFolderlist = new FolderMetadata[0];
		ArrayList<FolderMetadata> dst = new ArrayList<FolderMetadata>();
		Vector<View> l_viewsObj = null;
		View l_viewObj = null;
		super.initSession((NotesAccountData)acc,dest.getDatabase());
		try {
			l_viewsObj = getViews();
		  	/* Searches for the Folders in list of View */
			for (int l_intIndex=0; l_intIndex < l_viewsObj.size(); l_intIndex++) {
				l_viewObj = (View)l_viewsObj.elementAt(l_intIndex);
				if (l_viewObj.isFolder()) {
					FolderMetadata folder = new FolderMetadata();
					folder.setFolderName(l_viewObj.getName());
					dst.add(folder);
				}
			}
        }
		catch(NullPointerException l_nullExObj) {
			throw new StoreNotFound(l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
	    }
		finally {
			try {
				if(l_viewsObj != null) {
					l_viewObj.recycle();
				}
			}
			catch(Exception l_exceptionObj) {
				Log.error("",l_exceptionObj);
			}
			finally {
				super.recycleObjects();
			}
		}
		l_strFolderlist = (FolderMetadata[])dst.toArray(l_strFolderlist);
		return l_strFolderlist;
	}

	public IRecordObject[] getDataSet(IAccountData acc, ISearchDestination dest, IRecordObject template) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		View   l_viewObj = null;
		ViewEntryCollection col = null;
		ViewEntry entry = null;
		Document doc = null;
		NotesDBFieldSet l_docArray[] = null;
		IFolderIndex folder = dest.getFolder();
		super.initSession((NotesAccountData)acc,dest.getDatabase());
		try{
			l_viewObj = getView(dest.getFolder());
Log.debug("l_viewObj="+l_viewObj);
			col = l_viewObj.getAllEntries();
Log.debug("Number of Entries:"+col.getCount());
			l_docArray = new NotesDBFieldSet[col.getCount()];
			entry = col.getFirstEntry();
			int i = 0;
			while (entry!=null) {
				if (entry.isDocument()) {
					doc = entry.getDocument();
					l_docArray[i++] = new NotesDBFieldSet(doc.getUniversalID(),super.populateHashMap(doc));
				}
				entry=col.getNextEntry(entry);
			}
			
		}
		catch(NullPointerException l_nullExObj) {
			throw new StoreNotFound(folder.getIndex().toString()+l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		finally {
			try {
				if(l_viewObj !=null) {
					l_viewObj.recycle();
				}
				if(doc !=null) {
					doc.recycle();
				}
				if(entry !=null) {
					doc.recycle();
				}
				if(col !=null) {
					col.recycle();
				}
			}
			catch(NotesException l_excepObj)
			{
				throw new ServerDown(l_excepObj.id + l_excepObj.text);
			}
			finally
			{
				super.recycleObjects();
			}
		}
		return l_docArray;
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.eai.notes.store.EAINotesBase#getElmentName()
	 */
	protected String getElmentName()
	{
		// TODO ꂽ\bhEX^u
		return null;
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.db.IDBStore#delete(jp.co.connectone.user.IAccountData, jp.co.connectone.store.ISearchDestination, jp.co.connectone.store.IObjectIndex)
	 */
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		initSession((NotesAccountData)acc,dest.getDatabase());
		deleteDocument(oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.db.IDBStore#getUniqueIndexes(jp.co.connectone.user.IAccountData, jp.co.connectone.store.ISearchFormula)
	 */
	public IRecordObject[] getUniqueIndexes(IAccountData acc, ISearchFormula form) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("getUniqueIndexes is not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.db.IDBStore#search(jp.co.connectone.user.IAccountData, jp.co.connectone.store.ISearchFormula)
	 */
	public IRecordObject[] search(IAccountData acc, ISearchFormula col) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("search is not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.db.IDBStore#searchByDate(jp.co.connectone.user.IAccountData, jp.co.connectone.store.ISearchDestination, java.util.Date)
	 */
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date startDate) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("searchByDate is not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getAllDatas(jp.co.connectone.user.IAccountData, jp.co.connectone.store.ISearchDestination)
	 */
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		Log.debug("getAllDatas");
		IFolderIndex folder = dest.getFolder();
		View l_viewObj = null;
		DocumentCollection l_docCollectionObj = null;
		initSession((NotesAccountData)acc,dest.getDatabase());
		l_viewObj = getView(folder);
		l_docCollectionObj = l_viewObj.getAllDocumentsByKey("",false);
		
		if(l_viewObj.getAllEntries().getCount() != l_docCollectionObj.getCount()) {
			ViewEntryCollection evc = l_viewObj.getAllEntries();
			ViewEntry entry = evc.getFirstEntry();
			while(entry!=null) {
				Document doc = entry.getDocument();
				//ǉȂ悤
				if(doc != null && l_docCollectionObj.getDocument(doc) == null) {
					l_docCollectionObj.addDocument(doc);
				}
				entry = evc.getNextEntry();
			}
		}
		IRecordObject[] rc = null;
		try {
			rc = populateObjects(l_docCollectionObj);
		}
		catch (Exception e) {
			throw e;
		}
		finally {
			try {
				l_docCollectionObj.recycle();
				l_viewObj.recycle();
				l_docCollectionObj = null;
				l_viewObj = null;
			}
			catch (Exception e) {
			}
			finally {
				recycleObjects();
			}
		}
		
		return rc;
	}
	
	protected IRecordObject[] populateObjects(DocumentCollection col) throws HandleException
	{
		int cnt = 0;
		try {
			cnt = col.getCount();
		}
		catch (NotesException ne) {
			Log.error("error getting count of DocumentCollection", ne);
			throw new IncorrectData(ne.getMessage());
		}
		IRecordObject[] rc = new IRecordObject[cnt];
		try {
			for (int i=0;i<cnt;i++) {
				Document doc = col.getNthDocument(i+1);
				IRecordObject ro = populateObject(doc);
				doc.recycle();
				rc[i] = ro;
			}
		}
		catch (NotesException ne) {
			Log.error("error getting count of DocumentCollection", ne);
		}
		return rc;
	}
	
	protected IRecordObject populateObject(Document doc) throws NotesException,HandleException
	{
		String id = doc.getUniversalID();
		return new NotesDBFieldSet(id,super.populateHashMap(doc));
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getFolderList(jp.co.connectone.user.IAccountData, jp.co.connectone.store.ISearchDestination)
	 */
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getFolders(acc,dest);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getPresetDestination(jp.co.connectone.user.IAccountData, int)
	 */
	public ISearchDestination getPresetDestination(IAccountData acc, int type) throws IncorrectData, HandleException
	{
		throw new HandleException("no preset destination supported.");
	}
	

	public static Object getBean(IRecordObject beanToBuild,IRecordObject src) throws Exception
	{
		if (!(beanToBuild instanceof BasicRecordObject)) {
			throw new HandleException("beanToBuild must be an instance of BasicRecordObject.");
		}
		BasicRecordObject bro = (BasicRecordObject)beanToBuild;
		bro.setOid(src.getOid());
		Class<?> c = beanToBuild.getClass();
		while (c != null) {
Log.debug(c.getName());
			if (c.equals(BasicRecordObject.class)) {
				break;
			}
			Method[] m = c.getDeclaredMethods();
			for (int i=0;i<m.length;i++) {
				String name = m[i].getName();
				if (name.startsWith("set")) {
					Class<?>[] param = m[i].getParameterTypes();
					if (param.length==1) {
Log.debug("found setter:"+name);
						String fName = name.substring(3,4).toLowerCase() + name.substring(4);
						Object value[] = new Object[1];
						HashMap<String,Object> fieldSet = src.getFieldSet();
						value[0]= fieldSet.get(fName);
						if (value[0] == null) {
							fName = name.substring(3);
							value[0]= fieldSet.get(fName);
						}
						if (value != null) {
							Class<?> com =param[0].getComponentType();
							Class<?>[] interfaces = param[0].getInterfaces();
							boolean isArray = false;
							if (com != null) {
								interfaces = com.getInterfaces();
								isArray = true;
							}
							boolean isIRecordElement = false;
							for (int j=0;j<interfaces.length;j++) {
								Class<?> i_f = interfaces[j];
								if (i_f.equals(IRecordElement.class)) {
									isIRecordElement = true;
									break;
								}
							}
							try {
								if (isIRecordElement) {
									IRecordElement helper = null;
									if (isArray) {
										helper = (IRecordElement)com.newInstance();
										value[0] = helper.parseArrayInput(value[0]);
									}
									else {
										helper = (IRecordElement)param[0].newInstance();
										helper.setRecordElementValue(value[0]);
										value[0] = helper;
									}
								}
								m[i].invoke(beanToBuild,value);
								Log.debug("setting field "+fName+" value "+value[0]+" in method "+name+":OK");
							}
							catch (IllegalAccessException iae) {
								Log.debug("skip setter set"+fName+" because of it is not public.");
							}
							catch (Exception e) {
								Log.error("error setting field "+fName+" value "+value[0]+" in method "+name+". continue..",e);
							}
						}
					}
				}
			}
			c = c.getSuperclass();
		}
		return beanToBuild;
	}
	
	
	public HashMap<String,Object> generateHashMap(IRecordObject src) throws Exception
	{
		Class<?> c = src.getClass();
		HashMap<String,Object> h = new HashMap<String,Object>();
		while (c != null) {
			if (c.equals(BasicRecordObject.class)) {
				break;
			}
			Method[] m = c.getDeclaredMethods();
			for (int i=0;i<m.length;i++) {
				String name = m[i].getName();
				if (name.startsWith("get")) {
					Object param[] = m[i].getParameterTypes();
					if (param.length==0) {
Log.debug("found getter:"+name);
						String fName = name.substring(3,4).toLowerCase() + name.substring(4);
						Object value = m[i].invoke(src,(Object[])null);
						h.put(fName,value);
						fName = name.substring(3);
						value = m[i].invoke(src,(Object[])null);
						h.put(fName,value);
					}
				}
			}
			c = c.getSuperclass();
		}
		return h;
	}

	public IRecordObject[] getDataSet(IAccountData acc, ISearchDestination dest) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		try {
			return getAllDatas(acc, dest);
		}
		catch (Exception e) {
			Log.error("",e);
			throw new HandleException(e.getMessage());
		}
	}

	public IRecordObject getTableIndexFromString(String tableName) throws IncorrectData
	{
		if (tableName==null) {
			throw new NullPointerException("folder/table name must not be null");
		}
		BasicRecordObject rc = new BasicRecordObject();
		rc.setOid(new NotesFolderIndex(tableName));
		return rc;
	}

	public IRecordObject[] getUniqueIndexes(IAccountData acc, ISearchDestination dest) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("searchByDate is not yet supported.");
	}

	public IObjectIndex write(IAccountData acc, ISearchFormula form, IRecordObject data) throws Exception
	{
		ISearchDestination dest = form.getDest();
		return write(acc,dest,data);
		
	}
}
