/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.pop3.store;

import java.io.*;
import java.util.*;

import javax.mail.*;
import javax.mail.internet.*;

import jp.co.connectone.eai.pop3.user.Pop3AccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;


public class Pop3MailStoreImpl extends EAIPop3Base implements IMailStore
{
	public static final IStoreID storeID = new SimpleStoreID(Pop3MailStoreImpl.class.getName());
	public static final String storeName = "Pop3 Mail Store";
	
	public static Pop3MailStoreImpl getInstance()
	{
		return new Pop3MailStoreImpl();// implement resource_pool here if needed.
	}
	
	public IServiceInfo getServiceInfo() throws Exception
	{
		return new Pop3ServiceInfo(storeID, storeName);
	}

	public void deleteMail(IAccountData acc, IObjectIndex oidx) throws ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		Pop3AccountData pacc = (Pop3AccountData)acc;
        
		setUserID(pacc.getUserID());
		setPassword(pacc.getPassword());
		setServerAddress(acc.getServiceInfo().getServerAddress());
		Object oid = oidx.getIndex();

		Log.debug("deleteMail::oid="+oid);
		if (oid instanceof String) {
			super.delete(oid.toString());
		}
		else {
			super.delete((Pop3ObjectIndexRawDataDateAndSubject)oid);
		}
	}

	public IMailDTO[] getHeaders(IAccountData acc, ISearchDestination dest) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		IMailDTO[] headers = null;
		if (!(acc instanceof Pop3AccountData)) {
			throw new IncorrectData("account data type must be Pop3AccountData");
		}
		Pop3AccountData pacc = (Pop3AccountData)acc;
        
		Vector<String> v = new Vector<String>();
		setUserID(pacc.getUserID());
		setPassword(pacc.getPassword());
		setServerAddress(acc.getServiceInfo().getServerAddress());
        Message[] col = super.getList(dest.getFolder().getIndex().toString(),v);
        int len = col.length;
        if (len == 0) return new IMailDTO[0];
        headers = new IMailDTO[1];
        ArrayList<IMailDTO> ar = new ArrayList<IMailDTO>();
        for (int i=len-1;i>=0;i--) {
        	ar.add(populateIMailDTO((MimeMessage)col[i]));
        }
        headers = ar.toArray(headers);
		
        return headers;
	}

	public IMailDTO[] getHeadersByDate(IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		IMailDTO[] headers = null;
		if (!(acc instanceof Pop3AccountData)) {
			throw new IncorrectData("account data type must be Pop3AccountData");
		}
		Pop3AccountData pacc = (Pop3AccountData)acc;

		Vector<String> v = new Vector<String>();
		setUserID(pacc.getUserID());
		setPassword(pacc.getPassword());
		setServerAddress(acc.getServiceInfo().getServerAddress());
		Message[] col = super.getListByDate(dest.getFolder().getIndex().toString(),v,date);
		int len = col.length;
        if (len == 0) return new IMailDTO[0];
		headers = new IMailDTO[1];
        ArrayList<IMailDTO> ar = new ArrayList<IMailDTO>();
        for (int i=len-1;i>=0;i--) {
        	ar.add(populateIMailDTO((MimeMessage)col[i]));
        }
        headers = ar.toArray(headers);
		
        return headers;
	}

	public IMailDTO getMail(IAccountData acc, ISearchDestination dest, IObjectIndex oidx) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		IMailDTO l_IMailDTOObj = null;
		if (oidx == null) {
			throw new IncorrectData("message-id is null. cannot retrieve data.");
		}
		if (!(acc instanceof Pop3AccountData)) {
			throw new IncorrectData("account data type must be Pop3AccountData");
		}
		Pop3AccountData pacc = (Pop3AccountData)acc;
		setUserID(pacc.getUserID());
		setPassword(pacc.getPassword());
		setServerAddress(acc.getServiceInfo().getServerAddress());

		Object oid = oidx.getIndex();

		Log.debug("getMail::oid="+oid);

		MimeMessage mail = null;
		if (oid instanceof String) {
			mail = super.getMail(oid.toString());
		}
		else {
			mail = super.getMail((Pop3ObjectIndexRawDataDateAndSubject)oid);
		}

		l_IMailDTOObj = populateIMailDTO(mail);
		

		return l_IMailDTOObj;
	}

	public FolderMetadata[] getMailFolders(IAccountData arg0, ISearchDestination arg1) throws StoreNotFound, NoSuchRights, ServerDown, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		FolderMetadata folders[] = new FolderMetadata[1];
		folders[0] = populateFolder(new Pop3FolderIndex());
		return folders;
	}

	public IMailDTO[] getMailsByDate(IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		return getHeadersByDate(acc, dest, date);
	}

	public IMailDTO[] getSentMails(IAccountData arg0) throws ServerDown, HandleException
	{
		throw new HandleException(storeName+" does not support getSentMails.");
	}

	public IMailDTO[] getSentMailsByDate(IAccountData arg0, Date arg1) throws ServerDown, HandleException
	{
		throw new HandleException(storeName+" does not support getSentMailsByDate.");
	}

	public IObjectIndex[] getUIDLs(IAccountData acc, ISearchDestination dest) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		IMailDTO[] headers = getHeaders(acc, dest);
		if (headers==null) return new Pop3ObjectIndex[0];
		ArrayList<Pop3ObjectIndex> uidlarray = new ArrayList<Pop3ObjectIndex>();
		for (int i=0;i<headers.length;i++) {
			try {
				String uidl = headers[i].getHeader().getHeaderUIDL();
				if (uidl==null) {
					Log.debug("skip message "+headers[i].getHeader().getHeaderSubject()+" because of no Message-ID.");
				}
				else {
					uidlarray.add(new Pop3ObjectIndex(uidl));
				}
			}
			catch (Exception e) {
				Log.error("", e);
			}
		}
		Pop3ObjectIndex[] uidls = new Pop3ObjectIndex[1];
		uidls = (Pop3ObjectIndex[])uidlarray.toArray(uidls);
		
		return uidls;
	}

	public IObjectIndex[] getUIDLsByDate(IAccountData acc, ISearchDestination dest, Date date) throws StoreNotFound, ServerDown, NoSuchRights, IncorrectStore, DataNotFound, IncorrectData, HandleException
	{
		IMailDTO[] headers = getHeadersByDate(acc, dest, date);
		if (headers==null) return new Pop3ObjectIndex[0];
		ArrayList<Pop3ObjectIndex> uidlarray = new ArrayList<Pop3ObjectIndex>();
		for (int i=0;i<headers.length;i++) {
			try {
				String uidl = headers[i].getHeader().getHeaderUIDL();
				if (uidl==null) {
					Log.debug("skip message "+headers[i].getHeader().getHeaderSubject()+" because of no Message-ID.");
				}
				else {
					uidlarray.add(new Pop3ObjectIndex(uidl));
				}
			}
			catch (Exception e) {
				Log.error("", e);
			}
		}
		Pop3ObjectIndex[] uidls = new Pop3ObjectIndex[1];
		uidls = (Pop3ObjectIndex[])uidlarray.toArray(uidls);
		
		return uidls;
	}

	public IObjectIndex sendMail(IAccountData arg0, IMailDTO arg1) throws NoSuchRights, ServerDown, IncorrectData, HandleException
	{
		throw new HandleException(storeName+" does not support sendMail.");
	}

	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oidx) throws Exception
	{
		deleteMail(acc, oidx);
	}

	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getHeaders(acc,dest);
	}

	public IRecordObject getFolderIndexFromString(String arg0) throws IncorrectData
	{
		return (IRecordObject)(new Pop3FolderIndex());
	}

	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getMailFolders(acc,dest);
	}

	public String getName() throws Exception
	{
		return storeName;
	}

	public ISearchDestination getPresetDestination(IAccountData arg0, int arg2) throws IncorrectData, HandleException
	{
		BasicSearchDestination dst = new BasicSearchDestination(null,new Pop3FolderIndex());
		return dst;
	}

	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		Pop3ServiceInfo info = new Pop3ServiceInfo(storeID, storeName);
		String serverAddress = (String) h.get("server");
		if (serverAddress != null) info.setServerAddress(serverAddress);
		return info;
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oidx) throws Exception
	{
		return getMail(acc, dest, oidx);
	}

	public IRecordObject[] search(IAccountData arg0, ISearchFormula arg1) throws Exception
	{
		throw new HandleException(storeName+" does not support search.");
	}

	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date date) throws Exception
	{
		return getHeadersByDate(acc,dest,date);
	}

	public IObjectIndex write(IAccountData arg0, ISearchDestination arg1, IRecordObject arg2) throws Exception
	{
		throw new HandleException(storeName+" does not support write.");
	}

	private IMailDTO populateIMailDTO(MimeMessage src) throws HandleException
	{
    	IMailDTO l_IMailDTOObj = new BasicMailDTO();
		HeaderDTO l_headerDTOObj = new HeaderDTO();
		String tmp = null;
		try {
			l_headerDTOObj.setHeaderUIDL(src.getMessageID());
			String messageId = src.getMessageID();
			if (messageId==null) {
				String subject = src.getSubject();
				Date date = src.getReceivedDate();
				l_headerDTOObj.setOid(new Pop3ObjectIndex(Pop3ObjectIndex.makeIndexByDateAndSubject(date, subject)));
			}
			else {
				l_headerDTOObj.setOid(new BasicObjectIndex(messageId));
			}
			Address[] add = null;
			try {
				add = src.getFrom();
			}
			catch(AddressException e){}
			if (add!=null) {
				tmp = add[0].toString();
				try {
					tmp =MimeUtility.decodeText(add[0].toString());
				}
				catch (UnsupportedEncodingException e) {}
				l_headerDTOObj.setHeaderFrom(tmp);
			}
			l_headerDTOObj.setHeaderSubject(src.getSubject());
			int pri = 0;//0=normal
			
			String priStr[] = src.getHeader("importance");
			if (priStr==null) priStr = src.getHeader("priority");
			try {
				if (priStr!=null) pri = Integer.parseInt(priStr[0]);
			}
			catch(Exception e){};
			l_headerDTOObj.setHeaderPriority(pri);
			try {
				add = src.getReplyTo();
			}
			catch(AddressException e){}
			if (add!=null) {
				tmp = add[0].toString();
				try {
					tmp =MimeUtility.decodeText(add[0].toString());
				}
				catch (UnsupportedEncodingException e) {}
				l_headerDTOObj.setHeaderReplyTo(tmp);
			}

			l_headerDTOObj.setHeaderSize(src.getSize());
			l_headerDTOObj.setHeaderSendDate(src.getSentDate());

			try {
				add = src.getRecipients(Message.RecipientType.CC);
			}
			catch(AddressException e){}
			if (add != null) {
				MailAddress[] cc = new MailAddress[add.length];
				for (int i=0;i<add.length;i++) {
					tmp = add[i].toString();
					try {
						tmp =MimeUtility.decodeText(add[i].toString());
					}
					catch (UnsupportedEncodingException e) {}
					MailAddress address = new MailAddress(tmp);
					cc[i] = address;
				}
				l_headerDTOObj.setHeaderCC(cc);
			}
			try {
				add = src.getRecipients(Message.RecipientType.TO);
			}
			catch(AddressException e){}
			if (add != null) {
				MailAddress[] to = new MailAddress[add.length];
				for (int i=0;i<add.length;i++) {
					tmp = add[i].toString();
					try {
						tmp =MimeUtility.decodeText(add[i].toString());
					}
					catch (UnsupportedEncodingException e) {}
					MailAddress address = new MailAddress(tmp);
					to[i] = address;
				}
				l_headerDTOObj.setHeaderTo(to);
			}
		}
		catch (MessagingException e) {
			e.printStackTrace();
			if (!(e instanceof MessageRemovedException)) {
				throw new HandleException(e.getMessage());
			}
			Log.debug("skip this message...");
		}
		l_IMailDTOObj.setHeaderDTO(l_headerDTOObj);

		//ToDO: Ytt@C̎擾
		retrieveAllParts(src);
		try {
			bodyText = new String(bodyText.getBytes("Shift-JIS"),"MS932");
		}
		catch (Exception e){};
		l_IMailDTOObj.setBody(bodyText);
		
		if (attach != null) {
			AttachmentDTO[] list = attach.toArray(new AttachmentDTO[1]);
			l_IMailDTOObj.setAttachments(list);
			Log.debug(list.toString());
		}
		attach = null;

    	return l_IMailDTOObj;
    }
	
	private void retrieveAllParts(Part part) throws IncorrectData,HandleException
	{
		try {
			if (part.isMimeType("multipart/*")) {
				Multipart subPart = (Multipart)part.getContent();
				for (int i=0;i<subPart.getCount();i++) {
					retrieveAllParts(subPart.getBodyPart(i));
				}
			}
			else if (part.isMimeType("text/plain")) {
				if (part.getFileName()==null) {
					//{
					bodyText = (String)part.getContent();
				}
				else {
					//Yt
					addAttach(part);
				}
			}
			else {
				//Yt
				addAttach(part);
			}
			
		}
		catch (MessagingException me) {
			me.printStackTrace();
			throw new IncorrectData(me.getMessage());
		}
		catch (IOException ioe) {
			ioe.printStackTrace();
			throw new HandleException(ioe);
		}
	}
	
	private void addAttach(Part part) throws MessagingException
	{
		AttachmentDTO newAttach = new AttachmentDTO();
		String fileName = null;
		byte[] rawData = null;
		try {
			fileName = part.getFileName();
		}
		catch (ParseException e) {
			
		}
		if (fileName==null) return;// noName contents like html-part of html-mail
		try {
			fileName =MimeUtility.decodeText(fileName);
			InputStream bio = part.getInputStream();
			ByteArrayOutputStream boo = new ByteArrayOutputStream();
			byte[] buf = new byte[81920];
			int len = buf.length;
			while (len == buf.length) {
				len = bio.read(buf);
				if (len<0) break;
				try {
					boo.write(buf,0,len);
				}
				catch (IndexOutOfBoundsException e) {
					Log.error("", e);
				}
			}
			rawData = boo.toByteArray();
		}
		catch (UnsupportedEncodingException e) {}
		catch (IOException ioe) {
			ioe.printStackTrace();
		}
		Log.debug(fileName);
		if (attach == null) {
			attach = new ArrayList<AttachmentDTO>();
		}
		newAttach.setFileName(fileName);
		newAttach.setFileBody(rawData);
		attach.add(newAttach);
	}
	
	private String bodyText=null;
	private ArrayList<AttachmentDTO> attach=null;
}
