/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.pop3.socket;

import java.util.*;

import jp.co.connectone.eai.pop3.exception.ServerCommunicationException;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;

public class SocketBasedPop3BaseImpl
{
	protected Pop3ProtocolSocketImpl socket=null;
	protected Vector<UidlElement> cacheUidls=null;
	protected String userID;
	protected String password;
	protected String serverAddress;
	protected String serverAddress2;
	protected int port;
	
	public SocketBasedPop3BaseImpl()
	{
		try {
			socket=Pop3ProtocolSocketImpl.getInstance();
		}
		catch (HandleException e) {
			Log.error("couldn't instanciate Pop3ProtocolSocketImpl.", e);
		}
	}

	protected void setAccount(String userId, String passwd, String serverName1, String serverName2, int port)
	{
		this.userID = userId;
		this.password = passwd;
		this.serverAddress = serverName1;
		this.serverAddress2 = serverName2;
		this.port = port;
	}
	
	private boolean errorQuit() throws NoSuchRights
	{
		close();
		throw new NoSuchRights("POP Authentication Failed.");
	}
	
	protected boolean open() throws HandleException
	{
		if (userID==null) throw new HandleException("call setAccount first.");
		if (socket==null) throw new HandleException("socket is null. please make instance again.");
		socket.connect(serverAddress,serverAddress2);

		boolean f = socket.doUser(userID);
		if (!f) {
			return errorQuit();
		}
		f = socket.doPass(password);
		if (!f) {
			return errorQuit();
		}
		String[] rc = socket.doUidl();
		if(rc == null){
			throw new ServerCommunicationException("returns POP ERR on doUidl.");
		}
		cacheUidls = new Vector<UidlElement>();
		for (String uidl : rc) {
			cacheUidls.add(new UidlElement(uidl));
		}
		return true;
	}
	
	protected void close()
	{
		if (cacheUidls!=null) cacheUidls=null;
		// not executed more once.
		if (socket==null || socket.isClosed()) return;
		try {
			socket.doQuit();
		} 
		catch (HandleException e) {
			// Quit error, not notify user.
			Log.error("error on QUIT command error.", e);
		}
		socket.disconnect();
	}

	public Vector<UidlElement> getUIDLList() throws HandleException
	{
		Vector<UidlElement> list = null;
		try {
			open();
			list = cacheUidls;
			close();
			return list;
		} catch (HandleException e) {
			close();
			throw e;
		}
	}

	protected int reverseIndex(int idx, int size)
	{
		return size-idx-1;
	}
	
	public String getTopByUidlIndex(Vector<UidlElement> uidlList,int idx) throws HandleException
	{
		String rc = null;
		checkUidlListNotEmpty(uidlList);
		try {
			open();

			int actualUidlIdx = checkIdxActual(idx, uidlList);
			// -1: deleted UIDL
			// other:index of uidl_in_uidlList
			if (actualUidlIdx >= 0) {
				rc = socket.doTop(actualUidlIdx);
				if(rc == null){
					throw new ServerCommunicationException("returns POP ERR on doTop.");
				}
			}

			close();
			return rc;
		} catch (HandleException e) {
			close();
			throw e;
		}
	}
	
	public String getTopByUidlIndexR(Vector<UidlElement> uidlList,int idx) throws HandleException
	{
		checkUidlListNotEmpty(uidlList);

		idx = reverseIndex(idx,uidlList.size());

		return getTopByUidlIndex(uidlList,idx);
	}

	protected void checkUidlListNotEmpty(Vector<UidlElement> uidlList) throws HandleException
	{
		if (uidlList==null) {
			uidlList = getUIDLList();
		}
		else if (uidlList.size()==0) {
			uidlList = getUIDLList();
		}
	}

	protected Vector<UidlElement> checkUidlListNotEmptyWithoutOpen(Vector<UidlElement> uidlList) throws HandleException
	{
		if (uidlList==null) {
			uidlList = cacheUidls;
		}
		else if (uidlList.size()==0) {
			uidlList = cacheUidls;
		}
		return uidlList;
	}
	
	public String getRetrByUidlIndex(Vector<UidlElement> uidlList,int idx) throws HandleException
	{
		String rc = null;
		checkUidlListNotEmpty(uidlList);
		try{
			open();

			int actualUidlIdx = checkIdxActual(idx, uidlList);
			// -1: deleted UIDL
			// other:index of uidl_in_uidlList
			if (actualUidlIdx >= 0) {
				rc = socket.doRetr(actualUidlIdx);
				if(rc == null){
					throw new ServerCommunicationException("returns POP ERR on doRetr.");
				}
			}

			close();
			return rc;
		} catch (HandleException e) {
			close();
			throw e;
		}
	}
	
	public String getRetrByUidlIndexR(Vector<UidlElement> uidlList,int idx) throws HandleException
	{
		checkUidlListNotEmpty(uidlList);

		idx = reverseIndex(idx,uidlList.size());

		return getRetrByUidlIndex(uidlList,idx);
	}
	
	protected String getTopByUidlIndexWithoutOpen(Vector<UidlElement> uidlList,int idx) throws HandleException
	{
		String rc = null;

		int actualUidlIdx = checkIdxActual(idx,uidlList);
		//-1: deleted UIDL
		//other:index of uidl_in_uidlList
		if (actualUidlIdx>=0) {
			rc = socket.doTop(actualUidlIdx);
			if(rc == null){
				throw new ServerCommunicationException("returns POP ERR on doTop.");
			}
		}
		
		return rc;
	}

	private int convertUidl2Idx(Vector<UidlElement> uidlList, String uidl) throws HandleException
	{
		checkUidlListNotEmpty(uidlList);
		int idx = uidlList.indexOf(new UidlElement(uidl));
		if (idx==-1) throw new IncorrectData("given UIDL should be in given uidlList.");
		return idx;
	}
	
	public String getTopByUidl(Vector<UidlElement> uidlList,String uidl) throws HandleException
	{
		int idx = convertUidl2Idx(uidlList, uidl);
		return getTopByUidlIndex(uidlList, idx);
	}
	
	protected String getTopByUidlWithoutOpen(Vector<UidlElement> uidlList,String uidl) throws HandleException
	{
		int idx = convertUidl2Idx(uidlList, uidl);
		return getTopByUidlIndexWithoutOpen(uidlList, idx);
	}
	
	public String getRetrByUidl(Vector<UidlElement> uidlList,String uidl) throws HandleException
	{
		int idx = convertUidl2Idx(uidlList, uidl);
		return getRetrByUidlIndex(uidlList, idx);
	}
	
	public void delete(Vector<UidlElement> uidlList,String uidl) throws HandleException
	{
		int idx = convertUidl2Idx(uidlList, uidl);
		try{
			open();

			socket.doDelete(idx);

			close();
		} catch (HandleException e) {
			close();
			throw e;
		}
	}

	private int checkIdxActual(int idx, Vector<UidlElement> uidlList)
	{
		UidlElement target = uidlList.get(idx);
		return cacheUidls.indexOf(target);
	}

	public int getPort()
	{
		return port;
	}

	public void setPort(int port)
	{
		this.port = port;
	}

	public String getPassword()
	{
		return password;
	}

	public void setPassword(String password)
	{
		this.password = password;
	}

	public String getUserID()
	{
		return userID;
	}

	public void setUserID(String userID)
	{
		this.userID = userID;
	}

	public String getServerAddress()
	{
		return serverAddress;
	}

	public void setServerAddress(String serverAddress)
	{
		this.serverAddress = serverAddress;
	}

	public String getServerAddress2() {
		return serverAddress2;
	}

	public void setServerAddress2(String serverAddress2) {
		this.serverAddress2 = serverAddress2;
	}
}
