/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.exchews.store;

import java.util.*;

import com.microsoft.schemas.exchange.services._2006.types.*;

import jp.co.connectone.eai.exchews.ews.IEwsSearchFormula;
import jp.co.connectone.eai.exchews.ews.UpdateItemMap;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.*;

public class ExchewsMemoStoreImpl extends EAIExchewsBase implements IMemoStore
{
	public static final List<String> SEARCH_FIELDS = IEwsSearchFormula.ALL_PROPERTIES;
	public static final IStoreID storeID = new SimpleStoreID(ExchewsMemoStoreImpl.class.getName());
	public static final String storeName = "Exchange EWS memo store";
	protected static final String itemType="IPM.StickyNote";

	public String getName() throws Exception
	{
		return storeName;
	}
	
	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		return super.getServiceInfo(storeID, storeName, serviceData);
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new ExchewsServiceInfo(storeID,storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}


	public IMemoDTO[] getAllMemos(IAccountData acc) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		super.setAccountData(acc);
		ExchewsMemoDTO[] l_memoDtoArray = new ExchewsMemoDTO[0];
		
		ISearchDestination dest = getPresetDestination(acc, DEST_TYPE_DEFAULT_MEMO_FOLDER);
		List<String> fieldsToGet = SEARCH_FIELDS;
		
		List<? extends ItemType> items = new ArrayList<MessageType>();
		try {
			items = super.getList(dest, fieldsToGet);
		}
		catch (HandleException he) {
			throw he;
		}
		catch (Exception e) {
			Log.error("", e);
			throw new HandleException(e);
		}

		ArrayList<ExchewsMemoDTO> rc = new ArrayList<ExchewsMemoDTO>();
		for (ItemType item : items) {
			try {
				ExchewsMemoDTO dto = ExchewsMemoDTO.convertFromMemoItemType(item);
				rc.add(dto);
				if (item.isHasAttachments().booleanValue()) {
//					dto.convertAndSetEWSAttachments(getAttachments(message));
				}
			}
			catch (Exception e) {
				Log.debug("error on converting MessageType to ExchewsMailDTO. skip one data..", e);
			}
		}

		return rc.toArray(l_memoDtoArray);
	}
	
	public IMemoDTO[] getMemosByDate(IAccountData acc, Date dateObj) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		throw new UnsupportedOperationException("getMemosByDate not yet supported.");
	}
	
	public IMemoDTO getMemoByOID(IAccountData acc, IObjectIndex oid) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, HandleException
	{
		super.setAccountData(acc);
		ItemType item = getItem(getPresetDestination(acc, DEST_TYPE_DEFAULT_MEMO_FOLDER),(ItemIdType) oid.getIndex());
		ExchewsMemoDTO rc = null;
		try {
			rc = ExchewsMemoDTO.convertFromMemoItemType(item);
			if (rc!=null) {
				if (item.isHasAttachments().booleanValue()) {
					rc.convertAndSetEWSAttachments(getAttachments(item));
				}
			}
		}
		catch (HandleException he) {
			throw he;
		}
		catch (Exception e) {
			Log.error("memo convert error", e);
			throw new HandleException(e);
		}
		return rc;
	}
	
	public IObjectIndex createNewMemo(IAccountData acc, IMemoDTO dto) throws IncorrectData, NoSuchRights, DataNotFound, IncorrectStore, ServerDown, HandleException
	{
		super.setAccountData(acc);

		ItemType item = null;
		try {
			item = ExchewsMemoDTO.convertToMessageType(dto);
		}
		catch (Exception e) {
			Log.error("MemoDTO convert error.", e);
			throw new IncorrectData(e.getMessage());
		}
		item = super.createItem(memoElement, item);
		ExchewsObjectIndex oid = new ExchewsObjectIndex(item.getItemId());
		return oid;
	}

	public IObjectIndex updateMemo(IAccountData acc, IMemoDTO dto) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, SearchConditionFailed, UpdateFailed,HandleException
	{
		if (dto==null) {
			throw new IncorrectData("dto null.");
		}
		if (dto.getOid()==null) {
			throw new IncorrectData("oid in updateItem should not be null");
		}
		super.setAccountData(acc);
		IObjectIndex oid = dto.getOid();

		super.updateItem((ItemIdType) oid.getIndex(),generateUpdateMap(dto));

		return oid;
	}

	protected UpdateItemMap generateUpdateMap(IMemoDTO dto) throws IncorrectData
	{
		UpdateItemMap map = new UpdateItemMap();
		
		ItemType item = null;
		try {
			item = new ItemType();
			item.setSubject(dto.getSubject());
			map.put(UnindexedFieldURIType.ITEM_SUBJECT, item);
			item = new ItemType();
			BodyType value = new BodyType();
			value.setBodyType(BodyTypeType.TEXT);
			value.setValue(dto.getBody());
			item.setBody(value);
			map.put(UnindexedFieldURIType.ITEM_BODY, item);
			
		}
		catch (Exception e) {
			Log.error("MemoDTO convert error.", e);
			throw new IncorrectData(e.getMessage());
		}
		
		return map;
	}

	public void deleteMemo(IAccountData acc, IObjectIndex oid) throws IncorrectData, DeleteFailed, NoSuchRights, ServerDown,DataNotFound, IncorrectStore, SearchConditionFailed,HandleException
	{

		setAccountData((PasswordBasedAccountDataImpl)acc);
		super.delete(oid);

	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#delete(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		this.deleteMemo(acc,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getAllDatas(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getAllMemos(acc);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getFolderList(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		super.setAccountData(acc);

		ExchewsFolderIndex idx = (ExchewsFolderIndex) dest.getFolder();

		BaseFolderIdType src = (BaseFolderIdType) idx.getIndex();

		List<BaseFolderType> folders = super.getAllSubFolders(src,0);

		ArrayList<ExchewsFolderMetadata> rc = new ArrayList<ExchewsFolderMetadata>();

		ExchewsFolderMetadata a[] = new ExchewsFolderMetadata[0];
		for (BaseFolderType folder : folders) {
			rc.add(ExchewsFolderMetadata.convertFromEWSFolderType((FolderType) folder));
		}

		return rc.toArray(a);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#read(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return this.getMemoByOID(acc,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#search(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.ISearchFormula)
	 */
	public IRecordObject[] search(IAccountData acc, ISearchFormula col) throws Exception
	{
		throw new UnsupportedOperationException("search not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#searchByDate(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, java.util.Date)
	 */
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date date) throws Exception
	{
		throw new UnsupportedOperationException("searchByDate not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#write(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IRecordObject)
	 */
	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject data) throws Exception
	{
		IObjectIndex rc = null;
		if (data.isNew()) {
			rc = this.createNewMemo(acc,(IMemoDTO)data);
		}
		else {
			rc = this.updateMemo(acc,(IMemoDTO)data);
		}
		return rc;
	}


	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getPresetDestination(int)
	 */
	public ISearchDestination getPresetDestination(IAccountData acc,int type) throws IncorrectData, HandleException
	{
		super.setAccountData((PasswordBasedAccountDataImpl) acc);
		IFolderIndex folder = null;
		IDatabaseIndex db = null;
		DistinguishedFolderIdType src = new DistinguishedFolderIdType();
		switch (type) {
		case DEST_TYPE_DEFAULT_MEMO_FOLDER:
			src.setId(memoElement);
			break;
		default:
			throw new IncorrectData("illegal folderType");
		}
		folder = (IFolderIndex) populateFolder(new ExchewsFolderIndex(src)).getOid();
		return (ISearchDestination) new BasicSearchDestination(db, folder);
	}

	@Override
	protected UnindexedFieldURIType getDateField()
	{
		return UnindexedFieldURIType.ITEM_DATE_TIME_CREATED;
	}

	@Override
	protected TargetFolderIdType getTargetFolder()
	{
		TargetFolderIdType target = new TargetFolderIdType();
		DistinguishedFolderIdType id = new DistinguishedFolderIdType();
		id.setId(memoElement);
		target.setDistinguishedFolderId(id);
		return target;
	}

}

