/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.exch2k.store;

import java.util.*;
import java.io.*;
import javax.mail.Session;
import javax.mail.internet.*;

import jp.co.connectone.eai.exch2k.dav.DavSearchCondition;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.client.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;
import jp.co.connectone.user.PasswordBasedAccountDataImpl;

public class Exch2kScheduleStoreImpl extends EAIExch2kBase implements IScheduleStore
{
	private static final String DEFAULT_EXCHANGE_SCHEDULE_PATH_ELEMENT="\\";
//	private static final String DEFAULT_EXCHANGE_SCHEDULE_PATH_ELEMENT_EN="Calendar";
	private IFolderIndex scheduleElement = new Exch2kFolderIndex(DEFAULT_EXCHANGE_SCHEDULE_PATH_ELEMENT);
	protected static final String elmentName=DEFAULT_EXCHANGE_SCHEDULE_PATH_ELEMENT;

	public static final IStoreID storeID = new SimpleStoreID(Exch2kScheduleStoreImpl.class.getName());
	public static final String storeName = "Exchange2K schedule store";
	public String getName() throws Exception
	{
		return storeName;
	}
	
	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		return super.getServiceInfo(storeID, storeName, serviceData);
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new Exch2kServiceInfo(storeID,storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	
	
	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#delete(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		deleteSchedule(acc,oid);
		
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getAllDatas(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getAllSchedules(acc,null);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getFolderList(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		FolderMetadata[]  l_strFolderlist = new FolderMetadata[0]; 		   	 /* convert Mail Folders in Array of String[] */
		// Folder String format should be "root/parent/child/..." format.
		
		setAccountData((PasswordBasedAccountDataImpl)acc);
		IFolderIndex path = dest.getFolder();
		
		if (path==null) path=scheduleElement;
		Vector<Object> lists = super.getAllSubFolders((String)path.getIndex());
		ArrayList<FolderMetadata> arrFolderlist = populateFolderList(0,(String)path.getIndex(),lists);
		l_strFolderlist = (FolderMetadata[])arrFolderlist.toArray(l_strFolderlist);
		for (int i=0;i<l_strFolderlist.length;i++) {
//			l_strFolderlist[i].setTid(i);
			if (l_strFolderlist[i].getFolderName().length()==0) {
				l_strFolderlist[i].setFolderName((String)path.getIndex());
			}
		}
		return l_strFolderlist;

	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#read(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return getSchedule(acc,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#search(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.ISearchFormula)
	 */
	public IRecordObject[] search(IAccountData acc, ISearchFormula col) throws Exception
	{
		return searchSchedules(acc,col);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#searchByDate(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, java.util.Date)
	 */
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date startDate) throws Exception
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("search shcedules with date not yet supported.");

	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#write(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IRecordObject)
	 */
	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject data) throws Exception
	{
		IObjectIndex rc = null;
		if (data.isNew()) {
			rc = createNewSchedule(acc,(IScheduleDTO)data);
		}
		else {
			rc = updateSchedule(acc,(IScheduleDTO)data);
		}
		return rc;
	}

	public boolean isValidSchedule(IScheduleDTO IScheduleDTO) throws IncorrectStore, ServerDown, DataNotFound, HandleException
	{
		return true;
	}
	public boolean isValidSchedule(IAccountData acc, IScheduleDTO IScheduleDTO) throws IncorrectStore, ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		return true;
	}
	public boolean isDuplicateSchedule(IAccountData acc, IScheduleDTO IScheduleDTO) throws IncorrectStore, NoSuchRights, ServerDown, DataNotFound, IncorrectData, HandleException
	{
		return false;
	}

	public IScheduleDTO[] getAllSchedules(IAccountData acc, Date cutOffDate) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		Exch2kScheduleDTO[] l_IScheduleDTOArray = null;

		Vector<String> v = makeItemList();

		setAccountData((PasswordBasedAccountDataImpl)acc);
        Vector<HashMap<String,Object>> col = super.getList(scheduleElement,v);
        int len = col.size();
        l_IScheduleDTOArray = new Exch2kScheduleDTO[len];
        for (int i=0;i<len;i++) {
        	l_IScheduleDTOArray[i] = populateScheduleDTO(col.elementAt(i));
        }


		return l_IScheduleDTOArray;
	}// end getAllSchedules

	/**
	 * @return
	 */
	private Vector<String> makeItemList()
	{
		Vector<String> v = new Vector<String>();
		v.add("DAV:id");
		v.add("DAV:href");
		v.add("DAV:getcontentlength");
		v.add("DAV:ishidden");
		v.add("http://schemas.microsoft.com/exchange/sensitivity");
		v.add("http://schemas.microsoft.com/mapi/reminderset");
		v.add("http://schemas.microsoft.com/mapi/remindertime");
		v.add("urn:schemas:calendar:dtstart");
		v.add("urn:schemas:calendar:dtend");
		v.add("urn:schemas:calendar:location");
		v.add("urn:schemas:calendar:alldayevent");
		v.add("urn:schemas:calendar:busystatus");
		v.add("urn:schemas:calendar:created");
		v.add("urn:schemas:calendar:duration");
		v.add("urn:schemas:calendar:exdate");
		v.add("urn:schemas:calendar:exrule");
		v.add("urn:schemas:calendar:instancetype");
		v.add("urn:schemas:calendar:isorganizer");
		v.add("urn:schemas:calendar:lastmodified");
		v.add("urn:schemas:calendar:lastmodifiedtime");
		v.add("urn:schemas:calendar:meetingstatus");
		v.add("urn:schemas:calendar:organizer");
		v.add("urn:schemas:calendar:rdate");
		v.add("urn:schemas:calendar:recurrenceid");
		v.add("urn:schemas:calendar:reminderoffset");
		v.add("urn:schemas:calendar:remindernexttime");
		v.add("urn:schemas:calendar:resources");
		v.add("urn:schemas:calendar:rrule");
		v.add("urn:schemas:calendar:sequence");
		v.add("urn:schemas:calendar:timezone");
		v.add("urn:schemas:calendar:timezoneid");
		v.add("urn:schemas:calendar:uid");
		v.add("urn:schemas:calendar:version");

		v.add("urn:schemas:httpmail:subject");
		v.add("urn:schemas:httpmail:textdescription");
		v.add("urn:schemas:httpmail:hasattachment");

		v.add("urn:schemas:mailheader:to");
		v.add("urn:schemas:mailheader:cc");
		v.add("urn:schemas:mailheader:bcc");
		return v;
	}

	public IScheduleDTO[] searchSchedules(IAccountData acc,ISearchFormula form) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		SearchConditionCollection conds = form.getSearchConditions();
		if (conds==null) {
			throw new NullPointerException("Search Condition must not be null");
		}
		if (conds.size()==0) {
			return getAllSchedules(acc,null);
		}
		String idToSee = null;
		Date dateObj = null;
		IObjectIndex oid = null;
		Iterator<? extends ISearchConditionElement> it = conds.iterator();
		ScheduleSearchCondition cond = null;
//		String whereStr = "";
		while (it.hasNext()) {
			cond = (ScheduleSearchCondition)it.next();
			ISearchType type = cond.getType();
			Object value = cond.getValue();
Log.debug("condition:"+type+":"+value);
			if (type==ScheduleSearchCondition.CONDITION.TYPE_ACCOUNTNAME) {
				idToSee = ((IObjectIndex)value).getIndex().toString();
			}
			else if (type==ScheduleSearchCondition.CONDITION.TYPE_SINGLEDATE) {
				dateObj = (Date)value;
			}
			else if (type==ScheduleSearchCondition.CONDITION.TYPE_OID) {
				oid = (IObjectIndex)value;
			}
			else {
				throw new IllegalArgumentException("Illegal SearchCondition");
			}
		}
		if(idToSee==null) {
			return getSchedulesByDate(acc,dateObj);
		}

		Exch2kScheduleDTO[] l_IScheduleDTOArray = null;
		Vector<String> v = makeItemList();

		exSetAccountData((PasswordBasedAccountDataImpl)acc,idToSee);
        Vector<HashMap<String,Object>> col = super.exGetListByDate(DEPTH_1,scheduleElement,v,dateObj,"urn:schemas:calendar:dtstart");
        int len = col.size();
        l_IScheduleDTOArray = new Exch2kScheduleDTO[len];
        for (int i=0;i<len;i++) {
        	if (oid==null) {
            	l_IScheduleDTOArray[i] = populateScheduleDTO(col.elementAt(i));
        	}
        	else {
        		Exch2kScheduleDTO s = populateScheduleDTO(col.elementAt(i));
        		if (oid.equals(s.getOid())) {
        			l_IScheduleDTOArray = new Exch2kScheduleDTO[1];
        			l_IScheduleDTOArray[0] = s;
        			break;
        		}
        	}
        }


		return l_IScheduleDTOArray;
	}
	
	public IScheduleDTO[] getSchedulesByDate(IAccountData acc, Date dateObj) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		Exch2kScheduleDTO[] l_IScheduleDTOArray = null;

		Vector<String> v = makeItemList();
		setAccountData((PasswordBasedAccountDataImpl)acc);
        Vector<HashMap<String,Object>> col = super.getListByDate(scheduleElement,v,dateObj,"urn:schemas:calendar:dtstart");
        int len = col.size();
        l_IScheduleDTOArray = new Exch2kScheduleDTO[len];
        for (int i=0;i<len;i++) {
        	l_IScheduleDTOArray[i] = populateScheduleDTO(col.elementAt(i));
        }


		return l_IScheduleDTOArray;
	}// end getAllSchedules
	
	public IScheduleDTO getSchedule(IAccountData acc, IObjectIndex oid) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, HandleException
	{
Log.debug("getSchedule::oid="+oid);
		setAccountData((PasswordBasedAccountDataImpl)acc);
		HashMap<String,Object> sche = super.getProps((String)oid.getIndex(),makeItemList());
		return populateScheduleDTO(sche);
	} // end getSchedule

	public IObjectIndex createNewSchedule(IAccountData acc, IScheduleDTO scheduleDTO) throws IncorrectData, NoSuchRights, DataNotFound, IncorrectStore, ServerDown, HandleException
	{
		IObjectIndex oid = null;

		setAccountData((PasswordBasedAccountDataImpl)acc);
		Hashtable<String,String> h = populateHashtable(scheduleDTO);

		String url = generateBaseURL() + elmentName + "/" + new Date().getTime() +".EML";
		Hashtable<String,String> h2 = populateNewScheduleElements();
		h.putAll(h2);
		oid = super.saveData(escapeURL(url),h);
		if (scheduleDTO instanceof Exch2kScheduleDTO) {
			Exch2kScheduleDTO dto2 = (Exch2kScheduleDTO)scheduleDTO;
			if ("CONFIRMED".equals(dto2.getMeetingstatus())) {
				// send meeting request
				HashMap<String,Object> savedData = super.getFile(url);
				String uid = (String)savedData.get("uid");
				dto2.setUid(uid);
				h = populateHashtable(scheduleDTO);
				h2 = populateRequestElements();
				h.putAll(h2);
				url = generateBaseURL() + scheduleElement.getIndex().toString() + "/" + new Date().getTime() +"-request.EML";
				oid = super.saveData(escapeURL(url),h);
				super.sendSavedMsg(url);
			}
		}

		return oid;
	}

	public IObjectIndex updateSchedule(IAccountData acc, IScheduleDTO objIScheduleDTO) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, SearchConditionFailed, UpdateFailed,HandleException
	{
		IObjectIndex oid = null;
		Hashtable<String,String> h = populateHashtable(objIScheduleDTO);
		setAccountData((PasswordBasedAccountDataImpl)acc);

		String url = (String)objIScheduleDTO.getOid().getIndex();
		oid = super.saveData(url,h);

		return oid;
	}

	public void deleteSchedule(IAccountData acc, IObjectIndex oid) throws IncorrectData, DeleteFailed, NoSuchRights, ServerDown,DataNotFound, IncorrectStore, SearchConditionFailed,HandleException
	{
		
		setAccountData((PasswordBasedAccountDataImpl)acc);
		super.delete(oid);

	}

	private Exch2kScheduleDTO populateScheduleDTO(HashMap<String,Object> itemList) throws ServerDown, DataNotFound
	{
		Exch2kScheduleDTO l_IScheduleDTO = new Exch2kScheduleDTO();
		try {
			l_IScheduleDTO.setFieldSet(itemList);
		}
		catch (IllegalAccessException e) {
			Log.error("failed setting FieldSet",e);
		}
		l_IScheduleDTO.setOid(new Exch2kObjectIndex(itemList.get("href")));
		l_IScheduleDTO.setSubject((String)itemList.get("subject"));
		l_IScheduleDTO.setLocation((String)itemList.get("location"));
		l_IScheduleDTO.setBody((String)itemList.get("textdescription"));
		l_IScheduleDTO.setStartDate(davDateStr2JavaDate((String)itemList.get("dtstart")));
		l_IScheduleDTO.setEndDate(davDateStr2JavaDate((String)itemList.get("dtend")));
		try {
			boolean b = false;
			long l = Long.parseLong((String)itemList.get("alldayevent"));
			if (l!=0)b=true;
			l_IScheduleDTO.setAlldayevent(b);
		}catch(NumberFormatException e){};
		l_IScheduleDTO.setBusystatus((String)itemList.get("busystatus"));
		l_IScheduleDTO.setCreated(davDateStr2JavaDate((String)itemList.get("created")));

		try {l_IScheduleDTO.setDuration(Long.parseLong((String)itemList.get("duration")));}catch(NumberFormatException e){};
//		private String[] exdate;
		String[] exdate = new String[1];
		exdate[0] = (String)itemList.get("exdate");
		l_IScheduleDTO.setExdate(exdate);
		l_IScheduleDTO.setExrule((String)itemList.get("exrule"));
		try {l_IScheduleDTO.setInstancetype(Long.parseLong((String)itemList.get("instancetype")));}catch(NumberFormatException e){};
		try {
			boolean b = false;
			long l = Long.parseLong((String)itemList.get("isorganizer"));
			if (l!=0)b=true;
			l_IScheduleDTO.setIsorganizer(b);
		}catch(NumberFormatException e){};
		l_IScheduleDTO.setLastmodified(davDateStr2JavaDate((String)itemList.get("lastmodified")));
		l_IScheduleDTO.setLastmodifiedtime(davDateStr2JavaDate((String)itemList.get("lastmodifiedtime")));
		l_IScheduleDTO.setMeetingstatus((String)itemList.get("meetingstatus"));
		try {
			MailAddress[] organizer = MailAddress.parseAddressLine(MimeUtility.decodeText((String)itemList.get("organizer")));
			l_IScheduleDTO.setOrganizer(organizer[0]);
		}catch (Exception e) {};
		try {
			MailAddress[] TOs = MailAddress.parseAddressLine(MimeUtility.decodeText((String)itemList.get("to")));
			l_IScheduleDTO.setTo(TOs);
		}catch (Exception e) {};
		try {
			MailAddress[] CCs = MailAddress.parseAddressLine(MimeUtility.decodeText((String)itemList.get("cc")));
			l_IScheduleDTO.setCc(CCs);
		}catch (Exception e) {};
		try {
			MailAddress[] BCCs = MailAddress.parseAddressLine(MimeUtility.decodeText((String)itemList.get("bcc")));
			l_IScheduleDTO.setBcc(BCCs);
		}catch (Exception e) {};
//		private String[] rdate;
		l_IScheduleDTO.setRecurrenceid((String)itemList.get("recurrenceid"));
		try {l_IScheduleDTO.setReminderoffset(Long.parseLong((String)itemList.get("reminderoffset")));}catch(NumberFormatException e){};
		try {l_IScheduleDTO.setRemindernexttime(davDateStr2JavaDate((String)itemList.get("remindernexttime")));}catch(NumberFormatException e){};
		String reminderSet = (String)itemList.get("reminderset");
		if ("1".equals(reminderSet)) {
			l_IScheduleDTO.setReminderset(true);
		}
		else {
			l_IScheduleDTO.setReminderset(false);
		}

		l_IScheduleDTO.setResources((String)itemList.get("resources"));
		l_IScheduleDTO.setRrule((String)itemList.get("rrule"));
		try {l_IScheduleDTO.setSequence(Long.parseLong((String)itemList.get("sequence")));}catch(NumberFormatException e){};
		l_IScheduleDTO.setTimezone((String)itemList.get("timezone"));
		try {l_IScheduleDTO.setTimezoneid(Long.parseLong((String)itemList.get("timezoneid")));}catch(NumberFormatException e){};
		l_IScheduleDTO.setUid((String)itemList.get("uid"));
		l_IScheduleDTO.setVersion((String)itemList.get("version"));
		
		try {
			boolean b = false;
			long l = Long.parseLong((String)itemList.get("responserequested"));
			if (l!=0)b=true;
			l_IScheduleDTO.setResponserequested(b);
		}catch(NumberFormatException e){};

		return l_IScheduleDTO;
	}
	
	private Hashtable<String,String> populateRequestElements()
	{
		Hashtable<String,String> h = new Hashtable<String,String>();
		h.put("DAV:contentclass","urn:content-classes:calendarmessage");
		h.put("http://schemas.microsoft.com/exchange/outlookmessageclass","IPM.Schedule.Meeting.Request");
		
		h.put("http://schemas.microsoft.com/mapi/finvited","1");
		h.put("http://schemas.microsoft.com/mapi/responsestatus","1");
		h.put("http://schemas.microsoft.com/mapi/responsestate","0");
		h.put("http://schemas.microsoft.com/mapi/response_requested","1");
		h.put("http://schemas.microsoft.com/mapi/apptstateflags","3");
		h.put("http://schemas.microsoft.com/mapi/busystatus","1");
		h.put("http://schemas.microsoft.com/mapi/intendedbusystatus","2");
		return h;
	}
	
	private Hashtable<String,String> populateNewScheduleElements()
	{
		Hashtable<String,String> h = new Hashtable<String,String>();
		h.put("DAV:contentclass","urn:content-classes:appointment");
		h.put("http://schemas.microsoft.com/exchange/outlookmessageclass","IPM.Appointment");
		return h;
	}
	
	private Hashtable<String,String> populateHashtable(IScheduleDTO dto)
	{
		Hashtable<String,String> h = new Hashtable<String,String>();
Log.debug("dto="+dto);

		if (dto.getSubject()!=null) h.put("urn:schemas:httpmail:subject",dto.getSubject());
		if (dto.getLocation()!=null) h.put("urn:schemas:calendar:location",dto.getLocation());
		if (dto.getBody()!=null) h.put("urn:schemas:httpmail:textdescription",dto.getBody());
		if (dto.getStartDate()!=null) h.put("urn:schemas:calendar:dtstart",javaDate2MSDavDateStr(dto.getStartDate()));
		if (dto.getEndDate()!=null) h.put("urn:schemas:calendar:dtend",javaDate2MSDavDateStr(dto.getEndDate()));

		if (dto instanceof Exch2kScheduleDTO) {
			Exch2kScheduleDTO dto2 = (Exch2kScheduleDTO)dto;
			if (dto2.isResponserequested()) {
				h.put("urn:schemas:calendar:responserequested","1");
			}
			else {
				h.put("urn:schemas:calendar:responserequested","0");
			}
			if (dto2.isAlldayevent()) {
				h.put("urn:schemas:calendar:alldayevent","1");
			}
			else {
				h.put("urn:schemas:calendar:alldayevent","0");
			}
			if (dto2.getBusystatus()!=null) h.put("urn:schemas:calendar:busystatus",dto2.getBusystatus());
//			private String[] exdate;
//			private String exrule;
			h.put("urn:schemas:calendar:instancetype",""+dto2.getInstancetype());
			if (dto2.getMeetingstatus()!=null) h.put("urn:schemas:calendar:meetingstatus",dto2.getMeetingstatus());
//			private String[] rdate;
//			private String recurrenceid;
//			h.put("urn:schemas:calendar:reminderoffset",""+dto2.getReminderoffset());
			if (dto2.getRemindernexttime()!=null) h.put("urn:schemas:calendar:remindernexttime",javaDate2MSDavDateStr(dto2.getRemindernexttime()));
			if (dto2.isReminderset()) {
				h.put("http://schemas.microsoft.com/mapi/reminderset","1");
			}
			else {
				h.put("http://schemas.microsoft.com/mapi/reminderset","0");
			}
//			private String resources;
//			private String rrule;
//			private long sequence;
//			private String timezone;
//			private long timezoneid;
//			private String uid;
			if (dto2.getUid()!=null) h.put("urn:schemas:calendar:uid",dto2.getUid());
			MailAddress[] tmp = dto2.getTo();
			String addressStr = "";
			if (tmp != null) {
				for (int i=0;i<tmp.length;i++) {
					addressStr = addressStr + ((InternetAddress)populateAddress(tmp[i])).getAddress() + ",";
				}
				if (addressStr.length()>0) {
					addressStr = addressStr.substring(0,addressStr.length()-1);
					h.put("urn:schemas:mailheader:to",""+addressStr);
				}
			}
			tmp = dto2.getCc();
			addressStr = "";
			if (tmp != null) {
				for (int i=0;i<tmp.length;i++) {
					addressStr = addressStr + ((InternetAddress)populateAddress(tmp[i])).getAddress() + ",";
				}
				if (addressStr.length()>0) {
					addressStr = addressStr.substring(0,addressStr.length()-1);
					h.put("urn:schemas:mailheader:cc",""+addressStr);
				}
			}
			tmp = dto2.getBcc();
			addressStr = "";
			if (tmp != null) {
				for (int i=0;i<tmp.length;i++) {
					addressStr = addressStr + ((InternetAddress)populateAddress(tmp[i])).getAddress() + ",";
				}
				if (addressStr.length()>0) {
					addressStr = addressStr.substring(0,addressStr.length()-1);
					h.put("urn:schemas:mailheader:bcc",""+addressStr);
				}
			}
		}
		

//		h.put("urn:schemas:httpmail:sendername",super.getUserID());
		return h;
		
	}

	protected DavSearchCondition convertCondition(SearchCondition src) throws HandleException
	{
		throw new HandleException("Not yet implemented");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.eai.exch2k.logic.EAIExch2kBase#populateMessageFile(java.util.Hashtable)
	 */
	@SuppressWarnings("unchecked")
	protected String populateMessageFile(Hashtable<String,String> itemList)
	{
		String subject = itemList.get("urn:schemas:httpmail:subject");
		String body = itemList.get("urn:schemas:httpmail:textdescription");
//		String from = itemList.get("urn:schemas:httpmail:sendername");
		String location = itemList.get("urn:schemas:calendar:location");
		String startDate = itemList.get("urn:schemas:calendar:dtstart");
		String endDate = itemList.get("urn:schemas:calendar:dtend");
		String timeStampStr = javaDate2MSDavDateStr(new Date());

		String msg = "content-class: urn:content-classes:appointment\r\n";
		try {
//			msg = msg + "Subject: " + MimeUtility.encodeText(subject,"UTF-8","B") + "\r\n";
//			msg = msg + "Mime-Version: 1.0" + "\r\n";
//			msg = msg + "Content-Type: text/calendar; method=REQUEST; charset='utf-8'" + "\r\n";
			msg = msg + "Content-Transfer-Encoding: 8bit" + "\r\n";
			msg = msg + "Thread-Topic: " + MimeUtility.encodeText(subject,"UTF-8","B") + "\r\n";
		}
		catch (Exception e) {
			Log.error("Exception on creating MimeMessage",e);
		}
		String vcalStr = "";
		vcalStr = vcalStr + "BEGIN:VCALENDAR\r\n";
		vcalStr = vcalStr + "METHOD:REQUEST\r\n";
		vcalStr = vcalStr + "PRODID:Microsoft CDO for Microsoft Exchange\r\n";
		vcalStr = vcalStr + "VERSION:2.0\r\n";
		vcalStr = vcalStr + "BEGIN:VTIMEZONE\r\n";
		vcalStr = vcalStr + "TZID:Osaka\\, Sapporo\\, Tokyo\r\n";
		vcalStr = vcalStr + "X-MICROSOFT-CDO-TZID:20\r\n";
		vcalStr = vcalStr + "BEGIN:STANDARD\r\n";
		vcalStr = vcalStr + "DTSTART:16010101T000000\r\n";
		vcalStr = vcalStr + "TZOFFSETFROM:+0900\r\n";
		vcalStr = vcalStr + "TZOFFSETTO:+0900\r\n";
		vcalStr = vcalStr + "END:STANDARD\r\n";
		vcalStr = vcalStr + "BEGIN:DAYLIGHT\r\n";
		vcalStr = vcalStr + "DTSTART:16010101T000000\r\n";
		vcalStr = vcalStr + "TZOFFSETFROM:+0900\r\n";
		vcalStr = vcalStr + "TZOFFSETTO:+0900\r\n";
		vcalStr = vcalStr + "END:DAYLIGHT\r\n";
		vcalStr = vcalStr + "END:VTIMEZONE\r\n";
		vcalStr = vcalStr + "BEGIN:VEVENT\r\n";
			
		vcalStr = vcalStr + "DTSTAMP:"+timeStampStr+"\r\n";
		vcalStr = vcalStr + "DTSTART;TZID=\"Osaka, Sapporo, Tokyo\":" + startDate + "\r\n";
		vcalStr = vcalStr + "SUMMARY:" + subject + "\r\n";
		vcalStr = vcalStr + "UID:jp.co.amax." + timeStampStr + "\r\n";
		vcalStr = vcalStr + "DTEND;TZID=\"Osaka, Sapporo, Tokyo\":" + endDate + "\r\n";
		vcalStr = vcalStr + "LOCATION:" + location + "\r\n";
		vcalStr = vcalStr + "DESCRIPTION:" + body.replaceAll("\n","\\\\N").replaceAll("\r","") + "\r\n";
//		vcalStr = vcalStr + "DESCRIPTION:" + body + "\r\n";
		vcalStr = vcalStr + "SEQUENCE:0\r\n";
		vcalStr = vcalStr + "PRIORITY:5\r\n";
		vcalStr = vcalStr + "CLASS:\r\n";
		vcalStr = vcalStr + "CREATED:"+timeStampStr+"\r\n";
		vcalStr = vcalStr + "LAST-MODIFIED:"+timeStampStr+"\r\n";
		vcalStr = vcalStr + "STATUS:TENTATIVE\r\n";
		vcalStr = vcalStr + "TRANSP:OPAQUE\r\n";
		
		vcalStr = vcalStr + "X-MICROSOFT-CDO-BUSYSTATUS:BUSY\r\n";
		vcalStr = vcalStr + "X-MICROSOFT-CDO-INSTTYPE:0\r\n";
		vcalStr = vcalStr + "X-MICROSOFT-CDO-ALLDAYEVENT:FALSE\r\n";
		vcalStr = vcalStr + "X-MICROSOFT-CDO-IMPORTANCE:1\r\n";
		vcalStr = vcalStr + "X-MICROSOFT-CDO-OWNERAPPTID:-1\r\n";
		vcalStr = vcalStr + "END:VEVENT\r\n";
		vcalStr = vcalStr + "END:VCALENDAR\r\n";
Log.debug("vcalStr="+vcalStr);


		try {
			MimeMessage mmsg = new MimeMessage((Session)null);
			MimeMultipart part = new MimeMultipart();
			MimeBodyPart pbody = new MimeBodyPart();
			
			mmsg.setSubject(subject,"UTF-8");
//			mmsg.setHeader("Content-Type","text/calendar; method=REQUEST; charset='UTF-8'");

			pbody.setText(vcalStr,"UTF-8");
			part.addBodyPart(pbody);
			pbody.setHeader("Content-Type","text/calendar; method=REQUEST; charset='UTF-8'");
			mmsg.setContent(part);
			mmsg.saveChanges();

			
			Enumeration<String> lines = mmsg.getAllHeaderLines();
			while (lines.hasMoreElements()) {
				msg = msg + lines.nextElement() + "\r\n";
			}
			msg = msg + "\r\n";
			msg = msg + "content-class: urn:content-classes:appointment\r\n";
			InputStream is = mmsg.getInputStream();
			BufferedReader b = new BufferedReader(new InputStreamReader(is));
			String tmp;
			while ((tmp = b.readLine())!=null) {
				msg = msg + tmp + "\r\n";
			}
		
		}
		catch (Exception e) {
			Log.error("Exception on creating MimeMessage",e);
		}
		
		
//		msg = msg + "\r\n" + vcalStr;

		return msg;
	}

	public ISearchDestination getPresetDestination(IAccountData acc,int type) throws IncorrectData, HandleException
	{
		super.setAccountData((PasswordBasedAccountDataImpl)acc);
		IFolderIndex folder = null;
		IDatabaseIndex db = null;
		switch (type) {
		case IScheduleStore.DEST_TYPE_DEFAULT_SCHEDULE_FOLDER:
			folder = (IFolderIndex)populateFolder(scheduleElement).getOid();
			break;
		case IScheduleStore.DEST_TYPE_OTHERS_SCHEDULEBOX:
			folder = (IFolderIndex)populateFolder(scheduleElement).getOid();
			break;
		default:
			throw new IncorrectData("illegal folderType");
		}
		return (ISearchDestination)new BasicSearchDestination(db,folder);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.eai.exch2k.store.EAIExch2kBase#getElmentName()
	 */
	protected String getElmentName()
	{
		return DEFAULT_EXCHANGE_SCHEDULE_PATH_ELEMENT;
	}
}

