/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.exch2k.store;

import java.util.*;

import jp.co.connectone.eai.exch2k.dav.DavSearchCondition;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;
import jp.co.connectone.user.PasswordBasedAccountDataImpl;

public class Exch2kMemoStoreImpl extends EAIExch2kBase implements IMemoStore
{
	private static final String DEFAULT_EXCHANGE_MEMO_PATH_ELEMENT="";
	private IFolderIndex memoElement = new Exch2kFolderIndex(DEFAULT_EXCHANGE_MEMO_PATH_ELEMENT);
	protected static final String elmentName=DEFAULT_EXCHANGE_MEMO_PATH_ELEMENT;

	public static final IStoreID storeID = new SimpleStoreID(Exch2kMemoStoreImpl.class.getName());
	public static final String storeName = "Exchange2K memo store";
	public String getName() throws Exception
	{
		return storeName;
	}
	
	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		return super.getServiceInfo(storeID, storeName, serviceData);
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new Exch2kServiceInfo(storeID,storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}


	public IMemoDTO[] getAllMemos(IAccountData acc) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		IMemoDTO[] l_scheduleDtoArray = null;

		Vector<String> v = makeItemList();
		setAccountData((PasswordBasedAccountDataImpl)acc);
        Vector<HashMap<String,Object>> col = super.getList(memoElement,v);
        int len = col.size();
        l_scheduleDtoArray = new IMemoDTO[len];
        for (int i=0;i<len;i++) {
        	l_scheduleDtoArray[i] = populateMemoDTO(col.elementAt(i));
        }


		return l_scheduleDtoArray;
	}// end getAllMemos
	
	public IMemoDTO[] getMemosByDate(IAccountData acc, Date dateObj) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		IMemoDTO[] l_scheduleDtoArray = null;

		Vector<String> v = makeItemList();
		setAccountData((PasswordBasedAccountDataImpl)acc);
        Vector<HashMap<String,Object>> col = super.getListByDate(memoElement,v,dateObj,"urn:schemas:httpmail:datereceived");
        int len = col.size();
        l_scheduleDtoArray = new IMemoDTO[len];
        for (int i=0;i<len;i++) {
        	l_scheduleDtoArray[i] = populateMemoDTO(col.elementAt(i));
        }


		return l_scheduleDtoArray;
	}// end getAllMemos

	/**
	 * @return
	 */
	private Vector<String> makeItemList()
	{
		Vector<String> v = new Vector<String>();
		v.add("DAV:id");
		v.add("DAV:href");
		v.add("urn:schemas:httpmail:subject");
		v.add("urn:schemas:httpmail:textdescription");
		v.add("urn:schemas:httpmail:sendername");
		v.add("urn:schemas:httpmail:datereceived");
		return v;
	}
	
	public IMemoDTO getMemoByOID(IAccountData acc, IObjectIndex oid) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, HandleException
	{
Log.debug("getMemoByOID::oid="+oid);
		setAccountData((PasswordBasedAccountDataImpl)acc);
		HashMap<String,Object> memo = super.getProps((String)oid.getIndex(),makeItemList());
		return populateMemoDTO(memo);
	}
	
	public IObjectIndex createNewMemo(IAccountData acc, IMemoDTO IMemoDTO) throws IncorrectData, NoSuchRights, DataNotFound, IncorrectStore, ServerDown, HandleException
	{
		IObjectIndex oid = null;

		Hashtable<String,String> h = populateHashtable(IMemoDTO);
		setAccountData((PasswordBasedAccountDataImpl)acc);

		String url = generateBaseURL() + elmentName + "/" + new Date().getTime() +".EML";
		oid = super.saveData(escapeURL(url),h);

		return oid;
	}

	public IObjectIndex updateMemo(IAccountData acc, IMemoDTO objIMemoDTO) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound,IncorrectStore, SearchConditionFailed, UpdateFailed,HandleException
	{
		IObjectIndex oid = null;

		Hashtable<String,String> h = populateHashtable(objIMemoDTO);
		setAccountData((PasswordBasedAccountDataImpl)acc);

		String url = (String)objIMemoDTO.getOid().getIndex();
		oid = super.saveData(url,h);

		return oid;
	}

	public void deleteMemo(IAccountData acc, IObjectIndex oid) throws IncorrectData, DeleteFailed, NoSuchRights, ServerDown,DataNotFound, IncorrectStore, SearchConditionFailed,HandleException
	{

		setAccountData((PasswordBasedAccountDataImpl)acc);
		super.delete(oid);

	}

	private Hashtable<String,String> populateHashtable(IMemoDTO memo)
	{
		Hashtable<String,String> h = new Hashtable<String,String>();
Log.debug("memo="+memo);
		
		if (memo.getSubject()!=null) h.put("urn:schemas:httpmail:subject",memo.getSubject());
		if (memo.getBody()!=null) h.put("urn:schemas:httpmail:textdescription",memo.getBody());
		if (memo.getSenderName()!=null) h.put("urn:schemas:httpmail:sendername",memo.getSenderName());
		
		return h;
	}

	/**
	 * This method populates a IMemoDTO object by extracting data from a
	 * document Object
	 * 
	 * @param l_docObj Document Object representing the Memo
	 * @throws ServerDown, DataNotFound
	 */
	private IMemoDTO populateMemoDTO(HashMap<String,Object> itemList) throws ServerDown, DataNotFound
	{
		BasicMemoDTO l_IMemoDTO = new BasicMemoDTO();
		l_IMemoDTO.setNew(false);

		l_IMemoDTO.setTid((String)itemList.get("id"));
		l_IMemoDTO.setOid(new Exch2kObjectIndex(itemList.get("href")));
		l_IMemoDTO.setSubject((String)itemList.get("subject"));
		l_IMemoDTO.setBody((String)itemList.get("textdescription"));
		l_IMemoDTO.setSenderName((String)itemList.get("sendername"));
		l_IMemoDTO.setDate(davDateStr2JavaDate((String)itemList.get("datereceived")));
		
		return l_IMemoDTO;
	}

	protected DavSearchCondition convertCondition(SearchCondition src) throws HandleException
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("convertCondition not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#delete(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		deleteMemo(acc,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getAllDatas(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getAllMemos(acc);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getFolderList(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		FolderMetadata[]  l_strFolderlist = new FolderMetadata[0]; 		   	 /* convert Mail Folders in Array of String[] */
		// Folder String format should be "root/parent/child/..." format.
		
		setAccountData((PasswordBasedAccountDataImpl)acc);
		IFolderIndex path = dest.getFolder();
		if (path==null) path=memoElement;
		Vector<Object> lists = super.getAllSubFolders((String)path.getIndex());
		ArrayList<FolderMetadata> arrFolderlist = populateFolderList(0,(String)path.getIndex(),lists);
		l_strFolderlist = (FolderMetadata[])arrFolderlist.toArray(l_strFolderlist);
		for (int i=0;i<l_strFolderlist.length;i++) {
			if (l_strFolderlist[i].getFolderName().length()==0) {
				l_strFolderlist[i].setFolderName((String)path.getIndex());
			}
		}
		return l_strFolderlist;
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#read(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return getMemoByOID(acc,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#search(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.ISearchFormula)
	 */
	public IRecordObject[] search(IAccountData acc, ISearchFormula col) throws Exception
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("search not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#searchByDate(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, java.util.Date)
	 */
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date date) throws Exception
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("searchByDate not yet supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#write(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IRecordObject)
	 */
	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject data) throws Exception
	{
		IObjectIndex rc = null;
		if (data.isNew()) {
			rc = createNewMemo(acc,(IMemoDTO)data);
		}
		else {
			rc = updateMemo(acc,(IMemoDTO)data);
		}
		return rc;
	}


	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getPresetDestination(int)
	 */
	public ISearchDestination getPresetDestination(IAccountData acc,int type) throws IncorrectData, HandleException
	{
		super.setAccountData((PasswordBasedAccountDataImpl)acc);
		IFolderIndex folder = null;
		IDatabaseIndex db = null;
		switch (type) {
		case IMemoStore.DEST_TYPE_DEFAULT_MEMO_FOLDER:
			folder = (IFolderIndex)populateFolder(memoElement).getOid();
			break;
		default:
			throw new IncorrectData("illegal folderType");
		}
		return (ISearchDestination)new BasicSearchDestination(db,folder);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.eai.exch2k.store.EAIExch2kBase#getElmentName()
	 */
	protected String getElmentName()
	{
		return DEFAULT_EXCHANGE_MEMO_PATH_ELEMENT;
	}
	
}

