/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.notes.store;

import java.io.*;
import java.util.*;

import lotus.domino.*;

import jp.co.connectone.service.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;
import jp.co.connectone.eai.notes.user.NotesAccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;

public class NotesScheduleStoreImpl extends EAINotesBase implements IScheduleStore
{
	public static final IStoreID storeID = new SimpleStoreID(NotesScheduleStoreImpl.class.getName());

	public static final String storeName = NotesConstants.SCHEDULESTORE_NAME;

	public static final String DEFAULT_NOTES_CALENDAR_NAME = "($Calendar)";

	private IFolderIndex calendarElement = new NotesFolderIndex(DEFAULT_NOTES_CALENDAR_NAME);

	public String getName() throws Exception
	{
		return storeName;
	}

	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		NotesServiceInfo info = new NotesServiceInfo(storeID, storeName);
		String serverAddress = (String) h.get("server");
		String serverAddress2 = (String)h.get("server2");
		if (serverAddress!=null) info.setServerAddress(serverAddress);
		if (serverAddress2!=null) info.setServerAddress2(serverAddress2);
		return info;
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new NotesServiceInfo(storeID, storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#delete(jp.co.connectone.user.IAccountData,
	 *      java.lang.String, java.lang.String,
	 *      jp.co.connectone.store.IObjectIndex)
	 */
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		deleteSchedule(acc, oid);
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#write(jp.co.connectone.user.IAccountData,
	 *      java.lang.String, java.lang.String,
	 *      jp.co.connectone.store.IRecordObject)
	 */
	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject data) throws Exception
	{
		IObjectIndex rc = null;
		if (data.isNew()) {
			rc = createNewSchedule(acc, (IScheduleDTO) data);
		}
		else {
			rc = updateSchedule(acc, (IScheduleDTO) data);
		}
		return rc;
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#getAllDatas(jp.co.connectone.user.IAccountData,
	 *      java.lang.String, java.lang.String)
	 */
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("get All shcedules without date not yet supported.");
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#getFolderList(jp.co.connectone.user.IAccountData,
	 *      java.lang.String, java.lang.String)
	 */
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("get folder list on schedule not yet supported.");
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#read(jp.co.connectone.user.IAccountData,
	 *      java.lang.String, java.lang.String,
	 *      jp.co.connectone.store.IObjectIndex)
	 */
	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return getSchedule(acc, oid);
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#search(jp.co.connectone.user.IAccountData,
	 *      java.lang.String, java.lang.String,
	 *      jp.co.connectone.store.SearchConditionCollection)
	 */
	public IRecordObject[] search(IAccountData acc, ISearchFormula form) throws Exception
	{
		return searchSchedules(acc, form);
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#searchByDate(jp.co.connectone.user.IAccountData,
	 *      java.lang.String, java.lang.String, java.util.Date)
	 */
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date date) throws Exception
	{
		return getSchedulesByDate(acc, date);
	}

	public IScheduleDTO[] getAllSchedules(IAccountData acc, Date cutOffDate) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		IScheduleDTO[] l_scheduleDtoArray = null;
		Document l_docObj = null;
		String l_strQuery = null;
		DocumentCollection l_objDocCollection = null;
		String[] l_strQueryArgs = null;
		super.initSession((NotesAccountData) acc);
		try {
			l_strQueryArgs = new String[Integer.parseInt(ResourceBandleHandler.getString("notes.GETALLSCHEDULES_QUERY_TOTAL_ARGS"))];
			l_strQueryArgs[0] = createDateTime(cutOffDate).toString();
			// Get the Search Query from the Properties file
			l_strQuery = generateQuery(l_strQueryArgs, "notes.GETALLSCHEDULES_QUERY");
			l_objDocCollection = search(l_strQuery, null);// What's a
															// difference??
			int l_intDocCount = l_objDocCollection.getCount();
			if (l_intDocCount == 0) return new NotesScheduleDTO[0];

			// if total number of documents is greater than zero thefound
			if (l_objDocCollection.getCount() > 0) {
				l_scheduleDtoArray = new NotesScheduleDTO[l_intDocCount];
				l_docObj = l_objDocCollection.getFirstDocument();
				l_scheduleDtoArray[0] = populateScheduleDto(l_docObj);

				for (int intIndex = 1; intIndex < l_intDocCount; intIndex++) {
					l_docObj = l_objDocCollection.getNextDocument();
					l_scheduleDtoArray[intIndex] = populateScheduleDto(l_docObj);
				}
			}
		}
		catch (MissingResourceException l_missResExObj) {
			throw new DataNotFound();
		}
		catch (NotesException l_notesExceptionObj) {
			if (cutOffDate == null) throw new DataNotFound();
			else
				throw new ServerDown(acc.getServiceInfo().getServerAddress());
		}
		catch (NullPointerException l_nullPointerEx) {
			throw new DataNotFound();
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
				if (l_objDocCollection != null) {
					l_objDocCollection.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
		return l_scheduleDtoArray;
	}// end getAllSchedules

	public IScheduleDTO[] getSchedulesByDate(IAccountData acc, Date dateObj) throws ServerDown, DataNotFound, SearchConditionFailed, HandleException
	{
		IScheduleDTO[] l_scheduleDtoArray = null;
		View l_viewObj = null;
		DateTime l_dateTimeObj = null;
		Document l_docObj = null;
		DocumentCollection l_objDocCollection = null;
		super.initSession((NotesAccountData) acc);
		try {
			l_viewObj = getView(calendarElement);
			Log.debug("l_viewObj=" + l_viewObj);
			l_dateTimeObj = super.createDateTime(dateObj);
			// To get a mails on a particular day, Date should be in
			// "YYYY/MM/DD" format
			try {
				l_dateTimeObj = super.createDateTime(l_dateTimeObj.getDateOnly());
			}
			catch (java.text.ParseException pe) {
				Log.error("", pe);
				throw new IncorrectData(pe.getMessage());
			}
			Log.debug("l_viewObj=" + l_viewObj + ";l_dateTimeObj = "
					+ l_dateTimeObj);
			l_objDocCollection = l_viewObj.getAllDocumentsByKey((Object) l_dateTimeObj);
			int l_intDocCount = l_objDocCollection.getCount();

			if (l_intDocCount == 0) return new NotesScheduleDTO[0];
			// if total number of documents is greater than zero thefound
			if (l_objDocCollection.getCount() > 0) {
				l_scheduleDtoArray = new NotesScheduleDTO[l_intDocCount];
				l_docObj = l_objDocCollection.getFirstDocument();
				l_scheduleDtoArray[0] = populateScheduleDto(l_docObj);

				for (int intIndex = 1; intIndex < l_intDocCount; intIndex++) {
					l_docObj = l_objDocCollection.getNextDocument();
					l_scheduleDtoArray[intIndex] = populateScheduleDto(l_docObj);
				}
			}
		}
		catch (MissingResourceException l_missResExObj) {
			throw new DataNotFound();
		}
		catch (NotesException l_notesExceptionObj) {
			throw new ServerDown(acc.getServiceInfo().getServerAddress());
		}
		catch (NullPointerException l_nullPointerEx) {
			throw new DataNotFound();
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
				if (l_objDocCollection != null) {
					l_objDocCollection.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
		return l_scheduleDtoArray;
	}// end getAllSchedules

	public IScheduleDTO getSchedule(IAccountData acc, IObjectIndex oid) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, HandleException
	{
		IScheduleDTO l_scheduleDTO = null;
		Document l_docObj = null;
		// Create the session
		super.initSession((NotesAccountData) acc);
		try {
			// Get the document by UNID
			l_docObj = getDocumentByUNID(oid);
			// Populate the Schedule DTO
			l_scheduleDTO = populateScheduleDto(l_docObj);
		}
		catch (Exception l_excepObj) {
			throw new DataNotFound();
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
		return l_scheduleDTO;
	} // end getScheduleByUNID

	public IObjectIndex createNewSchedule(IAccountData acc, IScheduleDTO scheduleDTO) throws IncorrectData, NoSuchRights, DataNotFound, IncorrectStore, ServerDown, HandleException
	{
		String strUNID = null;
		Document l_docObj = null;
		NotesAccountData acc2 = (NotesAccountData) acc;
		String strUser = acc2.getUserID();
		String strServer = acc2.getServiceInfo().getServerAddress();
		int intSchType = NotesConstants.SCHEDULE_APPOINTMENT;
		super.initSession((NotesAccountData) acc);
		try {
			// Create DATETIME objects
			DateTime startDT = createDateTime(scheduleDTO.getStartDate());
			DateTime endDT = createDateTime(scheduleDTO.getEndDate());
			// Create a new schedule if only the schedule is a valid schedule
			if (this.isValidSchedule(scheduleDTO) == true) {
				l_docObj = CreateDocument();

				strUNID = l_docObj.getUniversalID();
				// Update start and end time.
				l_docObj.replaceItemValue(NotesConstants.APPEND_START_TIME, startDT);
				l_docObj.replaceItemValue(NotesConstants.APPEND_END_TIME, endDT);
				// Update other datetime fields
				l_docObj.replaceItemValue(NotesConstants.START_DATE_TIME, startDT);
				l_docObj.replaceItemValue(NotesConstants.CALANDER_DATE_TIME, startDT);
				l_docObj.replaceItemValue(NotesConstants.START__DATE, startDT);
				l_docObj.replaceItemValue(NotesConstants.NO_PURGE, endDT);
				// Update subject and body fields
				l_docObj.replaceItemValue(NotesConstants.SUBJECT, scheduleDTO.getSubject());
				// Create or modify, body rich text field
				RichTextItem rtItem = l_docObj.createRichTextItem(NotesConstants.BODY);
				if (scheduleDTO instanceof NotesScheduleDTO) {
					NotesScheduleDTO nScheduleDTO = (NotesScheduleDTO) scheduleDTO;
					// Get the schedule type
					intSchType = nScheduleDTO.getScheduleType();
					String[] l_description = nScheduleDTO.getDescription();
					if (l_description != null) {
						for (int i = 0; i < l_description.length; i++) {
							rtItem.appendText(l_description[i]);
						}
					}
				}
			}
			Vector<String> l_vecExcludeFromViewObj = new Vector<String>(2);
			// If schedule is not a Task (To Do item), then update the following
			// fields
			if (intSchType != NotesConstants.SCHEDULE_TASK) {
				l_docObj.replaceItemValue(NotesConstants.LOCATION, scheduleDTO.getLocation());
				l_docObj.replaceItemValue(NotesConstants.START_TIME, startDT);
				l_docObj.replaceItemValue(NotesConstants.END_TIME, endDT);
				l_docObj.replaceItemValue(NotesConstants.END_DATE_TIME, endDT);
				l_docObj.replaceItemValue(NotesConstants.END__DATE, endDT);
				// Update only if it is new schedule, do not update for
				// modification
				l_vecExcludeFromViewObj.add(NotesConstants.Letter_S);
				l_docObj.replaceItemValue(NotesConstants.FORM, NotesConstants.APPOINTMENT);
				l_docObj.replaceItemValue(NotesConstants.APPOINTMENT_TYPE, ""
						+ intSchType);
			}
			else {
				l_docObj.replaceItemValue(NotesConstants.DUE_STATE, new Integer(2));
				l_docObj.replaceItemValue(NotesConstants.DUE_DATE_TIME, endDT);
				l_docObj.replaceItemValue(NotesConstants.DUE_DATE, endDT);
				// Update only if it is new schedule, do not update for
				// modification
				l_docObj.replaceItemValue(NotesConstants.FORM, NotesConstants.TASK);
				l_docObj.replaceItemValue(NotesConstants.TASK_TYPE, NotesConstants.ONE);
				l_docObj.replaceItemValue(NotesConstants.APPOINTMENT_TYPE, NotesConstants.EMPTY_STRING);
			}
			l_vecExcludeFromViewObj.add(NotesConstants.Letter_D);
			l_docObj.replaceItemValue(NotesConstants.EXCLUDE_FROM_VIEW, l_vecExcludeFromViewObj);
			Item item = l_docObj.replaceItemValue(NotesConstants.BUSY_NAME, strUser);
			item.setNames(true);
			item = l_docObj.replaceItemValue(NotesConstants.FROM, strUser);
			item.setAuthors(true);
			item.setNames(true);
			// String trackString =
			// ResourceBandleHandler.getString("notes.VERSION") + new Date();
			// l_docObj.replaceItemValue(NotesConstants.CSTRACK, trackString);
			l_docObj.replaceItemValue(NotesConstants.PRINCIPAL, strUser);
			l_docObj.replaceItemValue(NotesConstants.ALT_PRINCIPAL, strUser);
			l_docObj.replaceItemValue(NotesConstants.CHAIR, strUser);
			l_docObj.replaceItemValue(NotesConstants.ALT_CHAIR, strUser);
			l_docObj.replaceItemValue(NotesConstants.APPT_UNID, strUNID);
			l_docObj.replaceItemValue(NotesConstants.VIEW_ICON, this.getScheduleViewIcon(intSchType));
			// Update other variables with null values and some properties.
			// This is to create the shcedules similar to
			this.setScheduleVariables(l_docObj);
			l_docObj.save();
		}
		catch (NullPointerException l_nullPtrEx) {
			Log.error("", l_nullPtrEx);
			throw new DataNotFound();
		}
		catch (DataNotFound l_exceptionObj) {
			Log.error("", l_exceptionObj);
			throw new DataNotFound();
		}
		catch (NotesException l_exceptionObj) {
			Log.error("", l_exceptionObj);
			throw new ServerDown(strServer);
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
		IObjectIndex idx = new NotesObjectIndex(strUNID);
		return idx;
	}// end createNewSchedule

	public IObjectIndex updateSchedule(IAccountData acc, IScheduleDTO objScheduleDTO) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, UpdateFailed, HandleException
	{
		DocumentCollection l_docCollectionObj = null;
		Document l_docObj = null;
		IObjectIndex idx = null;
		super.initSession((NotesAccountData) acc);
		try {
			l_docObj = getDocumentByUNID(objScheduleDTO.getOid());

			updateScheduleFields(l_docObj, objScheduleDTO);
			l_docObj.save();
			idx = new NotesObjectIndex(l_docObj.getUniversalID());
		}
		catch (NotesException l_notesExObj) {
			Log.error("", l_notesExObj);
			if (objScheduleDTO != null && l_docObj != null) {
				throw new UpdateFailed();
			}
			else
				throw new ServerDown(acc.getServiceInfo().getServerAddress());
		}
		catch (NullPointerException l_nullPointerExcepObj) {
			throw new DataNotFound();
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
				if (l_docCollectionObj != null) {
					l_docCollectionObj.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
		return idx;
	}// end updateSchedule

	public IScheduleDTO[] searchSchedules(IAccountData acc, ISearchFormula form) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		throw new HandleException("not yet implemented");
	}

	public void deleteSchedule(IAccountData acc, IObjectIndex strUNID) throws IncorrectData, DeleteFailed, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, SearchConditionFailed, HandleException
	{
		Document l_docObj = null;
		DocumentCollection l_docCollectionObj = null;
		super.initSession((NotesAccountData) acc);
		try {
			l_docObj = getDocumentByUNID(strUNID);
			l_docObj.remove(true);

		}
		catch (NotesException l_notesExObj) {
			if ((l_docCollectionObj != null) && (l_docObj != null)) throw new DeleteFailed();
			else
				throw new ServerDown(acc.getServiceInfo().getServerAddress());
		}
		catch (NullPointerException l_nullPointerExcepObj) {
			throw new DataNotFound();
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
				if (l_docCollectionObj != null) {
					l_docCollectionObj.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
	}// end deleteSchedule

	public boolean isValidSchedule(IAccountData acc, IScheduleDTO scheduleDTO) throws IncorrectData, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, HandleException
	{
		return true;
	}// end isValidSchedule with parameters

	public boolean isValidSchedule(IScheduleDTO scheduleDTO) throws ServerDown, DataNotFound, HandleException
	{
		return true;
	}// end isValidSchedule

	public boolean isDuplicateSchedule(IAccountData acc, IScheduleDTO scheduleDTO) throws NoSuchRights, ServerDown, DataNotFound, IncorrectData, HandleException
	{
		boolean l_boolIsDuplicateSchedule = false;
		DocumentCollection l_objDocCollection = null;
		String l_strQuery;
		String[] l_strQueryArgs;
		super.initSession((NotesAccountData) acc);
		try {
			// The query that is used to check if there is a duplciate instance
			l_strQueryArgs = new String[Integer.parseInt(ResourceBandleHandler.getString("notes.ISDUPLICATE_SCHEDULE_TOTAL_ARGS"))];
			l_strQueryArgs[0] = createDateTime(scheduleDTO.getStartDate()).toString();
			l_strQueryArgs[1] = createDateTime(scheduleDTO.getEndDate()).toString();
			if (scheduleDTO instanceof NotesScheduleDTO) l_strQueryArgs[2] = ""
					+ ((NotesScheduleDTO) scheduleDTO).getScheduleType();
			l_strQuery = generateQuery(l_strQueryArgs, "notes.ISDUPLICATE_SCHEDULE");
			// searching for the duplicate instance
			// l_objDocCollection = search(l_strQuery, null, 0);
			l_objDocCollection = search(l_strQuery, null);
			// if no documents found with the given condition duplicate schedule
			// = 0
			if (l_objDocCollection.getCount() > 0) {
				l_boolIsDuplicateSchedule = true;
			}
		}
		catch (NullPointerException l_nullPtrExObj) {
			throw new DataNotFound();
		}
		catch (NotesException l_notesExObj) {
			throw new ServerDown(l_notesExObj.id + l_notesExObj.text);
		}
		finally {
			try {
				if (l_objDocCollection != null) {
					l_objDocCollection.recycle();
				}

			}
			catch (NotesException l_exceptionObj) {
				throw new ServerDown(l_exceptionObj.id + l_exceptionObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}

		return l_boolIsDuplicateSchedule;
	} // end isDuplicateSchedule

	private String generateQuery(String[] queryArgs, String key) throws DataNotFound
	{
		String l_strQuery = null;
		try {
			l_strQuery = ResourceBandleHandler.getString(key);
			int l_intQryArgsLen = queryArgs.length;
			String l_strSepSubString = ResourceBandleHandler.getString("notes.FORMULA_ARGS_SEPARATOR");
			for (int i = 0; i < l_intQryArgsLen; i++) {
				l_strQuery = l_strQuery.replaceFirst(l_strSepSubString, queryArgs[i]);
			}
		}
		catch (MissingResourceException l_missResExcepObj) {
			throw new DataNotFound(l_missResExcepObj.getMessage());
		}
		return l_strQuery;
	}// end generateQuery

	/**
	 * Sets the values to Schedule variables in given document
	 * 
	 * @param l_docObj
	 *            Document object representing the Schedule
	 * @throws ServerDown
	 */
	private void setScheduleVariables(Document l_docObj) throws ServerDown
	{
		// Set constant fields
		try {
			l_docObj.replaceItemValue(NotesConstants.PUBLIC_ACCESS, ResourceBandleHandler.getString("notes.PUBLIC_ACCESS_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.BUSY_PRIORITY, ResourceBandleHandler.getString("notes.BUSY_PRIORITY_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.WEB_FLAGS, ResourceBandleHandler.getString("notes.WEB_FLAGS_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.EXPAND_GROUPS, ResourceBandleHandler.getString("notes.EXPAND_GROUPS_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.LOGO, ResourceBandleHandler.getString("notes.LOGO_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.SIGN, ResourceBandleHandler.getString("notes.SIGN_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.ENCRYPT, ResourceBandleHandler.getString("notes.ENCRYPT_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.TMP_OWNERHW, ResourceBandleHandler.getString("notes.TMP_OWNERHW_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.SEQUENCE_NUMBER, new Integer(Integer.parseInt(ResourceBandleHandler.getString("notes.SEQUENCE_NUMBER_VALUE"))));
			l_docObj.replaceItemValue(NotesConstants.CS_VERSION, ResourceBandleHandler.getString("notes.CS_VERSION_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.MAIL_FIRSTPASS, ResourceBandleHandler.getString("notes.MAIL_FIRSTPASS_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.SMPT_KEEP_NOTES_ITEMS, ResourceBandleHandler.getString("notes.SMPT_KEEP_NOTES_ITEMS_VALUE"));
			l_docObj.replaceItemValue(NotesConstants.WEB_DATETIME, ResourceBandleHandler.getString("notes.WEB_DATETIME_INITVALUE"));
			l_docObj.replaceItemValue(NotesConstants.ORG_TABLE, ResourceBandleHandler.getString("notes.ORG_TABLE_VALUE"));
			// Set other fields as null
			StringTokenizer parser = new StringTokenizer(getNullParameterList(), NotesConstants.COMMA);
			while (parser.hasMoreTokens()) {
				l_docObj.replaceItemValue(parser.nextToken().trim(), null);
			}
			// Set other fields as null and set as names
			parser = new StringTokenizer(getNullParameterListNames(), NotesConstants.COMMA);
			while (parser.hasMoreTokens()) {
				Item l_itemObj = l_docObj.replaceItemValue(parser.nextToken().trim(), null);
				l_itemObj.setNames(true);
			}
		}
		catch (NotesException l_notesExObj) {
			throw new ServerDown(l_notesExObj.id + l_notesExObj.text);
		}
	}// end setScheduleVariables

	private void updateScheduleFields(Document l_objDocument, IScheduleDTO objScheduleDTO) throws ServerDown, DataNotFound, HandleException
	{
		NotesScheduleDTO l_ScheduleDTOOfDoc = null;
		try {
			// l_objDocument = getDocumentByUNID(objScheduleDTO.getUNID());
			l_ScheduleDTOOfDoc = populateScheduleDto(l_objDocument);
			l_objDocument.replaceItemValue(NotesConstants.BODY, objScheduleDTO.getBody());
			l_objDocument.replaceItemValue(NotesConstants.START_DATE_TIME, createDateTime(objScheduleDTO.getStartDate()));
			l_objDocument.replaceItemValue(NotesConstants.END_DATE_TIME, createDateTime(objScheduleDTO.getEndDate()));
			l_objDocument.replaceItemValue(NotesConstants.LOCATION, objScheduleDTO.getLocation());
			l_objDocument.replaceItemValue(NotesConstants.SUBJECT, objScheduleDTO.getSubject());
			l_objDocument.replaceItemValue(NotesConstants.USER_DOMAINS, objScheduleDTO.getUserId());

			if (objScheduleDTO instanceof NotesScheduleDTO) {
				NotesScheduleDTO nScheduleDTO = (NotesScheduleDTO) objScheduleDTO;
				l_objDocument.replaceItemValue(NotesConstants.REPEAT_TYPE, nScheduleDTO.getRepeatType());
				l_objDocument.replaceItemValue(NotesConstants.APPOINTMENT_TYPE, ""
						+ nScheduleDTO.getScheduleType());
				l_objDocument.replaceItemValue(NotesConstants.CATEGORIES, nScheduleDTO.getCategory());
				if (nScheduleDTO.getRepeatFlag() == true) {
					l_objDocument.replaceItemValue(NotesConstants.REPEATS, NotesConstants.ONE);
				}
				else {
					l_objDocument.replaceItemValue(NotesConstants.REPEATS, NotesConstants.EMPTY_STRING);
					String[] l_description = nScheduleDTO.getDescription();
					if (l_description != null) {
						l_ScheduleDTOOfDoc.setDescription(l_description);
					}
				}
			}
			l_objDocument.save();
		}
		catch (NotesException l_notesExObj) {
			throw new ServerDown(l_notesExObj.id + l_notesExObj.text);
		}
	}

	private Integer getScheduleViewIcon(int intSchType)
	{
		Integer l_viewIcon = null;
		switch (intSchType) {
		// Appointment
		case 0:
			l_viewIcon = new Integer(160);
			break;
		// Anniversary
		case 1:
			l_viewIcon = new Integer(63);
			break;
		// Event
		case 2:
			l_viewIcon = new Integer(9);
			break;
		// Meeting
		case 3:
			l_viewIcon = new Integer(158);
			break;
		// Reminder
		case 4:
			l_viewIcon = new Integer(10);
			break;
		// Task
		case 5:
			l_viewIcon = new Integer(168);
			break;
		}
		return l_viewIcon;
	}

	@SuppressWarnings("unchecked")
	private NotesScheduleDTO populateScheduleDto(Document l_docObj) throws ServerDown, DataNotFound
	{
		NotesScheduleDTO l_scheduleDTO = new NotesScheduleDTO();
		DateTime l_datetimeStartDateTime = null;
		DateTime l_datetimeEndDateTime = null;

		try {
			String xmlStr = l_docObj.generateXML();
			Log.debug(xmlStr);

			l_scheduleDTO.setBody(l_docObj.getItemValueString(NotesConstants.BODY));
			l_scheduleDTO.setCategory(l_docObj.getItemValueString(NotesConstants.CATEGORIES));
			l_scheduleDTO.setLocation(l_docObj.getItemValueString(NotesConstants.LOCATION));
			String strType = l_docObj.getItemValueString(NotesConstants.APPOINTMENT_TYPE);
			try {
				int type = Integer.parseInt(strType);
				l_scheduleDTO.setScheduleType(type);
			}
			catch (Exception e) {
			}
			;
			l_scheduleDTO.setSubject(l_docObj.getItemValueString(NotesConstants.SUBJECT));
			l_scheduleDTO.setOid(new BasicObjectIndex(l_docObj.getUniversalID()));
			l_scheduleDTO.setUserId(l_docObj.getItemValueString(NotesConstants.FROM));
			// Getting the Start Date
			Vector<DateTime> l_vecObjStartDate = l_docObj.getItemValue(NotesConstants.START_DATE_TIME);
			if (l_vecObjStartDate.size() > 0) {
				l_datetimeStartDateTime = l_vecObjStartDate.get(0);
				l_scheduleDTO.setStartDate(l_datetimeStartDateTime.toJavaDate());
			}
			// Getting the End Date
			Vector<DateTime> l_vecObjEndDate = l_docObj.getItemValue(NotesConstants.END_DATE_TIME);
			if (l_vecObjEndDate.size() > 0) {
				l_datetimeEndDateTime = l_vecObjEndDate.get(0);
				l_scheduleDTO.setEndDate(l_datetimeEndDateTime.toJavaDate());
			}
			// Checking for repeat flag status
			if (NotesConstants.ONE.equals(l_docObj.getItemValueString(NotesConstants.REPEATS))) {
				l_scheduleDTO.setRepeatFlag(true);
				// Finding out the type of schedule - daily, weekly, monthly etc
				l_scheduleDTO.setRepeatType(l_docObj.getItemValueString(NotesConstants.REPEAT_TYPE));
			}
			else {
				l_scheduleDTO.setRepeatFlag(false);
			}
			// Getting the Attachments
			this.getDescription(l_docObj, l_scheduleDTO);
		}
		catch (ArrayIndexOutOfBoundsException l_arrIndexOutOfBound) {
			throw new DataNotFound();
		}
		catch (NullPointerException l_nullPtrException) {
			throw new DataNotFound();
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.text);
		}
		return l_scheduleDTO;
	}// end populateScheduleDto

	@SuppressWarnings("unchecked")
	private void getDescription(Document docObjlcl, IScheduleDTO l_scheduleDTOObj) throws ServerDown, DataNotFound
	{
		//Vector<EmbeddedObject> l_attachmentObj;
		String[] l_byteObjArray = null;
		EmbeddedObject l_embedObj = null;
		try {
			if (docObjlcl.hasEmbedded()) {
				Vector<Item> v = docObjlcl.getItems();
				Iterator<Item> ite = v.iterator();
				Vector<EmbeddedObject> attv = new Vector<EmbeddedObject>();
				while (ite.hasNext()) {
					Item item = ite.next();
					if (item.getType() == Item.ATTACHMENT) {
						l_embedObj = docObjlcl.getAttachment(item.getValueString());
						if (l_embedObj.getType() == EmbeddedObject.EMBED_ATTACHMENT) {
							attv.add(l_embedObj);
						}
					}
				}

				if (attv.size() > 0) {
					int l_intIndex = 0;
					l_byteObjArray = new String[attv.size()];
					Iterator<EmbeddedObject> attite = attv.iterator();

					while (attite.hasNext()) {
						l_embedObj = attite.next();
						ByteArrayOutputStream l_byteArr = new ByteArrayOutputStream();
						BufferedInputStream l_bufInStr = new BufferedInputStream(l_embedObj.getInputStream());
						BufferedOutputStream l_bufOutStr = new BufferedOutputStream(l_byteArr);
						int l_intBytExistence;
						while ((l_intBytExistence = l_bufInStr.read()) != -1) {
							l_bufOutStr.write(l_intBytExistence);
						}
						l_bufOutStr.flush();
						l_byteArr.flush();
						// ByteArrayOutputStream is moved to Byte Array
						l_byteObjArray[l_intIndex] = new String(l_byteArr.toByteArray());
						l_byteArr.close();
						l_bufOutStr.close();
						l_bufInStr.close();
					}
					if (l_scheduleDTOObj instanceof NotesScheduleDTO) {
						((NotesScheduleDTO) l_scheduleDTOObj).setDescription(l_byteObjArray);
					}
				}
			}
		}
		catch (NotesException l_notesExceptionObj) {
			throw new ServerDown(l_notesExceptionObj.text);
		}
		catch (IOException l_ioExceptionObj) {
			throw new DataNotFound();
		}
	} // end getAttachments

	private String getNullParameterList()
	{
		return m_nullParamList;
	} // end getNullParameterList

	private String getNullParameterListNames()
	{
		return m_nullParamListNames;
	} // end getNullParameterListNames

	private String m_nullParamList = NotesConstants.NULL_PARAMETER_LIST;

	private String m_nullParamListNames = NotesConstants.NULL_PARAMETER_NAMES;

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.eai.notes.store.EAINotesBase#getElmentName()
	 */
	protected String getElmentName()
	{
		return DEFAULT_NOTES_CALENDAR_NAME;
	}

	/*
	 * ( Javadoc)
	 * 
	 * @see jp.co.connectone.store.IStore#getPresetDestination(jp.co.connectone.user.IAccountData,
	 *      int)
	 */
	public ISearchDestination getPresetDestination(IAccountData acc, int type) throws IncorrectData, HandleException
	{
		IFolderIndex folder = null;
		IDatabaseIndex db = null;
		folder = calendarElement;
		return (ISearchDestination) new BasicSearchDestination(db, folder);
	}

}
