package coboled.editors.command;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.texteditor.IDocumentProvider;

import coboled.editors.CobolEditor;

public class JumpHandler extends AbstractHandler {

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		CobolEditor editor = (CobolEditor) HandlerUtil.getActiveEditor(event);
		String source = getDocument(editor).get();
		String selection = getSelection(editor);
		int offset = findDefinition(source, selection);
		if(offset != -1) {
			editor.selectAndReveal(offset, selection.length());
		} else {
			IWorkbenchWindow window = HandlerUtil.getActiveWorkbenchWindowChecked(event);
			MessageDialog.openInformation(window.getShell(),
					"CobolEditorエラー",
					selection + "は見つかりませんでした");
		}
		return null;
	}
	
	private String getSelection(CobolEditor editor) {
		ISelectionProvider selectionProvider = editor.getSelectionProvider();
		ITextSelection selection = (ITextSelection) selectionProvider.getSelection();
		if(selection.getLength() != 0) {
			return selection.getText();
		} else {
			return getSelectWord(editor);
		}
	}
	
	private String getSelectWord(CobolEditor editor) {
		IDocument doc = getDocument(editor);
		int caretPos = getCaretPos(editor);
		int startPos, endPos;

		try {
			int pos = caretPos;
			char c;

			while (pos >= 0) {
				c = doc.getChar(pos);
				if (!Character.isJavaIdentifierPart(c))
					break;
				--pos;
			}

			startPos = pos;

			pos = caretPos;
			int length = doc.getLength();

			while (pos < length) {
				c = doc.getChar(pos);
				if (!Character.isJavaIdentifierPart(c))
					break;
				++pos;
			}

			endPos = pos;
			int offset = startPos + 1;
			int wordLength = endPos - offset;
			return doc.get(offset, wordLength);
			
		} catch (BadLocationException e) {
		}
		return "";
	}

	private IDocument getDocument(CobolEditor editor) {
		IDocumentProvider provider = editor.getDocumentProvider();
		IDocument doc = provider.getDocument(editor.getEditorInput());
		return doc;
	}

	private int getCaretPos(CobolEditor editor) {
		ISelectionProvider selectionProvider = editor.getSelectionProvider();
		ITextSelection selection = (ITextSelection) selectionProvider.getSelection();
		int caretPos = selection.getOffset();
		return caretPos;
	}
	
	private int findDefinition(String source, String selection) {
		int offset = findSectionDefinition(source, selection);
		if(offset == -1) {
			offset = findDataDefinition(source, selection);
		}
		return offset;
	}
	
	private int findSectionDefinition(String source, String selection) {
		StringBuilder patternString = new StringBuilder();
		patternString.append("^.{6} ( {0,3})");
		patternString.append("(");
		patternString.append(selection);
		patternString.append(")");
		patternString.append("( +?SECTION)");
		return find(source, patternString.toString(), 2);
	}
	
	private int findDataDefinition(String source, String selection) {
		StringBuilder patternString = new StringBuilder();
		patternString.append("^.{6} ( *?)([0-9][0-9])( +?)");
		patternString.append("(");
		patternString.append(selection);
		patternString.append(")");
		return find(source, patternString.toString(), 4);
	}
	
	/**
	 * オフセットを返却する。検索結果が0の場合は-1を返却する。
	 * 
	 * @param source
	 * @param patternString
	 * @return offset
	 */
	private int find(String source, String patternString, int group) {
		Pattern pattern = Pattern.compile(patternString, Pattern.MULTILINE);
		Matcher matcher = pattern.matcher(source);
		if(matcher.find()) {
			return matcher.start(group);
		}
		return -1;
	}
}
