<?php
require_once __DIR__ . '/common.php';

# secret.phpは以下の項目を設定する
# OSDNのAPIのクライアントキーとシークレットキー
#const OSDN_CMJ_CLIENTKEY = '';
#const OSDN_CMJ_SECRETKEY = '';
# クライアントに保持するクッキーの妥当性検証のためのSID_KEY
#const CMD_UPLOADER_SID_KEY = '';
if (file_exists(__DIR__ . '/secret.php')) {
    include_once __DIR__ . '/secret.php';
} else {
    define('CMD_UPLOADER_SID_KEY', 'dNOMYxOyJG8NyMx113Wt0OHaeL/W/rNf');
}


/**
 * クライアント(リモートIP)からのアクセスの可否を制御するクラス
 */
class AccessCtrl {

    private $pdo;

    /**
     * コンストラクタ
     * @param pdo $pdo
     */
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    /**
     * 24時間以上経過したリモートIPの制御は削除する。
     */
    public function purge_expired_entries() {
        $stm = $this->pdo->prepare("delete from REMOTE_ENTRIES where REGDATE1 < datetime('now', '-1 days')");
        $stm->execute();
        $stm = $this->pdo->prepare("delete from REMOTE_ENTRIES where FAILDATE < datetime('now', '-1 days')");
        $stm->execute();
    }

    /**
     * COOKIEに設定されたSIDが期限切れでなく改ざんされていないことを確認する。
     * 期限内であり改ざんされていない場合のみtrueを返す。
     * COKKIEにSIDがない場合、もしくは期限切れ、改ざんされている場合はFalseを返す。
     * @return COKKIEに有効なSIDがある場合はTrue、そうでなければFalse
     */
    public function checkValidSid() {
        if (isset($_COOKIE['cmj-uploader-sid'])) {
            $sid = $_COOKIE['cmj-uploader-sid'];

            @list($credential_json_b64, $credential_hmac_b64) = explode('.', $sid);
            if (!isset($credential_hmac_b64)) {
                // ドットがない
                return false;
            }

            $credential_json = base64url_decode($credential_json_b64);
            $credential_hmac_b64_v = base64url_encode(hash_hmac('sha256', $credential_json, CMD_UPLOADER_SID_KEY, true));
            if ($credential_hmac_b64_v !== $credential_hmac_b64) {
                // 改ざんされている場合
                return false;
            }
            $credential = json_decode($credential_json, true);
            //append_log('credential: ' . var_export($credential, true) . ', time=' . time());

            if ($credential['expired'] > time()) {
                // 期限切れでない場合
                append_log('valid login_id:' . $credential['id']);
                return true;
            }
        }
        return false;
    }

    /**
     * リモートアドレスを登録する。
     * 一定時間内(1h)に所定5回数を超えて登録しようとした場合はFalseを返す。
     * @param string $remoteaddr リモートアドレス
     * @return 登録できた場合はTrue、そうでなければFalse
     */
    public function register($remoteaddr) {
        if ($this->checkValidSid()) {
            return true;
        }

        $stm = $this->pdo->prepare('update REMOTE_ENTRIES set
        REGDATE1 = CURRENT_TIMESTAMP,
        REGDATE2 = REGDATE1,
        REGDATE3 = REGDATE2,
        REGDATE4 = REGDATE3,
        REGDATE5 = REGDATE4
        where REMOTE_ADDR = ?
        ');
        $stm->execute([$remoteaddr]);

        if ($stm->rowCount() == 0) {
            // 初回登録の場合
            $stm = $this->pdo->prepare('insert into REMOTE_ENTRIES(regdate1, remote_addr) values(CURRENT_TIMESTAMP, ?)');
            $stm->execute([$remoteaddr]);
            $this->purge_expired_entries();
            return true;
        }

        $stm = $this->pdo->prepare("select regdate5 < datetime('now', '-1 hours') from REMOTE_ENTRIES where remote_addr = ?");
        $stm->execute([$remoteaddr]);
        $regdate5 = $stm->fetchColumn();
        if ($regdate5 == null || $regdate5 == 1) {
            // まだ5回登録していないか、1時間以上前であれば可
            return true;
        }
        // 1時間以内に5回登録しているか、もしくはエラー
        return false;
    }

    /**
     * 指定されたリモートアドレスで失敗した回数を取得する
     * @param string $remoteaddr リモートアドレス
     * @return 失敗回数
     */
    public function getFailCount($remoteaddr) {
        $this->purge_expired_entries();
        $stm = $this->pdo->prepare("select failcount, faildate > datetime('now', '-10 minutes') active 
            from REMOTE_ENTRIES where remote_addr = ?");
        $stm->execute([$remoteaddr]);
        $row = $stm->fetch();
        if ($row !== false) {
            $failcount = $row['failcount'];
            $active = $row['active'];
            if ($failcount != null && $active != null && $active == 1) {
                return $failcount;
            }
        }
        return 0;
    }

    /**
     * 指定したリモートアドレスでの失敗回数を+1する。
     * @param string $remoteaddr リモートアドレス
     * @return +1された失敗回数
     */
    public function failIncrement($remoteaddr) {
        $this->purge_expired_entries();

        $stm = $this->pdo->prepare('update REMOTE_ENTRIES set failcount = failcount + 1 where remote_addr = ?');
        $stm->execute([$remoteaddr]);
        if ($stm->rowCount() == 0) {
            // リモートアドレスの登録がない場合は新規カウントする
            $stm = $this->pdo->prepare('insert into REMOTE_ENTRIES(failcount, faildate, remote_addr) values(1, CURRENT_TIMESTAMP, ?)');
            $stm->execute([$remoteaddr]);
            return 1;
        }

        $stm = $this->pdo->prepare("select failcount, faildate > datetime('now', '-10 minutes') active 
            from REMOTE_ENTRIES where REMOTE_ADDR = ?");
        $stm->execute([$remoteaddr]);

        $row = $stm->fetch();
        $failcount = 0;
        $active = null;
        if ($row !== false) {
            $failcount = $row['failcount'];
            $active = $row['active'];
        }
        $row = null;

        if ($failcount == null || $active == null || $active != 1) {
            // 10分以内のアクセスが記録されていなければ、現時刻からカウントしなおす
            $stm = $this->pdo->prepare(
                'update REMOTE_ENTRIES set failcount = 1, faildate = CURRENT_TIMESTAMP where remote_addr = ?');
            $stm->execute([$remoteaddr]);
            return 1;
        }

        return $failcount;
    }
}
?>