#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <spu_mfcio.h>
#include "SpuDraw.h"
#include "polygon_pack.h"
#include "SpanPack.h"
#include "SpanPackList.h"
#include "error.h"
#include "CellScheduler.h"

#define PIXELSIZE 11520 //32 line pixel size

// 本来なら PPE から受け取る情報です
#define FRAMEBUFFER_X 1920
#define FRAMEBUFFER_Y 1080

// IMG_MAX_X x 8 が、spe が持てる 最大
// 現在、1920x8の zBuf は持てない
#define IMG_MAX_X 720

#define SPAN_PACK_LOAD 20

/**
 * SpanPack にある Span が持つ y 座標は
 * .. [-16 ~ -9] [-8 ~ -1] [0 ~ 7] [8 ~ 16] ..
 * のどれかの区域に入るように Pack されている。
 * sp->span[0].y から、その区域の一番上の y 座標を求める。
 */
#define YTOP(y) (y &~(TEXTURE_SPLIT_PIXEL-1))
#define YOFF(y) (y & (TEXTURE_SPLIT_PIXEL-1))

SpuDraw::~SpuDraw(void)
{
    //free(zRow);
    free(linebuf);
}

void
SpuDraw::linebuf_init(int rgb)
{
     int y = TEXTURE_SPLIT_PIXEL;
     int x = IMG_MAX_X;

     linebuf = (int*)memalign(DEFAULT_ALIGNMENT, sizeof(int)*x*y);
#if 1
     // memset は 1 バイトで描き込むから
     // 0x12345678 とか渡しても 0x12 (0x78だったかな) でしか
     // 描かないんだなー
     memset(linebuf, rgb, sizeof(int)*x*y);
     //bzero(linebuf, sizeof(int)*x*y);
#else
     for (int i = 0; i < x*y; i++) {
	 linebuf[i] = rgb;
     }
#endif
}

void
SpuDraw::zRow_init(void)
{
     int y = TEXTURE_SPLIT_PIXEL;
     int x = IMG_MAX_X;
     float z = 65535.0f;
     int length = x*y;

     zRow = (float*)malloc(sizeof(float)*length);

#if 1
     vector float indx = {0.0f, 0.0f, 0.0f, 0.0f};
     vector float maxf = {z, z, z, z};
     vector float *vbuf;

     int i;
     for (i = 0; i < length - 4; i += 4) {
	 vbuf = (vector float*)&zRow[i];
	 *vbuf = spu_madd(*vbuf, indx, maxf);
     }

     for (; i < length; i++) {
	 zRow[i] = z;
     }
#else
     for (int i = 0; i < length; i++) {
	  zRow[i] = z;
     }
#endif
}


char*
SpuDraw::get_pixel(int tx, int ty, void *texture_image)
{
     //return (char*)texture_image->pixels+(texture_image->format->BytesPerPixel*((texture_image->w)*ty+tx));
     return (char*)texture_image+(3*((128)*ty+tx));
}

Uint32
SpuDraw::get_rgb(int tx, int ty, void *texture)
{
     Uint8 red, green, blue, alpha;

     if (tx<0) tx = 0;
     if (128-1< tx) tx = 128-1 ;
     if (ty<0) ty = 0;
     if (128-1< ty) ty = 128-1 ;

     char *p = get_pixel(tx,ty,texture);

     blue  = (Uint8) p[0];
     green = (Uint8) p[1];
     red   = (Uint8) p[2];
     alpha = 255;

     return (red & 0xff) * 0x10000 + (green & 0xff) * 0x100
	  + (blue & 0xff) + (alpha << 24);
}

int
SpuDraw::run(void *rbuf, void *writebuf)
{
    SpanPack *sp = (SpanPack*)rbuf;
    SpanPack *next_sp = (SpanPack*)memalign(16, sizeof(SpanPack));

    SpanPack *free_sp = next_sp; // free()用
    SpanPack *tmp_sp = NULL;
    unsigned int fbdev_addr = task->out_addr;
    Span *span;
    int width = IMG_MAX_X;
    int linebuf_rgb;

    zRow = NULL;

    dma_tags = 0;
     
    __debug("[SpuDraw]:%s\n", __FUNCTION__);
     
    int render_y = 0;

     render_y = sp->span[0].y;
     render_y = YTOP(render_y);
     //render_y += FRAMEBUFFER_Y/2;

     if (sp->info.size < 0) {
	 // 塗りつぶしSpan
	 // あとでちゃんと定義すべし
	 // むしろ別Class で作る方がいいか。
	 linebuf_rgb = 0xffffff;
	 linebuf_init(linebuf_rgb);
	 goto WRITE;
     } else {
	 linebuf_rgb = 0x000000;
	 linebuf_init(linebuf_rgb);
     }
     zRow_init();


     /**
      * SpanPack->next が存在する場合が有るので
      * 現在の SpanPack を処理してる間に
      * 次の SpanPack の DMA 転送を行っています
      */
     do {
	 if (sp->next != NULL) {
	     connector->dma_load(next_sp, (uint32)sp->next, sizeof(SpanPack),
				 SPAN_PACK_LOAD);
	 } else {
	     next_sp = NULL;
	 }

	 for (int t = 0; t < sp->info.size; t++) {	  
	     span = &sp->span[t];

	     int end = span->length_x;
	     Uint32 rgb = 0x00ff00;
	     float tex1 = span->tex_x1;
	     float tex2 = span->tex_x2;
	     float tey1 = span->tex_y1;
	     float tey2 = span->tex_y2;
	     int tex_xpos;
	     int tex_ypos;
	     int tex_zpos;
	     int x = span->x;
	     int y = span->y;
	     float z = span->start_z;
	     float zpos = span->end_z;
	     float tex_x, tex_y, tex_z;

	     if (end == 1) {
		 tex_xpos = (int)((span->tex_height-1) * tex1);
		 tex_ypos = (int)((span->tex_width-1) * tey1);
		 tex_zpos = (int)z;

		 if (zpos < zRow[x + (width * YOFF(y))]) {
		     rgb = get_rgb(tex_xpos, tex_ypos, CellScheduler::tex);
		     zRow[x + (width * YOFF(y))] = zpos;
		     linebuf[x + (width * YOFF(y))] = rgb;
		 }
	     } else {
		 for (int j = 0; j < end; j++) {
		     tex_x = tex1*(end-1-j)/(end-1) + tex2*j/(end-1);
		     tex_y = tey1*(end-1-j)/(end-1) + tey2*j/(end-1);
		     tex_z = z*(end-1-j)/(end-1) + zpos*j/(end-1);
		     if (tex_x > 1) tex_x = 1;
		     if (tex_y > 1) tex_y = 1;
		     tex_xpos = (int)((span->tex_height-1) * tex_x);
		     tex_ypos = (int)((span->tex_width-1) * tex_y);

		     if (tex_z < zRow[x + j + (width * YOFF(y))]) {
			 rgb = get_rgb(tex_xpos,tex_ypos,CellScheduler::tex);
			 zRow[x + j + (width * YOFF(y))] = tex_z;
			 linebuf[x + j + (width * YOFF(y))] = rgb;
		     }
		 }
	     }
	 }
 
	 connector->dma_wait(SPAN_PACK_LOAD);
	 tmp_sp = sp;
	 sp = next_sp;
	 next_sp = tmp_sp;
     } while (sp);

WRITE:
     writebuffer(fbdev_addr, render_y);

     free(rbuf);
     free(free_sp);
     if (zRow != NULL) free(zRow);

     return 0;
}

void
SpuDraw::writebuffer(unsigned int fbdev_addr, int y)
{
    int end_y;
    int start_y;

    // 画面上の領域に描き込まないように
    // ・・・なんか変だな。
    // とりあえず if 文どうにかしようぜ俺
    
    if (y < 0) {
	if (y + TEXTURE_SPLIT_PIXEL < 0) {
	    return;
	} else {
	    start_y = -y;
	}
    } else {
	start_y = 0;
    }

    // 画面下の領域に書き込まないように
    end_y = FRAMEBUFFER_Y - y + TEXTURE_SPLIT_PIXEL;
    end_y = (end_y < 0) ? TEXTURE_SPLIT_PIXEL + end_y : TEXTURE_SPLIT_PIXEL;
    
    for (int i = start_y; i < end_y; i++) {
	spu_mfcdma32(&linebuf[i*IMG_MAX_X], fbdev_addr + (4*FRAMEBUFFER_X*(y+i)),
		     sizeof(int)*IMG_MAX_X, i+5, MFC_PUT_CMD);
	dma_tags |= 1 << (i+5);
    }
}

void
SpuDraw::write(void)
{
     SchedTask::write();
     
     spu_writech(MFC_WrTagMask, dma_tags);
     spu_mfcstat(MFC_TAG_UPDATE_ALL);
}

SchedTask*
createTask_spuDraw(TaskListPtr _taskList, TaskPtr _task,
		   void *rbuff, void *wbuff, DmaManager *dma)
{
    rbuff = memalign(16, sizeof(SpanPack));

     return new SpuDraw(_taskList, _task, rbuff, wbuff, dma);
}
