// fbinfo.c 
//
// Copyright (c) 2006, Mike Acton <macton@cellperformance.com>
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
// documentation files (the "Software"), to deal in the Software without restriction, including without
// limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
// conditions:
//
// The above copyright notice and this permission notice shall be included in all copies or substantial
// portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
// EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
// OR OTHER DEALINGS IN THE SOFTWARE.

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdint.h>
#include <sys/ioctl.h>
#include <linux/fb.h>
#include <asm/ps3fb.h>

int
main( void )
{
    const int   fb_file           = open( "/dev/fb0", O_RDWR );
    const int   open_fb_error     = (fb_file >> ((sizeof(int)*8)-1));

    if ( open_fb_error < 0 )
    {
        printf("Could not open /dev/fb0. Check permissions\n");
        return (-1);
    }

    struct fb_vblank vblank;

    int get_vblank_error = ioctl(fb_file, FBIOGET_VBLANK, (unsigned long)&vblank);

    if ( get_vblank_error == -1 )
    {
        printf("Warning: FBIOGET_VBLANK Failed\n");
    }
    else
    {
        printf("FBIOGET_VBLANK:\n");
        printf("  flags:\n");
        printf("    FB_VBLANK_VBLANKING   : %s\n",(vblank.flags&FB_VBLANK_VBLANKING)  ?"TRUE":"FALSE");
        printf("    FB_VBLANK_HBLANKING   : %s\n",(vblank.flags&FB_VBLANK_HBLANKING)  ?"TRUE":"FALSE");
        printf("    FB_VBLANK_HAVE_VBLANK : %s\n",(vblank.flags&FB_VBLANK_HAVE_VBLANK)?"TRUE":"FALSE");
        printf("    FB_VBLANK_HAVE_HBLANK : %s\n",(vblank.flags&FB_VBLANK_HAVE_HBLANK)?"TRUE":"FALSE");
        printf("    FB_VBLANK_HAVE_COUNT  : %s\n",(vblank.flags&FB_VBLANK_HAVE_COUNT) ?"TRUE":"FALSE");
        printf("    FB_VBLANK_HAVE_VCOUNT : %s\n",(vblank.flags&FB_VBLANK_HAVE_VCOUNT)?"TRUE":"FALSE");
        printf("    FB_VBLANK_HAVE_HCOUNT : %s\n",(vblank.flags&FB_VBLANK_HAVE_HCOUNT)?"TRUE":"FALSE");
        printf("    FB_VBLANK_VSYNCING    : %s\n",(vblank.flags&FB_VBLANK_VSYNCING)   ?"TRUE":"FALSE");
        printf("    FB_VBLANK_HAVE_VSYNC  : %s\n",(vblank.flags&FB_VBLANK_HAVE_VSYNC) ?"TRUE":"FALSE");
        printf("  count  : %d\n", vblank.count );    
        printf("  vcount : %d\n", vblank.vcount );    
        printf("  hcount : %d\n", vblank.hcount );    
    }
    printf("-------------------------------------\n");

    struct fb_fix_screeninfo fscreeninfo;

    int get_fscreeninfo_error = ioctl(fb_file, FBIOGET_FSCREENINFO, (unsigned long)&fscreeninfo);

    if ( get_fscreeninfo_error == -1 )
    {
        printf("Warning: FBIOGET_FSCREENINFO Failed\n");
    }
    else
    {
       printf("FBIOGET_FSCREENINFO:\n");
       printf("  id          : \"" "%c%c%c%c" "%c%c%c%c" "%c%c%c%c" "%c%c%c%c" "\"\n",
              fscreeninfo.id[0],  fscreeninfo.id[1],  fscreeninfo.id[2],  fscreeninfo.id[3],
              fscreeninfo.id[4],  fscreeninfo.id[5],  fscreeninfo.id[6],  fscreeninfo.id[7],
              fscreeninfo.id[8],  fscreeninfo.id[9],  fscreeninfo.id[10], fscreeninfo.id[11],
              fscreeninfo.id[12], fscreeninfo.id[13], fscreeninfo.id[14], fscreeninfo.id[15] );
       if ( sizeof(unsigned long) == 4 )
       {
           printf("  smem_start  : 0x%08x\n", (unsigned int)fscreeninfo.smem_start);
       }
       else if ( sizeof(unsigned long) == 8 )
       {
           union UL64 { uint32_t u32[2]; uint64_t u64; } smem_start = {.u64=fscreeninfo.smem_start};
           printf("  smem_start  : 0x%08x_%08x\n",smem_start.u32[1],smem_start.u32[0]);
       }
       else
       {
          printf("WTF! sizeof(unsigned long) = %d\n",sizeof(unsigned long));
       }
       printf("  smem_len    : %u\n", fscreeninfo.smem_len);
       printf("  type        : ");
       switch (fscreeninfo.type)
       {
           case FB_TYPE_PACKED_PIXELS:      printf("FB_TYPE_PACKED_PIXELS (0)\n");      break;
           case FB_TYPE_PLANES:             printf("FB_TYPE_PLANES (1)\n");             break;
           case FB_TYPE_INTERLEAVED_PLANES: printf("FB_TYPE_INTERLEAVED_PLANES (2)\n"); break;
           case FB_TYPE_TEXT:               printf("FB_TYPE_TEXT (3)\n");               break;
           case FB_TYPE_VGA_PLANES:         printf("FB_TYPE_VGA_PLANES (4)\n");         break;
           default:                         printf("UNKNOWN (See /usr/include/linux/fb.h)\n");
       }
       printf("  type_aux    : ");
       if ( fscreeninfo.type == FB_TYPE_TEXT )
       {
           switch (fscreeninfo.type_aux)
           {
               case FB_AUX_TEXT_MDA:        printf("FB_AUX_TEXT_MDA (0)\n");         break;
               case FB_AUX_TEXT_CGA:        printf("FB_AUX_TEXT_CGA (1)\n");         break;
               case FB_AUX_TEXT_S3_MMIO:    printf("FB_AUX_TEXT_S3_MMIO (2)\n");     break;
               case FB_AUX_TEXT_MGA_STEP16: printf("FB_AUX_TEXT_MGA_STEP16 (3)\n");  break;
               case FB_AUX_TEXT_MGA_STEP8:  printf("FB_AUX_TEXT_MGA_STEP8 (4)\n");   break;
               default:                     printf("UNKNOWN (See /usr/include/linux/fb.h)\n");
           }
       }
       else if ( fscreeninfo.type == FB_TYPE_INTERLEAVED_PLANES )
       {
           switch (fscreeninfo.type_aux)
           {
               case FB_AUX_VGA_PLANES_VGA4: printf("FB_AUX_VGA_PLANES_VGA4 (0)\n"); break;
               case FB_AUX_VGA_PLANES_CFB4: printf("FB_AUX_VGA_PLANES_CFB4 (1)\n"); break;
               case FB_AUX_VGA_PLANES_CFB8: printf("FB_AUX_VGA_PLANES_CFB8 (2)\n"); break;
               default:                     printf("UNKNOWN (See /usr/include/linux/fb.h)\n");
           }
       }
       else
       {
           printf("N/A\n");
       }
 
       printf("  visual      : ");
       switch (fscreeninfo.visual)
       {
           case FB_VISUAL_MONO01:             printf("FB_VISUAL_MONO01 (0)\n");             break;
           case FB_VISUAL_MONO10:             printf("FB_VISUAL_MONO10 (1)\n");             break;
           case FB_VISUAL_TRUECOLOR:          printf("FB_VISUAL_TRUECOLOR (2)\n");          break;
           case FB_VISUAL_PSEUDOCOLOR:        printf("FB_VISUAL_PSEUDOCOLOR (3)\n");        break;
           case FB_VISUAL_DIRECTCOLOR:        printf("FB_VISUAL_DIRECTCOLOR (4)\n");        break;
           case FB_VISUAL_STATIC_PSEUDOCOLOR: printf("FB_VISUAL_STATIC_PSEUDOCOLOR (5)\n"); break;
           default:                           printf("UNKNOWN (See /usr/include/linux/fb.h)\n");
       }

       printf("  xpanstep    : %u\n", fscreeninfo.xpanstep);
       printf("  ypanstep    : %u\n", fscreeninfo.ypanstep);
       printf("  ywrapstep   : %u\n", fscreeninfo.ywrapstep);
       printf("  line_length : %u\n", fscreeninfo.line_length);
       if ( sizeof(unsigned long) == 4 )
       {
           printf("  mmio_start  : 0x%08x\n", (unsigned int)fscreeninfo.mmio_start);
       }
       else if ( sizeof(unsigned long) == 8 )
       {
           union UL64 { uint32_t u32[2]; uint64_t u64; } mmio_start = {.u64=fscreeninfo.mmio_start};
           printf("  mmio_start  : 0x%08x_%08x\n",mmio_start.u32[1],mmio_start.u32[0]);
       }
       else
       {
          printf("WTF! sizeof(unsigned long) = %d\n",sizeof(unsigned long));
       }
       printf("  mmio_len    : %u\n", fscreeninfo.mmio_len);
       printf("  accel       : ");
       switch (fscreeninfo.accel)
       {
           case FB_ACCEL_NONE:                printf("FB_ACCEL_NONE (0)\n");                     break;
           case FB_ACCEL_ATARIBLITT:          printf("FB_ACCEL_ATARIBLITT (1)\n");               break;
           case FB_ACCEL_AMIGABLITT:          printf("FB_ACCEL_AMIGABLITT (2)\n");               break;
           case FB_ACCEL_S3_TRIO64:           printf("FB_ACCEL_S3_TRIO64 (3)\n");                break;
           case FB_ACCEL_NCR_77C32BLT:        printf("FB_ACCEL_NCR_77C32BLT (4)\n");             break;
           case FB_ACCEL_S3_VIRGE:            printf("FB_ACCEL_S3_VIRGE (5)\n");                 break;
           case FB_ACCEL_ATI_MACH64GX:        printf("FB_ACCEL_ATI_MACH64GX (6)\n");             break;
           case FB_ACCEL_DEC_TGA:             printf("FB_ACCEL_DEC_TGA (7)\n");                  break;
           case FB_ACCEL_ATI_MACH64CT:        printf("FB_ACCEL_ATI_MACH64CT (8)\n");             break;
           case FB_ACCEL_ATI_MACH64VT:        printf("FB_ACCEL_ATI_MACH64VT (9)\n");             break;
           case FB_ACCEL_ATI_MACH64GT:        printf("FB_ACCEL_ATI_MACH64GT (10)\n");            break;
           case FB_ACCEL_SUN_CREATOR:         printf("FB_ACCEL_SUN_CREATOR (11)\n");             break;
           case FB_ACCEL_SUN_CGSIX:           printf("FB_ACCEL_SUN_CGSIX (12)\n");               break;
           case FB_ACCEL_SUN_LEO:             printf("FB_ACCEL_SUN_LEO (13)\n");                 break;
           case FB_ACCEL_IMS_TWINTURBO:       printf("FB_ACCEL_IMS_TWINTURBO (14)\n");           break;
           case FB_ACCEL_3DLABS_PERMEDIA2:    printf("FB_ACCEL_3DLABS_PERMEDIA2 (15)\n");        break;
           case FB_ACCEL_MATROX_MGA2064W:     printf("FB_ACCEL_MATROX_MGA2064W (16)\n");         break;
           case FB_ACCEL_MATROX_MGA1064SG:    printf("FB_ACCEL_MATROX_MGA1064SG (17)\n");        break;
           case FB_ACCEL_MATROX_MGA2164W:     printf("FB_ACCEL_MATROX_MGA2164W (18)\n");         break;
           case FB_ACCEL_MATROX_MGA2164W_AGP: printf("FB_ACCEL_MATROX_MGA2164W_AGP (19)\n");     break;
           case FB_ACCEL_MATROX_MGAG100:      printf("FB_ACCEL_MATROX_MGAG100 (20)\n");          break;
           case FB_ACCEL_MATROX_MGAG200:      printf("FB_ACCEL_MATROX_MGAG200 (21)\n");          break;
           case FB_ACCEL_SUN_CG14:            printf("FB_ACCEL_SUN_CG14 (22)\n");                break;
           case FB_ACCEL_SUN_BWTWO:           printf("FB_ACCEL_SUN_BWTWO (23)\n");               break;
           case FB_ACCEL_SUN_CGTHREE:         printf("FB_ACCEL_SUN_CGTHREE (24)\n");             break;
           case FB_ACCEL_SUN_TCX:             printf("FB_ACCEL_SUN_TCX (25)\n");                 break;
           case FB_ACCEL_MATROX_MGAG400:      printf("FB_ACCEL_MATROX_MGAG400 (26)\n");          break;
           case FB_ACCEL_NV3:                 printf("FB_ACCEL_NV3 (27)\n");                     break;
           case FB_ACCEL_NV4:                 printf("FB_ACCEL_NV4 (28)\n");                     break;
           case FB_ACCEL_NV5:                 printf("FB_ACCEL_NV5 (29)\n");                     break;
           case FB_ACCEL_CT_6555x:            printf("FB_ACCEL_CT_6555x (30)\n");                break;
           case FB_ACCEL_3DFX_BANSHEE:        printf("FB_ACCEL_3DFX_BANSHEE (31)\n");            break;
           case FB_ACCEL_ATI_RAGE128:         printf("FB_ACCEL_ATI_RAGE128 (32)\n");             break;
           case FB_ACCEL_IGS_CYBER2000:       printf("FB_ACCEL_IGS_CYBER2000 (33)\n");           break;
           case FB_ACCEL_IGS_CYBER2010:       printf("FB_ACCEL_IGS_CYBER2010 (34)\n");           break;
           case FB_ACCEL_IGS_CYBER5000:       printf("FB_ACCEL_IGS_CYBER5000 (35)\n");           break;
           case FB_ACCEL_SIS_GLAMOUR:         printf("FB_ACCEL_SIS_GLAMOUR (36)\n");             break;
           default:                           printf("UNKNOWN (See /usr/include/linux/fb.h)\n");
       }
    }
    printf("-------------------------------------\n");

    struct ps3fb_ioctl_res res;

    int ps3_screeninfo_error = ioctl(fb_file, PS3FB_IOCTL_SCREENINFO, (unsigned long)&res);
    
    if ( ps3_screeninfo_error == -1 )
    {
        printf("Warning: PS3FB_IOCTL_SCREENINFO Failed\n");
    }
    else
    {
        printf("PS3FB_IOCTL_SCREENINFO:\n");
        printf("    xres        : %u\n",res.xres);
        printf("    yres        : %u\n",res.yres);
        printf("    xoff        : %u\n",res.xoff);
        printf("    yoff        : %u\n",res.yoff);
        printf("    num_frames  : %u\n",res.num_frames);
    }
    printf("-------------------------------------\n");

    int close_fb_error = close( fb_file );

    if ( close_fb_error == -1 )
    {
        printf("Warning: Could not close file handle used for /dev/fb0\n");
    }

    return (0);
}

