

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <string.h>
#include <stdint.h>
#include <avr/wdt.h>

#include <avr/sleep.h>
#include "macro.h"
#include "cc1100.h"
#include "cc1100_globals.h"
#include "spi1.h"
#include "i2c.h"





#define EEPROM __attribute__((section(".eeprom"))) 
ctl_data EEPROM gctl_data;
reg_slave EEPROM greg_slave[100]; 
int	EEPROM	eprom_level[14];
char EEPROM eprom_fixid[3];
char EEPROM test_mode;
char EEPROM monitor_state;
int EEPROM wdt_cnt;
int EEPROM cnt_loop_break;


const time_set_t time_set[4]={
							{0x09,0x59,0x57}, //ALARM_2SEC
							{0x09,0x59,0x00}, //ALARM_1MIN
							{0x09,0x58,0x00}, //ALARM_2MIN
							{0x09,0x55,0x00}, //ALARM_5MIN
							#if 0
							{0x09,0x57,0x00}, //ALARM_3MIN
							{0x09,0x56,0x00}, //ALARM_4MIN
							{0x09,0x55,0x00}, //ALARM_5MIN
							{0x09,0x30,0x00}, //ALARM_30MIN
							{0x09,0x00,0x00}, //ALARM_1hour
	 						{0x10,0X00,0x01}, //ALARM_24hour
							#endif 
							};

reg_t  reg_net_alarm[7]=
							{
								{0x0E,0x00},
								{0x0C,0x10},
								{0x0B,0x00},
								{0x02,0x00},
								{0x01,0x00},
								{0x00,0x00},
								{0x0E,0x40}
							};

reg_t reg_next_sch_alarm[9]=
{
	{0x0e,0x00},
	{0x08,0x00},
	{0x09,0x00},
	{0x0a,0x00},
	{0x03,0x00},
	{0x02,0x00},
	{0x01,0x00},
	{0x00,0x00},
	{0x0e,0x60}
};

u8 day_cnt[4]={ 0x40,0x20,0x08,0x04};


cc1101_client 	gClient;
char			gWCmd[6][16];
u8				wakeup_condition=0;
u8 				get_user0_data=0;
char 			rx_char[60];
u8 				uart_data;
u8 				uart_rx_index=0;
u8 				uart_rx_length;
reg_slave 		tSlave,tSlave2;
ctl_data  		tCtl;
u16				gTemperature;
char * 			pC;

u8 				sr_status=0x00,sr_status_old=0x00;
u8 				emg_status = 0x00;
u8 				tampa_status = 0x00,tampa_status_old = 0x00;
u8				setup_mode = SETUP_NONE;
u8				gslave_master_ctl = 0x00;
u8 				fix_id_mel[3]={0x4d,0x45,0x4c};
u16				gTemperature;
u8				gLevel;
u8				gRecvRssi=0x00, gRecvLqi= 0x00;
f_option_t		gOpt;
int	 			level_offset[14]={-3,-4,-4,-4,-3,-3,-1,0,1,2,3,4,5,5};
const int	 	default_offset[14]={-3,-4,-4,-4,-3,-3,-1,0,1,2,3,4,5,5};
int				SRBuf[2];
u8				SRBufIndex;
char			gTestMode;

void udelay(u16 usec)
{
	register u8 i;

	for(i = 0 ;   i < usec ; i++)
	{
		asm volatile("NOP");
	}
}


void mdelay(u16 msec)
{
	register u16 i;

	for(i = 0 ; i < msec  ; i++)
	{
		udelay(250);
		udelay(250);
		udelay(250);
		udelay(50);
	}
}


void sdelay(u16 msec)
{
	register u16 i;

	for(i=0;i<msec;i++)
	{
		mdelay(1000);
	}
}


void putc_(u8 data)
{
	while(!(UCSR0A & (1<<UDRE0)));
	UDR0 = data;
}

void putb(u8 data)
{
	if(data < 0x0a)
		putc_('0' + data);
	else 
		putc_('a' + (data-10));
}

void puthex(u8 data)
{
	putb(data/16);
	putb(data%16);
}


u8 putchr2hex(u8 data)
{
	u8 tmp;

	if(data&0x60)
		tmp = data - 0x60;
	else
		tmp = data - 0x30;

	return tmp;
}


u8 chr2hex(u8 data_h,u8 data_l)
{
	u8 test;

	data_h = putchr2hex(data_h);
	data_l = putchr2hex(data_l);
	
	test=(data_l&0x0F)|((data_h<<4)&0xF0);
	
	return test;	
}


void putstr(u8 * data)
{
	int i =0 ;

	for( i = 0 ;i < 50 ; i++)
	{
		putc_(data[i]);
		if(data[i] == 0x0d || data[i] == 0x00) break;
	}
}


char asc_to_hex(u8 asc)
{
	if((asc >= '0') && (asc <= '9')) return (asc - '0');
	else if((asc >= 'A') && (asc <= 'F')) return (asc - 'A' + 0x0a);
	else if((asc >= 'a') && (asc <= 'f')) return (asc - 'a' + 0x0a);
	else return 0xff;
}

u8 AscToHex(char * val)
{
	u8 tmp;
	
	tmp = asc_to_hex(val[0]) << 4 | asc_to_hex(val[1]);	
	
	return tmp;
}

char asc_to_int(u8 asc)
{
	if((asc >= '0') && (asc <= '9')) return (asc - '0');
	else return 0xff;
}

int AscToInt(char * val)
{
	u8 tmp;
	
	tmp = (asc_to_int(val[0]) * 10) +  asc_to_int(val[1]);	
	
	return tmp;
}


char hex_to_asc(u8 hex)
{
	char da;
	da = hex & 0x0f;
	if((da >= 0) && (da <= 9)) return ('0' + da);
	else return ('a' + da - 0x0a);
}

void hexToAsc(u8 hex,u8 * dest)
{
	dest[0] = hex_to_asc((hex >> 4 ) & 0x0f);
	dest[1] = hex_to_asc((hex >> 0 ) & 0x0f);
}



u8 my_eeprom_read_byte(u16 addr)
{
	eeprom_busy_wait();
	return eeprom_read_byte((u8 *)addr);
}

void my_eeprom_read_block(u8 * data,u8 *addr,u16 length)
{
	eeprom_busy_wait();
	return eeprom_read_block(data,addr,length);
}

void led_onoff(u8 onoff)
{
	if(onoff == 1)
		B_CTL_HIGH;
	else
		B_CTL_LOW;
}


void led_flasher(u8 count,u16 msec)
{
	u8 i=0;
	while(i<count)
	{
		B_CTL_HIGH;
		mdelay(msec);
		B_CTL_LOW;
		mdelay(msec);
		i++;
	}
}


void inituart()
{
	UCSR0A = 0x02;	//U2X = 1
	UCSR0B = 0x00; // Rx/Tx enable
	UCSR0C = 0x06; // 8bit, 1 stop bit, no parity
	UBRR0H = 0x00; 
	UBRR0L = 0x33; // Baudrate 9600
	UCSR0B = 0x98; // rx / tx enable , rx int enable 0x98
}


void hw_setup(void)
{

	CLKPR = _BV(CLKPCE);	// CLKPCErbg1ɂ
	CLKPR = 0b0001;			// 2ɂ



    DDRC=0x00;      
	DDRD=0x00;
	DDRB=0x0D;

	/*pull_up*/
	#if 0
	PORTC = (1<<PC0) | (1<<PC1);
	PORTD = 0xf3; 
	#else 
	PORTC = (1<<PC6) |(1<<PC0) | (1<<PC1);
	PORTD = 0xf7; 
	#endif 


	PORTB = 0x01;	//for yellow LED



	EICRA &= ~(1<<ISC00);	//INT0̊荞ݔ𗧉茟mŃgKŌŒ
	EICRA &= ~(1<<ISC01);

	EICRA &= ~(1<<ISC10);	//INT1̊荞ݔ𗧉茟mŃgKŌŒ
	EICRA &= ~(1<<ISC11);
}



u16 ad_get_val(u8 ad_ch,u8 adlar)
{
	u8 i=0;
	u8 ad_data_l=0;
	u8 ad_data_h=0;
	u16 ad=0;

	//u8 temp=0xc0;

	ADCSRA = 0b10000100; // AD:1 ADJn:0 ADN:0 AD:0 AD:0 ck/16

	// reg set
	ADMUX = 0xc0 | adlar | (ad_ch & 0x0f);

	// dummy
	for(i = 0 ; i  < 4 ; i++)
	{
		ADCSRA |= _BV(ADSC);						//ϊJn vٰߓ
		while((ADCSRA&0x10)==0x00);
		ADCSRA |= (1<<ADIF);						//ADIF clear
	}
	
	// read
	ADCSRA |= _BV(ADSC);						//ϊJn vٰߓ
	while((ADCSRA&0x10)==0x00);
	ADCSRA |= (1<<ADIF);						//ADIF clear

	ad_data_l = ADCL;
	ad_data_h = ADCH;

	ad = ad_data_h;
	ad = (ad << 2);
	ad = (ad & 0x03FC) | ((ad_data_l >> 6) & 0x03);

	ADCSRA = 0b00000100; // AD:0 ADJn:0 ADN:0 AD:0 AD:0 ck/16
	return ad;
}



void temperature_test_enable(void)
{
	cc1100_cmd_idle();
	PORTC &= ~(1<<PC2);	//pull-up disenable
	cc1100_write_reg(CC1100_REG_PTEST,0xBF);
	cc1100_cfg_gdo0(0x80);
}


void temperature_test_disenable(void)
{
	cc1100_cfg_gdo0(0x3F);
	cc1100_write_reg(CC1100_REG_PTEST,0x7F);
}


u16 temperature_get(void)
{
	PORTC &= ~(1<<PC2);	//pull-up disenable
	temperature_test_enable();

	DIDR0 |= (1<<ADC2D);		//ADC2 Digital disable

	gTemperature = ad_get_val(AD_PIN_TEMPERATURE,AD_VAL_FROM_L);

	DIDR0 &= ~(1<<ADC2D);		//ADC2 Digital enable

	temperature_test_disenable();
	PORTC |= (1<<PC2);		//pull-up enable

	return gTemperature;
}


int reg_slave_num_serch(u8 * id)
{
	u8 i=0;
	u8 Serial[6];

	memcpy(Serial,fix_id_mel,3);

	for(i=0;i<100;i++)
	{
		my_eeprom_read_block((u8 *)&tSlave,(u8 *)&greg_slave[i],sizeof(reg_slave));

		memcpy((u8 *)&Serial[3],(u8 *)tSlave.serial,3);
		if(memcmp((u8 *)Serial,(u8 *)id,6) == 0)
		{
			return i;
		}
	}
	return -1;
}
	 



u8 get_input_data(void)
{
	u8 reg = 0;

	reg = (reg & 0xFF) | (((PIND) << 2) & 0x80);	//F_SEL2
	reg = (reg & 0xFF) | (((PIND) << 2) & 0x40);	//F_SEL1

	reg = (reg & 0xFF) | (((PIND) >> 2) & 0x20);	//D_SEL2
	reg = (reg & 0xFF) | (((PIND) >> 2) & 0x10);	//D_SEL1

	reg = (reg & 0xFF) | (((PINC) << 2) & 0x08);	//M_SEL2
	reg = (reg & 0xFF) | (((PINC) << 2) & 0x04);	//M_SEL1


	return reg;
}

void level_change_hex(u8 * data)
{
	int i;

	for(i = 0 ;i < 14 ; i+=2)
	{
		 data[i/2] = (((level_offset[i] + 7) << 4) & 0xf0) | ((level_offset[i+1] + 7) & 0x0f);
	}
}

void hex_change_level(u8 * data)
{
	int i;

	for(i = 0 ;i < 7 ; i++)
	{
		level_offset[i * 2] = (int)((data[i] >> 4 ) & 0x0f) - 7;
		level_offset[i* 2 + 1] = (int)((data[i]  ) & 0x0f) - 7;
	}

	eeprom_busy_wait();		
	eeprom_write_block(level_offset,&eprom_level,sizeof(level_offset));
}

void slave_fifo_preset(fifo_t * fifo)
{

	memcpy(fifo->slave_id,gClient.status.serial,6);
	memset(fifo->master_id,0xff,3);
	fifo->user_data0 = gClient.status.user_data0;
	if(gClient.status.bat_val < 0xbf)
		fifo->user_data0 |= (1<<0);	// BATT_LOW active

	if(tampa_status == 1)
		fifo->user_data0 |= (0x01  << 1);		// Tampa
	else
		fifo->user_data0 &= ~(0x01 << 1);	// Tampa

	fifo->user_data1 = gslave_master_ctl;
	fifo->bat_val = gClient.status.bat_val;
	fifo->setup_mode = (setup_mode == SETUP_NONE ? 0 : 1) ;
	fifo->master_ctl = 0x00;
	fifo->version_l = PRG_VERSION;


	if(setup_mode != SETUP_NONE)
	{
		fifo->length = sizeof(fifo_t);
		fifo->opt.env_data[0] = (gTemperature >> 8) & 0xff;
		fifo->opt.env_data[1] = (gTemperature >> 0) & 0xff;
		fifo->opt.env_data[2] = my_eeprom_read_byte((u16)&gctl_data.adj_freq);	//serial no 01;
		fifo->opt.env_data[3] = my_eeprom_read_byte((u16)&gctl_data.freq_add_sub);	//serial no 01;
		fifo->opt.env_data[4] = my_eeprom_read_byte((u16)&gctl_data.adj_temper);	//serial no 01;
		fifo->opt.env_data[5] = my_eeprom_read_byte((u16)&gctl_data.temper_add_sub);	//serial no 01;
		level_change_hex(fifo->opt.level);
	}
	else
	{
		fifo->length = sizeof(fifo_t) - sizeof(f_option_t);
	}
}



void tx_fifo_write(u8 * data,u8 length)
{
	data[0] = length + 2;
	cc1100_fifo_put(data,1);
	cc1100_fifo_put(data,length+2);
}



u8 rx_fifo_read(u8 * data)
{
	u8 length;

	cc1100_fifo_get(&length,1); 

	cc1100_fifo_get(data,length);

	if(gClient.status.type != MASTER)
	{
		cc1100_fifo_get(&gClient.fifo.rssi,1);
		cc1100_fifo_get(&gClient.fifo.lqi,1);
	}
	else
	{
		cc1100_fifo_get(&gRecvRssi,1);
		cc1100_fifo_get(&gRecvLqi,1);
	}

	return length - 2;
}



#define GET_FR() (gClient.status.input_data & 0xC0) >> 6

enum {
	CA_WAVE_CH5=3,
	CA_WAVE_CH9=2,
	CA_WAVE_CH13=1,
	CA_WAVE_CH17=0
};

u8	get_frq()
{
	return (((~get_input_data()) >> 6) & 0x03);
}


void cc1101_carrier_wave_setup(void)
{
	
	volatile u16	frq;
	volatile int 	fr;
	volatile u16 	offset;
	volatile u16 	level;
	volatile u16 	temperature;
	

	fr = get_frq();//GET_FR();
	
	//frq = 0x68FC + (fr * 0x0040);		//25kHz next	Ver0xAD
	//frq = 0x68FC + (fr * 0x003E);		//25kHz next	Ver0xAE
	frq = 0x68FC + (fr * 0x003F);		//25kHz next



	if(my_eeprom_read_byte((u16)&gctl_data.freq_add_sub) == 0x80)
		frq -= my_eeprom_read_byte((u16)&gctl_data.adj_freq);
	else
		frq += my_eeprom_read_byte((u16)&gctl_data.adj_freq);


	temperature = temperature_get();


	if(my_eeprom_read_byte((u16)&gctl_data.temper_add_sub) == 0x80)
		temperature -= my_eeprom_read_byte((u16)&gctl_data.adj_temper);
	else
		temperature += my_eeprom_read_byte((u16)&gctl_data.adj_temper);


	offset = 0x0339 - temperature;
	
	level = 13 - (offset / 0x0c);
	

	if(offset & 0x8000)
		level = 13;
	else if((offset / 0x0c)>13)
		level = 0;

	gLevel = level;

	frq += level_offset[level];


	cc1100_write_reg(CC1100_REG_FREQ2, 0x10);
	cc1100_write_reg(CC1100_REG_FREQ1, (frq >> 8)  & 0xff);
	cc1100_write_reg(CC1100_REG_FREQ0, frq & 0xff);
}



void init_voltcomparator(void)
{
	ACSR = 0x4b;
	DIDR1 = 0x02;
}



	 


void add_batt_low(void)
{
	u16 bat_val;

	B_CTL_HIGH;
	mdelay(1);


	bat_val = ad_get_val(AD_PIN_BATT_LOW,AD_VAL_FROM_L);

	gClient.status.bat_val = (bat_val>>2);


	B_CTL_LOW;
}


u8 get_sr_status(void)
{
	if((PINB >> 7) & 0x01)
		return 1;

	return 0;

}



void sig_set(void)
{
	gClient.status.make_alert = 0x00;


	//emg_status_old = emg_status;
	if(!(PINB  & 0x02))
		emg_status = 0x01;
	else
		emg_status = 0x00;

	sr_status_old = sr_status;
	sr_status = get_sr_status();


	tampa_status_old = tampa_status;

//	tampa_status = ad_get_val(AD_PIN_TAMPA,AD_VAL_FROM_L);

	if(ad_get_val(AD_PIN_TAMPA,AD_VAL_FROM_L) > 0x80)
		tampa_status = 0;
	else
		tampa_status = 1;

	//tampa_status = ad_get_val(AD_PIN_TAMPA,AD_VAL_FROM_L);


	add_batt_low(); 


	if(emg_status == 0x01 && emg_status == 0x01 && gClient.status.status != SL_ALARM_WAIT )
		gClient.status.make_alert |= 0x08;

	if(sr_status_old ^ sr_status)
	{
		gClient.status.make_alert |= 0x04;
	}

	if((tampa_status_old ^ tampa_status))
		gClient.status.make_alert |= 0x02;
	
	if(gClient.status.bat_val < 0xbf)
		gClient.status.make_alert |= (1<<0);	// BATT_LOW active
}


u8 	sig_status(void)
{
	u8 status = 0x00;
	gClient.status.make_alert = 0x00;


	//emg_status_old = emg_status;
	if(!(PINB  & 0x02))
		status|= 0x08;


	if(get_sr_status())
		status|= 0x04;


	tampa_status = ad_get_val(AD_PIN_TAMPA,AD_VAL_FROM_L);

	if(tampa_status == 0x00)
		status|= 0x02;


	add_batt_low(); 


	if(gClient.status.bat_val < 0xbf)
		status |= 0x01;

	return status;
}

u8 cmd[40];






void carrier_sense()
{
	while(PINC&0x04);		//Carrier sense
}

void carrier_no_sense()
{
	while(!(PINC&0x04));		//Carrier sense
}

void cc1101_tx_carrier(cc1101_client * client)
{

	cc1101_carrier_wave_setup();

	cc1100_cfg_txoff_mode(CC1100_TXOFF_MODE_STAY_TX);
	cc1100_cfg_manchester_en(CC1100_MANCHESTER_DISABLE);
	cc1100_write_reg(CC1100_REG_MDMCFG3,0x33);
	cc1100_cfg_mod_format(CC1100_MODULATION_ASK);
	cc1100_write_reg(CC1100_REG_FREND0,0x10);
	cc1100_cfg_gdo0(CC1100_GDOx_CLK_XOSC_1);

	/* IDLE */
	cc1100_cmd_idle();
	/* MANCAL*/
	cc1100_cmd_calibrate();
	/* FS WAKEUP */
	cc1100_cmd_flush_tx();

	cc1100_cfg_gdo0(0x0e);
	carrier_sense();

	cc1100_cmd_tx();
}


void master_fifo_preset(void)
{

}

u8 gStatus=0x00;

u8 cc1101_tx(u8 * data , u8 length,u8 th)
{
	//int i;

	cc1101_rtx_reg_set(1); // rx
	cc1101_carrier_wave_setup();
	cc1100_cfg_gdo0(0x0e);

	cc1100_cmd_idle();
	cc1100_cmd_flush_rx();
	cc1100_cmd_calibrate();
	cc1100_cmd_rx();
	
	mdelay(5);

	if(PINC & 0x04)
	{
		cc1100_cmd_idle();
		mdelay(195);
		if(th == 0)
			return 0;

	}


	cc1101_rtx_reg_set(0); // tx
	cc1101_carrier_wave_setup();
	cc1100_cmd_idle();
	cc1100_cmd_calibrate();
	tx_fifo_write(data,length);
	
	cc1100_write_reg(CC1100_REG_MCSM1,0x00);
	cc1100_cfg_gdo0(0x09);



	cc1100_cfg_gdo0(CC1100_GDOx_SYNC_WORD);
	cc1100_cmd_tx();
	while(!(PINC&0x04));	//data send start
	while(PINC&0x04);		//data send end

	return 1;
}



u8 cc1101_rx(u8 * data,u8 loop)
{
	u8 cnt_wait_syn_ack=0;
	u8 length;

	cc1101_rtx_reg_set(1); // rx
	cc1101_carrier_wave_setup();
	cc1100_cfg_gdo0(CC1100_GDOx_SYNC_WORD);

	cc1100_cmd_idle();
	cc1100_cmd_flush_rx();
	cc1100_cmd_calibrate();
	cc1100_cmd_rx();
	
	while(!(PINC & 0x04))
	{
		mdelay(1);
		if(++cnt_wait_syn_ack > 50)
			return 0;
	}

	while(1)
	{
		if(!( PINC & 0x04) )
		{	
			if((cc1100_status_crc_lqi() & 0x80))
			{

				length = rx_fifo_read(data);
				return length;
			}
			break;
		}
	}

	return 0;
}




u8 my_memcmp(u8 * src,u8 * dest,u8 len)
{
	u8 i;

	for(i= 0 ; i < len ; i++)
	{
		if(src[i] != dest[i]) return i;
	}

	return 0;
}



void rtc_alarm_disenable(void)
{
	u8 ret;

	EIMSK &= ~(1<<INT0);
	ret = rtc_1byte_data_write(0x0E,0x00);
	if(ret == 0)
		ret = rtc_1byte_data_write(0x0E,0x00);
	if(ret == 0)
		ret = rtc_1byte_data_write(0x0E,0x00);

	//rtc_1byte_data_write(0x0E,0x00);		//Alarm enable disenable
}



ISR(USART_RX_vect)
{
	uart_data=UDR0;

	rx_char[uart_rx_index++]=uart_data;

	if(uart_data==0x0d)
	{
		uart_rx_length = uart_rx_index;
		gClient.status.status=MS_UART_RECV;
		uart_rx_index = 0;
	}
}


void insert_sr_buf()
{
	if(SRBuf[0] == -1)
	{
		SRBuf[0] = (get_user0_data >> 2) & 0x01;
	}
	else
	{
		if(SRBuf[0] == 1 && !(get_user0_data & 0x40) && SRBuf[1] == -1)
			SRBuf[1] =  0;
		else if(SRBuf[0] == 0 && (get_user0_data & 0x40) && SRBuf[1] == -1)
			SRBuf[1] =  1;
	}
}

ISR(INT1_vect)
{

	//u8 gmonitor_state;

	spi_signalpin_opendrain_nonactive();

	
	if(gClient.status.status == SL_SLEEP)
		rtc_alarm_disenable();
	

	gClient.status.input_data = ~get_input_data();

	sig_set();


	//if(gClient.status.make_alert != 0x00 && gClient.status.status != SL_SEND)
	if(gClient.status.make_alert != 0x00)
	{
		get_user0_data = gClient.status.make_alert;
		if(sr_status == 0)
			get_user0_data &= ~0x04;
		else
			get_user0_data |= 0x04;
	}

	if(gClient.status.status == SL_SLEEP )
	{
		gClient.status.status = SL_SEND;
		gClient.status.send_err_cnt = 0;
	}
	else if(gClient.status.status == SL_ALARM_WAIT  )
	{
		if(gClient.status.make_alert & 0x06)
		{
			wakeup_condition = 1;
			gClient.status.status = SL_SEND;
			gClient.status.send_err_cnt = 0;
		}

	}


	#if 0
	my_eeprom_read_block((u8 *)&gmonitor_state,(u8 *)&monitor_state,1);


	if(gClient.status.status != gmonitor_state)
	{
		eeprom_busy_wait();	
		eeprom_write_block((u8 *)&gClient.status.status,(u8 *)&monitor_state,1);
		led_flasher(10,40);
	}



	if(gClient.status.status == SL_SEND_WAIT || gClient.status.status == SL_ALARM_WAIT)
		led_flasher(4,200);

	#endif
	//if(setup_mode == SETUP_READY /*&& get_user0_data */)
		

}


ISR(INT0_vect)
{

	EIMSK &= ~(1<<INT0);

	spi_signalpin_opendrain_nonactive();

	wakeup_condition = 1;

	if(gClient.status.status == SL_SEND_WAIT)
		gClient.status.status = SL_SEND;


	//led_flasher(5,50);
}


/*
 	input_data
 	b0 b1 		type 00=master,01=slave,10=carrier,11=setup
*/

void device_status_setup(void)
{
	memcpy((u8 *)&gClient.status.serial,fix_id_mel,3);
	my_eeprom_read_block((u8 *)&gClient.status.serial[3],(u8 *)&gctl_data.serial,3);
	gClient.status.send_err_cnt=0;
	
	gClient.status.input_data = ~get_input_data();

	gClient.status.type = (gClient.status.input_data >> 2) & 0x3;
}


u8	get_alarm_time()
{
	return (((~get_input_data()) >> 4) & 0x03);
}




void rtc_set_next_alarm(u8 alarm)
{
	int i;
	u8 ret;

	if(alarm > 7)
	{
		led_flasher(20,500);
	}

	if(alarm < 4)
	{
		reg_net_alarm[3].data = time_set[alarm].hour;
		reg_net_alarm[4].data = time_set[alarm].min;
		reg_net_alarm[5].data = time_set[alarm].sec;
	
		for(i = 0 ;i < 7 ; i++)
		{
			ret = rtc_1byte_data_write(reg_net_alarm[i].addr,reg_net_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_net_alarm[i].addr,reg_net_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_net_alarm[i].addr,reg_net_alarm[i].data);
		}
	}
	else
	{
		reg_next_sch_alarm[3].data = day_cnt[alarm-4];

		for(i = 0 ;i < 9 ; i++)
		{
			ret = rtc_1byte_data_write(reg_next_sch_alarm[i].addr,reg_next_sch_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_next_sch_alarm[i].addr,reg_next_sch_alarm[i].data);
			if(ret == 0)
				ret = rtc_1byte_data_write(reg_next_sch_alarm[i].addr,reg_next_sch_alarm[i].data);
		}
	}
}


void temperature_carrier_test(void)
{
	cc1100_cfg_gdo2(0x30);		//26MHz XOSC/1  output
	while(1)
	{
		led_flasher(5,50);
		add_batt_low();
		cc1101_tx_carrier((cc1101_client *)&gClient);
		sdelay(5);
		temperature_test_enable();
		sdelay(1);
		temperature_test_disenable();
	}
}



void random_tx_test(void)
{
	//cc1100_cfg_gdo2(0x30);		//26MHz XOSC/1  output

	cc1101_rtx_reg_set(0); // tx
	cc1100_write_reg(CC1100_REG_PKTCTRL0,0x22);	//random TX mode
	cc1101_carrier_wave_setup();
	cc1100_cmd_idle();
	cc1100_cmd_calibrate();
	cc1100_cmd_tx();

	while(1)
	{
		//cc1100_cmd_tx();
		wdt_disable();
		sdelay(3);
	}
}




u8 gggLength;
void master_oper()
{
	int i;
	int length=0;
	int id=0;
	fifo_t * fifo;

	fifo = &gClient.fifo;

	while(1)
	{
		wdt_reset();

		length = cc1101_rx((u8 *)fifo,1);
		if(length > 0 )
		{
			fifo->master_ctl = my_eeprom_read_byte(&greg_slave[id].slave_ctl_set);
			memcpy(fifo->master_id,&gClient.status.serial[3],3);

			if(setup_mode == SETUP_NONE)
			{
				id = reg_slave_num_serch(fifo->slave_id);
				if( id >= 0 )
				{
					cc1101_tx((u8 *)fifo, sizeof(fifo_t) - sizeof(f_option_t),0);
	
					if(length < sizeof(fifo_t))
					{
						i=0;
						cmd[i++] = 'w';
						cmd[i++] = 'r';
						cmd[i++] = ' ';
						cmd[i++] = 'e';
						cmd[i++] = 'm';
						cmd[i++] = 'g';
						cmd[i++] = ' ';
						hexToAsc(fifo->slave_id[3],(u8 *)&cmd[i]);
						i+=2;
						hexToAsc(fifo->slave_id[4],(u8 *)&cmd[i]);
						i+=2;
						hexToAsc(fifo->slave_id[5],&cmd[i]);
						i+=2;
						cmd[i++] = ' ';
						hexToAsc(fifo->user_data0,&cmd[i]);
						i+=2;
						cmd[i++] = ' ';
						hexToAsc(fifo->user_data1,&cmd[i]);
						i+=2;
						cmd[i++] = ' ';
						hexToAsc(fifo->bat_val,&cmd[i]);
						i+=2;
						cmd[i++] = ' ';
						hexToAsc(gRecvRssi,&cmd[i]);
						i+=2;
						cmd[i++] = ' ';
						hexToAsc(fifo->rssi,&cmd[i]);
						i+=2;
						cmd[i++] = 0x0d;
						cmd[i]=0;
						putstr(cmd);
					}
					return;
				}
				//sdelay(2);		//ACK limit delay
			}
			else
			{
				u8 * pD;

				if(setup_mode == SETUP_WRITE)
				{
					fifo->setup_mode = SETUP_WRITE;

					memcpy((u8 *)&fifo->opt,(u8 *)&gOpt,sizeof(f_option_t));
					if(cc1101_tx((u8 *)fifo,sizeof(fifo_t),0) == true)
					{
						setup_mode = SETUP_READY;
						putstr((u8 *)"wr ok\r");
					}
				}
				else
				{
					putc_('w');
					putc_('r');
					putc_(' ');
					putc_('d');
					putc_('u');
					putc_('m');
					putc_('p');
					putc_(' ');
	
					pD = (u8 *) fifo;
					for(i =0 ;i < length ; i++)
					{
						hexToAsc(pD[i],(u8 *)&cmd[0]);
						putc_(cmd[0]);
						putc_(cmd[1]);
					}
					hexToAsc(gRecvRssi,(u8 *)&cmd[0]);
					putc_(cmd[0]);
					putc_(cmd[1]);
					hexToAsc(gRecvLqi,(u8 *)&cmd[0]);
					putc_(cmd[0]);
					putc_(cmd[1]);
					putc_(0x0d);
				}
			}
		}

		if(gClient.status.status==MS_UART_RECV)
			return;
	}
}



void master_main()
{
	u8  ret=0;
	int i;

	while(1)
	{
		master_oper((cc1101_client *)&gClient);
	
		if(gClient.status.status==MS_UART_RECV)
		{
			ret = 0;
			memset(gWCmd,0,sizeof(gWCmd));
			
			pC = strtok(rx_char," ");
			
			i=0;
			while(pC != NULL)
			{
				strcpy(&gWCmd[i][0],pC);
				pC = strtok(NULL," ");
				i++;
			}

			if(i != 0)
			{
				if(strstr(&gWCmd[0][0],"wr") != NULL)
				{
					if(strstr(&gWCmd[1][0],"env") != NULL)
					{
						int id=0;

						id = AscToInt(&gWCmd[2][0]);
						if(id >= 0 && id < 100)
						{
							tSlave.serial[0] = AscToHex(&gWCmd[3][0]);
							tSlave.serial[1] = AscToHex(&gWCmd[3][2]);
							tSlave.serial[2] = AscToHex(&gWCmd[3][4]);
							tSlave.slave_ctl_set = AscToHex(&gWCmd[4][0]);


							my_eeprom_read_block((u8 *)&tSlave2,(u8 *)&greg_slave[id],sizeof(reg_slave));

							if(memcmp((u8 *)&tSlave,&tSlave2,sizeof(reg_slave)) != 0)
							{
								eeprom_busy_wait();		//EEPROMgp\܂őҋ@
								eeprom_write_block(&tSlave, &greg_slave[id], sizeof(reg_slave)); //write 8byte 
							}

							ret = 1;
							putstr((u8 * )"ok\r");
						}
					}
				}
				else if(strstr(&gWCmd[0][0],"rd") != NULL)
				{
					if(strstr(&gWCmd[1][0],"env") != NULL)
					{
						int id=0;

						id = AscToInt(&gWCmd[2][0]);
						if(id >= 0 && id < 100)
						{
							my_eeprom_read_block((u8 *)&tSlave2,(u8 *)&greg_slave[id],sizeof(reg_slave));

							i=0;
							cmd[i++] = 'r';
							cmd[i++] = 'd';
							cmd[i++] = ' ';
							cmd[i++] = 'e';
							cmd[i++] = 'n';
							cmd[i++] = 'v';
							cmd[i++] = ' ';
							hexToAsc(tSlave2.serial[0],(u8 *)&cmd[i]);
							i+=2;
							hexToAsc(tSlave2.serial[1],(u8 *)&cmd[i]);
							i+=2;
							hexToAsc(tSlave2.serial[2],&cmd[i]);
							i+=2;
							cmd[i++] = ' ';
							hexToAsc(tSlave2.slave_ctl_set,&cmd[i]);
							i+=2;
							cmd[i++] = 0x0d;
							cmd[i]=0;
							putstr(cmd);
						}
					}
				}
				else if(strstr(&gWCmd[0][0],"setenv") != NULL)
				{
					if(strstr(&gWCmd[1][0],"enable") != NULL)
					{
						wdt_disable();
						setup_mode = SETUP_READY;
						putstr((u8 * )"ok\r");
					}
					else if(strstr(&gWCmd[1][0],"write") != NULL)
					{
						gOpt.env_data[0] = AscToHex(&gWCmd[2][0]); // serial 0
						gOpt.env_data[1] = AscToHex(&gWCmd[2][2]); // serial 1
						gOpt.env_data[2] = AscToHex(&gWCmd[2][4]); // serial 2
						gOpt.env_data[3] = AscToHex(&gWCmd[3][0]); // ctl

						gOpt.env_data[4] = AscToHex(&gWCmd[4][0]); // 
						gOpt.env_data[5] = AscToHex(&gWCmd[4][2]);
						gOpt.env_data[6] = AscToHex(&gWCmd[4][4]);
						gOpt.env_data[7] = AscToHex(&gWCmd[4][6]);

						gOpt.level[0] =  AscToHex(&gWCmd[5][0]);
						gOpt.level[1] =  AscToHex(&gWCmd[5][2]);
						gOpt.level[2] =  AscToHex(&gWCmd[5][4]);
						gOpt.level[3] =  AscToHex(&gWCmd[5][6]);
						gOpt.level[4] =  AscToHex(&gWCmd[5][8]);
						gOpt.level[5] =  AscToHex(&gWCmd[5][10]);
						gOpt.level[6] =  AscToHex(&gWCmd[5][12]);

						setup_mode = SETUP_WRITE;
						putstr((u8 * )"ok\r");
					}
					else if(strstr(&gWCmd[1][0],"save") != NULL)
					{
						tCtl.serial[0] = AscToHex(&gWCmd[2][0]);
						tCtl.serial[1] = AscToHex(&gWCmd[2][2]);
						tCtl.serial[2] = AscToHex(&gWCmd[2][4]);

						tCtl.adj_freq = AscToHex(&gWCmd[4][0]);
						tCtl.freq_add_sub = AscToHex(&gWCmd[4][2]);
						tCtl.adj_temper = AscToHex(&gWCmd[4][4]);
						tCtl.temper_add_sub = AscToHex(&gWCmd[4][6]);
						eeprom_busy_wait();		//EEPROMgp\܂őҋ
						eeprom_write_block(&tCtl, &gctl_data, sizeof(ctl_data)); //write 8byte 
						putstr((u8 * )"ok\r");
					}
					else
					{
						setup_mode = SETUP_NONE;
					}
				}
				else
				{

				}
			}

			memset(rx_char,0,60);
			gClient.status.status = MS_BOOT;
		}
	}
}



u8 slave_oper()
{
	u8	ret;
	u8	cnt;
	u8 i=0;
	fifo_t * fifo;
	int length;


	fifo = &gClient.fifo;

	led_onoff(1);
	cnt = 0;
	for(i = 0  ; i < 7 ;i++)
	{
		slave_fifo_preset(fifo);
		ret = cc1101_tx((u8 *)fifo,fifo->length,cnt > 5 ? 1 : 0);


		if(ret == 0)
			cnt++;

		if(ret == 1)
		{
			wdt_reset();
			length = cc1101_rx((u8 *)fifo,0);
			if(length > 0)
			{
				if(my_memcmp(gClient.status.serial,fifo->slave_id,6) == 0)
				{
					gslave_master_ctl = fifo->master_ctl;
					fifo->length=0;
					led_onoff(0);
					return 1;
				}
			}
		}
	}
	led_onoff(0);
	return 0;
}


void sleep_set(u8 alarm,u8 bo)
{
	WDTCSR &= ~(1 << WDIE);
	wdt_disable();
	if(bo == 0)
		rtc_set_next_alarm(alarm);

	cc1100_cmd_idle();
	cc1100_cmd_pwd();

	spi_signalpin_opendrain_active();
	EIMSK |= (1<<INT0);
	sleep_mode();
	wdt_enable(WDTO_4S);
	WDTCSR |= (1 << WDIE);
}

void slave_main()
{
	u8	throw=0;
	u8 ret=0;
	u8 alarm=0;
	//int gwdt_cnt=0;
	int i=0;
	int gcnt_loop_break=0;

	memset(SRBuf,0,3);
	SRBufIndex = 0;
	while(1)
	{
		throw = 0;
		i=0;
		wdt_reset();
		spi_signalpin_opendrain_nonactive();
		switch(gClient.status.status)
		{
			case SL_BOOT:
				// config read
				alarm = (gslave_master_ctl & 0x03) + 4;//ALARM_30MIN;
				gClient.status.status = SL_SLEEP;
				break;
			case SL_SLEEP:
				gClient.status.status = SL_SCH_SEND;
				gClient.status.send_err_cnt = 0;
				break;
			case SL_SCH_SEND:
				gClient.status.user_data0 = ((sig_status() & 0x1A) | 0x10);
				ret = slave_oper((cc1101_client *)&gClient);
				if(ret == 1)
				{
					throw = 1;
					gClient.status.status = SL_BOOT;
					#ifdef S3_DELAY
					sdelay(3);
					#endif
				}
				else
				{
					alarm = ALARM_2SEC;
					gClient.status.send_err_cnt++;
					if(gClient.status.send_err_cnt > 8)
					{
						throw = 1;
						gClient.status.status = SL_BOOT;
						#ifdef S3_DELAY
						sdelay(3);
						#endif
					}
				}
				break;
			case SL_SEND:
				if(gClient.status.send_err_cnt  == 0)
				{
					gClient.status.user_data0 = ((get_user0_data & 0x2F) | 0x20);	//EMG_SIG_SET
				}


				ret = slave_oper((cc1101_client *)&gClient);
				if(ret == 1)
				{
					#ifdef S3_DELAY
					sdelay(3);
					#endif
					// config read
					if((gClient.status.user_data0 & 0x04) != (get_user0_data & 0x04))
					{
						throw = 1;
						gClient.status.send_err_cnt = 0;
					}
					else
					{
						wakeup_condition = 0;
						alarm = get_alarm_time();
						gClient.status.status = SL_ALARM_WAIT;
					}
				}
				else
				{
					alarm = ALARM_2SEC;
					gClient.status.send_err_cnt++;
					gClient.status.status = SL_SEND_WAIT;
					if(gClient.status.send_err_cnt > 6)
					{
						gClient.status.send_err_cnt = 0;
						throw = 0;
						#ifdef S3_DELAY
						sdelay(3);
						#endif
						alarm = ALARM_1MIN;
						wakeup_condition = 0;
						gClient.status.status = SL_ALARM_WAIT;
					}
				}
				break;
			case SL_ALARM_WAIT:
				throw = 1;
				gClient.status.status = SL_BOOT;
				break;
			case SL_SEND_WAIT:
				//led_flasher(4,200);
				throw = 1;
				break;
			default:
				break;


		}

		if(throw == 0)
			sleep_set(alarm,0);

		if(gClient.status.status == SL_SEND_WAIT)
			sleep_set(alarm,1);			

		while(wakeup_condition == 0 && gClient.status.status == SL_ALARM_WAIT)
		{
			sleep_set(alarm,1);


			#if 1
			i++;
			if(i>100)
			{
				wakeup_condition = 1;

				led_flasher(10,40);
			}
			#endif
		}
	}
}



u8 setup_oper()
{
	u8	ret;
	int 	length;
	fifo_t * fifo;


	fifo = &gClient.fifo;

	

	temperature_get();

	slave_fifo_preset(fifo);
	ret = cc1101_tx((u8 *)fifo,sizeof(fifo_t),0);

	if(ret == 1)
	{
		length = cc1101_rx((u8 *)fifo,0);
		if(length > 0)
		{
			gggLength = length;
			if(length == sizeof(fifo_t) && fifo->setup_mode == SETUP_WRITE)
			{
				memcpy((u8 *)tCtl.serial,(u8 *)&fifo->opt.env_data[0],3);
				tCtl.adj_freq = fifo->opt.env_data[4];
				tCtl.freq_add_sub = fifo->opt.env_data[5];
				tCtl.adj_temper = fifo->opt.env_data[6];
				tCtl.temper_add_sub = fifo->opt.env_data[7];
	
				memcpy((u8 *)&gClient.status.serial[3],(u8 *)tCtl.serial,3);
				memcpy((u8 *)&fifo->slave_id[3],(u8 *)tCtl.serial,3);

				eeprom_busy_wait();		
				eeprom_write_block(&tCtl, &gctl_data, sizeof(ctl_data)); 

				hex_change_level(fifo->opt.level);

				gTestMode = 0xab;
				eeprom_busy_wait();		//EEPROMgp\܂őҋ
				eeprom_write_block(&gTestMode, &test_mode, 1); //write 8byte 
				gTestMode = 0;

				cc1101_tx_carrier((cc1101_client *)&gClient);
				sdelay(5);

				return 1;
			}
			fifo->length=0;
		}
	}
	return 0;
}

extern u8 pt_data[];
void setup_main()
{
	while(1)
	{
		if(setup_mode == SETUP_READY)
		{
			led_flasher(1,1000);
		}
		else
		{
			led_flasher(5,100);
			//pt_data[0] = 0xc0;			//10dBm
			//pt_data[0] = 0xc8;			//7dBm
			//pt_data[0] = 0x84;			//5dBm
			pt_data[0] = 0x60;			//0dBm
			cc1101_8PATABLE_write_reg();
			setup_oper();

			#if 0
			pt_data[0] = 0x60;
			cc1101_8PATABLE_write_reg();
			add_batt_low();
			cc1101_tx_carrier((cc1101_client *)&gClient);
	 		sdelay(2);
			#endif
		}
	}
}


void check_env(void)
{
	u8  i;
	u8 * pD;
	u8	fixid[3];
	u8	test;

	
	my_eeprom_read_block((u8 *)&tCtl,(u8 *)&gctl_data,sizeof(ctl_data));

	pD = (u8 *)&tCtl;

	for(i = 0 ;i  < 5 ; i++)
	{
		if(pD[i] != 0xff && pD[i] != 0x00) break;
	}

	if(i == 5)
	{
		// init;
		tCtl.adj_freq = 0x24;
		tCtl.freq_add_sub = 0x80;
		tCtl.adj_temper = 0x18;
		tCtl.temper_add_sub = 0x80;
		eeprom_busy_wait();		
		eeprom_write_block(&tCtl,&gctl_data,sizeof(ctl_data));
	}

	my_eeprom_read_block((u8 *)level_offset,(u8 *)&eprom_level,sizeof(level_offset));


	pD = (u8 *)&level_offset;

	for(i = 0 ;i  < sizeof(level_offset) ; i++)
	{
		if(pD[i] != 0xff && pD[i] != 0x00) break;
	}


	if(i == 5)
	{
		eeprom_busy_wait();		
		eeprom_write_block(default_offset,&eprom_level,sizeof(level_offset));
		memcpy(level_offset,default_offset,sizeof(level_offset));

	}

	my_eeprom_read_block((u8 *)fixid,(u8 *)&eprom_fixid,sizeof(eprom_fixid));

	if(fixid[0] == 0x00 || fixid[0] == 0xff )
	{

	}
	else
	{
		fix_id_mel[0] = fixid[0];
		fix_id_mel[1] = fixid[1];
		fix_id_mel[2] = fixid[2];
	}

	my_eeprom_read_block((u8 *)&test,(u8 *)&test_mode,1);

	if(test == 0xab)
	{
		gTestMode = 0;
	}
	else
	{
		gTestMode = 1;
	}
}

int main()
{



	hw_setup();
	inituart();
	i2c_init();
	check_env();

	device_status_setup();

    cc1100_init();

	cc1101_init_reg();
	pt_data[0] = 0xC0;				//10dB
	//pt_data[0] = 0xc8;			//7dBm
	//pt_data[0] = 0x84;			//5dBm
	//pt_data[0] = 0x60;			//0dB
	cc1101_8PATABLE_write_reg();

	


	set_sleep_mode(SLEEP_MODE_PWR_DOWN);

	led_flasher(2,100);

	if(gClient.status.type == SLAVE  )
	{
		EIMSK |= (1<<INT1);
		sr_status = get_sr_status();
		sr_status_old = sr_status;
	}

	if(gClient.status.type == SLAVE || gClient.status.type == MASTER)
	{
		wdt_enable(WDTO_4S);
		WDTCSR |= (1 << WDIE);
	}

	sei();
	spi_signalpin_opendrain_nonactive();

	if(gClient.status.type == SLAVE)
		rtc_alarm_disenable();


	/* Enter the infinite loop */
    while (1)
    {
		if(gClient.status.type == MASTER)
		{
			gClient.status.status = MS_BOOT;
			master_main();
		}
		else if(gClient.status.type == TEST_MDOE && gTestMode == 1)
		{
			temperature_carrier_test();
			//random_tx_test();
		}
		else if(gClient.status.type == SETUP)
		{
			setup_mode = SETUP_TRX;;
			setup_main();
		}
		else
		{
			slave_main();
		}
	}
}





