#include "straightSynthesizer.h"

#define VOWEL_MORPH_LENGTH 80

straightSynthesizer::~straightSynthesizer()
{
    if(synth)        straightSynthDestroy(synth);
    if(source)        straightSourceDestroy(source);
    if(specgram)    straightSpecgramDestroy(specgram);
    if(straight)    straightDestroy(straight);
}


bool straightSynthesizer::Initialize(string sFileName)
{
    bool bResult=false;

    straightInitConfig(&config);
    config.samplingFrequency=44100.0;
    config.f0Ceil=1000.0;

    if((straight=straightInitialize(&config))!=NULL){
        if((source=straightSourceInitialize(straight,NULL))!=NULL){
            if((specgram=straightSpecgramInitialize(straight,NULL))!=NULL){
                if((synth=straightSynthInitialize(straight,NULL))!=NULL){
                    bResult=true;
                }else{
                    cout << "error; straightSynthInitialize" << endl;
                }
            }else{
                cout << "error; straigthSpecgramInitialize" << endl;
            }
        }else{
            cout << "error; straightSourceInitialize" <<endl;
        }
    }else{
        cout << "error; straightInitialize" << endl;
    }

    straightSetCallbackFunc(straight, STRAIGHT_PERCENTAGE_CALLBACK, callbackFunc, NULL);

    if( bResult ){
        bResult = Sequencer.LoadFile( sFileName, config.frameShift );
    }

#ifdef USE_MULTI_SINGER
    sFilePaths = Sequencer.GetFilePaths();
#else
    sFilePath = Sequencer.GetFilePath();
#endif

    return bResult;
}

bool straightSynthesizer::Synthesize(string sFileName)
{
    bool bResult=false;

    long nBeginFrame=Sequencer.GetBeginFrame();
    long nEndFrame=Sequencer.GetEndFrame() + 500;
    long nFrameLength=nEndFrame-nBeginFrame;

    long nApLength,nSpLength;

    double dF0,dSrcF0,dBrethiness,dFormant,dClearness,dBrightness;
    NoteEvent*     pNoteEvent;
    straightFrame  stFrame;
    straightNoteManager stManager;

    straightSourceCreateF0(straight,source,nFrameLength);
    straightSourceCreateAperiodicity(straight,source,nFrameLength);
    straightSpecgramCreate(straight,specgram,nFrameLength);

    nApLength=straightSourceGetAperiodicityFrequencyLength(source);
    nSpLength=straightSpecgramGetFrequencyLength(specgram);

    double *pApBuffer = new double[nApLength];
    double *pSpBuffer = new double[nSpLength];

    memset(pApBuffer,0,sizeof(double)*nApLength);
    memset(pSpBuffer,0,sizeof(double)*nSpLength);
    dF0=0.0;
    dSrcF0=0.0;

#ifdef USE_MULTI_SINGER
    stManager.Initialize(sFilePaths,&straight);
#else
    stManager.Initialize(sFilePath,&straight);
#endif

    stFrame.SetFrame(dF0,pApBuffer,nApLength,pSpBuffer,nSpLength);

    for(long n=0;n<nFrameLength;n++){

        //ここに合成アルゴリズム
        long nFrame = n+nBeginFrame;
        long nSrcFrameLength;

        pNoteEvent=Sequencer.GetNoteEvent(nFrame);

        memset(pApBuffer,0,sizeof(double)*nApLength);
        memset(pSpBuffer,0,sizeof(double)*nSpLength);
        dF0=0.0;dSrcF0=0.0;dBrethiness=1.0;

        stFrame.SetFrame(dF0,pApBuffer,nApLength,pSpBuffer,nSpLength);

        // 歌手イベントはここで弾くべきか、それとも初めから消しておくべきか…
        if(pNoteEvent!=NULL){

            bool bReadResult=true;

            //ここで各波形データからデータを持ってくる
            long nAbsoluteFrame;

#ifdef USE_MULTI_SINGER
            nSrcFrameLength=stManager.GetSrcFrameLength(pNoteEvent->mLyric, pNoteEvent->mSingerIndex);
#else
            nSrcFrameLength=stManager.GetSrcFrameLength(pNoteEvent->mLyric);
#endif
            nAbsoluteFrame=CalculateAbsoluteFrame(pNoteEvent,nFrame,nSrcFrameLength);
            
#ifdef USE_MULTI_SINGER
            bReadResult &= stManager.GetStraightFrame( &stFrame, pNoteEvent->mLyric, nAbsoluteFrame, pNoteEvent->mSingerIndex );
#else
            bReadResult &= stManager.GetStraightFrame( &stFrame, pNoteEvent->mLyric, nAbsoluteFrame );
#endif
            bReadResult &= stFrame.GetApBuffer( pApBuffer, nApLength );
            bReadResult &= stFrame.GetSpBuffer( pSpBuffer, nSpLength );
            dSrcF0 = stFrame.GetF0();

            if(pNoteEvent->mIsContinuousBack){

                double* pTempAp = new double[nApLength];
                double* pTempSp = new double[nSpLength];
                double vMorphLength = VOWEL_MORPH_LENGTH;
                if( vMorphLength > pNoteEvent->mEndFrame - pNoteEvent->mBeginFrame ){
                    vMorphLength = (double)(pNoteEvent->mEndFrame - pNoteEvent->mBeginFrame) * config.frameShift;
                }

                /* Vowel Morph */
                if( pNoteEvent->mIsContinuousBack && pNoteEvent->mEndFrame - vMorphLength / config.frameShift < nFrame ){

                    nAbsoluteFrame = (long)((pNoteEvent->mNextNote->mSetting.iConsonant + pNoteEvent->mNextNote->mSetting.iLeftBlank + 1 ) / config.frameShift );

#ifdef USE_MULTI_SINGER
                    bReadResult &= stManager.GetStraightFrame( &stFrame, pNoteEvent->mNextNote->mLyric, nAbsoluteFrame, pNoteEvent->mNextNote->mSingerIndex );
#else
                    bReadResult &= stManager.GetStraightFrame( &stFrame, pNoteEvent->mNextNote->mLyric, nAbsoluteFrame );
#endif
                    bReadResult &= stFrame.GetApBuffer( pTempAp, nApLength );
                    bReadResult &= stFrame.GetSpBuffer( pTempSp, nSpLength );

                    if( bReadResult ){
                        double dMorphRate = (double)(pNoteEvent->mEndFrame - nFrame) * config.frameShift / vMorphLength;
                        double dTempBre = 1.0 - 0.8 * (double)(pNoteEvent->mDecay) / 100.0 * (1.0 - dMorphRate);

                        for( long i = 0; i < nApLength; i++ ){
                            pApBuffer[i] = pTempAp[i] * (1.0 - dMorphRate) + dMorphRate * pApBuffer[i] * dTempBre;
                        }

                        for( long i = 0; i < nSpLength; i++ ){
                            pSpBuffer[i] = pow( pTempSp[i], 1.0 - dMorphRate ) * pow( pSpBuffer[i] * dTempBre, dMorphRate );
                        }

                        if( (dF0 = stFrame.GetF0()) != 0.0 ){
                            if( dSrcF0 != 0.0 ){
                                dSrcF0 = pow( dF0, 1.0 - dMorphRate ) * pow( dSrcF0, dMorphRate );
                            }else{
                                dSrcF0 = dF0;
                            }
                        }
                    }
                }
                /* Consonant Morph */
                if( pNoteEvent->mNextNote->mBeginFrame < nFrame ){
                    
#ifdef USE_MULTI_SINGER
                    nSrcFrameLength = stManager.GetSrcFrameLength( pNoteEvent->mNextNote->mLyric, pNoteEvent->mNextNote->mSingerIndex );
#else
                    nSrcFrameLength = stManager.GetSrcFrameLength( pNoteEvent->mNextNote->mLyric );
#endif
                    nAbsoluteFrame = CalculateAbsoluteFrame( pNoteEvent->mNextNote, nFrame, nSrcFrameLength );

#ifdef USE_MULTI_SINGER
                    bReadResult &= stManager.GetStraightFrame( &stFrame, pNoteEvent->mNextNote->mLyric, nAbsoluteFrame, pNoteEvent->mNextNote->mSingerIndex );
#else
                    bReadResult &= stManager.GetStraightFrame( &stFrame, pNoteEvent->mNextNote->mLyric, nAbsoluteFrame );
#endif
                    bReadResult &= stFrame.GetApBuffer( pTempAp, nApLength );
                    bReadResult &= stFrame.GetSpBuffer( pTempSp, nSpLength );

                    if( bReadResult ){
                        double dMorphRate = (double)(pNoteEvent->mEndFrame - nFrame) / (double)(pNoteEvent->mEndFrame - pNoteEvent->mNextNote->mBeginFrame);

                        for( long i = 0; i < nApLength; i++ ){
                            pApBuffer[i] = pTempAp[i] * (1.0 - dMorphRate) + dMorphRate * pApBuffer[i];
                        }

                        for( long i = 0; i < nSpLength; i++ ){
                            pSpBuffer[i] = pow( pTempSp[i], 1.0 - dMorphRate ) * pow( pSpBuffer[i], dMorphRate );
                        }

                        if( (dF0 = stFrame.GetF0()) != 0.0 ){
                            if( dSrcF0 != 0.0 ){
                                dSrcF0 = pow( dF0, 1.0 - dMorphRate ) * pow( dSrcF0, dMorphRate );
                            }else{
                                dSrcF0 = dF0;
                            }
                        }
                    }

                }
                SAFE_DELETE_ARRAY( pTempAp );
                SAFE_DELETE_ARRAY( pTempSp );
            }

            //各種計算をさせる
            if( bReadResult ){
                dF0 = Sequencer.GetF0( nFrame, config.frameShift );

/*                if( pNoteEvent )
                    if( pNoteEvent->ucPortamentoUse == 0 )
                        dF0 = 0.0;
*/
/*                if(dSrcF0!=0.0 && dF0!=0.0){
                    dBrethiness=dF0/dSrcF0;
                } else {
                    dBrethiness = 1.0;
                }*/

                dBrethiness *= (double)(Sequencer.GetBrethiness( nFrame, config.frameShift ) );
                dFormant = pow( 0.75, ((double)(64.0 - Sequencer.GetControlValue( GENDOR, nFrame, config.frameShift ))) / 64.0 );
                dClearness = (double)(Sequencer.GetControlValue( CLEARNESS, nFrame, config.frameShift )) / 127.0;
                dBrightness = ((double)(Sequencer.GetControlValue( BRIGHTNESS, nFrame, config.frameShift )) - 64.0) / 64.0;

                stFrame.SetFrame( dF0, pApBuffer, nApLength, pSpBuffer, nSpLength );

                stFrame.ApplyBrethinessChange( dBrethiness );
                stFrame.ApplyFormantChange( dFormant );
                stFrame.ApplyClearnessChange( dClearness, config.samplingFrequency );
//                stFrame.ApplyBrightnessChange( dBrightness, config.samplingFrequency );
                if( dF0 != 0.0 ){
                    stFrame.ApplyBrightnessChange( -dBrightness, dF0, config.samplingFrequency );
                }

                stFrame.GetApBuffer( pApBuffer, nApLength );
                stFrame.GetSpBuffer( pSpBuffer, nSpLength );
            }else{
                memset( pApBuffer, 0, sizeof( double ) * nApLength );
                memset( pSpBuffer, 0, sizeof( double ) * nSpLength );
            }

            //最終フレームに自身と同じ発音のファイルを開放
            if( nFrame == pNoteEvent->mEndFrame - 1 ){
                if( pNoteEvent->mIsContinuousBack ){
                    if( pNoteEvent->mNextNote->mLyric.compare( pNoteEvent->mLyric ) != 0 ){
#ifdef USE_MULTI_SINGER
                        stManager.ReleaseSTF( pNoteEvent->mLyric, pNoteEvent->mSingerIndex );
#else
                        stManager.ReleaseSTF( pNoteEvent->mLyric );
#endif
                    }
                }else{
#ifdef USE_MULTI_SINGER
                    stManager.ReleaseSTF( pNoteEvent->mLyric, pNoteEvent->mSingerIndex );
#else
                    stManager.ReleaseSTF( pNoteEvent->mLyric );
#endif
                }
            }

        }

        straightSourceSetF0( straight, source, n, dF0 );
        straightSourceSetAperiodicity( straight, source, n, pApBuffer, nApLength );
        straightSpecgramSetSpectrum( straight, specgram, n, pSpBuffer, nSpLength );
    }

    SAFE_DELETE_ARRAY( pApBuffer );
    SAFE_DELETE_ARRAY( pSpBuffer );

    
    if( straightSynthCompute( straight, source, specgram, synth, 1.0, 1.0, 1.0 ) == ST_TRUE ){
        ApplyDynamics();
        Normalize();

/*        char cBuf[256];
#ifdef __GNUC__
        strcpy( cBuf, sFileName.c_str() );
#else
        strcpy_s( cBuf, sFileName.c_str() );
#endif
        straightWriteSynthAudioFile( straight, synth, cBuf, "output_wav", 16 );
*/
        OutputWave( sFileName.c_str() );
    }
    
    return bResult;
}

long straightSynthesizer::CalculateAbsoluteFrame( NoteEvent *pNoteEvent, long nFrame, long nSrcFrameLength ){
    if( pNoteEvent == NULL ){
        return false;
    }

    long nAbsoluteFrame = 0;
    long nRelativeFrame = nFrame - pNoteEvent->mBeginFrame;
    long nFrameLength = pNoteEvent->mEndFrame - pNoteEvent->mBeginFrame;

    long nConsonantEndFrame = (long)((double)(pNoteEvent->mSetting.iConsonant) * pow( 2.0, (64 - pNoteEvent->mVelocity) / 64.0 ) / config.frameShift);

    if( nConsonantEndFrame > nRelativeFrame ){
        //現在Consonant中
        nAbsoluteFrame = (long)(pNoteEvent->mSetting.iLeftBlank / config.frameShift ); //LeftBlankにいくらか加える

        nAbsoluteFrame += (long)((double)nRelativeFrame / (double)nConsonantEndFrame * (double)(pNoteEvent->mSetting.iConsonant) / config.frameShift);
    }else{
        //現在持続音中
        nAbsoluteFrame = (long)((pNoteEvent->mSetting.iLeftBlank + pNoteEvent->mSetting.iConsonant) / config.frameShift);

        nAbsoluteFrame += (long)((double)(nRelativeFrame - nConsonantEndFrame) / (double)(nFrameLength - nConsonantEndFrame)
            * (double)(nSrcFrameLength - (pNoteEvent->mSetting.iConsonant + pNoteEvent->mSetting.iLeftBlank + pNoteEvent->mSetting.iRightblank) / config.frameShift));
    }

    return nAbsoluteFrame;

}

void straightSynthesizer::Normalize(){
    long nWaveLength;
    double *pWaveBuffer;

    //波形の振幅の伸張が必要になる場合以外は伸張しない。
    //0.0でもよいがその場合Dynamicsで音量を揃えているとDAW側で対応しなくてはいけなくなってしまう。
    //外部から音量Volume指定に対応させるほうがよいかも…？
    double dMaxAbsoluteValue = 1.0;

    pWaveBuffer = straightSynthGetOutputWave( synth, &nWaveLength );

    if( pWaveBuffer != NULL ){

        for( long n = 0; n < nWaveLength; n++ ){
            double dTemp = pWaveBuffer[n];
            if( dTemp < 0 ){
                dTemp *= -1;
            }

            if( dMaxAbsoluteValue < dTemp ){
                dMaxAbsoluteValue = dTemp;
            }
        }

        for( long n = 0; n < nWaveLength; n++ ){
            pWaveBuffer[n] /= dMaxAbsoluteValue;
            pWaveBuffer[n] *= (32767.0 / 32768.0);
        }
    }

    return;
}

void straightSynthesizer::ApplyDynamics(){
    long nBeginFrame = Sequencer.GetBeginFrame();
    long nWaveLength;
    double *pWaveBuffer;

    pWaveBuffer = straightSynthGetOutputWave( synth, &nWaveLength );

#ifdef _DEBUG
    ofstream ofs;
    ofs.open("log.txt");
#endif

    if( pWaveBuffer != NULL ){

        long nFrame, nPreviousFrame;
        double dCurrent, dNext, dFrame, dRate;

        dFrame = (double)nBeginFrame;
        nPreviousFrame = nBeginFrame;
        dCurrent = Sequencer.GetDynamics( nBeginFrame, config.frameShift );
        dNext = Sequencer.GetDynamics( nBeginFrame + 1, config.frameShift );

        for( long n = 0; n < nWaveLength; n++ ){
            dFrame = (double)nBeginFrame + ((double)n * (1000.0 / config.samplingFrequency) / config.frameShift);
            nFrame = (long)dFrame;
            if( nFrame != nPreviousFrame ){
                dCurrent = dNext;
#ifdef _DEBUG
                ofs << dNext << endl;
#endif
                dNext = Sequencer.GetDynamics( nFrame + 1, config.frameShift );
            }
            dRate = dFrame - (double)nFrame;

            pWaveBuffer[n] *= dCurrent * (1.0 - dRate) + dNext * dRate;
            nPreviousFrame = nFrame;
        }
    }
#ifdef _DEBUG
    ofs.close();
#endif
}

void straightSynthesizer::OutputWave( const char *cBuf ){
    long nBeginFrame = Sequencer.GetBeginFrame();
    long nEmptyLength = (long)((double)nBeginFrame * config.frameShift / 1000.0 * config.samplingFrequency);

    long nWaveLength;
    double *pWaveBuffer = straightSynthGetOutputWave( synth, &nWaveLength );

    if( NULL == pWaveBuffer ){
        return;
    }
    
    FILE *fp;
#ifdef __GNUC__
    fp = fopen( cBuf, "wb" );
#else
    fopen_s( &fp, cBuf, "wb" );
#endif

    if( NULL == fp ){
        cout << "error; io error on file:" << cBuf << endl;
        return;
    }

    //WriteWaveHeader
    int LinearPCM = 16;
    short FormatID = 1;
    short ChannelNum = 1;
    int SamplingRate = (int)(config.samplingFrequency);
    short BitPerSample = 16;
    short BlockSize = BitPerSample / 8 * ChannelNum;
    int BytePerSec = BlockSize * SamplingRate;
    int WaveSize = (nEmptyLength + nWaveLength) * BlockSize;
    int FileSize = 44 + WaveSize;

    fprintf( fp, "RIFF" );
    fwrite( &FileSize, 4, 1, fp );
    fprintf( fp, "WAVEfmt " );
    fwrite( &LinearPCM, 4, 1, fp );
    fwrite( &FormatID, 2, 1, fp );
    fwrite( &ChannelNum, 2, 1, fp );
    fwrite( &SamplingRate, 4, 1, fp );
    fwrite( &BytePerSec, 4, 1, fp );
    fwrite( &BlockSize, 2, 1, fp );
    fwrite( &BitPerSample, 2, 1, fp );
    fprintf( fp, "data" );
    fwrite( &WaveSize, 4, 1, fp );

    //WriteWaveData

    for( long n =0; n < nEmptyLength; n++ ){
        short s = 0;
        fwrite( &s, 2, 1, fp );
    }

    for( long n = 0; n < nWaveLength; n++ ){
        short s = (short)(32767.0 * (pWaveBuffer[n]));
        fwrite( &s, 2, 1, fp );
    }

    fclose( fp );

    return;
}
