﻿/*
 * FormShortcutKeys.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Drawing;

using Boare.Lib.AppUtil;

namespace Boare.Cadencii {

    public partial class FormShortcutKeys : Form {
        private ToolStripMenuItem m_dumy;
        private Dictionary<string, ValuePair<string, Keys[]>> m_dict;
        private Dictionary<string, ValuePair<string, Keys[]>> m_first_dict;

        public FormShortcutKeys( Dictionary<string, ValuePair<string, Keys[]>> dict ) {
            InitializeComponent();
            m_dict = dict;
            m_dumy = new ToolStripMenuItem();
            m_dumy.ShowShortcutKeys = true;
            m_first_dict = new Dictionary<string, ValuePair<string, Keys[]>>();
            CopyDict( m_dict, ref m_first_dict );
            ApplyLanguage();
            UpdateList();
            Misc.ApplyFontRecurse( this, AppManager.EditorConfig.BaseFont ); 
        }

        public void ApplyLanguage() {
            this.Text = _( "Shortcut Config" );

            btnOK.Text = _( "OK" );
            btnCancel.Text = _( "Cancel" );
            btnRevert.Text = _( "Revert" );
            btnLoadDefault.Text = _( "Load Default" );

            columnCommand.Text = _( "Command" );
            columnShortcut.Text = _( "Shortcut Key" );

            toolTip.SetToolTip( list, _( "Select command and hit key(s) you want to set.\nHit Delete or Backspace if you want to remove shortcut key." ) );

            list.Groups["listGroupFile"].Header = _( "File" );
            list.Groups["listGroupEdit"].Header = _( "Edit" );
            list.Groups["listGroupVisual"].Header = _( "View" );
            list.Groups["listGroupJob"].Header = _( "Job" );
            list.Groups["listGroupLyric"].Header = _( "Lyrics" );
            list.Groups["listGroupSetting"].Header = _( "Setting" );
            list.Groups["listGroupHelp"].Header = _( "Help" );
            list.Groups["listGroupTrack"].Header = _( "Track" );
            list.Groups["listGroupScript"].Header = _( "Script" );
            list.Groups["listGroupOther"].Header = _( "Others" );
        }

        private static string _( string id ) {
            return Messaging.GetMessage( id );
        }

        public Dictionary<string, ValuePair<string, Keys[]>> Result {
            get {
                Dictionary<string, ValuePair<string, Keys[]>> ret = new Dictionary<string, ValuePair<string, Keys[]>>();
                CopyDict( m_dict, ref ret );
                return ret;
            }
        }

        private static void CopyDict( Dictionary<string, ValuePair<string, Keys[]>> src, ref Dictionary<string, ValuePair<string, Keys[]>> dest ) {
            dest.Clear();
            foreach ( string name in src.Keys ) {
                string key = src[name].Key;
                Keys[] values = src[name].Value;
                List<Keys> cp = new List<Keys>();
                foreach ( Keys k in values ) {
                    cp.Add( k );
                }
                dest.Add( name, new ValuePair<string, Keys[]>( key, cp.ToArray() ) );
            }
        }

        private void UpdateList() {
            list.Items.Clear();
            foreach ( string display in m_dict.Keys ) {
                Keys k = Keys.None;
                List<Keys> a = new List<Keys>();
                foreach( Keys key in m_dict[display].Value ){
                    a.Add( key );
                    k = k | key;
                }
                try {
                    m_dumy.ShortcutKeys = k;
                } catch {
                    k = Keys.None;
                }
                ListViewItem item = new ListViewItem( new string[] { display, AppManager.GetShortcutDisplayString( a.ToArray() ) } );
                string name = m_dict[display].Key;
                item.Name = name;
                item.Tag = k;
                if ( name.StartsWith( "menuFile" ) ) {
                    item.Group = list.Groups["listGroupFile"];
                } else if ( name.StartsWith( "menuEdit" ) ) {
                    item.Group = list.Groups["listGroupEdit"];
                } else if ( name.StartsWith( "menuVisual" ) ){
                    item.Group = list.Groups["listGroupVisual"];
                } else if ( name.StartsWith( "menuJob" ) ) {
                    item.Group = list.Groups["listGroupJob"];
                } else if ( name.StartsWith( "menuLyric" ) ){
                    item.Group = list.Groups["listGroupLyric"];
                } else if ( name.StartsWith( "menuTrack" ) ){
                    item.Group = list.Groups["listGroupTrack"];
                } else if ( name.StartsWith( "menuScript" ) ){
                    item.Group = list.Groups["listGroupScript"];
                } else if ( name.StartsWith( "menuSetting" ) ){
                    item.Group = list.Groups["listGroupSetting"];
                } else if ( name.StartsWith( "menuHelp" ) ) {
                    item.Group = list.Groups["listGroupHelp"];
                } else {
                    item.Group = list.Groups["listGroupOther"];
                }
                list.Items.Add( item );
            }
            UpdateColor();
        }

        private void list_PreviewKeyDown( object sender, PreviewKeyDownEventArgs e ) {
        }

        private void list_KeyDown( object sender, KeyEventArgs e ) {
            if ( list.SelectedIndices.Count <= 0 ) {
                return;
            }
            int index = list.SelectedIndices[0];
            Keys code = e.KeyCode;
            Keys capture = Keys.None;
            List<Keys> capturelist = new List<Keys>();
            foreach ( Keys k in AppManager._SHORTCUT_ACCEPTABLE ) {
                if ( code == k ) {
                    capturelist.Add( k );
                    capture = k;
                    break;
                }
            }
            Keys res = capture;
            if ( (e.Modifiers & Keys.Menu) == Keys.Menu ) {
                res = res | Keys.Menu;
                capturelist.Add( Keys.Menu );
            }
            if ( (e.Modifiers & Keys.Control) == Keys.Control ) {
                res = res | Keys.Control;
                capturelist.Add( Keys.Control );
            }
            if ( (e.Modifiers & Keys.Shift) == Keys.Shift ) {
                res = res | Keys.Shift;
                capturelist.Add( Keys.Shift );
            }
            if ( (e.Modifiers & Keys.Alt) == Keys.Alt ) {
                res = res | Keys.Alt;
                capturelist.Add( Keys.Alt );
            }

            // 指定されたキーの組み合わせが、ショートカットとして適切かどうか判定
            try {
                m_dumy.ShortcutKeys = res;
            } catch {
                return;
            }
            list.Items[index].Tag = res;
            list.Items[index].SubItems[1].Text = AppManager.GetShortcutDisplayString( capturelist.ToArray() );
            string display = list.Items[index].SubItems[0].Text;
            if ( m_dict.ContainsKey( display ) ) {
                m_dict[display].Value = capturelist.ToArray();
            }
            UpdateColor();
        }

        private void btnRevert_Click( object sender, EventArgs e ) {
            CopyDict( m_first_dict, ref m_dict );
            UpdateList();
        }

        private void btnLoadDefault_Click( object sender, EventArgs e ) {
            for ( int i = 0; i < EditorConfig.DEFAULT_SHORTCUT_KEYS.Count; i++ ) {
                string name = EditorConfig.DEFAULT_SHORTCUT_KEYS[i].Key;
                Keys[] keys = EditorConfig.DEFAULT_SHORTCUT_KEYS[i].Value;
                foreach ( string display in m_dict.Keys ) {
                    if ( name == m_dict[display].Key ) {
                        m_dict[display].Value = keys;
                        break;
                    }
                }
            }
            UpdateList();
        }

        private void UpdateColor() {
            for ( int i = 0; i < list.Items.Count; i++ ) {
                string compare = list.Items[i].SubItems[1].Text;
                if ( compare == "" ) {
                    list.Items[i].BackColor = SystemColors.Window;
                    continue;
                }
                bool found = false;
                for ( int j = 0; j < list.Items.Count; j++ ) {
                    if ( i == j ) {
                        continue;
                    }
                    if ( compare == list.Items[j].SubItems[1].Text ) {
                        found = true;
                        break;
                    }
                }
                if ( found ) {
                    list.Items[i].BackColor = Color.Yellow;
                } else {
                    list.Items[i].BackColor = SystemColors.Window;
                }
            }
        }
    }

}
