﻿/*
 * FormSynthesize.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Forms;

using Boare.Lib.AppUtil;
using Boare.Lib.Vsq;
using Boare.Lib.Media;

namespace Boare.Cadencii {

    /// <summary>
    /// レンダリングの進捗状況を表示しながら，バックグラウンドでレンダリングを行うフォーム．フォームのLoadと同時にレンダリングが始まる．
    /// </summary>
    public partial class FormSynthesize : Form {
        private VsqFileEx m_vsq;
        private int m_presend = 500;
        private int[] m_tracks;
        private string[] m_files;
        private int m_clock_start;
        private int m_clock_end;
        private bool m_partial_mode = false;
        private int m_temp_premeasure = 0;
        private int m_finished = -1;
        //private AppManager AppManager = null;
        private bool m_rendering_started = false;

        public FormSynthesize( /*AppManager manager, */
                               VsqFileEx vsq,
                               int presend,
                               int track,
                               string file,
                               int clock_start,
                               int clock_end,
                               int temp_premeasure ) {
            //AppManager = manager;
            m_vsq = vsq;
            m_presend = presend;
            m_tracks = new int[] { track };
            m_files = new string[] { file };
            InitializeComponent();
            lblProgress.Text = "1/" + 1;
            progressWhole.Maximum = 1;
            m_partial_mode = true;
            m_clock_start = clock_start;
            m_clock_end = clock_end;
            m_temp_premeasure = temp_premeasure;
            ApplyLanguage();
            Misc.ApplyFontRecurse( this, AppManager.EditorConfig.BaseFont );
        }
        
        public FormSynthesize( VsqFileEx vsq, int presend, int[] tracks, string[] files, int end ) {
            //AppManager = manager;
            m_vsq = vsq;
            m_presend = presend;
            m_tracks = tracks;
            m_files = files;
            InitializeComponent();
            lblProgress.Text = "1/" + m_tracks.Length;
            progressWhole.Maximum = m_tracks.Length;
            m_partial_mode = false;
            m_clock_end = end;
            ApplyLanguage();
            Misc.ApplyFontRecurse( this, AppManager.EditorConfig.BaseFont );
        }

        public void ApplyLanguage() {
            this.Text = _( "Synthesize" );
            lblSynthesizing.Text = _( "now synthesizing..." );
            btnCancel.Text = _( "Cancel" );
        }

        private static string _( string id ) {
            return Messaging.GetMessage( id );
        }

        /// <summary>
        /// レンダリングが完了したトラックのリストを取得します
        /// </summary>
        public int[] Finished {
            get {
                List<int> list = new List<int>();
                for ( int i = 0; i <= m_finished; i++ ) {
                    list.Add( m_tracks[i] );
                }
                return list.ToArray();
            }
        }

        private void FormSynthesize_Load( object sender, EventArgs e ) {
            Start();
        }

        private void Start() {
            if ( VSTiProxy.CurrentUser == "" ) {
                VSTiProxy.CurrentUser = AppManager.ID;
                timer.Enabled = true;
                m_rendering_started = true;
                bgWork.RunWorkerAsync();
            } else {
                m_rendering_started = false;
                this.DialogResult = DialogResult.Cancel;
            }
        }

        private void UpdateProgress( int value ) {
            progressWhole.Value = value > progressWhole.Maximum ? progressWhole.Maximum : value;
            lblProgress.Text = value + "/" + m_tracks.Length;
            m_finished = value - 1;
        }

        private void bgWork_DoWork( object sender, DoWorkEventArgs e ) {
            double amp_master = VocaloSysUtil.getAmplifyCoeffFromFeder( m_vsq.Mixer.MasterFeder );
            double pan_left_master = VocaloSysUtil.getAmplifyCoeffFromPanLeft( m_vsq.Mixer.MasterPanpot );
            double pan_right_master = VocaloSysUtil.getAmplifyCoeffFromPanRight( m_vsq.Mixer.MasterPanpot );
            if ( m_partial_mode ) {
                this.Invoke( new BSimpleDelegate<int>( this.UpdateProgress ), new object[] { (object)1 } );
                double amp_track = VocaloSysUtil.getAmplifyCoeffFromFeder( m_vsq.Mixer.Slave[m_tracks[0] - 1].Feder );
                double pan_left_track = VocaloSysUtil.getAmplifyCoeffFromPanLeft( m_vsq.Mixer.Slave[m_tracks[0] - 1].Panpot );
                double pan_right_track = VocaloSysUtil.getAmplifyCoeffFromPanRight( m_vsq.Mixer.Slave[m_tracks[0] - 1].Panpot );
                double amp_left = amp_master * amp_track * pan_left_master * pan_left_track;
                double amp_right = amp_master * amp_track * pan_right_master * pan_right_track;
                using ( WaveWriter ww = new WaveWriter( m_files[0] ) ) {
                    VSTiProxy.Render( m_vsq,
                                      m_tracks[0],
                                      ww,
                                      m_vsq.getSecFromClock( m_clock_start ),
                                      m_vsq.getSecFromClock( m_clock_end ),
                                      amp_left,
                                      amp_right,
                                      m_presend,
                                      false,
                                      new string[] { },
                                      0.0,
                                      false,
                                      AppManager.TempWaveDir );
                }
            } else {
                for ( int i = 0; i < m_tracks.Length; i++ ) {
                    this.Invoke( new BSimpleDelegate<int>( this.UpdateProgress ), new object[] { (object)(i + 1) } );
                    List<VsqNrpn> nrpn = new List<VsqNrpn>( VsqFile.generateNRPN( m_vsq, m_tracks[i], m_presend ) );
                    int count = m_vsq.Track[m_tracks[i]].getEventCount();
                    if ( count > 0 ) {
#if DEBUG
                        AppManager.DebugWriteLine( "FormSynthesize+bgWork_DoWork" );
                        AppManager.DebugWriteLine( "    System.IO.Directory.GetCurrentDirectory()=" + System.IO.Directory.GetCurrentDirectory() );
                        AppManager.DebugWriteLine( "    VsqUtil.VstiDllPath=" + VocaloSysUtil.getDllPathVsti2() );
#endif
                        double amp_track = VocaloSysUtil.getAmplifyCoeffFromFeder( m_vsq.Mixer.Slave[m_tracks[i] - 1].Feder );
                        double pan_left_track = VocaloSysUtil.getAmplifyCoeffFromPanLeft( m_vsq.Mixer.Slave[m_tracks[i] - 1].Panpot );
                        double pan_right_track = VocaloSysUtil.getAmplifyCoeffFromPanRight( m_vsq.Mixer.Slave[m_tracks[i] - 1].Panpot );
                        double amp_left = amp_master * amp_track * pan_left_master * pan_left_track;
                        double amp_right = amp_master * amp_track * pan_right_master * pan_right_track;
                        int total_clocks = m_vsq.TotalClocks;
                        double total_sec = m_vsq.getSecFromClock( total_clocks );
                        using ( WaveWriter ww = new WaveWriter( m_files[i] ) ) {
                            VSTiProxy.Render( m_vsq,
                                              m_tracks[i],
                                              ww,
                                              m_vsq.getSecFromClock( m_clock_start ),
                                              m_vsq.getSecFromClock( m_clock_end ),
                                              amp_left,
                                              amp_right,
                                              m_presend,
                                              false,
                                              new string[] { },
                                              0.0,
                                              false,
                                              AppManager.TempWaveDir );
                        }
                    }
                }
            }
        }

        private void FormSynthesize_FormClosing( object sender, FormClosingEventArgs e ) {
            timer.Enabled = false;
            if ( m_rendering_started ) {
                VSTiProxy.CurrentUser = "";
            }
            if ( bgWork.IsBusy ) {
                VSTiProxy.AbortRendering();
                bgWork.CancelAsync();
                while ( bgWork.IsBusy ) {
                    Application.DoEvents();
                }
                DialogResult = DialogResult.Cancel;
            } else {
                DialogResult = DialogResult.OK;
            }
        }

        private void bgWork_RunWorkerCompleted( object sender, RunWorkerCompletedEventArgs e ) {
            timer.Enabled = false;
            DialogResult = DialogResult.OK;
            this.Close();
        }

        private void timer_Tick( object sender, EventArgs e ) {
            double progress = (int)VSTiProxy.GetProgress();
            progressOne.Value = (int)progress > 100 ? 100 : (int)progress;
        }
    }

}
