﻿/*
 * MidiEvent.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
#if !__cplusplus
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
#endif

#if __cplusplus
namespace Boare{ namespace Lib{ namespace Vsq{
#else
namespace Boare.Lib.Vsq {
#endif

#if __cplusplus
    typedef enum {
#else
    public enum MidiEventType {
#endif
        /// <summary>
        /// channel = 0～15の値
        /// </summary>
        channel,
        /// <summary>
        /// note = 0～127の値
        /// </summary>
        note,
        /// <summary>
        /// dtime = 0～268,435,455 (0x0FFFFFFF)の値
        /// </summary>
        dtime,
        /// <summary>
        /// velocity = 0～127の値
        /// </summary>
        velocity,
        /// <summary>
        /// patch = 0～127の値
        /// </summary>
        patch,
        /// <summary>
        /// sequence = 0-65,535 (0xFFFF)の値
        /// </summary>
        sequence,
        /// <summary>
        /// text = 0byte以上のASCII文字列
        /// </summary>
        text,
        /// <summary>
        /// raw = 0byte以上のバイナリデータの文字列
        /// </summary>
        raw,
        /// <summary>
        /// pitch_wheel = -8192～8191 (0x1FFF)の値
        /// </summary>
        pitch_wheel,
        /// <summary>
        /// song_pos = 0～16,383 (0x3FFF)の値
        /// </summary>
        song_pos,
        /// <summary>
        /// song_number = 0～127の値
        /// </summary>
        song_number,
        /// <summary>
        /// tempo = マイクロ秒, 0～16,777,215 (0x00FFFFFF)の値
        /// </summary>
        tempo,
        time_signal,
        unknown,
        /// <summary>
        /// 勝手に追加。スイマセン
        /// </summary>
        parameter,
#if __cplusplus
    } MidiEventType;
#else
    }
#endif

#if __cplusplus
    class MidiEvent{
#else
    [Serializable]
    public class MidiEvent : IComparable<MidiEvent>, ICloneable {
#endif
#if __cplusplus
    public: 
        int index;
    public: 
        MidiEventType type;
#else
        public int index;
        public MidiEventType type;
#endif

#if __cplusplus
    public: 
        std::vector<int> intValue;
#else
        public int[] intValue;
#endif

#if __cplusplus
    public:
        std::string stringValue;
#else
        public string stringValue;
#endif

#if __cplusplus
    public:
        std::vector<unsigned char> byteValue;
#else
        public byte[] byteValue;
#endif

#if __cplusplus
    public:
        static MidiEvent *TempoChange( int clock, int tempo ){
            MidiEvent *res = new MidiEvent();
#else
        public static MidiEvent TempoChange( int clock, int tempo ) {
            MidiEvent res = new MidiEvent();
#endif
#if __cplusplus
            res->index = clock;
            res->type = Boare::Lib::Vsq::tempo;
            res->intValue = std::vector<int>( 1 );
            res->intValue[0] = tempo;
#else
            res.index = clock;
            res.type = MidiEventType.tempo;
            res.intValue = new int[1];
            res.intValue[0] = tempo;
#endif
            return res;
        }

#if __cplusplus
    public:
        static MidiEvent *TimeSig( int clock, int numerator, int denominator ) {
            MidiEvent *res = new MidiEvent();
#else
        public static MidiEvent TimeSig( int clock, int numerator, int denominator ) {
            MidiEvent res = new MidiEvent();
#endif
#if __cplusplus
            res->index = clock;
            res->type = Vsq::time_signal;
            res->intValue = std::vector<int>( 2 );
            res->intValue[0] = numerator;
            res->intValue[1] = denominator;
#else
            res.index = clock;
            res.type = MidiEventType.time_signal;
            res.intValue = new int[2];
            res.intValue[0] = numerator;
            res.intValue[1] = denominator;
#endif
            return res;
        }

#if __cplusplus
    public :
        MidiEvent *Clone(){
            MidiEvent *res = new MidiEvent();
#else
        public object Clone() {
            MidiEvent res = new MidiEvent();
#endif
#if __cplusplus
            res->index = index;
            res->type = type;
            int len = intValue.size();
            res->intValue = std::vector<int>( len );
            for ( int i = 0; i < len; i++ ) {
                res->intValue[i] = intValue[i];
            }
            res->stringValue = stringValue;
            len = byteValue.size();
            res->byteValue = std::vector<unsigned char>( len );
            for ( int i = 0; i < len; i++ ) {
                res->byteValue[i] = byteValue[i];
            }
#else
            res.index = index;
            res.type = type;
            if ( intValue != null ) {
                res.intValue = new int[intValue.Length];
                for ( int i = 0; i < intValue.Length; i++ ) {
                    res.intValue[i] = intValue[i];
                }
            }
            res.stringValue = stringValue;
            if ( byteValue != null ) {
                res.byteValue = new byte[byteValue.Length];
                for ( int i = 0; i < byteValue.Length; i++ ) {
                    res.byteValue[i] = byteValue[i];
                }
            }
#endif
            return res;
        }

#if __cplusplus
    public:
        int CompareTo( MidiEvent *obj ){
            return index - obj->index;
#else
        public int CompareTo( MidiEvent obj ) {
            return this.index - obj.index;
#endif
        }

#if __cplusplus
    public:
        bool Equals( MidiEvent *obj ){
            if ( index == obj->index ) {
#else
        public bool Equals( MidiEvent obj ) {
            if ( this.index == obj.index ) {
#endif
                return true;
            } else {
                return false;
            }
        }

        MidiEvent() {
        }

        /// <summary>
        /// かきかけ。メタテキスト以外のmidiイベントを取り扱う。
        /// </summary>
        /// <param name="line"></param>
#if __cplusplus
    public:
        MidiEvent( std::string line ){
#else
        public MidiEvent( string line ) {
#endif
            index = -1;
#if __cplusplus
            type = Vsq::unknown;
#else
            type = MidiEventType.unknown;
#endif
            stringValue = "";
#if __cplusplus
            byteValue = std::vector<unsigned char>( 0 );
            std::vector<std::string> spl = util_split( line, " " );
            index = util_parse_int( spl[0] );
#else
            byteValue = null;
            string[] spl = line.Split( new char[] { ' ' } );
            index = int.Parse( spl[0] );
#endif
            if ( spl[1] == "Tempo" ) {
#if __cplusplus
                type = Vsq::tempo;
                intValue = std::vector<int>( 1 );
                intValue[0] = util_parse_int( spl[2] );
#else
                type = MidiEventType.tempo;
                intValue = new int[1];
                intValue[0] = int.Parse( spl[2] );
#endif
            } else if ( spl[1] == "TimeSig" ) {
#if __cplusplus
                type = Vsq::time_signal;
                intValue = std::vector<int>( 4 );
                std::vector<std::string> spl2 = util_split( spl[2], "/" );
                intValue[0] = util_parse_int( spl2[0] );
                intValue[1] = util_parse_int( spl2[1] );
                intValue[2] = util_parse_int( spl[3] );
                intValue[3] = util_parse_int( spl[4] );
#else
                type = MidiEventType.time_signal;
                intValue = new int[4];
                string[] spl2 = spl[2].Split( new char[] { '/' } );
                intValue[0] = int.Parse( spl2[0] );
                intValue[1] = int.Parse( spl2[1] );
                intValue[2] = int.Parse( spl[3] );
                intValue[3] = int.Parse( spl[4] );
#endif
            } else if ( spl[0] == "Par" ) {
#if __cplusplus
                type = Vsq::parameter;
                intValue = std::vector<int>( 3 );
                std::vector<std::string> spl3 = util_split( spl[2], "=" );
                intValue[0] = util_parse_int( spl3[1] );
                spl3 = util_split( spl[3], "=" );
                intValue[1] = util_parse_int( spl3[1] );
                spl3 = util_split( spl[4], "=" );
                intValue[2] = util_parse_int( spl3[1] );
#else
                type = MidiEventType.parameter;
                intValue = new int[3];
                string[] spl3 = spl[2].Split( new char[] { '=' } );
                intValue[0] = int.Parse( spl3[1] );
                spl3 = spl[3].Split( new char[] { '=' } );
                intValue[1] = int.Parse( spl3[1] );
                spl3 = spl[4].Split( new char[] { '=' } );
                intValue[2] = int.Parse( spl3[1] );
#endif
            } else {
                stringValue = spl[2];
#if __cplusplus
                type = Vsq::unknown;
                for ( int i = 1; i < spl.size(); i++ ) {
                    stringValue += " " + spl[i];
                }
#else
                type = MidiEventType.unknown;
                for ( int i = 1; i < spl.Length; i++ ) {
                    stringValue += " " + spl[i];
                }
#endif
            }

        }
    };

#if __cplusplus
} } }
#else
}
#endif
