/*
 *
 *    standSpecgram.cpp                                         
 *                              (c) HAL 2010-           
 *
 *  This file is a part of STAND Library.
 * standSpecgram is a structure that contains
 * STAR specgram, PLATINUM aperiodicities and pitch contour.
 * This class provides analysis and synthesis
 * between WORLD and WAVE files.
 *
 * These files are distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
#include "standSpecgram.h"


#ifdef WIN32
#include <windows.h>
#include <process.h>

struct computeArg{
    double *f0;
    int tLen;
    double **specgram;
    double **aperiodicity;
    int fftl;
    double framePeriod;
    int fs;
    double *synthWave;
    long wave_len;
};
unsigned __stdcall compute_multithread_win32(void *arg)
{
    computeArg *p = (computeArg*)arg;

    synthesis(p->f0, p->tLen, p->specgram, p->aperiodicity, p->fftl, p->framePeriod, p->fs, p->synthWave, p->wave_len);

    _endthreadex(0);
    return 0;
}
#endif

standSpecgram::standSpecgram()
{
    aperiodicity = NULL;
    specgram = NULL;
    f0 = NULL;
    t = NULL;
    targetF0 = 200.0;    // とりあえず
    tLen = 0;
    fftl = 0;
    isFast = true;
    ap_len = 4;

	synthWave = NULL;
	wave_len = 0;
}

standSpecgram::~standSpecgram()
{
    destroy();
}

int standSpecgram::computeWaveFile( string_t input, bool fast )
{
    int ret = 0;
    waveFileEx wave;
    double* tmp;
    unsigned long length;

    string t_input;
    mb_conv( input, t_input );
    int check;
    check = wave.readWaveFile(t_input);

    if( check == 1 ){
        length = wave.getWaveLength();
        tmp = new double[length];
        wave.getWaveBuffer( tmp, &length );
        tLen = getSamplesForDIO( fs, (int)length, framePeriod );
        isFast = fast;
        if( fast ){
            ap_len =4;
        }else{
            ap_len = getBands_v3( fs );
        }
        f0 = new double[tLen];
        t  = new double[tLen];
        dio( tmp, (int)length, fs, framePeriod, t, f0 );

        fftl = getFFTLengthForStar( fs );
        specgram = new double*[tLen];
        aperiodicity = new double*[tLen];
        for( int i = 0; i < tLen; i++ ){
            specgram[i] = new double[fftl];
            aperiodicity[i] = new double[ap_len];
            memset( specgram[i], 0, sizeof(double) * fftl );
            memset( aperiodicity[i], 0, sizeof(double) * ap_len );
        }
        star( tmp, (int)length, fs, t, f0, specgram );
        if( isFast ){
            platinum( fs, f0, tLen, aperiodicity );
        }else{
            aperiodicityRatio_v3( tmp, length, fs, f0, tLen, framePeriod, aperiodicity, &targetF0 );
        }

        SAFE_DELETE_ARRAY( tmp );
        ret = 1;
    }
    return ret;
}

int standSpecgram::computeWaveFile( string_t input, utauParameters& parameters, bool fast )
{
    int ret = 0;
    waveFileEx wave;
    double *tmp;
//    double *utauF0;
//    double *utauDYN;
    unsigned long length;

    string t_input;
    mb_conv( input, t_input );
    if( wave.readWaveFile( t_input + ".wav" ) == 1 ){
//        utauFreq frqFile;
//        frqFile.readFrqFile( input + _T("_wav.frq") );

        // 諸々変数の準備＆計算
        length = wave.getWaveLength( parameters.msLeftBlank, parameters.msRightBlank );
        if( length < 1 ) length = 1;
        tmp = new double[length];
        wave.getWaveBuffer( tmp, parameters.msLeftBlank, parameters.msRightBlank, length );

        tLen = getSamplesForDIO( fs, (int)length, framePeriod );

        f0 = new double[tLen];
        t  = new double[tLen];
//        utauF0 = new double[tLen];
//        utauDYN = new double[tLen];

        // 音量の正規化を行う．
        long index = (long)( (double)fs * parameters.msFixedLength / 1000.0 );
        double sum1 = 0.0, sum2 = 0.0;
        // 固定長終了位置の音量を得る．
        for( int i = index - 1024; 0 <= i && i < index + 1024 && i < length; i++ ){
            sum1 += tmp[i] * tmp[i];
        }
        // 左ブランク終了位置の音量を得る．
        for( int i = 0; i < 2048 && i < length; i++){
            sum2 += tmp[i] * tmp[i];
        }
        // 大きい方が正規化のための音量．
        sum1 = max(sum1, sum2);
        sum1 = VOL_NORMALIZE / sqrt( ( sum1 / 2048.0 ) );

        for( int i = 0; i < length; i++ ){
            tmp[i] *= sum1;
        }

        dio( tmp, (int)length, fs, framePeriod, t, f0 );

        // 固定長以下は定常部と仮定 -> 推定ミスは有声を無声と判断するためなのでカット．
/*        for( int i = (int)(parameters.msFixedLength / framePeriod); i < tLen; i++ )
            if( !i && !f0[i] )
                f0[i] = f0[i-1];
                */

/*        frqFile.getF0Contour( utauF0, parameters.msLeftBlank, parameters.msRightBlank, tLen );

        // UTAU周波数表からの補完
        for( int i = 0; i < tLen; i++ )
            if( !f0[i] && utauF0[i] || utauF0[i] / f0[i] < 1 / 1.25 || utauF0[i] / f0[i] > 1.25 )
                f0[i] = utauF0[i];
*/
        fftl = getFFTLengthForStar( fs );
        isFast = fast;
        if( isFast )
            ap_len = 4;
        else
            ap_len = getBands_v3( fs );
        specgram = new double*[tLen];
        aperiodicity = new double*[tLen];
        for( int i = 0; i < tLen; i++ ){
            specgram[i] = new double[fftl];
            aperiodicity[i] = new double[ap_len];
            memset( specgram[i], 0, sizeof(double) * fftl );
            memset( aperiodicity[i], 0, sizeof(double) * ap_len );
        }
        star( tmp, (int)length, fs, t, f0, specgram );
        if( isFast )
            platinum( fs, f0, tLen, aperiodicity );
        else
            aperiodicityRatio_v3( tmp, length, fs, f0, tLen, framePeriod, aperiodicity, &targetF0 );

        SAFE_DELETE_ARRAY( tmp );
//        SAFE_DELETE_ARRAY( utauF0 );
        ret = 1;
    }
    return ret;
}

int standSpecgram::writeWaveFile( string_t output, long beginFrame, vector<double>* dynamics ){
    int ret = 0;
    waveFileEx wave;

	if( synthWave ){
        double secOffset = (double)beginFrame * framePeriod / 1000.0;

        wave.setWaveBuffer( synthWave, (unsigned long)wave_len );
        if( dynamics ){
            wave.applyDynamics( *dynamics, fs, framePeriod );
        }
        wave.setOffset( secOffset );
        wave.normalize();
        string t_output;
        mb_conv( output, t_output );
        wave.writeWaveFile( t_output );
        ret = 1;
    }

    return ret;
}

double* standSpecgram::synthesizeWave( long *length ){
    wave_len = (int)( (double)tLen * fs * framePeriod / 1000.0 );
    SAFE_DELETE_ARRAY( synthWave );
    synthWave = new double[wave_len];

    if( synthWave ){
        // 教訓： メモリのゼロクリアは重要である．
        memset( synthWave, 0, sizeof(double) * wave_len );
        if( isFast ){
#ifdef WIN32
            // 十分に処理が遅ければ破綻しない．ただし真ん中の数サンプルに誤差は出てるはず．
            computeArg arg1, arg2;
            HANDLE hThread[2];
            hFFTWMutex = CreateMutex(NULL,FALSE,NULL);
            arg1.f0 = f0;
            arg1.specgram = specgram;
            arg1.aperiodicity = aperiodicity;
            arg1.fftl = fftl;
            arg1.framePeriod = framePeriod;
            arg1.fs = fs;
            arg1.synthWave = synthWave;
            arg1.wave_len = wave_len;

            // ここで波形の位置を正しく計算しておきたい．
            // wave のバッファが重なる fft / 2 の長さだけスレッドセーフでない．
            double currentTime = 0.0;
            int currentPosition = 0;//currentTime / framePeriod;
            int currentFrame = 0;
            for(; ;){
                currentPosition = (int)(currentTime*(double)fs);
                currentTime += 1.0/(f0[currentFrame] == 0.0 ? 160.0 : f0[currentFrame]);
                currentFrame = (int)(currentTime/(framePeriod/1000.0) + 0.5);
                currentPosition = (int)(currentTime*(double)fs);
                if(currentFrame >= tLen / 2) break; 
            }
            arg1.tLen = currentFrame;

            arg2.f0 = &f0[currentFrame];
            arg2.tLen = tLen - currentFrame;
            arg2.specgram = &specgram[currentFrame];
            arg2.aperiodicity = &aperiodicity[currentFrame];
            arg2.fftl = fftl;
            arg2.framePeriod = framePeriod;
            arg2.fs = fs;
            arg2.synthWave = &synthWave[currentPosition];
            arg2.wave_len = wave_len - currentPosition;

            hThread[0] = (HANDLE)_beginthreadex(NULL, 0, compute_multithread_win32, &arg1, 0, NULL);
            hThread[1] = (HANDLE)_beginthreadex(NULL, 0, compute_multithread_win32, &arg2, 0, NULL);

            WaitForMultipleObjects(2, hThread, TRUE, INFINITE);
            CloseHandle(hFFTWMutex);
            hFFTWMutex = 0;
            CloseHandle(hThread[0]);
            CloseHandle(hThread[1]);
#else
            synthesis( f0, tLen, specgram, aperiodicity, fftl, framePeriod, fs, synthWave, wave_len, 0 );
#endif
        }else{
            double maxF0 = 0.0;
            for( int i = 0; i < tLen; i++ ){
                if( maxF0 < f0[i] ){
                    maxF0 = f0[i];
                }
            }
            targetF0 = max( 32.0, min( 200.0, maxF0 ) );

#ifdef WIN32
            synthesis_ap( f0, tLen, specgram, fftl, aperiodicity, ap_len, targetF0, framePeriod, fs, synthWave, wave_len );
#else
            synthesis_ap( f0, tLen, specgram, fftl, aperiodicity, ap_len, targetF0, framePeriod, fs, synthWave, wave_len );
#endif
        }
        *length = wave_len;
    }else{
        *length = 0;
    }
    return synthWave;
}

void standSpecgram::destroy( void )
{
    for( int i = 0; i < tLen; i++ ){
        SAFE_DELETE_ARRAY( aperiodicity[i] );
        SAFE_DELETE_ARRAY( specgram[i] );
    }
    SAFE_DELETE_ARRAY( aperiodicity );
    SAFE_DELETE_ARRAY( specgram );
    SAFE_DELETE_ARRAY( t );
    SAFE_DELETE_ARRAY( f0 );
    SAFE_DELETE_ARRAY( synthWave );
    tLen = 0;
    wave_len = 0;
    ap_len = 0;
    fftl = 0;
    isFast = true;
}

void standSpecgram::setFrameLength( long length, bool fast )
{
    destroy();

    tLen = length; //getSamplesForDIO( fs, (int)length, framePeriod );
    fftl = getFFTLengthForStar( fs );
    
    // this function does NOT initialize array values.
    f0 = new double[tLen];
    t = new double[tLen];
    aperiodicity = new double*[tLen];
    specgram = new double*[tLen];
    isFast = fast;
    if( isFast )
        ap_len = 4;
    else
        ap_len = getBands_v3( fs );
    for( int i = 0; i < tLen; i++ ){
        aperiodicity[i] = new double[ap_len];
        specgram[i] = new double[fftl];
        memset( specgram[i], 0, sizeof(double) * fftl );
        memset( aperiodicity[i], 0, sizeof(double) * ap_len );
    }
}

void standSpecgram::getFramePointer( long frame, standFrame& target )
{
    target.aperiodicity = aperiodicity[frame];
    target.ap_len = ap_len;
    target.isFast = isFast;
    target.spectrum = specgram[frame];
    target.f0 = &(f0[frame]);
    target.t = &(t[frame]);
    target.fftl = this->fftl;
}

void standSpecgram::setFrame( long frame, standFrame& src )
{
    // without safe guard
    memcpy( specgram[frame], src.spectrum, sizeof(double)*src.fftl );
    memcpy( aperiodicity[frame], src.aperiodicity, sizeof(double)*src.ap_len );

    f0[frame] = *(src.f0);
    t[frame] = *(src.t);
}
