﻿/*
 * TrackSelector.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
//#define MONITOR_FPS
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Threading;
using System.Windows.Forms;
using System.Drawing.Drawing2D;

using Boare.Lib.Vsq;
using Boare.Lib.AppUtil;

namespace Boare.Cadencii {

    partial class TrackSelector : UserControl {
        #region Constants and internal enums
        private enum MouseDownMode {
            None,
            CurveEdit,
            TrackList,
            SingerList,
            /// <summary>
            /// マウス長押しによるVELの編集。マウスがDownされ、MouseHoverが発生するのを待っている状態
            /// </summary>
            VelWaitHover,
            /// <summary>
            /// マウス長押しによるVELの編集。MouseHoverが発生し、編集している状態
            /// </summary>
            VelEdit,
            /// <summary>
            /// ベジエカーブのデータ点または制御点を移動させているモード
            /// </summary>
            BezierMove,
            /// <summary>
            /// ベジエカーブのデータ点の範囲選択をするモード
            /// </summary>
            BezierSelect,
            /// <summary>
            /// ベジエカーブのデータ点を新規に追加し、マウスドラッグにより制御点の位置を変えているモード
            /// </summary>
            BezierAddNew,
            /// <summary>
            /// 既存のベジエカーブのデータ点を追加し、マウスドラッグにより制御点の位置を変えているモード
            /// </summary>
            BezierEdit,
        }
        private static readonly Color _DOT_COLOR_MASTER = Color.Red;
        private static readonly Color _CURVE_COLOR = Color.White;
        private static readonly Color _CONROL_LINE = Color.Orange;
        private static readonly Color _DOT_COLOR_NORMAL = Color.FromArgb( 237, 107, 158 );
        private static readonly Color _DOT_COLOR_BASE = Color.FromArgb( 125, 198, 34 );
        private static readonly Color _DOT_COLOR_NORMAL_AROUND = Color.FromArgb( 231, 50, 122 );
        private static readonly Color _DOT_COLOR_BASE_AROUND = Color.FromArgb( 90, 143, 24 );
        private static readonly SolidBrush s_brs_a244_255_023_012 = new SolidBrush( Color.FromArgb( 244, 255, 23, 12 ) );
        private static readonly SolidBrush s_brs_a144_255_255_255 = new SolidBrush( Color.FromArgb( 144, 255, 255, 255 ) );
        private static readonly SolidBrush s_brs_a072_255_255_255 = new SolidBrush( Color.FromArgb( 72, 255, 255, 255 ) );
        private static readonly SolidBrush s_brs_a127_008_166_172 = new SolidBrush( Color.FromArgb( 127, 8, 166, 172 ) );
        private static readonly SolidBrush s_brs_a098_000_000_000 = new SolidBrush( Color.FromArgb( 98, 0, 0, 0 ) );
        private static readonly SolidBrush s_brs_dot_master = new SolidBrush( _DOT_COLOR_MASTER );
        private static readonly SolidBrush s_brs_dot_normal = new SolidBrush( _DOT_COLOR_NORMAL );
        private static readonly SolidBrush s_brs_dot_base = new SolidBrush( _DOT_COLOR_BASE );
        private static readonly Pen s_pen_dot_normal = new Pen( _DOT_COLOR_NORMAL_AROUND );
        private static readonly Pen s_pen_dot_base = new Pen( _DOT_COLOR_BASE_AROUND );
        private static readonly Pen s_pen_050_140_150 = new Pen( Color.FromArgb( 50, 140, 150 ) );
        private static readonly Pen s_pen_128_128_128 = new Pen( Color.FromArgb( 128, 128, 128 ) );
        private static readonly Pen s_pen_246_251_010 = new Pen( Color.FromArgb( 246, 251, 10 ) );
        private static readonly Pen s_pen_curve = new Pen( _CURVE_COLOR );
        private static readonly Pen s_pen_control_line = new Pen( _CONROL_LINE );
        /// <summary>
        /// ベロシティを画面に描くときの，棒グラフの幅(pixel)
        /// </summary>
        public const int _VEL_BAR_WIDTH = 8;
        /// <summary>
        /// パフォーマンスカウンタ用バッファの容量
        /// </summary>
        const int _NUM_PCOUNTER = 50;
        /// <summary>
        /// コントロールの下辺から、TRACKタブまでのオフセット(px)
        /// </summary>
        public const int _OFFSET_TRACK_TAB = 19;
        const int _FOOTER = 7;
        public const int _HEADER = 8;
        const int _BUF_LEN = 512;
        /// <summary>
        /// 歌手変更イベントの表示矩形の幅
        /// </summary>
        const int _SINGER_ITEM_WIDTH = 66;
        /*/// <summary>
        /// コントロールの下辺から、SINGERタブまでのオフセット(px)
        /// </summary>
        const int _OFFSET_SINGER_TAB = 35;*/
        /*/// <summary>
        /// RENDER ALLボタンの幅(px)
        /// </summary>
        const int _PX_WIDTH_RENDERALL = 70;*/
        /// <summary>
        /// RENDERボタンの幅(px)
        /// </summary>
        const int _PX_WIDTH_RENDER = 10;
        /// <summary>
        /// カーブ制御点の幅（実際は_DOT_WID * 2 + 1ピクセルで描画される）
        /// </summary>
        const int _DOT_WID = 3;
        static readonly Rectangle _PLUS_MARK = new Rectangle( 7, 38, 23, 21 );
        static readonly Rectangle _MINUS_MARK = new Rectangle( 40, 38, 23, 21 );
        const int _VSCROLL_WIDTH = 16;
        const int _ZOOMPANEL_HEIGHT = 33;
        #endregion

        //private int m_start_to_draw = 0;
        //private float m_scale = 1f;
        private CurveType m_selected_curve = CurveType.VEL;
        private CurveType m_last_selected_curve = CurveType.DYN;
        private bool m_curve_visible = true;
        /// <summary>
        /// 現在のマウス位置におけるカーブの値
        /// </summary>
        private int m_mouse_value;
        private Point[] m_points = new Point[_BUF_LEN];
        /// <summary>
        /// 編集しているBezierChainのID
        /// </summary>
        public int EditingChainID = -1;
        /// <summary>
        /// 編集しているBezierPointのID
        /// </summary>
        public int EditingPointID = -1;
#if MONITOR_FPS
        /// <summary>
        /// パフォーマンスカウンタ
        /// </summary>
        private float[] m_performance = new float[_NUM_PCOUNTER];
        /// <summary>
        /// 最後にpictureBox1_Paintが実行された時刻
        /// </summary>
        private DateTime m_last_ignitted;
        /// <summary>
        /// パフォーマンスカウンタから算出される画面の更新速度
        /// </summary>
        private float m_fps = 0f;
#endif
        /// <summary>
        /// ラインツールが降りた位置。x座標はクロック、y座標はカーブのコントロール値
        /// </summary>
        private Point m_line_start;
        /// <summary>
        /// マウスがカーブ部分に下ろされている最中かどうかを表すフラグ
        /// </summary>
        private bool m_mouse_downed = false;
        /// <summary>
        /// マウスのトレース。
        /// </summary>
        private SortedList<int, int> m_mouse_trace;
        /// <summary>
        /// マウスのトレース時、前回リストに追加されたx座標の値
        /// </summary>
        private int m_mouse_trace_last_x;
        private int m_mouse_trace_last_y;
        private Point m_selecting_region;
        private Point m_selected_region;
        private bool m_selected_region_enabled = false;
        private bool m_pencil_moved = false;
        private Thread m_mouse_hover_thread = null;
        /// <summary>
        /// cmenuSingerのメニューアイテムを初期化するのに使用したRenderer。
        /// </summary>
        private string m_cmenu_singer_prepared = "";
        /// <summary>
        /// マウスがDownしてからUpするまでのモード
        /// </summary>
        private MouseDownMode m_mouse_down_mode = MouseDownMode.None;
        /// <summary>
        /// マウスがDownしてからマウスが移動したかどうかを表す。
        /// </summary>
        private bool m_mouse_moved = false;
        /// <summary>
        /// マウスドラッグで歌手変更イベントの矩形を移動開始した時の、マウス位置におけるクロック
        /// </summary>
        private int m_singer_move_started_clock;
        /// <summary>
        /// cmenuSinger用のツールチップの幅を記録しておく。
        /// </summary>
        private int[] m_cmenusinger_tooltip_width;
        /*/// <summary>
        /// cmenuSingerのメニューアイテムが、どのプログラムチェンジかを記録しておく
        /// </summary>
        private int[] m_cmenusinger_map;*/
        /// <summary>
        /// マウス長押しによるVELの編集。選択されている音符のInternalID
        /// </summary>
        private int m_veledit_last_selectedid = -1;
        /// <summary>
        /// マウス長押しによるVELの編集。棒グラフのてっぺんの座標と、マウスが降りた座標の差分。プラスなら、マウスの方が下になる。
        /// </summary>
        private int m_veledit_shifty = 0;
        /// <summary>
        /// マウス長押しによるVELの編集。編集対象の音符のリスト。
        /// </summary>
        private Dictionary<int, SelectedEventEntry> m_veledit_selected = new Dictionary<int, SelectedEventEntry>();
        /// <summary>
        /// 現在編集操作が行われているBezierChainの、編集直前のオリジナル
        /// </summary>
        private BezierChain m_editing_bezier_original = null;
        /// <summary>
        /// CTRLキー。MacOSXの場合はMenu
        /// </summary>
        private Keys s_modifier_key = Keys.Control;
        private EditorManager m_manager = null;
        /// <summary>
        /// このコントロールが表示を担当しているカーブのリスト
        /// </summary>
        private List<CurveType> m_viewing_curves = new List<CurveType>();
        //private Font m_font8 = new Font( "MS UI Gothic", 8 );
        //private Font m_font9 = new Font( "MS UI Gothic", 9 );
        private Pen m_generic_line = new Pen( Color.FromArgb( 118, 123, 138 ) );
        /// <summary>
        /// スペースキーが押されているかどうか。
        /// MouseDown時に範囲選択モードをスキップする必要があるので、FormMainでの処理に加えてこのクラス内部でも処理する必要がある
        /// </summary>
        private bool m_spacekey_downed = false;
        /// <summary>
        /// CurveType.Pitchが選択されている場合の縦方向の倍率
        /// </summary>
        private int m_internal_pbs = 1;

        public delegate void CommandExecutedEventHandler();

        public event BSimpleDelegate<CurveType> SelectedCurveChanged;
        public event BSimpleDelegate<int> SelectedTrackChanged;
        public event CommandExecutedEventHandler CommandExecuted;
        public event BSimpleDelegate<int[]> RenderRequired;

        public TrackSelector() {
            this.SetStyle( ControlStyles.DoubleBuffer, true );
            this.SetStyle( ControlStyles.UserPaint, true );
            InitializeComponent();
            s_modifier_key = (AppManager.EditorConfig.Platform == Platform.Macintosh) ? Keys.Menu : Keys.Control;
            cmenuCurveAccent.Tag = CurveType.Accent;
            cmenuCurveBreathiness.Tag = CurveType.BRE;
            cmenuCurveBrightness.Tag = CurveType.BRI;
            cmenuCurveClearness.Tag = CurveType.CLE;
            cmenuCurveDecay.Tag = CurveType.Decay;
            cmenuCurveDynamics.Tag = CurveType.DYN;
            cmenuCurveEffect2Depth.Tag = CurveType.fx2depth;
            cmenuCurveGenderFactor.Tag = CurveType.GEN;
            cmenuCurveHarmonics.Tag = CurveType.harmonics;
            cmenuCurveOpening.Tag = CurveType.OPE;
            cmenuCurvePitch.Tag = CurveType.Pitch;
            cmenuCurvePortamentoTiming.Tag = CurveType.POR;
            cmenuCurveReso1Amp.Tag = CurveType.reso1amp;
            cmenuCurveReso1BW.Tag = CurveType.reso1bw;
            cmenuCurveReso1Freq.Tag = CurveType.reso1freq;
            cmenuCurveReso2Amp.Tag = CurveType.reso2amp;
            cmenuCurveReso2BW.Tag = CurveType.reso2bw;
            cmenuCurveReso2Freq.Tag = CurveType.reso2freq;
            cmenuCurveReso3Amp.Tag = CurveType.reso3amp;
            cmenuCurveReso3BW.Tag = CurveType.reso3bw;
            cmenuCurveReso3Freq.Tag = CurveType.reso3freq;
            cmenuCurveReso4Amp.Tag = CurveType.reso4amp;
            cmenuCurveReso4BW.Tag = CurveType.reso4bw;
            cmenuCurveReso4Freq.Tag = CurveType.reso4freq;
            cmenuCurveVelocity.Tag = CurveType.VEL;
            cmenuCurveVibratoDepth.Tag = CurveType.VibratoDepth;
            cmenuCurveVibratoRate.Tag = CurveType.VibratoRate;
        }

        protected override void OnResize( EventArgs e ) {
            base.OnResize( e );
            vScroll.Width = _VSCROLL_WIDTH;
            vScroll.Height = this.Height - _ZOOMPANEL_HEIGHT - 2;
            vScroll.Left = this.Width - _VSCROLL_WIDTH;
            vScroll.Top = 0;

            panelZoomButton.Width = _VSCROLL_WIDTH;
            panelZoomButton.Height = _ZOOMPANEL_HEIGHT;
            panelZoomButton.Left = this.Width - _VSCROLL_WIDTH;
            panelZoomButton.Top = this.Height - _ZOOMPANEL_HEIGHT - 2;
        }

        public void ApplyLanguage() {
        }

        public void ApplyFont( Font font ) {
            foreach ( Control c in Controls ) {
                Boare.Lib.AppUtil.Misc.ApplyFontRecurse( c, font );
            }
            Misc.ApplyContextMenuFontRecurse( cmenuSinger, font );
            Misc.ApplyContextMenuFontRecurse( cmenuCurve, font );
        }

        /// <summary>
        /// このコントロールの推奨最小表示高さを取得します
        /// </summary>
        public int PreferredMinSize {
            get {
                return 240 + 18 * (m_viewing_curves.Count - 10) + 16;
            }
        }

        public EditorManager EditorManager {
            set {
                m_manager = value;
            }
        }

        /// <summary>
        /// このコントロールに担当させるカーブを追加します
        /// </summary>
        /// <param name="curve"></param>
        public void AddViewingCurve( CurveType curve ) {
            AddViewingCurve( new CurveType[] { curve } );
        }

        public void AddViewingCurve( CurveType[] curve ) {
            for ( int j = 0; j < curve.Length; j++ ) {
                bool found = false;
                for ( int i = 0; i < m_viewing_curves.Count; i++ ) {
                    if ( m_viewing_curves[i] == curve[j] ) {
                        found = true;
                        break;
                    }
                }
                if ( !found ) {
                    m_viewing_curves.Add( curve[j] );
                }
            }
            if ( m_viewing_curves.Count >= 2 ) {
                bool changed = true;
                while ( changed ) {
                    changed = false;
                    for ( int i = 0; i < m_viewing_curves.Count - 1; i++ ) {
                        if ( m_viewing_curves[i].Index > m_viewing_curves[i + 1].Index ) {
                            CurveType b = m_viewing_curves[i];
                            m_viewing_curves[i] = m_viewing_curves[i + 1];
                            m_viewing_curves[i + 1] = b;
                            changed = true;
                        }
                    }
                }
            }
        }

        public void ClearViewingCurve() {
            m_viewing_curves.Clear();
        }

        /// <summary>
        /// このコントロールに担当させるカーブを削除します
        /// </summary>
        /// <param name="curve"></param>
        public void RemoveViewingCurve( CurveType curve ) {
            for ( int i = 0; i < m_viewing_curves.Count; i++ ) {
                if ( m_viewing_curves[i] == curve ) {
                    m_viewing_curves.RemoveAt( i );
                    break;
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="command"></param>
        /// <param name="register">Undo/Redo用バッファにExecuteの結果を格納するかどうかを指定するフラグ</param>
        private void ExecuteCommand( CadenciiCommand command, bool register ) {
            if ( register ) {
                m_manager.Register( m_manager.VsqFile.executeCommand( command ) );
            } else {
                m_manager.VsqFile.executeCommand( command );
            }
            if ( CommandExecuted != null ) {
                CommandExecuted();
            }
        }

        public KeyValuePair<int, int> SelectedRegion {
            get {
                int x0 = m_selected_region.X;
                int x1 = m_selected_region.Y;
                int min = Math.Min( x0, x1 );
                int max = Math.Max( x0, x1 );
                return new KeyValuePair<int, int>( min, max );
            }
        }

        public void DisableSelectedRegion() {
            m_selected_region_enabled = false;
        }

        public bool SelectedRegionEnabled {
            get {
                if ( m_selected_curve == CurveType.VEL || m_selected_curve == CurveType.Accent || m_selected_curve == CurveType.Decay ) {
                    return false;
                } else {
                    return m_selected_region_enabled;
                }
            }
        }

        /// <summary>
        /// 現在最前面に表示され，編集可能となっているカーブの種類を取得または設定します
        /// </summary>
        public CurveType SelectedCurve {
            get {
                return m_selected_curve;
            }
            set {
                CurveType old = m_selected_curve;
                m_selected_curve = value;
                if ( old != m_selected_curve ) {
                    m_last_selected_curve = old;
                    if ( SelectedCurveChanged != null ) {
                        SelectedCurveChanged( m_selected_curve );
                    }
                }
            }
        }

        /// <summary>
        /// エディタのx方向の位置からクロック数を求めます
        /// </summary>
        /// <param name="x"></param>
        /// <returns></returns>
        public int ClockFromXCoord( int x ) {
            return (int)((x - AppManager._KEY_LENGTH - 6 + m_manager.StartToDrawX) / m_manager.ScaleX);
        }

        /// <summary>
        /// クロック数から、画面に描くべきx座標の値を取得します。
        /// </summary>
        /// <param name="clocks"></param>
        /// <returns></returns>
        public int XCoordFromClocks( int clocks ) {
            return (int)(clocks * m_manager.ScaleX - m_manager.StartToDrawX) + AppManager._KEY_LENGTH + 6;
        }

        /// <summary>
        /// エディタのy方向の位置から，カーブの値を求めます
        /// </summary>
        /// <param name="y"></param>
        /// <returns></returns>
        public int ValueFromYCoord( int y ) {
            int max = m_selected_curve.Maximum;
            int min = m_selected_curve.Minimum;
            if ( m_selected_curve == CurveType.Pitch ) {
                max = m_internal_pbs * 10000;
                min = -m_internal_pbs * 10000;
            }
            return ValueFromYCoord( y, max, min );
        }

        public int ValueFromYCoord( int y, int max, int min ) {
            int oy = this.Height - 42;
            float order = GraphHeight / (float)(max - min);
            return (int)((oy - y) / order) + min;
        }

        public int YCoordFromValue( int value ) {
            int max = m_selected_curve.Maximum;
            int min = m_selected_curve.Minimum;
            if ( m_selected_curve == CurveType.Pitch ) {
                max = m_internal_pbs * 10000;
                min = -m_internal_pbs * 10000;
            }
            return YCoordFromValue( value, max, min );
        }

        public int YCoordFromValue( int value, int max, int min ) {
            int oy = this.Height - 42;
            float order = GraphHeight / (float)(max - min);
            return oy - (int)((value - min) * order);
        }

        /*private int YCoordFromValue( int value, CurveType curve ) {
            int max = curve.Maximum;
            int min = curve.Minimum;
            int oy = this.Height - 42;
            float order = GraphHeight / (float)(max - min);
            return oy - (int)((value - min) * order);
        }*/

        /// <summary>
        /// カーブエディタを表示するかどうかを取得または設定します
        /// </summary>
        public bool CurveVisible {
            get {
                return m_curve_visible;
            }
            set {
                m_curve_visible = value;
            }
        }

        private void _config_SelectedToolChanged( EditTool tool ) {
            m_selected_region_enabled = false;
        }

        protected override void OnPaint( PaintEventArgs e ) {
            DrawTo( e.Graphics, new Size( this.Width - vScroll.Width + 2, this.Height ) );
#if MONITOR_FPS
            DateTime dnow = DateTime.Now;
            for ( int i = 0; i < _NUM_PCOUNTER - 1; i++ ) {
                m_performance[i] = m_performance[i + 1];
            }
            m_performance[_NUM_PCOUNTER - 1] = (float)dnow.Subtract( m_last_ignitted ).TotalSeconds;
            m_last_ignitted = dnow;
            float sum = 0f;
            for ( int i = 0; i < _NUM_PCOUNTER; i++ ) {
                sum += m_performance[i];
            }
            m_fps = _NUM_PCOUNTER / sum;
            e.Graphics.DrawString(
                m_fps.ToString( "000.000" ),
                new Font( "Verdana", 40, FontStyle.Bold ),
                Brushes.Red,
                new PointF( 0, 0 ) );
#endif
        }

        /// <summary>
        /// x軸方向の表示倍率。pixel/clock
        /// </summary>
        public float ScaleY {
            get {
                int max = m_selected_curve.Maximum;
                int min = m_selected_curve.Minimum;
                int oy = this.Height - 42;
                return GraphHeight / (float)(max - min);
            }
        }

        /// <summary>
        /// 第index番目のトラックが現在選択されているかどうかを判定します。
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public bool IsSelected( int index ) {
            if ( index >= 0 && index == m_manager.Selected ) {
                return true;
            } else {
                return false;
            }
        }

        public Rectangle RectFromCurveType( CurveType curve ) {
            int centre = 17 + GraphHeight / 2 + 8;
            int index = 100;
            for ( int i = 0; i < m_viewing_curves.Count; i++ ) {
                if ( m_viewing_curves[i] == curve ) {
                    index = i;
                    break;
                }
            }
            int y = centre - m_viewing_curves.Count * 9 + 2 + 18 * index;
            return new Rectangle( 7, y, 56, 14 );
        }

        public void DrawTo( Graphics g, Size size ) {
#if DEBUG
            //Common.DebugWriteLine( "TrackSelector+DrawTo(Graphics,Size)" );
            //Common.DebugWriteLine( "    m_manager.Selected=" + m_manager.Selected );
            //Common.DebugWriteLine( "    m_manager.VsqFile.Tracks.Count=" + m_manager.VsqFile.Tracks.Count );
            try{
#endif
            SolidBrush brs_string = new SolidBrush( Color.Black );
            using ( Pen rect_curve = new Pen( Color.FromArgb( 41, 46, 55 ) ) ) {
                int centre = 8 + GraphHeight / 2;
                g.FillRectangle( Brushes.DarkGray,
                                 new Rectangle( 0, size.Height - 2 * _OFFSET_TRACK_TAB, size.Width, 2 * _OFFSET_TRACK_TAB ) );
                int numeric_view = m_mouse_value;

                #region SINGER
                Region last = g.Clip;
                g.DrawLine( m_generic_line,
                            new Point( 2, size.Height - 2 * _OFFSET_TRACK_TAB ),
                            new Point( size.Width - 3, size.Height - 2 * _OFFSET_TRACK_TAB ) );
                g.DrawLine( m_generic_line,
                            new Point( AppManager._KEY_LENGTH, size.Height - 2 * _OFFSET_TRACK_TAB + 1 ),
                            new Point( AppManager._KEY_LENGTH, size.Height - 2 * _OFFSET_TRACK_TAB + 15 ) );
                g.DrawString( "SINGER",
                              AppManager.BaseFont8,
                              brs_string,
                              new PointF( 9, size.Height - 2 * _OFFSET_TRACK_TAB + _OFFSET_TRACK_TAB / 2 - AppManager.BaseFont8OffsetHeight ) );
                g.SetClip( new Rectangle( AppManager._KEY_LENGTH,
                                          size.Height - 2 * _OFFSET_TRACK_TAB,
                                          size.Width - AppManager._KEY_LENGTH,
                                          _OFFSET_TRACK_TAB ),
                                          CombineMode.Replace );
                if ( m_manager.VsqFile != null ) {
                    StringFormat sf = new StringFormat();
                    sf.LineAlignment = StringAlignment.Center;
                    sf.Alignment = StringAlignment.Near;
                    for ( int i = 0; i < m_manager.VsqFile.Track[m_manager.Selected].getEventCount(); i++ ) {
                        VsqEvent ve = m_manager.VsqFile.Track[m_manager.Selected].getEvent( i );
                        if ( ve.ID.type == VsqIDType.Singer ) {
                            int clock = ve.Clock;
                            int x = XCoordFromClocks( clock );
                            Rectangle rc = new Rectangle( x, size.Height - 2 * _OFFSET_TRACK_TAB + 1, _SINGER_ITEM_WIDTH, _OFFSET_TRACK_TAB - 5 );
                            g.FillRectangle( Brushes.White, rc );
                            if ( m_manager.SelectedEvent.ContainsKey( m_manager.Selected, ve.InternalID ) ) {
                                g.DrawRectangle( new Pen( AppManager.HilightBrush ), rc );
                                g.DrawString( ve.ID.IconHandle.IDS,
                                              AppManager.BaseFont8,
                                              brs_string,
                                              rc, 
                                              sf );
                            } else {
                                g.DrawRectangle( new Pen( Color.FromArgb( 182, 182, 182 ) ), rc );
                                g.DrawString( ve.ID.IconHandle.IDS,
                                              AppManager.BaseFont8,
                                              brs_string,
                                              rc,
                                              sf );
                            }
                        }
                    }
                }
                g.SetClip( last, CombineMode.Replace );
                #endregion

                #region トラック選択欄
                int selecter_width = SelectorWidth;
                g.DrawLine( m_generic_line,
                            new Point( 1, size.Height - _OFFSET_TRACK_TAB ),
                            new Point( size.Width - 2, size.Height - _OFFSET_TRACK_TAB ) );
                g.DrawString( "TRACK",
                              AppManager.BaseFont8,
                              brs_string,
                              new PointF( 9, size.Height - _OFFSET_TRACK_TAB + _OFFSET_TRACK_TAB / 2 - AppManager.BaseFont8OffsetHeight ) );
                /*g.DrawString( "RENDER ALL",
                              new Font( "MS UI Gothic", 7 ),
                              brs_string,
                              new PointF( size.Width - 59, size.Height - _OFFSET_TRACK_TAB + 3 ) );*/
                /*g.DrawLine( m_generic_line,
                            new Point( size.Width - _PX_WIDTH_RENDERALL, size.Height - _OFFSET_TRACK_TAB ),
                            new Point( size.Width - _PX_WIDTH_RENDERALL, size.Height - 1 ) );*/
                if ( m_manager.VsqFile != null ) {
                    for ( int i = 0; i < 16; i++ ) {
                        int x = AppManager._KEY_LENGTH + i * selecter_width;
#if DEBUG
                        try {
#endif
                            DrawTrackTab( g,
                                          new Rectangle( x, size.Height - _OFFSET_TRACK_TAB + 1, selecter_width, _OFFSET_TRACK_TAB - 1 ),
                                          (i + 1 < m_manager.VsqFile.Track.Count) ? (i + 1) + " " + m_manager.VsqFile.Track[i + 1].Name : "",
                                          (i == m_manager.Selected - 1),
                                          m_manager.VsqFile.Track[m_manager.Selected].getCommon().PlayMode >= 0,
                                          m_manager.getRenderRequired( i + 1 ),
                                          AppManager.s_HILIGHT[i],
                                          AppManager.s_RENDER[i] );
#if DEBUG
                        } catch ( Exception ex ) {
                            Common.DebugWriteLine( "TrackSelector.DrawTo; ex=" + ex );
                        }
#endif
                    }
                }
                #endregion

                if ( CurveVisible ) {
                    #region カーブエディタ
                    // カーブエディタの下の線
                    g.DrawLine( new Pen( Color.FromArgb( 156, 161, 169 ) ),
                                new Point( AppManager._KEY_LENGTH, size.Height - 42 ),
                                new Point( size.Width - 3, size.Height - 42 ) );

                    // カーブエディタの上の線
                    g.DrawLine( new Pen( Color.FromArgb( 46, 47, 50 ) ),
                                new Point( AppManager._KEY_LENGTH, 8 ),
                                new Point( size.Width - 3, 8 ) );

                    g.DrawLine( new Pen( Color.FromArgb( 125, 123, 124 ) ),
                                new Point( AppManager._KEY_LENGTH, 0 ),
                                new Point( AppManager._KEY_LENGTH, size.Height - 35 ) );
                    if ( m_manager.VsqFile != null && m_manager.Selected >= 0 ) {
                        Color front = Color.FromArgb( 150, AppManager.HilightColor );
                        Color back = Color.FromArgb( 44, 255, 249, 255 );
                        Color vel_color = Color.FromArgb( 64, 78, 30 );

                        // 後ろに描くカーブ
                        if ( m_last_selected_curve == CurveType.VEL || m_last_selected_curve == CurveType.Accent || m_last_selected_curve == CurveType.Decay ) {
                            DrawVEL( g, m_manager.VsqFile.Track[m_manager.Selected], back, false, m_last_selected_curve );
                        } else if ( m_last_selected_curve == CurveType.VibratoRate || m_last_selected_curve == CurveType.VibratoDepth ) {
                            DrawVibratoControlCurve( g, m_last_selected_curve, back, false );
                        } else {
                            VsqBPList list_back = null;
                            if ( m_last_selected_curve == CurveType.Pitch ) {
                                list_back = m_manager.VsqFile.getPitchCurve( m_manager.Selected );
                            } else {
                                list_back = m_manager.VsqFile.Track[m_manager.Selected].getCurve( m_last_selected_curve.Name );
                            }
                            if ( list_back != null ) {
                                DrawVsqBPList( g, list_back, back, false, (m_last_selected_curve == CurveType.Pitch) );
                            }
                        }

                        // 手前に描くカーブ
                        if ( m_selected_curve == CurveType.VEL || m_selected_curve == CurveType.Accent || m_selected_curve == CurveType.Decay ) {
                            DrawVEL( g, m_manager.VsqFile.Track[m_manager.Selected], vel_color, true, m_selected_curve );
                        } else if ( m_selected_curve == CurveType.VibratoRate || m_selected_curve == CurveType.VibratoDepth ) {
                            DrawVibratoControlCurve( g, m_selected_curve, front, true );
                        } else {
                            VsqBPList list_front;
                            if ( m_selected_curve == CurveType.Pitch ) {
                                list_front = m_manager.VsqFile.getPitchCurve( m_manager.Selected );
                            } else {
                                list_front = m_manager.VsqFile.Track[m_manager.Selected].getCurve( m_selected_curve.Name );
                            }

                            if ( list_front != null ) {
                                DrawVsqBPList( g, list_front, front, true, (m_selected_curve == CurveType.Pitch) );
                            }
                            if ( m_selected_curve == CurveType.Pitch ) {
                                int max = m_internal_pbs * 10000;
                                int min = -m_internal_pbs * 10000;
                                for ( int i = min; i <= max; i += 10000 ) {
                                    int y = YCoordFromValue( i );
                                    g.DrawLine( new Pen( Color.FromArgb( 128, Color.Black ) ), new Point( 0, y ), new Point( this.Width, y ) );
                                }
                            }
                            DrawAttachedCurve( g, m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1][m_selected_curve] );
                        }
                    }

                    #region 小節ごとのライン
                    if ( m_manager.VsqFile != null ) {
                        int dashed_line_step = AppManager.GetPositionQuantizeClock();
                        g.SetClip( new Rectangle( AppManager._KEY_LENGTH, _HEADER, size.Width - AppManager._KEY_LENGTH, size.Height - 2 * _OFFSET_TRACK_TAB ) );
                        using ( Pen white100 = new Pen( Color.FromArgb( 100, Color.Black ) ) ) {
                            for( Iterator itr = m_manager.VsqFile.getBarLineIterator( ClockFromXCoord( Width ) ); itr.hasNext(); ) {
                                 VsqBarLineType blt = (VsqBarLineType)itr.next();
                                int x = XCoordFromClocks( blt.clock() );
                                int local_clock_step = 480 * 4 / blt.getLocalDenominator();
                                if ( blt.isSeparator() ) {
                                    g.DrawLine( white100,
                                                new Point( x, size.Height - 42 - 1 ),
                                                new Point( x, 8 + 1 ) );
                                } else {
                                    g.DrawLine( white100,
                                                new Point( x, centre - 5 ),
                                                new Point( x, centre + 6 ) );
                                    using ( Pen pen = new Pen( Color.FromArgb( 12, 12, 12 ) ) ) {
                                        g.DrawLine( pen,
                                                    new Point( x, 8 ),
                                                    new Point( x, 14 ) );
                                        g.DrawLine( pen,
                                                    new Point( x, size.Height - 43 ),
                                                    new Point( x, size.Height - 42 - 6 ) );
                                    }
                                }
                                if ( dashed_line_step > 1 && m_manager.GridVisible ) {
                                    int numDashedLine = local_clock_step / dashed_line_step;
                                    using ( Pen pen = new Pen( Color.FromArgb( 65, 65, 65 ) ) ) {
                                        for ( int i = 1; i < numDashedLine; i++ ) {
                                            int x2 = XCoordFromClocks( blt.clock() + i * dashed_line_step );
                                            g.DrawLine( white100,
                                                        new Point( x2, centre - 2 ),
                                                        new Point( x2, centre + 3 ) );
                                            g.DrawLine( white100,
                                                        new Point( x2, 8 ),
                                                        new Point( x2, 12 ) );
                                            g.DrawLine( white100,
                                                        new Point( x2, size.Height - 43 ),
                                                        new Point( x2, size.Height - 43 - 4 ) );
                                        }
                                    }
                                }
                            }
                        }
                        g.ResetClip();
                    }
                    #endregion

                    if ( m_manager.SelectedRegionEnabled ) {
                        int stdx = m_manager.StartToDrawX;
                        int start = m_manager.SelectedRegion.Start - stdx;
                        int end = m_manager.SelectedRegion.End - stdx;
                        g.FillRectangle( s_brs_a098_000_000_000,
                                         new Rectangle( start, _HEADER, end - start, GraphHeight ) );
                    }

                    if ( m_mouse_downed ) {
                        #region 選択されたツールに応じて描画
                        Point mouse = this.PointToClient( Control.MousePosition );
                        int clock = ClockFromXCoord( mouse.X );
                        int value = ValueFromYCoord( mouse.Y );
                        if ( clock < m_manager.VsqFile.getPreMeasure() ) {
                            clock = m_manager.VsqFile.getPreMeasure();
                        }
                        int max = m_selected_curve.Maximum;
                        int min = m_selected_curve.Minimum;
                        if ( value < min ) {
                            value = min;
                        } else if ( max < value ) {
                            value = max;
                        }
                        switch ( m_manager.SelectedTool ) {
                            case EditTool.Line:
                                int xini = XCoordFromClocks( m_line_start.X );
                                int yini = YCoordFromValue( m_line_start.Y );
                                g.DrawLine( s_pen_050_140_150,
                                            new Point( xini, yini ),
                                            new Point( XCoordFromClocks( clock ), YCoordFromValue( value ) ) );
                                break;
                            case EditTool.Pencil:
                                if ( m_mouse_trace != null && !m_manager.IsCurveMode ) {
                                    List<Point> pt = new List<Point>();
                                    int stdx = m_manager.StartToDrawX;
                                    int lastx = m_mouse_trace.Keys[0] - stdx;
                                    int lasty = m_mouse_trace[m_mouse_trace.Keys[0]];
                                    int height = this.Height - 42;
                                    if ( lasty < 8 ) {
                                        lasty = 8;
                                    } else if ( height < lasty ) {
                                        lasty = height;
                                    }
                                    pt.Add( new Point( lastx, height ) );
                                    pt.Add( new Point( lastx, lasty ) );
                                    for ( int i = 1; i < m_mouse_trace.Keys.Count; i++ ) {
                                        int key = m_mouse_trace.Keys[i];
                                        int new_x = key - stdx;
                                        int new_y = m_mouse_trace[key];
                                        if ( new_y < 8 ) {
                                            new_y = 8;
                                        } else if ( height < new_y ) {
                                            new_y = height;
                                        }
                                        pt.Add( new Point( new_x, lasty ) );
                                        pt.Add( new Point( new_x, new_y ) );
                                        lastx = new_x;
                                        lasty = new_y;
                                    }
                                    pt.Add( new Point( lastx, height ) );
                                    g.FillPolygon( new SolidBrush( Color.FromArgb( 127, 8, 166, 172 ) ),
                                                   pt.ToArray() );
                                }
                                break;
                            case EditTool.Eraser:
                            case EditTool.Arrow:
                                if ( m_mouse_down_mode == MouseDownMode.CurveEdit && m_mouse_moved && m_selecting_region.X != m_selecting_region.Y ) {
                                    xini = XCoordFromClocks( m_selecting_region.X );
                                    int xend = XCoordFromClocks( m_selecting_region.Y );
                                    int start = Math.Min( xini, xend );
                                    if ( start < AppManager._KEY_LENGTH ) {
                                        start = AppManager._KEY_LENGTH;
                                    }
                                    int end = Math.Max( xini, xend );
                                    if ( start < end ) {
                                        g.FillRectangle( s_brs_a144_255_255_255,
                                                         new Rectangle( start, 8, end - start, this.Height - 42 - 8 ) );
                                    }
                                } else if ( m_mouse_down_mode == MouseDownMode.VelEdit && m_veledit_selected.ContainsKey( m_veledit_last_selectedid ) ) {
                                    if ( m_selected_curve == CurveType.VEL ) {
                                        numeric_view = m_veledit_selected[m_veledit_last_selectedid].Editing.ID.Dynamics;
                                    } else if ( m_selected_curve == CurveType.Accent ) {
                                        numeric_view = m_veledit_selected[m_veledit_last_selectedid].Editing.ID.DEMaccent;
                                    } else if ( m_selected_curve == CurveType.Decay ) {
                                        numeric_view = m_veledit_selected[m_veledit_last_selectedid].Editing.ID.DEMdecGainRate;
                                    }
                                }
                                break;
                        }
                        if ( m_mouse_down_mode == MouseDownMode.SingerList && m_manager.SelectedTool != EditTool.Eraser ) {
                            foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ) {
                                int x = XCoordFromClocks( item.Editing.Clock );
                                g.DrawLines( s_pen_128_128_128,
                                             new Point[] { new Point( x, size.Height - _OFFSET_TRACK_TAB ),
                                                           new Point( x, size.Height - 2 * _OFFSET_TRACK_TAB + 1 ),
                                                           new Point( x + _SINGER_ITEM_WIDTH, size.Height - 2 * _OFFSET_TRACK_TAB + 1 ),
                                                           new Point( x + _SINGER_ITEM_WIDTH, size.Height - _OFFSET_TRACK_TAB ) } );
                                g.DrawLine( s_pen_246_251_010,
                                            new Point( x, size.Height - _OFFSET_TRACK_TAB ),
                                            new Point( x + _SINGER_ITEM_WIDTH, size.Height - _OFFSET_TRACK_TAB ) );
                            }
                        }
                        #endregion
                    }
                    #endregion
                }

                if ( CurveVisible ) {
                    #region カーブの種類一覧
                    Color font_color_normal = Color.Black;
                    g.FillRectangle( new SolidBrush( Color.FromArgb( 212, 212, 212 ) ),
                                     new Rectangle( 0, 0, AppManager._KEY_LENGTH, size.Height - 2 * _OFFSET_TRACK_TAB ) );

                    // 数値ビュー
                    Rectangle num_view = new Rectangle( 13, 4, 38, 16 );
                    g.DrawRectangle( new Pen( Color.FromArgb( 125, 123, 124 ) ),
                                     num_view );
                    StringFormat sf = new StringFormat();
                    sf.Alignment = StringAlignment.Far;
                    sf.LineAlignment = StringAlignment.Far;
                    g.DrawString( numeric_view.ToString(),
                                  AppManager.BaseFont9,
                                  brs_string,
                                  num_view,
                                  sf );

                    // 現在表示されているカーブの名前
                    sf.Alignment = StringAlignment.Center;
                    sf.LineAlignment = StringAlignment.Near;
                    g.DrawString( m_selected_curve.Name, AppManager.BaseFont9, brs_string, new Rectangle( 7, 24, 56, 14 ), sf );

                    foreach ( CurveType curve in m_viewing_curves ) {
                        Rectangle rc = RectFromCurveType( curve );
                        if ( curve == m_selected_curve || curve == m_last_selected_curve ) {
                            g.FillRectangle( new SolidBrush( Color.FromArgb( 108, 108, 108 ) ), rc );
                        }
                        g.DrawRectangle( rect_curve, rc );
                        Rectangle rc_str = new Rectangle( rc.X, rc.Y + rc.Height / 2 - AppManager.BaseFont9OffsetHeight, rc.Width, rc.Height );
                        rc_str.Y += 2;
                        if ( curve == m_selected_curve ) {
                            g.DrawString( curve.Name,
                                          AppManager.BaseFont9,
                                          Brushes.White,
                                          rc_str,
                                          sf );
                        } else {
                            g.DrawString( curve.Name,
                                          AppManager.BaseFont9,
                                          new SolidBrush( font_color_normal ),
                                          rc_str,
                                          sf );
                        }
                    }
                    #endregion
                }

                #region 現在のマーカー
                int marker_x = XCoordFromClocks( m_manager.CurrentClock );
                if ( AppManager._KEY_LENGTH <= marker_x && marker_x <= size.Width ) {
                    g.DrawLine( new Pen( Color.White, 2f ),
                                new Point( marker_x, 0 ),
                                new Point( marker_x, size.Height - 18 ) );
                }
                #endregion

                #region 外枠
                // 左外側
                g.DrawLine( new Pen( Color.FromArgb( 160, 160, 160 ) ),
                            new Point( 0, 0 ),
                            new Point( 0, size.Height - 2 ) );
                // 左内側
                g.DrawLine( new Pen( Color.FromArgb( 105, 105, 105 ) ),
                            new Point( 1, 0 ),
                            new Point( 1, size.Height - 1 ) );
                // 下内側
                g.DrawLine( new Pen( Color.FromArgb( 192, 192, 192 ) ),
                            new Point( 1, size.Height - 2 ),
                            new Point( size.Width + 20, size.Height - 2 ) );
                // 下外側
                g.DrawLine(
                    Pens.White,
                    new Point( 0, size.Height - 1 ),
                    new Point( size.Width + 20, size.Height - 1 ) );
                /*/ 右外側
                g.DrawLine( Pens.White,
                            new Point( size.Width - 1, 0 ),
                            new Point( size.Width - 1, size.Height - 1 ) );
                // 右内側
                g.DrawLine( new Pen( Color.FromArgb( 227, 227, 227 ) ),
                            new Point( size.Width - 2, 0 ),
                            new Point( size.Width - 2, size.Height - 2 ) );*/
                #endregion
            }
#if DEBUG
            } catch( Exception ex ){
                Common.DebugWriteLine( "    ex=" + ex );
            }
#endif
        }

        private void DrawTrackTab( Graphics g, Rectangle destRect, string name, bool selected, bool enabled, bool render_required, Color hilight, Color render_button_hilight ) {
            int x = destRect.X;
            int panel_width = render_required ? destRect.Width - 10 : destRect.Width;
            Color panel_color = enabled ? hilight : Color.FromArgb( 125, 123, 124 );
            Color button_color = enabled ? render_button_hilight : Color.FromArgb( 125, 123, 124 );
            Color panel_title = Color.Black;
            Color button_title = selected ? Color.White : Color.Black;
            Color border = selected ? Color.White : Color.FromArgb( 118, 123, 138 );

            // 背景(選択されている場合)
            if ( selected ) {
                g.FillRectangle( new SolidBrush( panel_color ),
                                 destRect );
                if ( render_required && enabled ) {
                    g.FillRectangle( new SolidBrush( render_button_hilight ),
                                     new Rectangle( destRect.X + destRect.Width - 10, destRect.Y, 10, destRect.Height ) );
                }
            }

            // 左縦線
            g.DrawLine( new Pen( border ),
                        new Point( destRect.X, destRect.Y ),
                        new Point( destRect.X, destRect.Y + destRect.Height ) );
            if ( name.Length > 0 ) {
                // 上横線
                g.DrawLine( new Pen( border ),
                            new Point( destRect.X + 1, destRect.Y ),
                            new Point( destRect.X + destRect.Width, destRect.Y ) );
            }
            if ( render_required ) {
                g.DrawLine(
                    new Pen( border ),
                    new Point( destRect.X + destRect.Width - 10, destRect.Y ),
                    new Point( destRect.X + destRect.Width - 10, destRect.Y + destRect.Height ) );
            }
            g.SetClip( destRect );
            string title = Common.TrimString( name, AppManager.BaseFont8, panel_width );
            g.DrawString( title,
                          AppManager.BaseFont8,
                          new SolidBrush( panel_title ),
                          new PointF( destRect.X + 2, destRect.Y + destRect.Height / 2 - AppManager.BaseFont8OffsetHeight ) );
            if ( render_required ) {
                g.DrawString( "R",
                              AppManager.BaseFont8,
                              new SolidBrush( button_title ),
                              new PointF( destRect.X + destRect.Width - _PX_WIDTH_RENDER, destRect.Y + destRect.Height / 2 - AppManager.BaseFont8OffsetHeight ) );
            }
            if ( selected ) {
                g.DrawLine( new Pen( border ),
                            new Point( destRect.X + destRect.Width - 1, destRect.Y ),
                            new Point( destRect.X + destRect.Width - 1, destRect.Y + destRect.Height ) );
                g.DrawLine( new Pen( border ),
                            new Point( destRect.X, destRect.Y + destRect.Height - 1 ),
                            new Point( destRect.X + destRect.Width, destRect.Y + destRect.Height - 1 ) );
            }
            g.ResetClip();
            g.DrawLine( m_generic_line,
                        new Point( destRect.X + destRect.Width, destRect.Y ),
                        new Point( destRect.X + destRect.Width, destRect.Y + destRect.Height ) );
        }

        /// <summary>
        /// トラック選択部分の、トラック1個分の幅を調べます。pixel
        /// </summary>
        public int SelectorWidth {
            get {
                return (int)((this.Width - vScroll.Width) / 16.0f);
            }
        }

        /// <summary>
        /// ベロシティを、与えられたグラフィックgを用いて描画します
        /// </summary>
        /// <param name="g"></param>
        /// <param name="track"></param>
        /// <param name="color"></param>
        public void DrawVEL( Graphics g, VsqTrack track, Color color, bool is_front, CurveType type ) {
            int HEADER = 8;
            int height = GraphHeight;
            float order = (type == CurveType.VEL) ? height / 127f : height / 100f;
            int oy = this.Height - 42;
            Region last_clip = g.Clip;
            int xoffset = 6 + AppManager._KEY_LENGTH - m_manager.StartToDrawX;
            g.SetClip( new Rectangle( AppManager._KEY_LENGTH, HEADER, this.Width - AppManager._KEY_LENGTH - vScroll.Width, height ) );
            float scale = m_manager.ScaleX;
            for ( int i = 0; i < track.getEventCount(); i++ ) {
                VsqEvent ve = track.getEvent( i );
                if ( ve.ID.type != VsqIDType.Anote ) {
                    continue;
                }
                int clock = ve.Clock;
                int x = (int)(clock * scale) + xoffset;
                if ( x + _VEL_BAR_WIDTH < 0 ) {
                    continue;
                } else if ( this.Width - vScroll.Width < x ) {
                    break;
                } else {
                    int value = 0;
                    if ( type == CurveType.VEL ) {
                        value = ve.ID.Dynamics;
                    } else if ( type == CurveType.Accent ) {
                        value = ve.ID.DEMaccent;
                    } else if ( type == CurveType.Decay ) {
                        value = ve.ID.DEMdecGainRate;
                    }
                    int y = oy - (int)(value * order);
                    if ( is_front && m_manager.SelectedEvent.ContainsKey( m_manager.Selected, ve.InternalID ) ) {
                        g.FillRectangle( s_brs_a127_008_166_172,
                                         new Rectangle( x, y, _VEL_BAR_WIDTH, oy - y ) );
                        if ( m_mouse_down_mode == MouseDownMode.VelEdit ) {
                            int editing = 0;
                            if ( m_veledit_selected.ContainsKey( ve.InternalID ) ) {
                                if ( m_selected_curve == CurveType.VEL ) {
                                    editing = m_veledit_selected[ve.InternalID].Editing.ID.Dynamics;
                                } else if ( m_selected_curve == CurveType.Accent ) {
                                    editing = m_veledit_selected[ve.InternalID].Editing.ID.DEMaccent;
                                } else if ( m_selected_curve == CurveType.Decay ) {
                                    editing = m_veledit_selected[ve.InternalID].Editing.ID.DEMdecGainRate;
                                }
                                int edit_y = oy - (int)(editing * order);
                                g.FillRectangle( s_brs_a244_255_023_012,
                                                 new Rectangle( x, edit_y, _VEL_BAR_WIDTH, oy - edit_y ) );
                            }
                        }
                    } else {
                        g.FillRectangle( new SolidBrush( color ),
                                         new Rectangle( x, y, _VEL_BAR_WIDTH, oy - y ) );
                    }
                }
            }
            g.SetClip( last_clip, CombineMode.Replace );
        }

        private void DrawAttachedCurve( Graphics g, List<BezierChain> chains ) {
#if DEBUG
            try {
                BezierCurves t;
#endif
                //foreach ( int chain_id in chains.Keys ) {
                for( int i = 0; i < chains.Count; i++ ){
                    int chain_id = chains[i].ID;
                    if ( chains[i].points.Count <= 0 ) {
                        continue;
                    }
                    BezierPoint next;
                    BezierPoint current = chains[i].points[0];
                    Point pxNext;
                    Point pxCurrent = GetScreenCoord( current.Base );
                    for ( int j = 0; j < chains[i].points.Count; j++ ) {
                        next = chains[i].points[j];
                        int next_x = XCoordFromClocks( (int)next.Base.X );
                        pxNext = new Point( next_x, YCoordFromValue( (int)next.Base.Y ) );
                        Point pxControlCurrent = GetScreenCoord( current.ControlRight );
                        Point pxControlNext = GetScreenCoord( next.ControlLeft );

                        // ベジエ曲線本体を描く
                        g.SmoothingMode = SmoothingMode.AntiAlias;
                        if ( current.ControlRightType == BezierControlType.None &&
                             next.ControlLeftType == BezierControlType.None ) {
                            g.DrawLine( s_pen_curve, pxCurrent, pxNext );
                        } else {
                            g.DrawBezier( s_pen_curve,
                                          pxCurrent,
                                          (current.ControlRightType == BezierControlType.None) ? pxCurrent : pxControlCurrent,
                                          (next.ControlLeftType == BezierControlType.None) ? pxNext : pxControlNext,
                                          pxNext );
                        }

                        if ( current.ControlRightType != BezierControlType.None ) {
                            g.DrawLine( s_pen_control_line, pxCurrent, pxControlCurrent );
                        }
                        if ( next.ControlLeftType != BezierControlType.None ) {
                            g.DrawLine( s_pen_control_line, pxNext, pxControlNext );
                        }
                        g.SmoothingMode = SmoothingMode.Default;

                        // コントロール点
                        if ( current.ControlRightType == BezierControlType.Normal ) {
                            Rectangle rc = new Rectangle( pxControlCurrent.X - _DOT_WID,
                                                          pxControlCurrent.Y - _DOT_WID,
                                                          _DOT_WID * 2 + 1,
                                                          _DOT_WID * 2 + 1 );
                            if ( chain_id == EditingChainID && current.ID == EditingPointID ) {
                                g.FillEllipse( AppManager.HilightBrush, rc );
                            } else {
                                g.FillEllipse( s_brs_dot_normal, rc );
                            }
                            g.DrawEllipse( s_pen_dot_normal, rc );
                        }

                        // コントロール点
                        if ( next.ControlLeftType == BezierControlType.Normal ) {
                            Rectangle rc = new Rectangle( pxControlNext.X - _DOT_WID,
                                                          pxControlNext.Y - _DOT_WID,
                                                          _DOT_WID * 2 + 1,
                                                          _DOT_WID * 2 + 1 );
                            if ( chain_id == EditingChainID && next.ID == EditingPointID ) {
                                g.FillEllipse( AppManager.HilightBrush, rc );
                            } else {
                                g.FillEllipse( s_brs_dot_normal, rc );
                            }
                            g.DrawEllipse( s_pen_dot_normal, rc );
                        }

                        // データ点
                        Rectangle rc2 = new Rectangle( pxCurrent.X - _DOT_WID,
                                                        pxCurrent.Y - _DOT_WID,
                                                        _DOT_WID * 2 + 1,
                                                        _DOT_WID * 2 + 1 );
                        if ( chain_id == EditingChainID && current.ID == EditingPointID ) {
                            g.FillRectangle( AppManager.HilightBrush, rc2 );
                        } else {
                            g.FillRectangle( s_brs_dot_base, rc2 );
                        }
                        g.DrawRectangle( s_pen_dot_base, rc2 );
                        pxCurrent = pxNext;
                        current = next;
                    }
                    next = chains[i].points[chains[i].points.Count - 1];
                    pxNext = GetScreenCoord( next.Base );
                    Rectangle rc_last = new Rectangle( pxNext.X - _DOT_WID,
                                                       pxNext.Y - _DOT_WID,
                                                       _DOT_WID * 2 + 1,
                                                       _DOT_WID * 2 + 1 );
                    if ( chain_id == EditingChainID && next.ID == EditingPointID ) {
                        g.FillRectangle( AppManager.HilightBrush, rc_last );
                    } else {
                        g.FillRectangle( s_brs_dot_base, rc_last );
                    }
                    g.DrawRectangle( s_pen_dot_base, rc_last );
                }
#if DEBUG
            } catch ( Exception ex ) {
                Common.DebugWriteLine( "TrackSelector+DrawAttatchedCurve" );
                Common.DebugWriteLine( "    ex=" + ex );
            }
#endif
        }

        private Point GetScreenCoord( PointD pt ) {
            return new Point( XCoordFromClocks( (int)pt.X ), YCoordFromValue( (int)pt.Y ) );
        }

        // TODO: TrackSelector+DrawVibratoControlCurve; かきかけ
        public void DrawVibratoControlCurve( Graphics g, CurveType type, Color color, bool is_front ) {
#if DEBUG
            //Common.DebugWriteLine( "TrackSelector+DrawVibratoControlCurve" );
#endif
            Region last_clip = g.Clip;
            int graph_height = GraphHeight;
            g.SetClip( new Rectangle( AppManager._KEY_LENGTH,
                                      _HEADER,
                                      this.Width - AppManager._KEY_LENGTH - vScroll.Width,
                                      graph_height ),
                       CombineMode.Replace );

            int track = m_manager.Selected;
            int cl_start = ClockFromXCoord( AppManager._KEY_LENGTH );
            int cl_end = ClockFromXCoord( this.Width - vScroll.Width );
            if ( is_front ) {
                // draw shadow of non-note area
                Region last_clip2 = g.Clip;
                for ( Iterator itr = m_manager.VsqFile.Track[track].getNoteEventIterator(); itr.hasNext(); ) {
                    VsqEvent ve = (VsqEvent)itr.next();
                    int start = ve.Clock + ve.ID.VibratoDelay;
                    int end = ve.Clock + ve.ID.Length;
                    if ( end < cl_start ) {
                        continue;
                    }
                    if ( cl_end < start ) {
                        break;
                    }
                    if ( ve.ID.VibratoHandle == null ) {
                        continue;
                    }
                    int x1 = XCoordFromClocks( start );
                    int x2 = XCoordFromClocks( end );
                    if ( x1 < AppManager._KEY_LENGTH ) {
                        x1 = AppManager._KEY_LENGTH;
                    }
                    if ( this.Width - vScroll.Width < x2 ) {
                        x2 = this.Width;
                    }
                    if ( x1 < x2 ) {
                        g.SetClip( new Rectangle( x1, _HEADER, x2 - x1, graph_height ), CombineMode.Exclude );
                    }
                }
                g.FillRectangle( new SolidBrush( Color.FromArgb( 127, Color.Black ) ),
                                 new Rectangle( AppManager._KEY_LENGTH, _HEADER, this.Width - AppManager._KEY_LENGTH - vScroll.Width, graph_height ) );

                g.SetClip( last_clip, CombineMode.Replace );

                // draw curve
                for ( Iterator itr =  m_manager.VsqFile.Track[track].getNoteEventIterator(); itr.hasNext(); ) {
                    VsqEvent ve = (VsqEvent)itr.next();
                    int start = ve.Clock + ve.ID.VibratoDelay;
                    int end = ve.Clock + ve.ID.Length;
                    if ( end < cl_start ) {
                        continue;
                    }
                    if ( cl_end < start ) {
                        break;
                    }
                    if ( ve.ID.VibratoHandle == null ) {
                        continue;
                    }
                    int x1 = XCoordFromClocks( start );
                    int x2 = XCoordFromClocks( end );
                    if ( x1 < x2 ) {
                        g.SetClip( new Rectangle( x1, _HEADER, x2 - x1, graph_height ), CombineMode.Replace );
                        int points = 0;
                        List<Point> poly = new List<Point>();
                        int draw_width = x2 - x1;
                        poly.Add( new Point( x1, YCoordFromValue( 0, type.Maximum, type.Minimum ) ) );
                        if ( type == CurveType.VibratoRate ) {
                            int last_y = YCoordFromValue( ve.ID.VibratoHandle.StartRate, type.Maximum, type.Minimum );
                            poly.Add( new Point( x1, last_y ) );
                            for ( int i = 0; i < ve.ID.VibratoHandle.RateBP.getCount(); i++ ) {
                                int x = x1 + (int)(ve.ID.VibratoHandle.RateBP.getElement( i ).X * draw_width);
                                int y = YCoordFromValue( ve.ID.VibratoHandle.RateBP.getElement( i ).Y, type.Maximum, type.Minimum );
                                poly.Add( new Point( x, last_y ) );
                                poly.Add( new Point( x, y ) );
                                last_y = y;
                            }
                            poly.Add( new Point( x2, last_y ) );
                        } else {
                            int last_y = YCoordFromValue( ve.ID.VibratoHandle.StartDepth, type.Maximum, type.Minimum );
                            poly.Add( new Point( x1, last_y ) );
                            for ( int i = 0; i < ve.ID.VibratoHandle.DepthBP.getCount(); i++ ) {
                                int x = x1 + (int)(ve.ID.VibratoHandle.DepthBP.getElement( i ).X * draw_width);
                                int y = YCoordFromValue( ve.ID.VibratoHandle.DepthBP.getElement( i ).Y, type.Maximum, type.Minimum );
                                poly.Add( new Point( x, last_y ) );
                                poly.Add( new Point( x, y ) );
                                last_y = y;
                            }
                            poly.Add( new Point( x2, last_y ) );
                        }
                        poly.Add( new Point( x2, YCoordFromValue( 0, type.Maximum, type.Minimum ) ) );
                        g.FillPolygon( new SolidBrush( color ), poly.ToArray() );
                    }
                }
            }
            g.SetClip( last_clip, CombineMode.Replace );
        }

        /// <summary>
        /// Draws VsqBPList using specified Graphics "g", toward rectangular region "rect".
        /// </summary>
        /// <param name="g"></param>
        /// <param name="rect"></param>
        /// <param name="list"></param>
        /// <param name="color"></param>
        public void DrawVsqBPList( Graphics g, VsqBPList list, Color color, bool is_front, bool list_represents_pitch ) {
            int max = list.getMaximum();
            int min = list.getMinimum();
            if ( list_represents_pitch ) {
                max = m_internal_pbs * 10000;
                min = -m_internal_pbs * 10000;
            }
            int height = GraphHeight;
            float order = height / (float)(max - min);
            int oy = this.Height - 42;
            Region last_clip = g.Clip.Clone();
            g.SetClip( new Rectangle( AppManager._KEY_LENGTH, _HEADER, this.Width - AppManager._KEY_LENGTH - vScroll.Width, this.Height - 2 * _OFFSET_TRACK_TAB ), CombineMode.Replace );
            List<Point> hilight = new List<Point>();

            int hilight_start = Math.Min( m_selected_region.X, m_selected_region.Y );
            int hilight_end = Math.Max( m_selected_region.X, m_selected_region.Y );
            int hilight_start_x = XCoordFromClocks( hilight_start );
            int hilight_end_x = XCoordFromClocks( hilight_end );
            int start = AppManager._KEY_LENGTH;
            int start_clock = ClockFromXCoord( start );
            int end = this.Width - vScroll.Width;
            int end_clock = ClockFromXCoord( end );
            Rectangle clip_hilight = Rectangle.Empty;
            bool hilight_enabled = false;
            if ( is_front && m_selected_region_enabled ) {
                if ( start < hilight_end_x && hilight_start_x < end ) {
                    // ハイライト表示する部分をクリップからはずす
                    hilight_enabled = true;
                    clip_hilight = new Rectangle( hilight_start_x,
                                                  _HEADER,
                                                  hilight_end_x - hilight_start_x,
                                                  this.Height - 2 * _OFFSET_TRACK_TAB );
                    g.SetClip( clip_hilight, CombineMode.Exclude );
                }
            }

            SolidBrush brush = new SolidBrush( color );
            List<Point> points = new List<Point>();
            points.Add( new Point( this.Width - vScroll.Width, oy ) );
            points.Add( new Point( AppManager._KEY_LENGTH, oy ) );
            int first_y = list.getValue( start_clock );
            int last_y = oy - (int)((first_y - min) * order);

            bool first = true;
            if ( list.getCount() > 0 ) {
                int first_clock = list.getKeyClock( 0 );
                int last_x = XCoordFromClocks( first_clock );
                first_y = list.getValue( first_clock );
                last_y = oy - (int)((first_y - min) * order);
                bool hilight_first = true;
                bool hilight_entered = false;
                for ( int i = 0; i < list.getCount(); i++ ) {
                    int clock = list.getKeyClock( i );
                    if ( clock < start_clock ) {
                        continue;
                    }
                    if ( end_clock < clock ) {
                        break;
                    }
                    if ( first ) {
                        last_y = YCoordFromValue( list.getValue( ClockFromXCoord( AppManager._KEY_LENGTH ) ),
                                                  max,
                                                  min );
                        points.Add( new Point( AppManager._KEY_LENGTH, last_y ) );
                        first = false;
                    }
                    int x = XCoordFromClocks( clock );
                    int y = oy - (int)((list.getElement( i ) - min) * order);
                    if ( hilight_enabled ) {
                        if ( hilight_start_x <= x && x <= hilight_end_x ) {
                            if ( hilight_first ) {
                                hilight.Add( new Point( hilight_start_x, oy ) );
                                hilight.Add( new Point( hilight_start_x, last_y ) );
                                hilight_first = false;
                                hilight_entered = true;
                            }
                            hilight.Add( new Point( x, last_y ) );
                            hilight.Add( new Point( x, y ) );
                        } else if ( hilight_entered && hilight_end_x < x ) {
                            hilight.Add( new Point( hilight_end_x, last_y ) );
                            hilight.Add( new Point( hilight_end_x, oy ) );
                        }
                    }
                    points.Add( new Point( x, last_y ) );
                    points.Add( new Point( x, y ) );
                    last_y = y;
                }
            }
            if ( first ) {
                last_y = YCoordFromValue( list.getValue( ClockFromXCoord( AppManager._KEY_LENGTH ) ),
                                          max,
                                          min );
                points.Add( new Point( AppManager._KEY_LENGTH, last_y ) );
            }
            last_y = oy - (int)((list.getValue( end_clock ) - min) * order);
            points.Add( new Point( this.Width - vScroll.Width, last_y ) );
            g.FillPolygon( brush, points.ToArray() );
            brush.Dispose();

            if ( hilight_enabled ) {
                int start_y = YCoordFromValue( list.getValue( hilight_start ) );
                int end_y = YCoordFromValue( list.getValue( hilight_end ) );
                hilight.Add( new Point( hilight_end_x, end_y ) );
                hilight.Add( new Point( hilight_end_x, oy ) );
                hilight.Add( new Point( hilight_start_x, oy ) );
                hilight.Add( new Point( hilight_start_x, start_y ) );
                g.SetClip( clip_hilight, CombineMode.Replace );
                g.FillPolygon(
                    s_brs_a072_255_255_255,
                    hilight.ToArray() );
            }

            g.SetClip( last_clip, CombineMode.Replace );
        }

        /// <summary>
        /// VsqBPListを与えられたグラフィックgを用いて、領域rectに描画します。
        /// </summary>
        /// <param name="g"></param>
        /// <param name="rect"></param>
        /// <param name="list"></param>
        /// <param name="color"></param>
        public void NEW_DrawVsqBPList( Graphics g, VsqBPList list, Color color, int start_x, int end_x, Point position ) {
            int max = list.getMaximum();
            int min = list.getMinimum();
            int height = GraphHeight;
            float order = height / (float)(max - min);
            int oy = this.Height - 42;
            Region last_clip = g.Clip;
            g.SetClip( new Rectangle( position.X + AppManager._KEY_LENGTH, position.Y + _HEADER, this.Width - AppManager._KEY_LENGTH - vScroll.Width, this.Height - 2 * _OFFSET_TRACK_TAB ) );
            SolidBrush brush = new SolidBrush( color );
            int count = -1;
            count++;
            m_points[count].X = position.X + AppManager._KEY_LENGTH;
            m_points[count].Y = position.Y + oy;
            int first_y = list.getValue( 0 );
            int last_y = oy - (int)((first_y - min) * order) + position.Y;
            count++;
            m_points[count].X = AppManager._KEY_LENGTH;
            m_points[count].Y = last_y;
            count--;

            int xoffset = -m_manager.StartToDrawX + 73 + position.X;

            if ( list.getCount() > 0 ) {
                int[] keys = list.getKeys();
                int first_clock = keys[0];
                first_y = list.getValue( first_clock );
                last_y = oy - (int)((first_y - min) * order) + position.Y;
                int last_x = XCoordFromClocks( first_clock ) + position.X;
                bool first = true;
                for ( int i = 1; i < keys.Length; i++ ) {
                    int clock = keys[i];
                    int x = (int)(clock * m_manager.ScaleX) + xoffset;
                    int y = oy - (int)((list.getValue( clock ) - min) * order) + position.Y;
                    if ( x < start_x ) {
                        last_y = y;
                        continue;
                    }
                    if ( first ) {
                        count++;
                        m_points[count].X = position.X + AppManager._KEY_LENGTH;
                        m_points[count].Y = last_y;
                        first = false;
                    }
                    if ( count == _BUF_LEN - 1 ) { //このif文の位置，BUF_LENが偶数であることを利用
                        m_points[_BUF_LEN - 1].Y = position.Y + oy;
                        g.FillPolygon( brush, m_points );
                        m_points[0].X = m_points[_BUF_LEN - 1].X;
                        m_points[0].Y = position.Y + oy;
                        m_points[1].X = m_points[_BUF_LEN - 1].X;
                        m_points[1].Y = m_points[_BUF_LEN - 1].Y;
                        count = 1;
                    }
                    count++;
                    m_points[count].X = x;
                    m_points[count].Y = last_y;
                    count++;
                    m_points[count].X = x;
                    m_points[count].Y = y;
                    if ( end_x < last_x ) {
                        break;
                    }
                    last_y = y;
                    last_x = x;
                }
            }
            count++;
            m_points[count].X = position.X + this.Width - vScroll.Width;
            m_points[count].Y = position.Y + last_y;
            count++;
            m_points[count].X = position.X + this.Width - vScroll.Width;
            m_points[count].Y = position.Y + oy;
            Point[] pbuf = new Point[count];
            Array.Copy( m_points, pbuf, count );
            g.FillPolygon( brush, pbuf );
            brush.Dispose();
            g.SetClip( last_clip, CombineMode.Replace );
        }

        /// <summary>
        /// カーブエディタのグラフ部分の高さを取得します(pixel)
        /// </summary>
        public int GraphHeight {
            get {
                return this.Height - 42 - 8;
            }
        }

        /// <summary>
        /// カーブエディタのグラフ部分の幅を取得します。(pixel)
        /// </summary>
        public int GraphWidth {
            get {
                return this.Width - AppManager._KEY_LENGTH - vScroll.Width;
            }
        }

        private void TrackSelector_Load( object sender, EventArgs e ) {
            this.SetStyle( ControlStyles.DoubleBuffer, true );
            this.SetStyle( ControlStyles.UserPaint, true );
            this.SetStyle( ControlStyles.AllPaintingInWmPaint, true );
        }

        private void TrackSelector_MouseClick( object sender, MouseEventArgs e ) {
            if ( CurveVisible ) {
                if ( e.Button == MouseButtons.Left ) {
                    // カーブの種類一覧上で発生したイベントかどうかを検査
                    foreach ( CurveType curve in m_viewing_curves ) {
                        Rectangle r = RectFromCurveType( curve );
                        if ( IsInRect( e.Location, r ) ) {
                            ChangeCurve( curve );
                            return;
                        }
                    }
                } else if ( e.Button == MouseButtons.Right ) {
                    if ( 0 <= e.X && e.X <= AppManager._KEY_LENGTH &&
                         0 <= e.Y && e.Y <= this.Height - 2 * _OFFSET_TRACK_TAB ) {
                        foreach ( ToolStripItem tsi in cmenuCurve.Items ) {
                            if ( tsi is ToolStripMenuItem ) {
                                ToolStripMenuItem tsmi = (ToolStripMenuItem)tsi;
                                tsmi.Checked = false;
                                foreach ( ToolStripItem tsi2 in tsmi.DropDownItems ) {
                                    if ( tsi2 is ToolStripMenuItem ) {
                                        ToolStripMenuItem tsmi2 = (ToolStripMenuItem)tsi2;
                                        tsmi2.Checked = false;
                                    }
                                }
                            }
                        }
                        string version = m_manager.VsqFile.Track[m_manager.Selected].getCommon().Version;
                        if ( version.StartsWith( VSTiProxy.RENDERER_DSB2 ) ) {
                            cmenuCurveBreathiness.Text = "Noise";
                            cmenuCurveEffect2Depth.Visible = true;
                            cmenuCurveOpening.Visible = false;
                            cmenuCurveReso1.Visible = true;
                            cmenuCurveReso2.Visible = true;
                            cmenuCurveReso3.Visible = true;
                            cmenuCurveReso4.Visible = true;
                            cmenuCurveSeparator1.Visible = true;
                            cmenuCurveHarmonics.Visible = true;
                        } else {
                            cmenuCurveBreathiness.Text = "Breathiness";
                            cmenuCurveEffect2Depth.Visible = false;
                            cmenuCurveOpening.Visible = false;
                            cmenuCurveReso1.Visible = false;
                            cmenuCurveReso2.Visible = false;
                            cmenuCurveReso3.Visible = false;
                            cmenuCurveReso4.Visible = false;
                            cmenuCurveSeparator1.Visible = false;
                            cmenuCurveHarmonics.Visible = false;
                        }
                        foreach ( ToolStripItem tsi in cmenuCurve.Items ) {
                            if ( tsi is ToolStripMenuItem ) {
                                ToolStripMenuItem tsmi = (ToolStripMenuItem)tsi;
                                if ( tsmi.Tag is CurveType ) {
                                    CurveType ct = (CurveType)tsmi.Tag;
                                    if ( ct == m_selected_curve ) {
                                        tsmi.Checked = true;
                                        break;
                                    }
                                }
                                foreach ( ToolStripItem tsi2 in tsmi.DropDownItems ) {
                                    if ( tsi2 is ToolStripMenuItem ) {
                                        ToolStripMenuItem tsmi2 = (ToolStripMenuItem)tsi2;
                                        if ( tsmi2.Tag is CurveType ) {
                                            CurveType ct2 = (CurveType)tsmi2.Tag;
                                            if ( ct2 == m_selected_curve ) {
                                                tsmi2.Checked = true;
                                                if ( ct2 == CurveType.reso1amp || ct2 == CurveType.reso1bw || ct2 == CurveType.reso1freq ||
                                                     ct2 == CurveType.reso2amp || ct2 == CurveType.reso2bw || ct2 == CurveType.reso2freq ||
                                                     ct2 == CurveType.reso3amp || ct2 == CurveType.reso3bw || ct2 == CurveType.reso3freq ||
                                                     ct2 == CurveType.reso4amp || ct2 == CurveType.reso4bw || ct2 == CurveType.reso4freq ){
                                                    tsmi.Checked = true;//親アイテムもチェック。Resonance*用
                                                }
                                                break;
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        cmenuCurve.Show( this, e.Location );
                    }
                }
            }
        }

        public void SelectNextCurve() {
            int index = 0;
            if ( m_viewing_curves.Count >= 2 ) {
                for ( int i = 0; i < m_viewing_curves.Count; i++ ) {
                    if ( m_viewing_curves[i] == m_selected_curve ) {
                        index = i;
                        break;
                    }
                }
                index++;
                if ( m_viewing_curves.Count <= index ) {
                    index = 0;
                }
                ChangeCurve( m_viewing_curves[index] );
            }
        }

        public void SelectPreviousCurve() {
            int index = 0;
            if ( m_viewing_curves.Count >= 2 ) {
                for ( int i = 0; i < m_viewing_curves.Count; i++ ) {
                    if ( m_viewing_curves[i] == m_selected_curve ) {
                        index = i;
                        break;
                    }
                }
                index--;
                if ( index < 0 ) {
                    index = m_viewing_curves.Count - 1;
                }
                ChangeCurve( m_viewing_curves[index] );
            }
        }

        private BezierPoint HandleMouseMoveForBezierMove( int clock, int value, int value_raw, BezierPickedSide picked ) {
            BezierChain target = m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].GetBezierChain( m_selected_curve, m_manager.LastSelectedBezier.ChainID );
            int point_id = m_manager.LastSelectedBezier.PointID;
            /*BezierChain target = null;
            List<BezierChain> list = m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1][m_selected_curve];
            for ( int i = 0; i < list.Count; i++ ) {
                if ( list[i].ID == m_manager.LastSelectedBezier.ChainID ) {
                    target = list[i];
                }
            }*/
            int index = -1;
            for ( int i = 0; i < target.points.Count; i++ ) {
                if ( target.points[i].ID == point_id ) {
                    index = i;
                    break;
                }
            }
            float scale_x = m_manager.ScaleX;
            float scale_y = ScaleY;
            BezierPoint ret = new BezierPoint( 0, 0 );
            if ( index >= 0 ) {
                if ( picked == BezierPickedSide.Base ) {
                    Point old = target.points[index].Base.ToPoint();
                    target.points[index].Base = new PointD( clock, value );
                    if ( !BezierChain.IsBezierImplicit( target ) ) {
                        target.points[index].Base = new PointD( old.X, old.Y );
                    }
                    ret = (BezierPoint)target.points[index].Clone();
                } else if ( picked == BezierPickedSide.Left ) {
                    if ( target.points[index].ControlLeftType != BezierControlType.Master ) {
                        Point old1 = target.points[index].ControlLeft.ToPoint();
                        Point old2 = target.points[index].ControlRight.ToPoint();
                        Point b = target.points[index].Base.ToPoint();
                        target.points[index].ControlLeft = new PointD( clock, value_raw );
                        double dx = (b.X - target.points[index].ControlLeft.X) * scale_x;
                        double dy = (b.Y - target.points[index].ControlLeft.Y) * scale_y;
                        BezierPoint original = m_manager.LastSelectedBezier.Original;
                        double theta = Math.Atan2( dy, dx );
                        double dx2 = (original.ControlRight.X - b.X) * scale_x;
                        double dy2 = (original.ControlRight.Y - b.Y) * scale_y;
                        double l = Math.Sqrt( dx2 * dx2 + dy2 * dy2 );
                        dx2 = l * Math.Cos( theta ) / scale_x;
                        dy2 = l * Math.Sin( theta ) / scale_y;
                        target.points[index].ControlRight = new PointD( b.X + (int)dx2, b.Y + (int)dy2 );
                        if ( !BezierChain.IsBezierImplicit( target ) ) {
                            target.points[index].ControlLeft = new PointD( old1.X, old1.Y );
                            target.points[index].ControlRight = new PointD( old2.X, old2.Y );
                        }
                    } else {
                        Point old = target.points[index].ControlLeft.ToPoint();
                        target.points[index].ControlLeft = new PointD( clock, value_raw );
                        if ( !BezierChain.IsBezierImplicit( target ) ) {
                            target.points[index].ControlLeft = new PointD( old.X, old.Y );
                        }
                    }
                    ret = (BezierPoint)target.points[index].Clone();
                } else if ( picked == BezierPickedSide.Right ) {
                    if ( target.points[index].ControlRightType != BezierControlType.Master ) {
                        Point old1 = target.points[index].ControlLeft.ToPoint();
                        Point old2 = target.points[index].ControlRight.ToPoint();
                        Point b = target.points[index].Base.ToPoint();
                        target.points[index].ControlRight = new PointD( clock, value_raw );
                        double dx = (target.points[index].ControlRight.X - b.X) * scale_x;
                        double dy = (target.points[index].ControlRight.Y - b.Y) * scale_y;
                        BezierPoint original = m_manager.LastSelectedBezier.Original;
                        double theta = Math.Atan2( dy, dx );
                        double dx2 = (b.X - original.ControlLeft.X) * scale_x;
                        double dy2 = (b.Y - original.ControlLeft.Y) * scale_y;
                        double l = Math.Sqrt( dx2 * dx2 + dy2 * dy2 );
                        dx2 = l * Math.Cos( theta ) / scale_x;
                        dy2 = l * Math.Sin( theta ) / scale_y;
                        target.points[index].ControlLeft = new PointD( b.X - (int)dx2, b.Y - (int)dy2 );
                        if ( !BezierChain.IsBezierImplicit( target ) ) {
                            target.points[index].ControlLeft = new PointD( old1.X, old1.Y );
                            target.points[index].ControlRight = new PointD( old2.X, old2.Y );
                        }
                    } else {
                        Point old = target.points[index].ControlRight.ToPoint();
                        target.points[index].ControlRight = new PointD( clock, value );
                        if ( !BezierChain.IsBezierImplicit( target ) ) {
                            target.points[index].ControlRight = new PointD( old.X, old.Y );
                        }
                    }
                    ret = (BezierPoint)target.points[index].Clone();
                }
            }
            return ret;
        }

        public BezierPoint HandleMouseMoveForBezierMove( MouseEventArgs e, BezierPickedSide picked ) {
            int clock = ClockFromXCoord( e.X );
            int value = ValueFromYCoord( e.Y );
            int value_raw = value;

            if ( clock < m_manager.VsqFile.getPreMeasure() ) {
                clock = m_manager.VsqFile.getPreMeasure();
            }
            int max = m_selected_curve.Maximum;
            int min = m_selected_curve.Minimum;
            if ( value < min ) {
                value = min;
            } else if ( max < value ) {
                value = max;
            }
            return HandleMouseMoveForBezierMove( clock, value, value_raw, picked );
        }

        private void TrackSelector_MouseMove( object sender, MouseEventArgs e ) {
#if DEBUG
            //Common.DebugWriteLine( "TrackSelectro_MouseMove" );
#endif
            if ( e.Button == MouseButtons.None ) {
                return;
            }
            if ( m_mouse_hover_thread != null ) {
                if ( m_mouse_hover_thread.IsAlive ) {
                    m_mouse_hover_thread.Abort();
                }
            }
            if ( m_manager.Playing ) {
                return;
            }
            m_mouse_moved = true;
            int clock = ClockFromXCoord( e.X );
            int value = ValueFromYCoord( e.Y );
            int value_raw = value;

            if ( clock < m_manager.VsqFile.getPreMeasure() ) {
                clock = m_manager.VsqFile.getPreMeasure();
            }
            int max = m_selected_curve.Maximum;
            int min = m_selected_curve.Minimum;
            if ( value < min ) {
                value = min;
            } else if ( max < value ) {
                value = max;
            }
            m_mouse_value = value;

            if ( e.Button == MouseButtons.Left &&
                 0 <= e.Y && e.Y <= Height - 2 * _OFFSET_TRACK_TAB &&
                 m_mouse_down_mode == MouseDownMode.CurveEdit ) {
                switch ( m_manager.SelectedTool ) {
                    case EditTool.Pencil:
                        m_pencil_moved = true;
                        if ( m_mouse_trace != null ) {
                            List<int> removelist = new List<int>();
                            int x = e.X + m_manager.StartToDrawX;
                            if ( x < m_mouse_trace_last_x ) {
                                foreach ( int key in m_mouse_trace.Keys ) {
                                    if ( x <= key && key < m_mouse_trace_last_x ) {
                                        removelist.Add( key );
                                    }
                                }
                            } else if ( m_mouse_trace_last_x < x ) {
                                foreach ( int key in m_mouse_trace.Keys ) {
                                    if ( m_mouse_trace_last_x < key && key <= x ) {
                                        removelist.Add( key );
                                    }
                                }
                            }
                            for ( int i = 0; i < removelist.Count; i++ ) {
                                m_mouse_trace.Remove( removelist[i] );
                            }
                            if ( x == m_mouse_trace_last_x ) {
                                m_mouse_trace[x] = e.Y;
                                m_mouse_trace_last_y = e.Y;
                            } else {
                                float a = (e.Y - m_mouse_trace_last_y) / (float)(x - m_mouse_trace_last_x);
                                float b = m_mouse_trace_last_y - a * m_mouse_trace_last_x;
                                int start = Math.Min( x, m_mouse_trace_last_x );
                                int end = Math.Max( x, m_mouse_trace_last_x );
                                for ( int xx = start; xx <= end; xx++ ) {
                                    int yy = (int)(a * xx + b);
                                    if ( m_mouse_trace.ContainsKey( xx ) ) {
                                        m_mouse_trace[xx] = yy;
                                    } else {
                                        m_mouse_trace.Add( xx, yy );
                                    }
                                }
                                m_mouse_trace_last_x = x;
                                m_mouse_trace_last_y = e.Y;
                            }
                        }
                        break;
                    case EditTool.Arrow:
                    case EditTool.Eraser:
                        if ( AppManager.EditorConfig.CurveSelectingQuantized ) {
                            int unit = AppManager.GetPositionQuantizeClock();
                            int odd = clock % unit;
                            int nclock = clock;
                            nclock -= odd;
                            if ( odd > unit / 2 ) {
                                nclock += unit;
                            }
                            m_selecting_region.Y = nclock;
                        } else {
                            m_selecting_region.Y = clock;
                        }
                        break;
                }
            } else if ( m_mouse_down_mode == MouseDownMode.SingerList ) {
                int dclock = clock - m_singer_move_started_clock;
                foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ){
                    item.Editing.Clock = item.Original.Clock + dclock;
                }
            } else if ( m_mouse_down_mode == MouseDownMode.VelEdit ) {
                int t_value = ValueFromYCoord( e.Y - m_veledit_shifty );
                int d_vel = 0;
                if ( m_selected_curve == CurveType.VEL ) {
                    d_vel = t_value - m_veledit_selected[m_veledit_last_selectedid].Original.ID.Dynamics;
                } else if ( m_selected_curve == CurveType.Accent ) {
                    d_vel = t_value - m_veledit_selected[m_veledit_last_selectedid].Original.ID.DEMaccent;
                } else if ( m_selected_curve == CurveType.Decay ) {
                    d_vel = t_value - m_veledit_selected[m_veledit_last_selectedid].Original.ID.DEMdecGainRate;
                }
                foreach ( int id in m_veledit_selected.Keys ) {
                    if ( m_selected_curve == CurveType.VEL ) {
                        int new_vel = m_veledit_selected[id].Original.ID.Dynamics + d_vel;
                        if ( new_vel < m_selected_curve.Minimum ) {
                            new_vel = m_selected_curve.Minimum;
                        } else if ( m_selected_curve.Maximum < new_vel ) {
                            new_vel = m_selected_curve.Maximum;
                        }
                        m_veledit_selected[id].Editing.ID.Dynamics = new_vel;
                    } else if ( m_selected_curve == CurveType.Accent ) {
                        int new_vel = m_veledit_selected[id].Original.ID.DEMaccent + d_vel;
                        if ( new_vel < m_selected_curve.Minimum ) {
                            new_vel = m_selected_curve.Minimum;
                        } else if ( m_selected_curve.Maximum < new_vel ) {
                            new_vel = m_selected_curve.Maximum;
                        }
                        m_veledit_selected[id].Editing.ID.DEMaccent = new_vel;
                    } else if ( m_selected_curve == CurveType.Decay ) {
                        int new_vel = m_veledit_selected[id].Original.ID.DEMdecGainRate + d_vel;
                        if ( new_vel < m_selected_curve.Minimum ) {
                            new_vel = m_selected_curve.Minimum;
                        } else if ( m_selected_curve.Maximum < new_vel ) {
                            new_vel = m_selected_curve.Maximum;
                        }
                        m_veledit_selected[id].Editing.ID.DEMdecGainRate = new_vel;
                    }
                }
            } else if ( m_mouse_down_mode == MouseDownMode.BezierMove ) {
                HandleMouseMoveForBezierMove( clock, value, value_raw, m_manager.LastSelectedBezier.Picked );
            } else if ( m_mouse_down_mode == MouseDownMode.BezierAddNew || m_mouse_down_mode == MouseDownMode.BezierEdit ) {
                BezierChain target = m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].GetBezierChain( m_selected_curve, m_manager.LastSelectedBezier.ChainID );
                int point_id = m_manager.LastSelectedBezier.PointID;
                int index = -1;
                for ( int i = 0; i < target.points.Count; i++ ) {
                    if ( target.points[i].ID == point_id ) {
                        index = i;
                        break;
                    }
                }
                if ( index >= 0 ) {
                    Point old_right = target.points[index].ControlRight.ToPoint();
                    Point old_left = target.points[index].ControlLeft.ToPoint();
                    BezierControlType old_right_type = target.points[index].ControlRightType;
                    BezierControlType old_left_type = target.points[index].ControlLeftType;
                    int cl = clock;
                    int va = value_raw;
                    int dx = (int)target.points[index].Base.X - cl;
                    int dy = (int)target.points[index].Base.Y - va;
                    if ( target.points[index].Base.X + dx >= 0 ) {
                        target.points[index].ControlRight = new PointD( clock, value_raw );
                        target.points[index].ControlLeft = new PointD( clock + 2 * dx, value_raw + 2 * dy );
                        target.points[index].ControlRightType = BezierControlType.Normal;
                        target.points[index].ControlLeftType = BezierControlType.Normal;
                        if ( !BezierChain.IsBezierImplicit( target ) ) {
                            target.points[index].ControlLeft = new PointD( old_left.X, old_left.Y );
                            target.points[index].ControlRight = new PointD( old_right.X, old_right.Y );
                            target.points[index].ControlLeftType = old_left_type;
                            target.points[index].ControlRightType = old_right_type;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 指定した位置にあるBezierPointを検索します。
        /// </summary>
        /// <param name="location"></param>
        /// <param name="list"></param>
        /// <param name="found_chain"></param>
        /// <param name="found_point"></param>
        /// <param name="found_side"></param>
        /// <param name="dot_width"></param>
        /// <param name="px_tolerance"></param>
        private void FindBezierPointAt( Point location,
                                        List<BezierChain> list,
                                        out BezierChain found_chain,
                                        out BezierPoint found_point,
                                        out BezierPickedSide found_side,
                                        int dot_width,
                                        int px_tolerance ) {
            found_chain = null;
            found_point = null;
            found_side = BezierPickedSide.Base;
            int shift = dot_width + px_tolerance;
            int width = (dot_width + px_tolerance) * 2;
            for ( int i = 0; i < list.Count; i++ ) {
                BezierChain bc = list[i];
                foreach ( BezierPoint bp in bc.points ) {
                    Point p = GetScreenCoord( bp.Base );
                    Rectangle r = new Rectangle( p.X - shift, p.Y - shift, width, width );
                    if ( IsInRect( location, r ) ) {
                        found_chain = bc;
                        found_point = bp;
                        found_side = BezierPickedSide.Base;
                        return;
                    }

                    if ( bp.ControlLeftType != BezierControlType.None ) {
                        p = GetScreenCoord( bp.ControlLeft );
                        r = new Rectangle( p.X - shift, p.Y - shift, width, width );
                        if ( IsInRect( location, r ) ) {
                            found_chain = bc;
                            found_point = bp;
                            found_side = BezierPickedSide.Left;
                            return;
                        }
                    }

                    if ( bp.ControlRightType != BezierControlType.None ) {
                        p = GetScreenCoord( bp.ControlRight );
                        r = new Rectangle( p.X - shift, p.Y - shift, width, width );
                        if ( IsInRect( location, r ) ) {
                            found_chain = bc;
                            found_point = bp;
                            found_side = BezierPickedSide.Right;
                            return;
                        }
                    }
                }
            }
        }

        public void TrackSelector_MouseDown( object sender, MouseEventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "TrackSelector_MouseDown" );
#endif
            int clock = ClockFromXCoord( e.X );
            m_mouse_moved = false;
            m_mouse_downed = true;
            if ( AppManager._KEY_LENGTH < e.X && clock < m_manager.VsqFile.getPreMeasure() ) {
                System.Media.SystemSounds.Asterisk.Play();
                return;
            }
            Keys modifier = Control.ModifierKeys;
            int max = m_selected_curve.Maximum;
            int min = m_selected_curve.Minimum;
            int value = ValueFromYCoord( e.Y );
            if ( value < min ) {
                value = min;
            } else if ( max < value ) {
                value = max;
            }

            if ( Height - _OFFSET_TRACK_TAB <= e.Y && e.Y < Height ) {
                if ( e.Button == MouseButtons.Left ) {
                    #region MouseDown occured on track list
                    m_mouse_down_mode = MouseDownMode.TrackList;
                    m_selected_region_enabled = false;
                    m_mouse_trace = null;
                    int selecter_width = SelectorWidth;
                    if ( m_manager.VsqFile != null ) {
                        for ( int i = 0; i < 16; i++ ) {
                            int x = AppManager._KEY_LENGTH + i * selecter_width;
                            if ( m_manager.VsqFile.Track.Count > i + 1 ) {
                                if ( x <= e.X && e.X < x + selecter_width ) {
                                    int new_selected = i + 1;
                                    if ( m_manager.Selected != new_selected ) {
                                        m_manager.Selected = i + 1;
                                        if ( SelectedTrackChanged != null ) {
                                            SelectedTrackChanged( i + 1 );
                                        }
                                        Invalidate();
                                        return;
                                    } else if ( x + selecter_width - _PX_WIDTH_RENDER <= e.X && e.X < e.X + selecter_width ) {
                                        if ( m_manager.getRenderRequired( m_manager.Selected ) && !m_manager.Playing ) {
                                            if ( RenderRequired != null ) {
                                                RenderRequired( new int[] { m_manager.Selected } );
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        /*// RENDER ALL
                        if ( Width - _PX_WIDTH_RENDERALL <= e.X && e.X <= Width && !m_manager.Playing ) {
                            List<int> tracks = new List<int>();
                            for ( int i = 1; i < m_manager.VsqFile.Track.Count; i++ ) {
                                if ( m_manager.getRenderRequired( i ) ) {
                                    tracks.Add( i );
                                }
                            }
                            if ( tracks.Count > 0 && RenderRequired != null ) {
                                RenderRequired( tracks.ToArray() );
                            }
                        }*/
                    }
                    #endregion
                }
            } else if ( Height - 2 * _OFFSET_TRACK_TAB <= e.Y && e.Y < Height - _OFFSET_TRACK_TAB ) {
                #region MouseDown occured on singer tab
                m_mouse_down_mode = MouseDownMode.SingerList;
                m_selected_region_enabled = false;
                m_mouse_trace = null;
                int x;
                VsqEvent ve = GetItemAtClickedPosition( e.Location, out x );
                if ( m_manager.SelectedTool == EditTool.Eraser ) {
                    #region EditTool.Eraser
                    if ( ve != null && ve.Clock > 0 ) {
                        CadenciiCommand run = new CadenciiCommand( VsqCommand.generateCommandEventDelete( m_manager.Selected, ve.InternalID ) );
                        ExecuteCommand( run, true );
                    }
                    #endregion
                } else {
                    if ( ve != null ) {
                        if ( (Control.ModifierKeys & s_modifier_key) == s_modifier_key ) {
                            if ( m_manager.SelectedEvent.ContainsKey( m_manager.Selected, ve.InternalID ) ) {
                                List<int> old = new List<int>();
                                foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ){
                                    int id = item.Original.InternalID;
                                    if ( id != ve.InternalID ) {
                                        old.Add( id );
                                    }
                                }
                                m_manager.ClearSelectedEvent();
                                foreach ( int id in old ) {
                                    m_manager.AddSelectedEvent( id );
                                }
                            } else {
                                m_manager.AddSelectedEvent( ve.InternalID );
                            }
                        } else if ( (Control.ModifierKeys & Keys.Shift) == Keys.Shift ) {
                            int last_clock = m_manager.SelectedEvent.LastSelected.Original.Clock;
                            int tmin = Math.Min( ve.Clock, last_clock );
                            int tmax = Math.Max( ve.Clock, last_clock );
                            for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getEventIterator(); itr.hasNext(); ) {
                                VsqEvent item = (VsqEvent)itr.next();
                                if ( item.ID.type == VsqIDType.Singer && tmin <= item.Clock && item.Clock <= tmax ) {
                                    m_manager.AddSelectedEvent( item.InternalID );
                                }
                            }
                            m_manager.AddSelectedEvent( ve.InternalID );
                        } else {
                            if ( !m_manager.SelectedEvent.ContainsKey( m_manager.Selected, ve.InternalID ) ) {
                                m_manager.ClearSelectedEvent();
                            }
                            m_manager.AddSelectedEvent( ve.InternalID );
                        }
                        m_singer_move_started_clock = clock;
                    } else {
                        m_manager.ClearSelectedEvent();
                    }
                }
                #endregion
            } else {
                #region MouseDown occred on other position
                bool clock_inner_note = false; //マウスの降りたクロックが，ノートの範囲内かどうかをチェック
                int left_clock = ClockFromXCoord( AppManager._KEY_LENGTH );
                int right_clock = ClockFromXCoord( this.Width - vScroll.Width );
                for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getEventIterator(); itr.hasNext(); ) {
                    VsqEvent ve = (VsqEvent)itr.next();
                    if ( ve.ID.type == VsqIDType.Anote ) {
                        int start = ve.Clock;
                        if ( right_clock < start ) {
                            break;
                        }
                        int end = ve.Clock + ve.ID.Length;
                        if ( end < left_clock ) {
                            continue;
                        }
                        if ( start <= clock && clock < end ) {
                            clock_inner_note = true;
                            break;
                        }
                    }
                }
#if DEBUG
                Common.DebugWriteLine( "    clock_inner_note=" + clock_inner_note );
#endif
                if ( AppManager._KEY_LENGTH <= e.X ) {
                    if ( e.Button == MouseButtons.Left && !m_spacekey_downed ) {
                        m_selected_region_enabled = false;
                        m_mouse_down_mode = MouseDownMode.CurveEdit;
                        if ( m_manager.SelectedTool != EditTool.Arrow ) {
                            m_manager.ClearSelectedEvent();
                        }
                        int quantized_clock = clock;
                        int unit = AppManager.GetPositionQuantizeClock();
                        int odd = clock % unit;
                        quantized_clock -= odd;
                        if ( odd > unit / 2 ) {
                            quantized_clock += unit;
                        }

                        // 
                        int px_shift = _DOT_WID + AppManager.EditorConfig.PxToleranceBezier;
                        int px_width = px_shift * 2 + 1;

                        if ( m_manager.SelectedTool == EditTool.Line ) {
                            #region Line
                            m_line_start = new Point( clock, value );
                            #endregion
                        } else if ( m_manager.SelectedTool == EditTool.Pencil ) {
                            #region Pencil
                            if ( m_manager.IsCurveMode ) {
                                #region CurveMode
                                if ( m_selected_curve == CurveType.VibratoRate || m_selected_curve == CurveType.VibratoDepth ) {
                                    // todo: TrackSelector_MouseDownのベジエ曲線
                                } else if ( m_selected_curve != CurveType.VEL &&
                                            m_selected_curve != CurveType.Accent &&
                                            m_selected_curve != CurveType.Decay ) {
                                    int track = m_manager.Selected;
                                    bool too_near = false; // clicked position is too near to existing bezier points
                                    bool is_middle = false;

                                    // check whether bezier point exists on clicked position
                                    List<BezierChain> dict = m_manager.VsqFile.AttachedCurves[track - 1][m_selected_curve];
                                    BezierChain found_chain;
                                    BezierPoint found_point;
                                    BezierPickedSide found_side;
                                    FindBezierPointAt( e.Location, dict, out found_chain, out found_point, out found_side, _DOT_WID, AppManager.EditorConfig.PxToleranceBezier );

                                    if ( found_chain != null ) {
                                        m_manager.AddSelectedBezier( new SelectedBezierPoint( found_chain.ID, found_point.ID, found_side, found_point ) );
                                        m_editing_bezier_original = (BezierChain)found_chain.Clone();
                                        m_mouse_down_mode = MouseDownMode.BezierMove;
                                    } else {
                                        BezierChain target_chain = null;
                                        for ( int j = 0; j < dict.Count; j++ ) {
                                            BezierChain bc = dict[j];
                                            //foreach ( BezierChain bc in dict.Values ) {
                                            for ( int i = 1; i < bc.Count; i++ ) {
                                                if ( !is_middle && bc.points[i - 1].Base.X <= clock && clock <= bc.points[i].Base.X ) {
                                                    target_chain = (BezierChain)bc.Clone();
                                                    is_middle = true;
                                                }
                                                if ( !too_near ) {
                                                    foreach ( BezierPoint bp in bc.points ) {
                                                        Point pt = GetScreenCoord( bp.Base );
                                                        Rectangle rc = new Rectangle( pt.X - px_shift, pt.Y - px_shift, px_width, px_width );
                                                        if ( IsInRect( e.Location, rc ) ) {
                                                            too_near = true;
                                                            break;
                                                        }
                                                    }
                                                }
                                            }
                                        }

                                        if ( !too_near ) {
                                            if ( (modifier & s_modifier_key) != s_modifier_key && target_chain == null ) {
                                                // search BezierChain just before the clicked position
                                                int tmax = -1;
                                                for ( int i = 0; i < dict.Count; i++ ) {
                                                    BezierChain bc = dict[i];
                                                    bc.points.Sort();
                                                    // check most nearest data point from clicked position
                                                    int last = (int)bc.points[bc.points.Count - 1].Base.X;
                                                    if ( tmax < last && last < clock ) {
                                                        tmax = last;
                                                        target_chain = (BezierChain)bc.Clone();
                                                    }
                                                }
                                            }

#if DEBUG
                                            Common.DebugWriteLine( "    (target_chain==null)=" + (target_chain == null) );
#endif
                                            // fork whether target_chain is null or not
                                            PointD pt = new PointD( clock, value );
                                            BezierPoint bp = null;
                                            int chain_id = -1;
                                            int point_id = -1;
                                            if ( target_chain == null ) {
                                                // generate new BezierChain
                                                BezierChain adding = new BezierChain();
                                                bp = new BezierPoint( pt, pt, pt );
                                                point_id = adding.GetNextId();
                                                bp.ID = point_id;
                                                adding.Add( bp );
                                                chain_id = m_manager.VsqFile.AttachedCurves[track - 1].GetNextId( m_selected_curve );
#if DEBUG
                                                Common.DebugWriteLine( "    new chain_id=" + chain_id );
#endif
                                                CadenciiCommand run = VsqFileEx.generateCommandAddBezierChain( track,
                                                                                                        m_selected_curve,
                                                                                                        chain_id,
                                                                                                        AppManager.EditorConfig.ControlCurveResolution.Value,
                                                                                                        adding );
                                                ExecuteCommand( run, false );
                                                m_mouse_down_mode = MouseDownMode.BezierAddNew;
                                            } else {
                                                m_editing_bezier_original = (BezierChain)target_chain.Clone();
                                                bp = new BezierPoint( pt, pt, pt );
                                                point_id = target_chain.GetNextId();
                                                bp.ID = point_id;
                                                target_chain.Add( bp );
                                                target_chain.points.Sort();
                                                chain_id = target_chain.ID;
                                                CadenciiCommand run = VsqFileEx.generateCommandReplaceBezierChain( track,
                                                                                                            m_selected_curve,
                                                                                                            target_chain.ID,
                                                                                                            target_chain,
                                                                                                            AppManager.EditorConfig.ControlCurveResolution.Value );
                                                ExecuteCommand( run, false );
                                                m_mouse_down_mode = MouseDownMode.BezierEdit;
                                            }
                                            m_manager.ClearSelectedBezier();
                                            m_manager.AddSelectedBezier( new SelectedBezierPoint( chain_id, point_id, BezierPickedSide.Base, bp ) );
                                        } else {
                                            m_mouse_down_mode = MouseDownMode.None;
                                        }
                                    }
                                } else {
                                    m_mouse_down_mode = MouseDownMode.None;
                                }
                                #endregion
                            } else {
                                #region NOT CurveMode
                                m_mouse_trace = null;
                                m_mouse_trace = new SortedList<int, int>();
                                int x = e.X + m_manager.StartToDrawX;
                                m_mouse_trace.Add( x, e.Y );
                                m_mouse_trace_last_x = x;
                                m_mouse_trace_last_y = e.Y;
                                m_pencil_moved = false;
                                m_mouse_hover_thread = new Thread( new ThreadStart( MouseHoverEventGenerator ) );
                                m_mouse_hover_thread.Start();
                                #endregion
                            }
                            #endregion
                        } else if ( m_manager.SelectedTool == EditTool.Arrow ) {
                            #region Arrow
                            if ( m_manager.IsCurveMode && (m_selected_curve.IsScalar || m_selected_curve.IsAttachNote) ) {
                                m_mouse_down_mode = MouseDownMode.None;
                            } else {
                                bool found = false;
                                // カーブモードなら，まずベジエ曲線の点にヒットしてないかどうかを検査
                                if ( m_manager.IsCurveMode ) {
                                    #region _T_m_selected_curve != CurveType.VEL
                                    List<BezierChain> dict = m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1][m_selected_curve];
                                    m_manager.ClearSelectedBezier();
                                    for ( int i = 0; i < dict.Count; i++ ) {
                                        BezierChain bc = dict[i];
                                        foreach ( BezierPoint bp in bc.points ) {
                                            Point pt = GetScreenCoord( bp.Base );
                                            Rectangle rc = new Rectangle( pt.X - px_shift, pt.Y - px_shift, px_width, px_width );
                                            if ( IsInRect( e.Location, rc ) ) {
                                                m_manager.AddSelectedBezier( new SelectedBezierPoint( bc.ID, bp.ID, BezierPickedSide.Base, bp ) );
                                                m_editing_bezier_original = (BezierChain)bc.Clone();
                                                found = true;
                                                break;
                                            }

                                            if ( bp.ControlLeftType != BezierControlType.None ) {
                                                pt = GetScreenCoord( bp.ControlLeft );
                                                rc = new Rectangle( pt.X - px_shift, pt.Y - px_shift, px_width, px_width );
                                                if ( IsInRect( e.Location, rc ) ) {
                                                    m_manager.AddSelectedBezier( new SelectedBezierPoint( bc.ID, bp.ID, BezierPickedSide.Left, bp ) );
                                                    m_editing_bezier_original = (BezierChain)bc.Clone();
                                                    found = true;
                                                    break;
                                                }
                                            }

                                            if ( bp.ControlRightType != BezierControlType.None ) {
                                                pt = GetScreenCoord( bp.ControlRight );
                                                rc = new Rectangle( pt.X - px_shift, pt.Y - px_shift, px_width, px_width );
                                                if ( IsInRect( e.Location, rc ) ) {
                                                    m_manager.AddSelectedBezier( new SelectedBezierPoint( bc.ID, bp.ID, BezierPickedSide.Right, bp ) );
                                                    m_editing_bezier_original = (BezierChain)bc.Clone();
                                                    found = true;
                                                    break;
                                                }
                                            }
                                        }
                                        if ( found ) {
                                            break;
                                        }
                                    }
                                    if ( found ) {
                                        m_mouse_down_mode = MouseDownMode.BezierMove;
                                    }/* else {
                                        m_mouse_down_mode = MouseDownMode.BezierSelect;
                                    }*/
                                    #endregion
                                }

                                if ( !found ) {
                                    #region NOT CurveMode
                                    VsqEvent ve = GetItemAtClickedPosition( e.Location );
                                    if ( ve != null ) {
                                        bool found2 = false;
                                        if ( (Control.ModifierKeys & s_modifier_key) == s_modifier_key ) {
                                            // clicked with CTRL key
                                            List<int> list = new List<int>();
                                            foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ) {
                                                VsqEvent ve2 = item.Original;
                                                if ( ve.InternalID == ve2.InternalID ) {
                                                    found2 = true;
                                                } else {
                                                    list.Add( ve2.InternalID );
                                                }
                                            }
                                            m_manager.ClearSelectedEvent();
                                            foreach ( int id in list ) {
                                                m_manager.AddSelectedEvent( id );
                                            }
                                        } else if ( (Control.ModifierKeys & Keys.Shift) == Keys.Shift ) {
                                            // clicked with Shift key
                                            int last_clock = m_manager.SelectedEvent.LastSelected.Original.Clock;
                                            int tmin = Math.Min( ve.Clock, last_clock );
                                            int tmax = Math.Max( ve.Clock, last_clock );
                                            for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                                                VsqEvent item = (VsqEvent)itr.next();
                                                if ( tmin <= item.Clock && item.Clock <= tmax ) {
                                                    m_manager.AddSelectedEvent( item.InternalID );
                                                }
                                            }
                                        } else {
                                            // no modefier key
                                            if ( !m_manager.SelectedEvent.ContainsKey( m_manager.Selected, ve.InternalID ) ) {
                                                m_manager.ClearSelectedEvent();
                                            }
                                        }
                                        if ( !found2 ) {
                                            m_manager.AddSelectedEvent( ve.InternalID );
                                        }

                                        m_mouse_down_mode = MouseDownMode.VelWaitHover;
                                        m_veledit_last_selectedid = ve.InternalID;
                                        if ( m_selected_curve == CurveType.VEL ) {
                                            m_veledit_shifty = e.Y - YCoordFromValue( ve.ID.Dynamics );
                                        } else if ( m_selected_curve == CurveType.Accent ) {
                                            m_veledit_shifty = e.Y - YCoordFromValue( ve.ID.DEMaccent );
                                        } else if ( m_selected_curve == CurveType.Decay ) {
                                            m_veledit_shifty = e.Y - YCoordFromValue( ve.ID.DEMdecGainRate );
                                        }
                                        m_veledit_selected.Clear();
                                        if ( m_manager.SelectedEvent.ContainsKey( m_manager.Selected, m_veledit_last_selectedid ) ) {
                                            foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ) {
                                                m_veledit_selected.Add( item.Original.InternalID, 
                                                                        new SelectedEventEntry( m_manager.Selected,
                                                                                                item.Original,
                                                                                                item.Editing ) );
                                            }
                                        } else {
                                            m_veledit_selected.Add( m_veledit_last_selectedid,
                                                                    new SelectedEventEntry( m_manager.Selected, 
                                                                                            (VsqEvent)ve.Clone(),
                                                                                            (VsqEvent)ve.Clone() ) );
                                        }
                                        m_mouse_hover_thread = new Thread( new ThreadStart( MouseHoverEventGenerator ) );
                                        m_mouse_hover_thread.Start();
                                    } else {
                                        m_manager.ClearSelectedEvent();
                                        if ( (modifier & Keys.Shift) != Keys.Shift && (modifier & s_modifier_key) != s_modifier_key ) {
                                            m_selected_region_enabled = false;
                                        }
                                        if ( m_manager.SelectedTool == EditTool.Eraser ) {
                                            m_selected_region_enabled = false;
                                        }
                                        if ( AppManager.EditorConfig.CurveSelectingQuantized ) {
                                            m_selecting_region = new Point( quantized_clock, quantized_clock );
                                        } else {
                                            m_selecting_region = new Point( clock, clock );
                                        }
                                    }
                                    #endregion
                                }
                            }

                            /*if ( m_manager.IsCurveMode ) {
                                #region CurveMode
                                if ( m_selected_curve == CurveType.VEL ) {
                                    m_mouse_down_mode = MouseDownMode.None;
                                } else {
                                }
                                #endregion
                            } else {
                            }*/
                            #endregion
                        } else if ( m_manager.SelectedTool == EditTool.Eraser ) {
                            #region Eraser
                            VsqEvent ve3 = GetItemAtClickedPosition( e.Location );
                            if ( ve3 != null ) {
                                m_manager.ClearSelectedEvent();
                                CadenciiCommand run = new CadenciiCommand( VsqCommand.generateCommandEventDelete( m_manager.Selected,
                                                                                                           ve3.InternalID ) );
                                ExecuteCommand( run, true );
                            } else {
                                if ( (modifier & Keys.Shift) != Keys.Shift && (modifier & s_modifier_key) != s_modifier_key ) {
                                    m_selected_region_enabled = false;
                                }
                                if ( m_manager.SelectedTool == EditTool.Eraser ) {
                                    m_selected_region_enabled = false;
                                }
                                if ( AppManager.EditorConfig.CurveSelectingQuantized ) {
                                    m_selecting_region = new Point( quantized_clock, quantized_clock );
                                } else {
                                    m_selecting_region = new Point( clock, clock );
                                }
                            }
                            #endregion
                        }
                    } else if ( e.Button == MouseButtons.Right ) {
                        if ( m_manager.IsCurveMode ) {
                            if ( m_selected_curve != CurveType.VEL ) {
                                List<BezierChain> dict = m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1][m_selected_curve];
                                m_manager.ClearSelectedBezier();
                                bool found = false;
                                //foreach ( BezierChain bc in dict.Values ) {
                                for ( int i = 0; i < dict.Count; i++ ) {
                                    BezierChain bc = dict[i];
                                    foreach ( BezierPoint bp in bc.points ) {
                                        Point pt = GetScreenCoord( bp.Base );
                                        Rectangle rc = new Rectangle( pt.X - _DOT_WID, pt.Y - _DOT_WID, 2 * _DOT_WID + 1, 2 * _DOT_WID + 1 );
                                        if ( IsInRect( e.Location, rc ) ) {
                                            m_manager.AddSelectedBezier( new SelectedBezierPoint( bc.ID, bp.ID, BezierPickedSide.Base, bp ) );
                                            found = true;
                                            break;
                                        }
                                    }
                                    if ( found ) {
                                        break;
                                    }
                                }
                            }
                        }
                    }
                } else {
                    m_selected_region_enabled = false;
                }
                #endregion
            }
            Invalidate();
        }

        private void ChangeCurve( CurveType curve ) {
            if ( m_selected_curve != curve ) {
                m_last_selected_curve = m_selected_curve;
                m_selected_curve = curve;
                if ( SelectedCurveChanged != null ) {
                    SelectedCurveChanged( curve );
                }
            }
        }

        private static bool IsInRect( Point pt, Rectangle rc ) {
            if ( rc.X <= pt.X && pt.X <= rc.X + rc.Width ) {
                if ( rc.Y <= pt.Y && pt.Y <= rc.Y + rc.Height ) {
                    return true;
                } else {
                    return false;
                }
            } else {
                return false;
            }
        }
        
        /// <summary>
        /// クリックされた位置にある歌手変更イベントを取得します
        /// </summary>
        /// <param name="location"></param>
        /// <param name="position_x"></param>
        /// <returns></returns>
        private VsqEvent GetItemAtClickedPosition( Point location, out int position_x ) {
            position_x = 0;
            if ( m_manager.VsqFile != null ) {
                for ( int i = 0; i < m_manager.VsqFile.Track[m_manager.Selected].getEventCount(); i++ ) {
                    VsqEvent ve = m_manager.VsqFile.Track[m_manager.Selected].getEvent( i );
                    if ( ve.ID.type == VsqIDType.Singer ) {
                        int x = XCoordFromClocks( ve.Clock );
                        if ( Height - 2 * _OFFSET_TRACK_TAB <= location.Y && 
                             location.Y <= Height - _OFFSET_TRACK_TAB && 
                             x <= location.X && location.X <= x + _SINGER_ITEM_WIDTH ) {
                            position_x = x;
                            return ve;
                        } else if ( Width < x ) {
                            //return null;
                        }
                    } else if ( ve.ID.type == VsqIDType.Anote ) {
                        int x = XCoordFromClocks( ve.Clock );
                        int y = 0;
                        if ( m_selected_curve == CurveType.VEL ) {
                            y = YCoordFromValue( ve.ID.Dynamics );
                        } else if ( m_selected_curve == CurveType.Accent ) {
                            y = YCoordFromValue( ve.ID.DEMaccent );
                        } else if ( m_selected_curve == CurveType.Decay ) {
                            y = YCoordFromValue( ve.ID.DEMdecGainRate );
                        } else {
                            continue;
                        }
                        if ( 0 <= location.Y && location.Y <= Height - 2 * _OFFSET_TRACK_TAB && 
                             AppManager._KEY_LENGTH <= location.X && location.X <= Width ) {
                            if ( y <= location.Y && location.Y <= Height - _FOOTER && x <= location.X && location.X <= x + _VEL_BAR_WIDTH ) {
                                position_x = x;
                                return ve;
                            }
                        }
                    }
                }
            }
            return null;
        }
        
        /// <summary>
        /// クリックされた位置にある歌手変更イベントを取得します
        /// </summary>
        /// <param name="location"></param>
        /// <returns></returns>
        private VsqEvent GetItemAtClickedPosition( Point location ) {
            int x;
            return GetItemAtClickedPosition( location, out x );
        }

        public void TrackSelector_MouseUp( object sender, MouseEventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "TrackSelector_MouseUp" );
#endif
            m_mouse_downed = false;
            if ( m_mouse_hover_thread != null ) {
                if ( m_mouse_hover_thread.IsAlive ) {
                    m_mouse_hover_thread.Abort();
                }
            }
            if ( CurveVisible ) {
                int max = m_selected_curve.Maximum;
                int min = m_selected_curve.Minimum;
                if ( m_selected_curve == CurveType.Pitch ) {
                    max = m_internal_pbs * 10000;
                    min = -m_internal_pbs * 10000;
                }
#if DEBUG
                Common.DebugWriteLine( "    max,min=" + max + "," + min );
#endif
                if ( m_mouse_down_mode == MouseDownMode.BezierAddNew ||
                     m_mouse_down_mode == MouseDownMode.BezierMove ||
                     m_mouse_down_mode == MouseDownMode.BezierEdit ) {
                    if ( e.Button == MouseButtons.Left ) {
                        if ( m_manager.IsCurveMode ) {
                            if ( m_manager.SelectedTool == EditTool.Arrow && sender is TrackSelector ) {
                                #region Arrow
                                int chain_id = m_manager.LastSelectedBezier.ChainID;
                                BezierChain edited = (BezierChain)m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].GetBezierChain( m_selected_curve, chain_id ).Clone();
                                m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].SetBezierChain( m_selected_curve, chain_id, m_editing_bezier_original );
                                CadenciiCommand run = VsqFileEx.generateCommandReplaceBezierChain( m_manager.Selected,
                                                                                                   m_selected_curve,
                                                                                                   chain_id,
                                                                                                   edited,
                                                                                                   AppManager.EditorConfig.ControlCurveResolution.Value );
                                ExecuteCommand( run, true );
#if DEBUG
                                Common.DebugWriteLine( "    m_mouse_down_mode=" + m_mouse_down_mode );
                                Common.DebugWriteLine( "    chain_id=" + chain_id );
#endif
                                #endregion
                            } else if ( m_manager.SelectedTool == EditTool.Pencil && sender is TrackSelector ) {
                                #region Pencil
                                int chain_id = m_manager.LastSelectedBezier.ChainID;
                                BezierChain edited = (BezierChain)m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].GetBezierChain( m_selected_curve, chain_id ).Clone();
                                if ( m_mouse_down_mode == MouseDownMode.BezierAddNew ) {
                                    edited.ID = chain_id;
                                    CadenciiCommand pre = VsqFileEx.generateCommandDeleteBezierChain( m_manager.Selected,
                                                                                                      m_selected_curve,
                                                                                                      chain_id,
                                                                                                      AppManager.EditorConfig.ControlCurveResolution.Value );
                                    ExecuteCommand( pre, false );
                                    CadenciiCommand run = VsqFileEx.generateCommandAddBezierChain( m_manager.Selected,
                                                                                                   m_selected_curve,
                                                                                                   chain_id,
                                                                                                   AppManager.EditorConfig.ControlCurveResolution.Value,
                                                                                                   edited );
                                    ExecuteCommand( run, true );
                                } else if ( m_mouse_down_mode == MouseDownMode.BezierEdit ) {
                                    CadenciiCommand pre = VsqFileEx.generateCommandReplaceBezierChain( m_manager.Selected,
                                                                                                m_selected_curve,
                                                                                                chain_id,
                                                                                                m_editing_bezier_original,
                                                                                                AppManager.EditorConfig.ControlCurveResolution.Value );
                                    ExecuteCommand( pre, false );
                                    CadenciiCommand run = VsqFileEx.generateCommandReplaceBezierChain( m_manager.Selected,
                                                                                                m_selected_curve,
                                                                                                chain_id,
                                                                                                edited,
                                                                                                AppManager.EditorConfig.ControlCurveResolution.Value );
                                    ExecuteCommand( run, true );
                                } else if ( m_mouse_down_mode == MouseDownMode.BezierMove ) {
                                    m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].SetBezierChain( m_selected_curve, chain_id, m_editing_bezier_original );
                                    CadenciiCommand run = VsqFileEx.generateCommandReplaceBezierChain( m_manager.Selected,
                                                                                                m_selected_curve,
                                                                                                chain_id,
                                                                                                edited,
                                                                                                AppManager.EditorConfig.ControlCurveResolution.Value );
                                    ExecuteCommand( run, true );
#if DEBUG
                                    Common.DebugWriteLine( "    m_mouse_down_mode=" + m_mouse_down_mode );
                                    Common.DebugWriteLine( "    chain_id=" + chain_id );
#endif

                                }
                                #endregion
                            }
                        }
                    }
                } else if ( m_mouse_down_mode == MouseDownMode.CurveEdit ||
                            m_mouse_down_mode == MouseDownMode.VelWaitHover ) {
                    if ( e.Button == MouseButtons.Left ) {
                        if ( m_manager.SelectedTool == EditTool.Arrow ) {
                            #region Arrow
                            if ( m_selected_curve != CurveType.VEL && m_selected_curve != CurveType.Accent && m_selected_curve != CurveType.Decay ) {
                                if ( m_selecting_region.X == m_selecting_region.Y ) {
                                    m_selected_region_enabled = false;
                                } else {
                                    if ( !m_selected_region_enabled ) {
                                        m_selected_region = m_selecting_region;
#if DEBUG
                                        Common.DebugWriteLine( "    selected_region ENABLED" );
#endif
                                        m_selected_region_enabled = true;
                                    } else {
                                        int start = Math.Min( m_selecting_region.X, m_selecting_region.Y );
                                        int end = Math.Max( m_selecting_region.X, m_selecting_region.Y );
                                        int old_start = Math.Min( m_selected_region.X, m_selected_region.Y );
                                        int old_end = Math.Max( m_selected_region.X, m_selected_region.Y );
                                        m_selected_region.X = Math.Min( start, old_start );
                                        m_selected_region.Y = Math.Max( end, old_end );
                                    }
                                }
                            }
                            #endregion
                        } else if ( !m_manager.IsCurveMode && m_manager.SelectedTool == EditTool.Eraser ) {
                            #region Eraser
                            if ( m_selected_curve == CurveType.VEL || m_selected_curve == CurveType.Accent || m_selected_curve == CurveType.Decay ) {
                                #region VEL Accent Delay
                                int start = Math.Min( m_selecting_region.X, m_selecting_region.Y );
                                int end = Math.Max( m_selecting_region.X, m_selecting_region.Y );
                                int old_start = Math.Min( m_selected_region.X, m_selected_region.Y );
                                int old_end = Math.Max( m_selected_region.X, m_selected_region.Y );
                                m_selected_region.X = Math.Min( start, old_start );
                                m_selected_region.Y = Math.Max( end, old_end );
                                m_manager.ClearSelectedEvent();
                                List<int> deleting = new List<int>();
                                for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                                    VsqEvent ev = (VsqEvent)itr.next();
                                    if ( start <= ev.Clock && ev.Clock <= end ) {
                                        deleting.Add( ev.InternalID );
                                    }
                                }
                                if ( deleting.Count > 0 ) {
                                    CadenciiCommand er_run = new CadenciiCommand(
                                        VsqCommand.generateCommandEventDeleteRange( m_manager.Selected, deleting.ToArray() ) );
                                    ExecuteCommand( er_run, true );
                                }
                                #endregion
                            } else if ( m_selected_curve == CurveType.VibratoRate || m_selected_curve == CurveType.VibratoDepth ) {
                                #region VibratoRate ViratoDepth
                                int er_start = Math.Min( m_selecting_region.X, m_selecting_region.Y );
                                int er_end = Math.Max( m_selecting_region.X, m_selecting_region.Y );
                                List<int> internal_ids = new List<int>();
                                List<VsqID> items = new List<VsqID>();
                                for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                                    VsqEvent ve = (VsqEvent)itr.next();
                                    if ( ve.ID.VibratoHandle == null ){
                                        continue;
                                    }
                                    int cl_vib_start = ve.Clock + ve.ID.VibratoDelay;
                                    int cl_vib_length = ve.ID.Length - ve.ID.VibratoDelay;
                                    int cl_vib_end = cl_vib_start + cl_vib_length;
                                    int clear_start = int.MaxValue;
                                    int clear_end = int.MinValue;
                                    if ( er_start < cl_vib_start && cl_vib_start < er_end && er_end <= cl_vib_end ) {
                                        // cl_vib_startからer_endまでをリセット
                                        clear_start = cl_vib_start;
                                        clear_end = er_end;
                                    } else if ( cl_vib_start <= er_start && er_end <= cl_vib_end ) {
                                        // er_startからer_endまでをリセット
                                        clear_start = er_start;
                                        clear_end = er_end;
                                    } else if ( cl_vib_start < er_start && er_start < cl_vib_end && cl_vib_end < er_end ) {
                                        // er_startからcl_vib_endまでをリセット
                                        clear_start = er_start;
                                        clear_end = cl_vib_end;
                                    } else if ( er_start < cl_vib_start && cl_vib_end < er_end ) {
                                        // 全部リセット
                                        clear_start = cl_vib_start;
                                        clear_end = cl_vib_end;
                                    }
                                    if ( clear_start < clear_end ) {
                                        float f_clear_start = (clear_start - cl_vib_start) / (float)cl_vib_length;
                                        float f_clear_end = (clear_end - cl_vib_start) / (float)cl_vib_length;
                                        VsqID item = (VsqID)ve.ID.Clone();
                                        VibratoBPList target = null;
                                        if ( m_selected_curve == CurveType.VibratoDepth ) {
                                            target = item.VibratoHandle.DepthBP;
                                        } else {
                                            target = item.VibratoHandle.RateBP;
                                        }
                                        List<float> bpx = new List<float>();
                                        List<int> bpy = new List<int>();
                                        bool start_added = false;
                                        bool end_added = false;
                                        for ( int i = 0; i < target.getCount(); i++ ) {
                                            VibratoBPPair vbpp = target.getElement( i );
                                            if ( vbpp.X < f_clear_start ) {
                                                bpx.Add( vbpp.X );
                                                bpy.Add( vbpp.Y );
                                            } else if ( f_clear_start == vbpp.X ) {
                                                bpx.Add( vbpp.X );
                                                bpy.Add( 64 );
                                                start_added = true;
                                            } else if ( f_clear_start < vbpp.X && !start_added ) {
                                                bpx.Add( f_clear_start );
                                                bpy.Add( 64 );
                                                start_added = true;
                                            } else if ( f_clear_end == vbpp.X ) {
                                                bpx.Add( vbpp.X );
                                                bpy.Add( vbpp.Y );
                                                end_added = true;
                                            } else if ( f_clear_end < vbpp.X && !end_added ) {
                                                int y = vbpp.Y;
                                                if ( i > 0 ) {
                                                    y = target.getElement( i - 1 ).Y;
                                                }
                                                bpx.Add( f_clear_end );
                                                bpy.Add( y );
                                                end_added = true;
                                                bpx.Add( vbpp.X );
                                                bpy.Add( vbpp.Y );
                                            } else if ( f_clear_end < vbpp.X ) {
                                                bpx.Add( vbpp.X );
                                                bpy.Add( vbpp.Y );
                                            }
                                        }
                                        if ( m_selected_curve == CurveType.VibratoDepth ) {
                                            item.VibratoHandle.DepthBP = new VibratoBPList( bpx.ToArray(), bpy.ToArray() );
                                        } else {
                                            item.VibratoHandle.RateBP = new VibratoBPList( bpx.ToArray(), bpy.ToArray() );
                                        }
                                        internal_ids.Add( ve.InternalID );
                                        items.Add( item );
                                    }
                                }
                                CadenciiCommand run = new CadenciiCommand( 
                                    VsqCommand.generateCommandEventChangeIDContaintsRange( m_manager.Selected, internal_ids.ToArray(), items.ToArray() ) );
                                ExecuteCommand( run, true );
                                #endregion
                            } else {
                                #region Other Curves
                                List<BPPair> er_edit = new List<BPPair>();
                                int er_start = Math.Min( m_selecting_region.X, m_selecting_region.Y );
                                int er_end = Math.Max( m_selecting_region.X, m_selecting_region.Y );
                                int er_lastv;
                                if ( m_selected_curve == CurveType.Pitch ) {
                                    er_lastv = m_manager.VsqFile.getPitchCurve( m_manager.Selected ).getValue( er_start );
                                } else {
                                    er_lastv = m_manager.VsqFile.Track[m_manager.Selected].getCurve( m_selected_curve.Name ).getValue( er_start );
                                }
                                er_edit.Add( new BPPair( er_start, er_lastv ) );
                                er_edit.Add( new BPPair( er_end, er_lastv ) );
                                CadenciiCommand run_eraser = new CadenciiCommand( VsqCommand.generateCommandTrackEditCurve( m_manager.Selected, m_selected_curve.Name, er_edit ) );
                                ExecuteCommand( run_eraser, true );
                                if ( m_selected_curve == CurveType.Pitch ) {
                                    m_manager.VsqFile.reflectPitch();
                                }
                                #endregion
                            }
                            #endregion
                        } else if ( !m_manager.IsCurveMode && m_manager.SelectedTool == EditTool.Pencil ) {
                            #region Pencil
#if DEBUG
                            Common.DebugWriteLine( "    Pencil" );
                            Common.DebugWriteLine( "    m_selected_curve=" + m_selected_curve );
#endif
                            if ( m_pencil_moved ) {
                                int stdx = m_manager.StartToDrawX;
                                if ( m_selected_curve == CurveType.VEL || m_selected_curve == CurveType.Accent || m_selected_curve == CurveType.Decay ) {
                                    #region VEL Accent Decay
                                    int start = m_mouse_trace.Keys[0];
                                    int end = m_mouse_trace.Keys[m_mouse_trace.Count - 1];
                                    start = ClockFromXCoord( start - stdx );
                                    end = ClockFromXCoord( end - stdx );
#if DEBUG
                                    Common.DebugWriteLine( "        start=" + start );
                                    Common.DebugWriteLine( "        end=" + end );
#endif
                                    Dictionary<int, int> velocity = new Dictionary<int, int>();
                                    for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                                        VsqEvent ve = (VsqEvent)itr.next();
                                        if ( start <= ve.Clock && ve.Clock < end ) {
                                            for ( int i = 0; i < m_mouse_trace.Keys.Count - 1; i++ ) {
                                                int key0 = m_mouse_trace.Keys[i];
                                                int key1 = m_mouse_trace.Keys[i + 1];
                                                int key0_clock = ClockFromXCoord( key0 - stdx );
                                                int key1_clock = ClockFromXCoord( key1 - stdx );
#if DEBUG
                                                Common.DebugWriteLine( "        key0,key1=" + key0 + "," + key1 );
#endif
                                                if ( key0_clock < ve.Clock && ve.Clock < key1_clock ) {
                                                    int key0_value = ValueFromYCoord( m_mouse_trace[key0] );
                                                    int key1_value = ValueFromYCoord( m_mouse_trace[key1] );
                                                    float a = (key1_value - key0_value) / (float)(key1_clock - key0_clock);
                                                    float b = key0_value - a * key0_clock;
                                                    int new_value = (int)(a * ve.Clock + b);
                                                    if ( velocity.ContainsKey( ve.InternalID ) ) {
                                                        velocity[ve.InternalID] = new_value;
                                                    } else {
                                                        velocity.Add( ve.InternalID, new_value );
                                                    }
                                                } else if ( key0_clock == ve.Clock ) {
                                                    if ( velocity.ContainsKey( ve.InternalID ) ) {
                                                        velocity[ve.InternalID] = ValueFromYCoord( m_mouse_trace[key0] );
                                                    } else {
                                                        velocity.Add( ve.InternalID, ValueFromYCoord( m_mouse_trace[key0] ) );
                                                    }
                                                } else if ( key1_clock == ve.Clock ) {
                                                    if ( velocity.ContainsKey( ve.InternalID ) ) {
                                                        velocity[ve.InternalID] = ValueFromYCoord( m_mouse_trace[key1] );
                                                    } else {
                                                        velocity.Add( ve.InternalID, ValueFromYCoord( m_mouse_trace[key1] ) );
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    if ( velocity.Count > 0 ) {
                                        List<KeyValuePair<int, int>> cpy = new List<KeyValuePair<int, int>>();
                                        foreach ( KeyValuePair<int, int> item in velocity ) {
                                            cpy.Add( item );
                                        }
                                        CadenciiCommand run = null;
                                        if ( m_selected_curve == CurveType.VEL ) {
                                            run = new CadenciiCommand( VsqCommand.generateCommandEventChangeVelocity( m_manager.Selected, cpy ) );
                                        } else if ( m_selected_curve == CurveType.Accent ) {
                                            run = new CadenciiCommand( VsqCommand.generateCommandEventChangeAccent( m_manager.Selected, cpy ) );
                                        } else if ( m_selected_curve == CurveType.Decay ) {
                                            run = new CadenciiCommand( VsqCommand.generateCommandEventChangeDecay( m_manager.Selected, cpy ) );
                                        }
                                        ExecuteCommand( run, true );
                                    }
                                    #endregion
                                } else if ( m_selected_curve == CurveType.VibratoRate || m_selected_curve == CurveType.VibratoDepth ) {
                                    #region VibratoRate || VibratoDepth
                                    int step_clock = AppManager.EditorConfig.ControlCurveResolution.Value;
                                    int step_px = (int)(step_clock * m_manager.ScaleX);
                                    if ( step_px <= 0 ) {
                                        step_px = 1;
                                    }
                                    int start = m_mouse_trace.Keys[0];
                                    int end = m_mouse_trace.Keys[m_mouse_trace.Count - 1];
#if DEBUG
                                    Common.DebugWriteLine( "    start,end=" + start + " " + end );
#endif
                                    List<int> internal_ids = new List<int>();
                                    List<VsqID> items = new List<VsqID>();
                                    for( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                                        VsqEvent ve = (VsqEvent)itr.next();
                                        int cl_vib_start = ve.Clock + ve.ID.VibratoDelay;
                                        float cl_vib_length = ve.ID.Length - ve.ID.VibratoDelay;
                                        int vib_start = XCoordFromClocks( cl_vib_start ) + stdx;          // 仮想スクリーン上の、ビブラートの描画開始位置
                                        int vib_end = XCoordFromClocks( ve.Clock + ve.ID.Length ) + stdx; // 仮想スクリーン上の、ビブラートの描画終了位置
                                        int chk_start = Math.Max( vib_start, start );       // マウスのトレースと、ビブラートの描画範囲がオーバーラップしている部分を検出
                                        int chk_end = Math.Min( vib_end, end );
                                        float add_min = (ClockFromXCoord( chk_start - stdx ) - cl_vib_start) / cl_vib_length;
                                        float add_max = (ClockFromXCoord( chk_end - stdx ) - cl_vib_start) / cl_vib_length;
#if DEBUG
                                        Common.DebugWriteLine( "    vib_start,vib_end=" + vib_start + " " + vib_end );
                                        Common.DebugWriteLine( "    chk_start,chk_end=" + chk_start + " " + chk_end );
                                        Common.DebugWriteLine( "    add_min,add_max=" + add_min + " " + add_max );
#endif
                                        if ( chk_start < chk_end ) {    // オーバーラップしている。
                                            List<ValuePair<float, int>> edit = new List<ValuePair<float, int>>();
                                            for ( int i = chk_start; i < chk_end; i += step_px ) {
                                                if ( m_mouse_trace.ContainsKey( i ) ) {
                                                    edit.Add( new ValuePair<float, int>( (ClockFromXCoord( i - stdx ) - cl_vib_start) / cl_vib_length,
                                                                                            ValueFromYCoord( m_mouse_trace[i] ) ) );
                                                } else {
                                                    for ( int j = 0; j < m_mouse_trace.Keys.Count - 1; j++ ) {
                                                        if ( m_mouse_trace.Keys[j] <= i && i < m_mouse_trace.Keys[j + 1] ) {
                                                            int key0 = m_mouse_trace.Keys[j];
                                                            int key1 = m_mouse_trace.Keys[j + 1];
                                                            float a = (m_mouse_trace[key1] - m_mouse_trace[key0]) / (float)(key1 - key0);
                                                            int newy = (int)(m_mouse_trace[key0] + a * (i - key0));
                                                            int clock = ClockFromXCoord( i - stdx );
                                                            int value = ValueFromYCoord( newy );
                                                            if ( value < min ) {
                                                                value = min;
                                                            } else if ( max < value ) {
                                                                value = max;
                                                            }
                                                            edit.Add( new ValuePair<float, int>( (clock - cl_vib_start) / cl_vib_length, value ) );
                                                            break;
                                                        }
                                                    }
                                                }
                                            }
                                            int c = ClockFromXCoord( end - stdx );
                                            float lastx = (c - cl_vib_start) / cl_vib_length;
                                            if ( 0 <= lastx && lastx <= 1 ) {
                                                int v = ValueFromYCoord( m_mouse_trace[end] );
                                                if ( v < min ) {
                                                    v = min;
                                                } else if ( max < v ) {
                                                    v = max;
                                                }
                                                edit.Add( new ValuePair<float, int>( lastx, v ) );
                                            }
                                            if ( ve.ID.VibratoHandle != null ) {
                                                VibratoBPList target = null;
                                                if ( m_selected_curve == CurveType.VibratoRate ) {
                                                    target = ve.ID.VibratoHandle.RateBP;
                                                } else {
                                                    target = ve.ID.VibratoHandle.DepthBP;
                                                }
                                                if ( target.getCount() > 0 ) {
                                                    for ( int i = 0; i < target.getCount(); i++ ) {
                                                        if ( target.getElement( i ).X < add_min || add_max < target.getElement( i ).X ) {
                                                            edit.Add( new ValuePair<float, int>( target.getElement( i ).X,
                                                                                                     target.getElement( i ).Y ) );
                                                        }
                                                    }
                                                }
                                                edit.Sort();
                                                VsqID id = (VsqID)ve.ID.Clone();
                                                float[] bpx = new float[edit.Count];
                                                int[] bpy = new int[edit.Count];
                                                for ( int i = 0; i < edit.Count; i++ ) {
                                                    bpx[i] = edit[i].Key;
                                                    bpy[i] = edit[i].Value;
                                                }
                                                if ( m_selected_curve == CurveType.VibratoDepth ) {
                                                    id.VibratoHandle.DepthBP = new VibratoBPList( bpx, bpy );
                                                } else {
                                                    id.VibratoHandle.RateBP = new VibratoBPList( bpx, bpy );
                                                }
                                                internal_ids.Add( ve.InternalID );
                                                items.Add( id );
                                            }
                                        }
                                    }
                                    if ( internal_ids.Count > 0 ) {
                                        CadenciiCommand run = new CadenciiCommand(
                                            VsqCommand.generateCommandEventChangeIDContaintsRange( m_manager.Selected,
                                                                                                   internal_ids.ToArray(),
                                                                                                   items.ToArray() ) );
                                        ExecuteCommand( run, true );
                                    }
                                    #endregion
                                } else {
                                    #region Other Curves
                                    List<BPPair> edit = new List<BPPair>();
                                    int step_clock = AppManager.EditorConfig.ControlCurveResolution.Value;
                                    int step_px = (int)(step_clock * m_manager.ScaleX);
                                    if ( step_px <= 0 ) {
                                        step_px = 1;
                                    }
                                    int start = m_mouse_trace.Keys[0];
                                    int end = m_mouse_trace.Keys[m_mouse_trace.Count - 1];
                                    int last = start;
#if DEBUG
                                    Common.DebugWriteLine( "    start=" + start );
                                    Common.DebugWriteLine( "    end=" + end );
#endif
                                    for ( int i = start; i <= end; i += step_px ) {
                                        if ( m_mouse_trace.ContainsKey( i ) ) {
                                            int clock = ClockFromXCoord( i - stdx );
                                            int value = ValueFromYCoord( m_mouse_trace[i] );
                                            if ( value < min ) {
                                                value = min;
                                            } else if ( max < value ) {
                                                value = max;
                                            }
                                            edit.Add( new BPPair( clock, value ) );
                                        } else {
                                            for ( int j = 0; j < m_mouse_trace.Keys.Count - 1; j++ ) {
                                                if ( m_mouse_trace.Keys[j] <= i && i < m_mouse_trace.Keys[j + 1] ) {
                                                    int key0 = m_mouse_trace.Keys[j];
                                                    int key1 = m_mouse_trace.Keys[j + 1];
                                                    float a = (m_mouse_trace[key1] - m_mouse_trace[key0]) / (float)(key1 - key0);
                                                    int newy = (int)(m_mouse_trace[key0] + a * (i - key0));
                                                    int clock = ClockFromXCoord( i - stdx );
                                                    int value = ValueFromYCoord( newy, max, min );
                                                    if ( value < min ) {
                                                        value = min;
                                                    } else if ( max < value ) {
                                                        value = max;
                                                    }
                                                    edit.Add( new BPPair( clock, value ) );
                                                    break;
                                                }
                                            }
                                        }
                                    }
                                    if ( (end - start) % step_px != 0 ) {
                                        int clock = ClockFromXCoord( end - stdx );
                                        int value = ValueFromYCoord( m_mouse_trace[end] );
                                        if ( value < min ) {
                                            value = min;
                                        } else if ( max < value ) {
                                            value = max;
                                        }
                                        edit.Add( new BPPair( clock, value ) );
                                    }
                                    int end_clock = ClockFromXCoord( end - stdx );
                                    int last_value = 0;
                                    if ( m_selected_curve == CurveType.Pitch ) {
                                        last_value = m_manager.VsqFile.getPitchCurve( m_manager.Selected ).getValue( end_clock );
                                    } else {
                                        last_value = m_manager.VsqFile.Track[m_manager.Selected].getCurve( m_selected_curve.Name ).getValue( end_clock );
                                    }
                                    edit[edit.Count - 1].Value = last_value;
                                    CadenciiCommand pen_run = new CadenciiCommand( VsqCommand.generateCommandTrackEditCurve( m_manager.Selected, m_selected_curve.Name, edit ) );
                                    ExecuteCommand( pen_run, true );
                                    if ( m_selected_curve == CurveType.Pitch ) {
                                        m_manager.VsqFile.reflectPitch();
                                    }
                                    #endregion
                                }
                            }
                            m_mouse_trace = null;
                            #endregion
                        } else if ( !m_manager.IsCurveMode && m_manager.SelectedTool == EditTool.Line ) {
                            #region Line
                            int x0 = m_line_start.X;
                            int y0 = m_line_start.Y;
                            int x1 = ClockFromXCoord( e.X );
                            int y1 = ValueFromYCoord( e.Y );
                            if ( y1 < min ) {
                                y1 = min;
                            } else if ( max < y1 ) {
                                y1 = max;
                            }
                            if ( x1 < x0 ) {
                                int b = x0;
                                x0 = x1;
                                x1 = b;
                                b = y0;
                                y0 = y1;
                                y1 = b;
                            }
                            float a0 = (y1 - y0) / (float)(x1 - x0);
                            float b0 = y0 - a0 * x0;
                            if ( m_selected_curve == CurveType.VEL || m_selected_curve == CurveType.Accent || m_selected_curve == CurveType.Decay ) {
                                List<KeyValuePair<int, int>> velocity = new List<KeyValuePair<int, int>>();
                                for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                                    VsqEvent ve = (VsqEvent)itr.next();
                                    if ( x0 <= ve.Clock && ve.Clock < x1 ) {
                                        int new_value = (int)(a0 * ve.Clock + b0);
                                        velocity.Add( new KeyValuePair<int, int>( ve.InternalID, new_value ) );
                                    }
                                }
                                CadenciiCommand run = null;
                                if ( velocity.Count > 0 ) {
                                    if ( m_selected_curve == CurveType.VEL ) {
                                        run = new CadenciiCommand( VsqCommand.generateCommandEventChangeVelocity( m_manager.Selected, velocity ) );
                                    } else if ( m_selected_curve == CurveType.Accent ) {
                                        run = new CadenciiCommand( VsqCommand.generateCommandEventChangeAccent( m_manager.Selected, velocity ) );
                                    } else if ( m_selected_curve == CurveType.Decay ) {
                                        run = new CadenciiCommand( VsqCommand.generateCommandEventChangeDecay( m_manager.Selected, velocity ) );
                                    }
                                }
                                if ( run != null ) {
                                    ExecuteCommand( run, true );
                                }
                            } else if ( m_selected_curve == CurveType.VibratoDepth || m_selected_curve == CurveType.VibratoRate ) {
                                int stdx = m_manager.StartToDrawX;
                                int step_clock = AppManager.EditorConfig.ControlCurveResolution.Value;
                                int cl_start = x0;
                                int cl_end = x1;
#if DEBUG
                                Common.DebugWriteLine( "    cl_start,cl_end=" + cl_start + " " + cl_end );
#endif
                                List<int> internal_ids = new List<int>();
                                List<VsqID> items = new List<VsqID>();
                                for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                                    VsqEvent ve = (VsqEvent)itr.next();
                                    int cl_vib_start = ve.Clock + ve.ID.VibratoDelay;
                                    int cl_vib_length = ve.ID.Length - ve.ID.VibratoDelay;
                                    int cl_vib_end = cl_vib_start + cl_vib_length;
                                    //int vib_start = XCoordFromClocks( cl_vib_start ) + stdx;          // 仮想スクリーン上の、ビブラートの描画開始位置
                                    //int vib_end = XCoordFromClocks( ve.Clock + ve.ID.Length ) + stdx; // 仮想スクリーン上の、ビブラートの描画終了位置
                                    int chk_start = Math.Max( cl_vib_start, cl_start );       // マウスのトレースと、ビブラートの描画範囲がオーバーラップしている部分を検出
                                    int chk_end = Math.Min( cl_vib_end, cl_end );
                                    float add_min = (chk_start - cl_vib_start) / (float)cl_vib_length;
                                    float add_max = (chk_end - cl_vib_start) / (float)cl_vib_length;
#if DEBUG
                                    Common.DebugWriteLine( "    cl_vib_start,cl_vib_end=" + cl_vib_start + " " + cl_vib_end );
                                    Common.DebugWriteLine( "    chk_start,chk_end=" + chk_start + " " + chk_end );
                                    Common.DebugWriteLine( "    add_min,add_max=" + add_min + " " + add_max );
#endif
                                    if ( chk_start < chk_end ) {    // オーバーラップしている。
                                        List<ValuePair<float, int>> edit = new List<ValuePair<float, int>>();
                                        for ( int i = chk_start; i < chk_end; i += step_clock ) {
                                            float x = (i - cl_vib_start) / (float)cl_vib_length;
                                            if ( 0 <= x && x <= 1 ) {
                                                int y = (int)(a0 * i + b0);
                                                edit.Add( new ValuePair<float, int>( x, y ) );
                                            }
                                        }
                                        edit.Add( new ValuePair<float, int>( (chk_end - cl_vib_start) / (float)cl_vib_length, (int)(a0 * chk_end + b0) ) );
                                        if ( ve.ID.VibratoHandle != null ) {
                                            VibratoBPList target = null;
                                            if ( m_selected_curve == CurveType.VibratoRate ) {
                                                target = ve.ID.VibratoHandle.RateBP;
                                            } else {
                                                target = ve.ID.VibratoHandle.DepthBP;
                                            }
                                            if ( target.getCount() > 0 ) {
                                                for ( int i = 0; i < target.getCount(); i++ ) {
                                                    if ( target.getElement( i ).X < add_min || add_max < target.getElement( i ).X ) {
                                                        edit.Add( new ValuePair<float, int>( target.getElement( i ).X,
                                                                                                 target.getElement( i ).Y ) );
                                                    }
                                                }
                                            }
                                            edit.Sort();
                                            VsqID id = (VsqID)ve.ID.Clone();
                                            float[] bpx = new float[edit.Count];
                                            int[] bpy = new int[edit.Count];
                                            for ( int i = 0; i < edit.Count; i++ ) {
                                                bpx[i] = edit[i].Key;
                                                bpy[i] = edit[i].Value;
                                            }
                                            if ( m_selected_curve == CurveType.VibratoDepth ) {
                                                id.VibratoHandle.DepthBP = new VibratoBPList( bpx, bpy );
                                            } else {
                                                id.VibratoHandle.RateBP = new VibratoBPList( bpx, bpy );
                                            }
                                            internal_ids.Add( ve.InternalID );
                                            items.Add( id );
                                        }
                                    }
                                }
                                if ( internal_ids.Count > 0 ) {
                                    CadenciiCommand run = new CadenciiCommand(
                                        VsqCommand.generateCommandEventChangeIDContaintsRange( m_manager.Selected,
                                                                                        internal_ids.ToArray(),
                                                                                        items.ToArray() ) );
                                    ExecuteCommand( run, true );
                                }
                            } else {
                                List<BPPair> edit = new List<BPPair>();
                                int step_clock = AppManager.EditorConfig.ControlCurveResolution.Value;
                                for ( int i = x0; i <= x1; i += step_clock ) {
                                    int y = (int)(a0 * i + b0);
                                    edit.Add( new BPPair( i, y ) );
                                }
                                int lasty;
                                if ( m_selected_curve == CurveType.Pitch ) {
                                    lasty = m_manager.VsqFile.getPitchCurve( m_manager.Selected ).getValue( x1 );
                                } else {
                                    lasty = m_manager.VsqFile.Track[m_manager.Selected].getCurve( m_selected_curve.Name ).getValue( x1 );
                                }
                                if ( x1 == edit[edit.Count - 1].Clock ) {
                                    edit[edit.Count - 1].Value = lasty;
                                } else {
                                    edit.Add( new BPPair( x1, lasty ) );
                                }
                                CadenciiCommand pen_run = new CadenciiCommand( VsqCommand.generateCommandTrackEditCurve( m_manager.Selected, m_selected_curve.Name, edit ) );
                                ExecuteCommand( pen_run, true );
                                if ( m_selected_curve == CurveType.Pitch ) {
                                    m_manager.VsqFile.reflectPitch();
                                }
                            }
                            #endregion
                        }
                    }
                    m_mouse_downed = false;
                } else if ( m_mouse_down_mode == MouseDownMode.SingerList ) {
                    if ( m_mouse_moved ) {
                        int count = m_manager.SelectedEvent.Count;
                        if ( count > 0 ) {
                            int[] ids = new int[count];
                            int[] clocks = new int[count];
                            VsqID[] values = new VsqID[count];
                            int i = -1;
                            bool is_valid = true;
                            bool contains_first_singer = false;
                            int premeasure = m_manager.VsqFile.getPreMeasureClocks();
                            foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ) {
                                i++;
                                ids[i] = item.Original.InternalID;
                                clocks[i] = item.Editing.Clock;
                                values[i] = item.Original.ID;
                                if ( clocks[i] < premeasure ) {
                                    is_valid = false;
                                    // breakしてはいけない。clock=0のを確実に検出するため
                                }
                                if ( item.Original.Clock == 0 ) {
                                    contains_first_singer = true;
                                    break;
                                }
                            }
                            if ( contains_first_singer ) {
                                System.Media.SystemSounds.Asterisk.Play();
                            } else {
                                if ( !is_valid ) {
                                    int tmin = clocks[0];
                                    for ( int j = 1; j < count; j++ ) {
                                        tmin = Math.Min( tmin, clocks[j] );
                                    }
                                    int dclock = premeasure - tmin;
                                    for ( int j = 0; j < count; j++ ) {
                                        clocks[j] += dclock;
                                    }
                                    System.Media.SystemSounds.Asterisk.Play();
                                }
                                bool changed = false;
                                for ( int j = 0; j < ids.Length; j++ ) {
                                    foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ) {
                                        if ( item.Original.InternalID == ids[j] && item.Original.Clock != clocks[j] ) {
                                            changed = true;
                                            break;
                                        }
                                    }
                                    if ( changed ) {
                                        break;
                                    }
                                }
                                if ( changed ) {
                                    CadenciiCommand run = new CadenciiCommand(
                                        VsqCommand.generateCommandEventChangeClockAndIDContaintsRange( m_manager.Selected, ids, clocks, values ) );
                                    ExecuteCommand( run, true );
                                }
                            }
                        }
                    }
                } else if ( m_mouse_down_mode == MouseDownMode.VelEdit ) {
                    int count = m_veledit_selected.Count;
                    int[] ids = new int[count];
                    VsqID[] values = new VsqID[count];
                    int i = -1;
                    foreach ( int id in m_veledit_selected.Keys ) {
                        i++;
                        ids[i] = id;
                        values[i] = (VsqID)m_veledit_selected[id].Editing.ID.Clone();
                    }
                    CadenciiCommand run = new CadenciiCommand( VsqCommand.generateCommandEventChangeIDContaintsRange( m_manager.Selected, ids, values ) );
                    ExecuteCommand( run, true );
                    if ( m_veledit_selected.Count == 1 ) {
                        m_manager.ClearSelectedEvent();
                        m_manager.AddSelectedEvent( m_veledit_last_selectedid );
                    }
                }
            }
            m_mouse_down_mode = MouseDownMode.None;
            this.Invalidate();
        }

        private void TrackSelector_MouseHover( object sender, EventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "TrackSelector_MouseHover" );
            Common.DebugWriteLine( "    m_mouse_downed=" + m_mouse_downed );
#endif
            if ( m_mouse_downed && !m_pencil_moved && m_manager.SelectedTool == EditTool.Pencil && 
                 m_selected_curve != CurveType.VEL && m_selected_curve != CurveType.Accent && m_selected_curve != CurveType.Decay ) {
                Point mouse = this.PointToClient( Control.MousePosition );
                int clock = ClockFromXCoord( mouse.X );
                int value = ValueFromYCoord( mouse.Y );
                int min = m_selected_curve.Minimum;
                int max = m_selected_curve.Maximum;
                if ( value < min ) {
                    value = min;
                } else if ( max < value ) {
                    value = max;
                }
                if ( m_selected_curve == CurveType.VibratoRate || m_selected_curve == CurveType.VibratoDepth ) {
                    // マウスの位置がビブラートの範囲かどうかを調べる
                    float x = -1f;
                    VsqID edited = null;
                    int event_id = -1;
                    for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getNoteEventIterator(); itr.hasNext(); ) {
                        VsqEvent ve = (VsqEvent)itr.next();
                        if ( ve.ID.VibratoHandle == null ){
                            continue;
                        }
                        int cl_vib_start = ve.Clock + ve.ID.VibratoDelay;
                        int cl_vib_end = ve.Clock + ve.ID.Length;
                        if ( cl_vib_start <= clock && clock < cl_vib_end ) {
                            x = (clock - cl_vib_start) / (float)(cl_vib_end - cl_vib_start);
                            edited = (VsqID)ve.ID.Clone();
                            event_id = ve.InternalID;
                            break;
                        }
                    }
#if DEBUG
                    Common.DebugWriteLine( "    x=" + x );
#endif
                    if ( 0f <= x && x <= 1f ) {
                        if ( m_selected_curve == CurveType.VibratoRate ) {
                            if ( x == 0f ) {
                                edited.VibratoHandle.StartRate = value;
                            } else {
                                if ( edited.VibratoHandle.RateBP.getCount() <= 0 ) {
                                    edited.VibratoHandle.RateBP = new VibratoBPList( new float[] { x },
                                                                                     new int[] { value } );
                                } else {
                                    List<float> xs = new List<float>();
                                    List<int> vals = new List<int>();
                                    bool first = true;
                                    for ( int i = 0; i < edited.VibratoHandle.RateBP.getCount(); i++ ) {
                                        if ( edited.VibratoHandle.RateBP.getElement( i ).X < x ) {
                                            xs.Add( edited.VibratoHandle.RateBP.getElement( i ).X );
                                            vals.Add( edited.VibratoHandle.RateBP.getElement( i ).Y );
                                        } else if ( edited.VibratoHandle.RateBP.getElement( i ).X == x ) {
                                            xs.Add( x );
                                            vals.Add( value );
                                            first = false;
                                        } else {
                                            if ( first ) {
                                                xs.Add( x );
                                                vals.Add( value );
                                                first = false;
                                            }
                                            xs.Add( edited.VibratoHandle.RateBP.getElement( i ).X );
                                            vals.Add( edited.VibratoHandle.RateBP.getElement( i ).Y );
                                        }
                                    }
                                    if ( first ) {
                                        xs.Add( x );
                                        vals.Add( value );
                                    }
                                    edited.VibratoHandle.RateBP = new VibratoBPList( xs.ToArray(), vals.ToArray() );
                                }
                            }
                        } else {
                            if ( x == 0f ) {
                                edited.VibratoHandle.StartDepth = value;
                            } else {
                                if ( edited.VibratoHandle.DepthBP.getCount() <= 0 ) {
                                    edited.VibratoHandle.DepthBP = new VibratoBPList( new float[] { x },
                                                                                      new int[] { value } );
                                } else {
                                    List<float> xs = new List<float>();
                                    List<int> vals = new List<int>();
                                    bool first = true;
                                    for ( int i = 0; i < edited.VibratoHandle.DepthBP.getCount(); i++ ) {
                                        if ( edited.VibratoHandle.DepthBP.getElement( i ).X < x ) {
                                            xs.Add( edited.VibratoHandle.DepthBP.getElement( i ).X );
                                            vals.Add( edited.VibratoHandle.DepthBP.getElement( i ).Y );
                                        } else if ( edited.VibratoHandle.DepthBP.getElement( i ).X == x ) {
                                            xs.Add( x );
                                            vals.Add( value );
                                            first = false;
                                        } else {
                                            if ( first ) {
                                                xs.Add( x );
                                                vals.Add( value );
                                                first = false;
                                            }
                                            xs.Add( edited.VibratoHandle.DepthBP.getElement( i ).X );
                                            vals.Add( edited.VibratoHandle.DepthBP.getElement( i ).Y );
                                        }
                                    }
                                    if ( first ) {
                                        xs.Add( x );
                                        vals.Add( value );
                                    }
                                    edited.VibratoHandle.DepthBP = new VibratoBPList( xs.ToArray(), vals.ToArray() );
                                }
                            }
                        }
                        CadenciiCommand run = new CadenciiCommand( VsqCommand.generateCommandEventChangeIDContaints( m_manager.Selected,
                                                                                                              event_id,
                                                                                                              edited ) );
                        ExecuteCommand( run, true );
                    }
                } else {
                    List<BPPair> edit = new List<BPPair>();
                    edit.Add( new BPPair( clock, value ) );
                    CadenciiCommand run = new CadenciiCommand( VsqCommand.generateCommandTrackEditCurve( m_manager.Selected, 
                                                                                                         m_selected_curve.Name, 
                                                                                                         edit ) );
                    ExecuteCommand( run, true );
                }
            } else if ( m_mouse_down_mode == MouseDownMode.VelWaitHover ) {
#if DEBUG
                Common.DebugWriteLine( "    entered VelEdit" );
                Common.DebugWriteLine( "    m_veledit_selected.Count=" + m_veledit_selected.Count );
                Common.DebugWriteLine( "    m_veledit_last_selectedid=" + m_veledit_last_selectedid );
                Common.DebugWriteLine( "    m_veledit_selected.ContainsKey(m_veledit_last_selectedid" + m_veledit_selected.ContainsKey( m_veledit_last_selectedid ) );
#endif
                m_mouse_down_mode = MouseDownMode.VelEdit;
                this.Invalidate();
            }
        }

        private void MouseHoverEventGenerator() {
#if DEBUG
            Common.DebugWriteLine( "MouseHoverEventGenerator" );
#endif
            System.Threading.Thread.Sleep( (int)(SystemInformation.MouseHoverTime * 0.8) );
#if DEBUG
            Common.DebugWriteLine( "   firing" );
#endif
            Invoke( new EventHandler( TrackSelector_MouseHover ) );
        }

        private void TrackSelector_MouseDoubleClick( object sender, MouseEventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "TrackSelector_MouseDoubleClick" );
#endif
            Keys modifier = Control.ModifierKeys;
            int x;
            if ( 0 <= e.Y && e.Y <= Height - 2 * _OFFSET_TRACK_TAB ) {
                #region MouseDown occured on curve-pane
                if ( AppManager._KEY_LENGTH <= e.X && e.X <= Width ) {
                    if ( m_selected_curve != CurveType.VEL && m_selected_curve != CurveType.Accent && m_selected_curve != CurveType.Decay ) {
                        // check whether bezier point exists on clicked position
                        int track = m_manager.Selected;
                        int clock = ClockFromXCoord( e.X );
                        List<BezierChain> dict = m_manager.VsqFile.AttachedCurves[track - 1][m_selected_curve];
                        BezierChain target_chain = null;
                        BezierPoint target_point = null;
                        bool found = false;
                        //foreach ( BezierChain bc in dict.Values ) {
                        for ( int i = 0; i < dict.Count; i++ ) {
                            BezierChain bc = dict[i];
                            foreach ( BezierPoint bp in bc.points ) {
                                Point pt = GetScreenCoord( bp.Base );
                                Rectangle rc = new Rectangle( pt.X - _DOT_WID, pt.Y - _DOT_WID, 2 * _DOT_WID + 1, 2 * _DOT_WID + 1 );
                                if ( IsInRect( e.Location, rc ) ) {
                                    found = true;
                                    target_point = (BezierPoint)bp.Clone();
                                    target_chain = (BezierChain)bc.Clone();
                                    break;
                                }

                                if ( bp.ControlLeftType != BezierControlType.None ) {
                                    pt = GetScreenCoord( bp.ControlLeft );
                                    rc = new Rectangle( pt.X - _DOT_WID, pt.Y - _DOT_WID, 2 * _DOT_WID + 1, 2 * _DOT_WID + 1 );
                                    if ( IsInRect( e.Location, rc ) ) {
                                        found = true;
                                        target_point = (BezierPoint)bp.Clone();
                                        target_chain = (BezierChain)bc.Clone();
                                        break;
                                    }
                                }
                                if ( bp.ControlRightType != BezierControlType.None ) {
                                    pt = GetScreenCoord( bp.ControlRight );
                                    rc = new Rectangle( pt.X - _DOT_WID, pt.Y - _DOT_WID, 2 * _DOT_WID + 1, 2 * _DOT_WID + 1 );
                                    if ( IsInRect( e.Location, rc ) ) {
                                        found = true;
                                        target_point = (BezierPoint)bp.Clone();
                                        target_chain = (BezierChain)bc.Clone();
                                        break;
                                    }
                                }
                            }
                            if ( found ) {
                                break;
                            }
                        }
                        if ( found ) {
                            int chain_id = target_chain.ID;
                            BezierChain before = (BezierChain)target_chain.Clone();
                            using ( FormBezierPointEdit fbpe = new FormBezierPointEdit( this,
                                                                                        m_manager,
                                                                                        m_selected_curve,
                                                                                        chain_id,
                                                                                        target_point.ID ) ) {
                                EditingChainID = chain_id;
                                EditingPointID = target_point.ID;
                                DialogResult ret = fbpe.ShowDialog();
                                EditingChainID = -1;
                                EditingPointID = -1;
                                BezierChain after = m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].GetBezierChain( m_selected_curve, chain_id );
                                // 編集前の状態に戻す
                                CadenciiCommand revert = VsqFileEx.generateCommandReplaceBezierChain( track,
                                                                                               m_selected_curve,
                                                                                               chain_id,
                                                                                               before,
                                                                                               AppManager.EditorConfig.ControlCurveResolution.Value );
                                ExecuteCommand( revert, false );
                                if ( ret == DialogResult.OK ) {
                                    // ダイアログの結果がOKで、かつベジエ曲線が単調増加なら編集を適用
                                    if ( BezierChain.IsBezierImplicit( target_chain ) ) {
                                        CadenciiCommand run = VsqFileEx.generateCommandReplaceBezierChain( track,
                                                                                                    m_selected_curve,
                                                                                                    chain_id,
                                                                                                    after,
                                                                                                    AppManager.EditorConfig.ControlCurveResolution.Value );
                                        ExecuteCommand( run, true );
                                    }
                                }
                            }
                        }
                    }
                }
                #endregion
            } else if ( Height - 2 * _OFFSET_TRACK_TAB <= e.Y && e.Y <= Height - _OFFSET_TRACK_TAB ) {
                #region MouseDown occured on singer list
                if ( m_manager.SelectedTool != EditTool.Eraser ) {
                    VsqEvent ve = GetItemAtClickedPosition( e.Location, out x );
                    string renderer = m_manager.VsqFile.Track[m_manager.Selected].getCommon().Version;
                    if ( ve != null ) {
                        if ( ve.ID.type != VsqIDType.Singer ) {
                            return;
                        }
                        if ( m_cmenu_singer_prepared != renderer ) {
                            InitCmenuSinger( renderer );
                        }
                        TagForCMenuSinger tag = new TagForCMenuSinger();
                        tag.SingerChangeExists = true;
                        tag.InternalID = ve.InternalID;
                        cmenuSinger.Tag = tag;//                        new KeyValuePair<bool, int>( true, ve.InternalID );
                        foreach ( ToolStripMenuItem tsmi in cmenuSinger.Items ) {
                            TagForCMenuSingerDropDown tag2 = (TagForCMenuSingerDropDown)tsmi.Tag;
                            if ( tag2.SingerName == ve.ID.IconHandle.IDS ) {
                                tsmi.Checked = true;
                            } else {
                                tsmi.Checked = false;
                            }
                        }
                        cmenuSinger.Show( this, e.Location );
                    } else {
                        if ( m_cmenu_singer_prepared != renderer ) {
                            InitCmenuSinger( renderer );
                        }
                        string singer = AppManager.EditorConfig.DefaultSingerName;
                        int clock = ClockFromXCoord( e.X );
                        int last_clock = 0;
                        for ( Iterator itr = m_manager.VsqFile.Track[m_manager.Selected].getSingerEventIterator(); itr.hasNext(); ) {
                            VsqEvent ve2 = (VsqEvent)itr.next();
                            if ( last_clock <= clock && clock < ve2.Clock ) {
                                singer = ve2.ID.IconHandle.IDS;
                                break;
                            }
                            last_clock = ve2.Clock;
                        }
                        TagForCMenuSinger tag = new TagForCMenuSinger();
                        tag.SingerChangeExists = false;
                        tag.Clock = clock;
                        cmenuSinger.Tag = tag;//                        new KeyValuePair<bool, int>( false, clock );
                        foreach ( ToolStripMenuItem tsmi in cmenuSinger.Items ) {
                            tsmi.Checked = false;
                        }
                        cmenuSinger.Show( this, e.Location );
                    }
                }
                #endregion
            }
        }

        private void InitCmenuSinger( string renderer ) {
            cmenuSinger.Items.Clear();
            //List<int> list = new List<int>();
            List<SingerConfig> items = null;
            if ( renderer.StartsWith( VSTiProxy.RENDERER_UTU0 ) ) {
                items = AppManager.EditorConfig.UtauSingers;
            } else if ( renderer.StartsWith( VSTiProxy.RENDERER_DSB2 ) ) {
                items = new List<SingerConfig>( VocaloSysUtil.getSingerConfigs1() );
            } else if ( renderer.StartsWith( VSTiProxy.RENDERER_DSB3 ) ) {
                items = new List<SingerConfig>( VocaloSysUtil.getSingerConfigs2() );
            } else {
                return;
            }
            int count = 0;
            foreach( SingerConfig sc in items ) {
                string tip = "";
                if ( renderer.StartsWith( VSTiProxy.RENDERER_UTU0 ) ){
                    //sc = AppManager.getSingerInfoUtau( i );
                    if ( sc != null ) {
                        tip = "Name: " + sc.VOICENAME +
                              "\nDirectory: " + sc.VOICEIDSTR;
                    }
                } else if ( renderer.StartsWith( VSTiProxy.RENDERER_DSB2 ) ){
                    //sc = VocaloSysUtil.getSingerInfo1( i );
                    if ( sc != null ) {
                        tip = "Original: " + VocaloSysUtil.getOriginalSinger1( sc.VOICENAME ) +
                              "\nBRE: " + sc.Breathiness +
                              "\nBRI: " + sc.Brightness +
                              "\nCLE: " + sc.Clearness +
                              "\nGEN: " + sc.GenderFactor +
                              "\nOPE: " + sc.Opening;
                    }
                } else if ( renderer.StartsWith( VSTiProxy.RENDERER_DSB3 ) ) {
                    //sc = VocaloSysUtil.getSingerInfo2( i );
                    if ( sc != null ) {
                        tip = "Original: " + VocaloSysUtil.getOriginalSinger2( sc.VOICENAME ) +
                              "\nBRE: " + sc.Breathiness +
                              "\nBRI: " + sc.Brightness +
                              "\nCLE: " + sc.Clearness +
                              "\nGEN: " + sc.GenderFactor +
                              "\nOPE: " + sc.Opening;
                    }
                }
                if ( sc != null ) {
                    ToolStripMenuItem tsmi = new ToolStripMenuItem( sc.VOICENAME );
                    TagForCMenuSingerDropDown tag = new TagForCMenuSingerDropDown();
                    tag.SingerName = sc.VOICENAME;
                    tag.ToolTipText = tip;
                    tag.ToolTipPxWidth = 0;
                    tsmi.Tag = tag;
                    tsmi.Click += new EventHandler( tsmi_Click );
                    if ( AppManager.EditorConfig.Platform == Platform.Windows ) {
                        // TODO: cmenuSinger.ItemsのToolTip。monoで実行するとMouseHoverで落ちる
                        tsmi.MouseEnter += new EventHandler( tsmi_MouseEnter );
                    }
                    cmenuSinger.Items.Add( tsmi );
                    //list.Add( i );
                    count++;
                }
            }
            cmenuSinger.VisibleChanged += new EventHandler( cmenuSinger_VisibleChanged );
            m_cmenusinger_tooltip_width = new int[count];
            //m_cmenusinger_map = list.ToArray();
            for ( int i = 0; i < count; i++ ) {
                m_cmenusinger_tooltip_width[i] = 0;
            }
            m_cmenu_singer_prepared = renderer;
        }

        private void cmenuSinger_VisibleChanged( object sender, EventArgs e ) {
            toolTip.Hide( cmenuSinger );
        }
        
        private void tsmi_MouseEnter( object sender, EventArgs e ) {
            toolTip.Hide( cmenuSinger );
        }
        
        private void tsmi_MouseHover( object sender, EventArgs e ) {
#if DEBUG
            try {
#endif
                TagForCMenuSingerDropDown tag = (TagForCMenuSingerDropDown)((ToolStripMenuItem)sender).Tag;
                string tip = tag.ToolTipText;
                string singer = tag.SingerName;

                // tooltipを表示するy座標を決める
                int y = 0;
                for ( int i = 0; i < cmenuSinger.Items.Count; i++ ) {
                    TagForCMenuSingerDropDown tag2 = (TagForCMenuSingerDropDown)cmenuSinger.Items[i].Tag;
                    string singer2 = tag2.SingerName;
                    if ( singer == singer2 ) {
                        break;
                    }
                    y += cmenuSinger.Items[i].Height;
                }

                int tip_width = tag.ToolTipPxWidth;
                Point pts = cmenuSinger.PointToScreen( new Point( 0, 0 ) );
                Rectangle rc = Screen.GetBounds( this );
                toolTip.Tag = singer;
#if DEBUG
                //Common.DebugWriteLine( "    screen widthr=" + rc.Width );
                //Common.DebugWriteLine( "    right=" + (pts.X + cmenuSinger.Width + tip_width) );
#endif
                if ( pts.X + cmenuSinger.Width + tip_width > rc.Width ) {
                    toolTip.Show( tip, cmenuSinger, new Point( -tip_width, y ), 5000 );
                } else {
                    toolTip.Show( tip, cmenuSinger, new Point( cmenuSinger.Width, y ), 5000 );
                }
#if DEBUG
            } catch ( Exception ex ) {
                Console.WriteLine( "TarckSelectro.tsmi_MouseHover; ex=" + ex );
                Common.DebugWriteLine( "TarckSelectro.tsmi_MouseHover; ex=" + ex );
            }
#endif
        }

        private struct TagForCMenuSinger {
            public bool SingerChangeExists;
            public int Clock;
            public int InternalID;
            //public string Renderer;
        }

        private struct TagForCMenuSingerDropDown {
            public string SingerName;
            public int ToolTipPxWidth;
            public string ToolTipText;
        }

        private void tsmi_Click( object sender, EventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "CmenuSingerClick" );
            Common.DebugWriteLine( "    sender.GetType()=" + sender.GetType() );
            Common.DebugWriteLine( "    ((ToolStripMenuItem)sender).Tag.GetType()=" + ((ToolStripMenuItem)sender).Tag.GetType() );
#endif
            if ( sender is ToolStripMenuItem ) {
                TagForCMenuSinger tag = (TagForCMenuSinger)cmenuSinger.Tag;
                TagForCMenuSingerDropDown tag_dditem = (TagForCMenuSingerDropDown)((ToolStripMenuItem)sender).Tag;
                string singer = tag_dditem.SingerName;
                VsqID item = null;
                if ( m_cmenu_singer_prepared.StartsWith( VSTiProxy.RENDERER_DSB2 ) ) {
                    item = VocaloSysUtil.getSingerID1( singer );
                } else if ( m_cmenu_singer_prepared.StartsWith( VSTiProxy.RENDERER_DSB3 ) ) {
                    item = VocaloSysUtil.getSingerID2( singer );
                } else if ( m_cmenu_singer_prepared.StartsWith( VSTiProxy.RENDERER_UTU0 ) ) {
                    item = AppManager.getSingerIDUtau( singer );
                }
                if ( item != null ) {
                    if ( tag.SingerChangeExists ) {
                        int id = tag.InternalID;
                        CadenciiCommand run = new CadenciiCommand(
                            VsqCommand.generateCommandEventChangeIDContaints( m_manager.Selected, id, item ) );
#if DEBUG
                        Common.DebugWriteLine( "tsmi_Click; item.IconHandle.Program" + item.IconHandle.Program );
#endif
                        ExecuteCommand( run, true );
                    } else {
                        int clock = tag.Clock;
                        VsqEvent ve = new VsqEvent( clock, item );
                        CadenciiCommand run = new CadenciiCommand( VsqCommand.generateCommandEventAdd( m_manager.Selected, ve ) );
                        ExecuteCommand( run, true );
                    }
                }
            }
        }
        
        private void toolTip_Draw( object sender, DrawToolTipEventArgs e ) {
            Rectangle rc = e.Bounds;
#if DEBUG
            Common.DebugWriteLine( "toolTip_Draw" );
            Common.DebugWriteLine( "    sender.GetType()=" + sender.GetType() );
#endif
            string singer = (string)((ToolTip)sender).Tag;
            foreach ( ToolStripItem tsi in cmenuSinger.Items ) {
                if ( tsi is ToolStripMenuItem ) {
                    TagForCMenuSingerDropDown tag = (TagForCMenuSingerDropDown)((ToolStripMenuItem)tsi).Tag;
                    if ( tag.SingerName == singer ) {
                        tag.ToolTipPxWidth = rc.Width;
                        ((ToolStripMenuItem)tsi).Tag = tag;
                        break;
                    }
                }
            }
            e.DrawBackground();
            e.DrawBorder();
            e.DrawText( TextFormatFlags.VerticalCenter | TextFormatFlags.Left | TextFormatFlags.NoFullWidthCharacterBreak );
        }

        private void TrackSelector_KeyDown( object sender, KeyEventArgs e ) {
            if ( (e.KeyCode & Keys.Space) == Keys.Space ) {
                m_spacekey_downed = true;
            }
        }

        private void TrackSelector_KeyUp( object sender, KeyEventArgs e ) {
            if ( (e.KeyCode & Keys.Space) == Keys.Space ) {
                m_spacekey_downed = false;
            }
        }

        private void cmenuCurveCommon_Click( object sender, EventArgs e ) {
            if ( sender is ToolStripMenuItem ) {
                ToolStripMenuItem tsmi = (ToolStripMenuItem)sender;
                if ( tsmi.Tag is CurveType ) {
                    CurveType curve = (CurveType)tsmi.Tag;
                    ChangeCurve( curve );
                }
            }
        }

        private void panelZoomButton_Paint( object sender, PaintEventArgs e ) {
            using ( Pen p = new Pen( Color.FromArgb( 118, 123, 138 ) ) ) {
                // 外枠
                int halfheight = panelZoomButton.Height / 2;
                e.Graphics.DrawRectangle( p, new Rectangle( 0, 0, panelZoomButton.Width - 1, panelZoomButton.Height - 1 ) );
                e.Graphics.DrawLine( p, new Point( 0, halfheight ), new Point( panelZoomButton.Width, halfheight ) );
                if ( m_selected_curve == CurveType.Pitch ) {
                    int halfwidth = panelZoomButton.Width / 2;
                    int quoterheight = panelZoomButton.Height / 4;
                    // +の文字
                    e.Graphics.DrawLine( Pens.Black, new Point( halfwidth - 4, quoterheight ), new Point( halfwidth + 4, quoterheight ) );
                    e.Graphics.DrawLine( Pens.Black, new Point( halfwidth, quoterheight - 4 ), new Point( halfwidth, quoterheight + 4 ) );

                    // -の文字
                    e.Graphics.DrawLine( Pens.Black, new Point( halfwidth - 4, quoterheight + halfheight ), new Point( halfwidth + 4, quoterheight + halfheight ) );
                }
            }
        }

        private void panelZoomButton_MouseDown( object sender, MouseEventArgs e ) {
            // Pitch表示のとき、縦方向倍率変更ボタン上のMouseDownかどうかを検査
            if ( m_selected_curve == CurveType.Pitch && e.Button == MouseButtons.Left ) {
                if ( 0 <= e.X && e.X < panelZoomButton.Width ) {
                    int halfheight = panelZoomButton.Height / 2;
                    if ( 0 <= e.Y && e.Y < halfheight ) {
                        if ( m_internal_pbs - 1 >= 1 ) {
                            m_internal_pbs--;
                            this.Invalidate();
                        }
                    } else if ( halfheight < e.Y && e.Y < panelZoomButton.Height ) {
                        if ( m_internal_pbs + 1 <= 24 ) {
                            m_internal_pbs++;
                            this.Invalidate();
                        }
                    }
                }
            }
        }
    }

}
