﻿/*
 * VsqCommand.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.VECapture is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;

namespace Boare.Lib.Vsq {

    /// <summary>
    /// 
    /// </summary>
    [Serializable]
    public class VsqCommand {
        public VsqCommandType Type;
        /// <summary>
        /// コマンドの処理内容を保持します。Args具体的な内容は、処理するクラスごとに異なります
        /// </summary>
        public object[] Args;
        /// <summary>
        /// 後続するコマンド
        /// </summary>
        public List<VsqCommand> Children = new List<VsqCommand>();
        /// <summary>
        /// このコマンドの親
        /// </summary>
        public VsqCommand Parent = null;

        /// <summary>
        /// VsqCommandはgenerateCommand*からコンストラクトしなければならない。
        /// </summary>
        public VsqCommand() {
        }

        public static VsqCommand generateCommandRoot() {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.Root;
            command.Args = null;
            return command;
        }

        public static VsqCommand generateCommandReplace( VsqFile vsq ) {
            VsqCommand command = new VsqCommand();
            command.Args = new object[1];
            command.Type = VsqCommandType.Replace;
            command.Args[0] = (VsqFile)vsq.Clone();
            return command;
        }

        public static VsqCommand generateCommandTrackReplace( int track, VsqTrack item ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.TrackReplace;
            command.Args = new object[2];
            command.Args[0] = track;
            command.Args[1] = (VsqTrack)item.Clone();
            return command;
        }

        public static VsqCommand generateCommandUpdateTimesig( int bar_count, int new_barcount, int numerator, int denominator ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.UpdateTimesig;
            command.Args = new object[4];
            command.Args[0] = bar_count;
            command.Args[1] = numerator;
            command.Args[2] = denominator;
            command.Args[3] = new_barcount;
            return command;
        }

        public static VsqCommand generateCommandUpdateTimesigRange( int[] bar_counts, int[] new_barcounts, int[] numerators, int[] denominators ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.UpdateTimesigRange;
            command.Args = new object[4];
            command.Args[0] = (int[])bar_counts.Clone();
            command.Args[1] = (int[])numerators.Clone();
            command.Args[2] = (int[])denominators.Clone();
            command.Args[3] = (int[])new_barcounts.Clone();
            return command;
        }

        public static VsqCommand generateCommandUpdateTempoRange( int[] clocks, int[] new_clocks, int[] tempos ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.UpdateTempoRange;
            command.Args = new object[3];
            command.Args[0] = (int[])clocks.Clone();
            command.Args[1] = (int[])tempos.Clone();
            command.Args[2] = (int[])new_clocks.Clone();
            return command;
        }

        public static VsqCommand generateCommandUpdateTempo( int clock, int new_clock, int tempo ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.UpdateTempo;
            command.Args = new object[3];
            command.Args[0] = clock;
            command.Args[1] = tempo;
            command.Args[2] = new_clock;
            return command;
        }

        public static VsqCommand generateCommandChangePreMeasure( int pre_measure ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.ChangePreMeasure;
            command.Args = new object[1];
            command.Args[0] = pre_measure;
            return command;
        }

        public static VsqCommand generateCommandDeleteTrack( int track ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.DeleteTrack;
            command.Args = new object[1];
            command.Args[0] = track;
            return command;
        }

        /// <summary>
        /// トラックを追加するコマンドを発行します．trackはClone()して渡さなくてもよい
        /// </summary>
        /// <param name="track"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandAddTrack( VsqTrack track, VsqMixerEntry mixer, int position ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.AddTrack;
            command.Args = new object[5];
            command.Args[0] = track;
            command.Args[1] = mixer;
            command.Args[2] = position;
            return command;
        }

        /// <summary>
        /// トラック名を変更するコマンドを作成します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="new_name"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandTrackChangeName( int track, string new_name ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.TrackChangeName;
            command.Args = new object[2];
            command.Args[0] = track;
            command.Args[1] = new_name;
            return command;
        }

        public static VsqCommand generateCommandTrackChangePlayMode( int track, int play_mode ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.TrackChangePlayMode;
            command.Args = new object[2];
            command.Args[0] = track;
            command.Args[1] = play_mode;
            return command;
        }

        /// <summary>
        /// VsqIDとClockを同時に変更するコマンドを発行します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_ids"></param>
        /// <param name="clocks"></param>
        /// <param name="values"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeClockAndIDContaintsRange( int track, int[] internal_ids, int[] clocks, VsqID[] values ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeClockAndIDContaintsRange;
            int count = internal_ids.Length;
            command.Args = new object[4];
            command.Args[0] = track;
            command.Args[1] = (int[])internal_ids.Clone();
            command.Args[2] = (int[])clocks.Clone();
            command.Args[3] = (VsqID[])values.Clone();
            return command;
        }

        /// <summary>
        /// VsqIDとClockを同時に変更するコマンドを発行します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="clock"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeClockAndIDContaints( int track, int internal_id, int clock, VsqID value ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeClockAndIDContaints;
            command.Args = new object[4];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = clock;
            command.Args[3] = (VsqID)value.Clone();
            return command;
        }

        /// <summary>
        /// VsqIDの内容を変更するコマンドを発行します。
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_ids"></param>
        /// <param name="values"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeIDContaintsRange( int track, int[] internal_ids, VsqID[] values ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeIDContaintsRange;
            command.Args = new object[3];
            command.Args[0] = track;
            command.Args[1] = (int[])internal_ids.Clone();
            VsqID[] list = new VsqID[values.Length];
            for ( int i = 0; i < values.Length; i++ ) {
                list[i] = (VsqID)values[i].Clone();
            }
            command.Args[2] = list;
            return command;
        }

        /// <summary>
        /// VsqIDの内容を変更するコマンドを発行します。
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeIDContaints( int track, int internal_id, VsqID value ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeIDContaints;
            command.Args = new object[3];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = (VsqID)value.Clone();
            return command;
        }

        /// <summary>
        /// ノートの長さを変更するコマンドを発行します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="new_clock"></param>
        /// <param name="new_length"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeClockAndLength( int track, int internal_id, int new_clock, int new_length ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeClockAndLength;
            command.Args = new object[4];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = new_clock;
            command.Args[3] = new_length;
            return command;
        }

        /// <summary>
        /// ノートの長さを変更するコマンドを発行します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="new_length"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeLength( int track, int internal_id, int new_length ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeLength;
            command.Args = new object[3];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = new_length;
            return command;
        }

        /// <summary>
        /// 指定したトラックの，音符のベロシティ(VEL)を変更するコマンドを発行します．
        /// リストvelocityには，音符を指定するInteralIDと，変更したいベロシティの値のペアを登録します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="velocity"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeVelocity( int track, List<KeyValuePair<int, int>> velocity ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeVelocity;
            command.Args = new object[2];
            command.Args[0] = track;
            List<KeyValuePair<int, int>> list = new List<KeyValuePair<int, int>>();
            foreach ( KeyValuePair<int, int> item in velocity ) {
                list.Add( new KeyValuePair<int, int>( item.Key, item.Value ) );
            }
            command.Args[1] = list;
            return command;
        }

        public static VsqCommand generateCommandEventReplace( int track, VsqEvent item ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventReplace;
            command.Args = new object[2];
            command.Args[0] = track;
            command.Args[1] = item.Clone();
            return command;
        }

        public static VsqCommand generateCommandEventReplaceRange( int track, VsqEvent[] items ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventReplaceRange;
            command.Args = new object[2];
            command.Args[0] = track;
            VsqEvent[] objs = new VsqEvent[items.Length];
            for( int i = 0; i < items.Length; i++ ){
                objs[i] = (VsqEvent)items[i].Clone();
            }
            command.Args[1] = objs;
            return command;
        }

        /// <summary>
        /// 指定したトラックの、音符のアクセント(Accent)を変更するコマンドを発行します。
        /// リストaccent_listには、音符を指定するInternalIDと、変更したいアクセント値のペアを登録します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="accent_list"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeAccent( int track, List<KeyValuePair<int, int>> accent_list ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeAccent;
            command.Args = new object[2];
            command.Args[0] = track;
            List<KeyValuePair<int, int>> list = new List<KeyValuePair<int, int>>();
            foreach ( KeyValuePair<int, int> item in accent_list ) {
                list.Add( new KeyValuePair<int, int>( item.Key, item.Value ) );
            }
            command.Args[1] = list;
            return command;
        }

        /// <summary>
        /// 指定したトラックの、音符のディケイ(Decay)を変更するコマンドを発行します。
        /// リストdecay_listには、音符を指定するInternalIDと、変更したいディケイ値のペアを登録します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="decay_list"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeDecay( int track, List<KeyValuePair<int, int>> decay_list ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeDecay;
            command.Args = new object[2];
            command.Args[0] = track;
            List<KeyValuePair<int, int>> list = new List<KeyValuePair<int, int>>();
            foreach ( KeyValuePair<int, int> item in decay_list ) {
                list.Add( new KeyValuePair<int, int>( item.Key, item.Value ) );
            }
            command.Args[1] = list;
            return command;
        }

        /// <summary>
        /// vsqファイルのカーブを編集するコマンドを発行します．
        /// </summary>
        /// <param name="track"></param>
        /// <param name="target"></param>
        /// <param name="edit"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandTrackEditCurve( int track, string target, List<BPPair> edit ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.TrackEditCurve;
            command.Args = new object[5];
            command.Args[0] = track;
            command.Args[1] = target;
            List<BPPair> copied = new List<BPPair>();
            foreach ( BPPair item in edit ) {
                copied.Add( item );
            }
            command.Args[2] = copied;
            return command;
        }

        public static VsqCommand generateCommandTrackEditCurveRange( int track, string[] targets, List<BPPair>[] edits ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.TrackEditCurveRange;
            command.Args = new object[3];
            command.Args[0] = track;
            command.Args[1] = (string[])targets.Clone();
            List<BPPair>[] cpy = new List<BPPair>[targets.Length];
            for ( int i = 0; i < edits.Length; i++ ) {
                List<BPPair> copied = new List<BPPair>();
                foreach ( BPPair item in edits[i] ) {
                    copied.Add( new BPPair( item.Clock, item.Value ) );
                }
                cpy[i] = copied;
            }
            command.Args[2] = cpy;
            return command;
        }

        /// <summary>
        /// 特定位置のイベントの歌詞と発音記号を変更するコマンドを発行します。
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="phrase"></param>
        /// <param name="phonetic_symbol"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeLyric( int track, int internal_id, string phrase, string phonetic_symbol, bool protect_symbol ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeLyric;
            command.Args = new object[5];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = phrase;
            command.Args[3] = phonetic_symbol;
            command.Args[4] = protect_symbol;
            return command;
        }

        /// <summary>
        /// ノートのクロック位置を変更するコマンドを発行します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeClock( int track, int internal_id, int value ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeClock;
            command.Args = new object[3];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = value;
            return command;
        }

        public static VsqCommand generateCommandEventDeleteRange( int track, int[] internal_ids ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventDeleteRange;
            command.Args = new object[2];
            command.Args[0] = (int[])internal_ids.Clone();
            command.Args[1] = track;
            return command;
        }

        /// <summary>
        /// ノートを削除するコマンドを発行します
        /// </summary>
        /// <param name="clock"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventDelete( int track, int internal_id ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventDelete;
            command.Args = new object[2];
            command.Args[1] = track;
            command.Args[0] = internal_id;
            return command;
        }

        public static VsqCommand generateCommandEventAddRange( int track, VsqEvent[] items ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventAddRange;
            command.Args = new object[2];
            command.Args[0] = track;
            command.Args[1] = (VsqEvent[])items.Clone();
            return command;
        }

        /// <summary>
        /// ノートを追加するコマンドを発行します。
        /// </summary>
        /// <param name="track"></param>
        /// <param name="item"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventAdd( int track, VsqEvent item ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventAdd;
            command.Args = new object[2];
            command.Args[0] = track;
            command.Args[1] = (VsqEvent)item.Clone();
            return command;
        }

        /// <summary>
        /// ノートの音程を変更するコマンドを発行します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeNote( int track, int internal_id, int note ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeNote;
            command.Args = new object[3];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = note;
            return command;
        }

        /// <summary>
        /// ノートの音程とクロックを変更するコマンドを発行します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="internal_id"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static VsqCommand generateCommandEventChangeClockAndNote( int track, int internal_id, int clock, int note ) {
            VsqCommand command = new VsqCommand();
            command.Type = VsqCommandType.EventChangeClockAndNote;
            command.Args = new object[4];
            command.Args[0] = track;
            command.Args[1] = internal_id;
            command.Args[2] = clock;
            command.Args[3] = note;
            return command;
        }
    }

}
