﻿/*
 * Preference.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Drawing;
using System.IO;
using System.Collections.Generic;
using System.Windows.Forms;

using Boare.Lib.Vsq;
using Boare.Lib.AppUtil;

namespace Boare.Cadencii {

    partial class Preference : Form {
        public enum PreferenceTab {
            Sequence,
            Another,
            Appearance,
            Operation,
            Platform,
        }

        Font m_base_font;
        Font m_screen_font;
        Font m_counter_font;
        List<int> m_program_change = null;
        private Platform m_platform = Platform.Windows;

        public Preference() {
            InitializeComponent();
            ApplyLanguage();
            tabSequence.Tag = PreferenceTab.Sequence;
            tabAnother.Tag = PreferenceTab.Another;
            tabAppearance.Tag = PreferenceTab.Appearance;
            tabOperation.Tag = PreferenceTab.Operation;
            tabPlatform.Tag = PreferenceTab.Platform;

            comboVibratoLength.Items.Clear();
            foreach ( DefaultVibratoLength dvl in Enum.GetValues( typeof( DefaultVibratoLength ) ) ) {
                comboVibratoLength.Items.Add( (int)dvl + "" );
            }
            comboVibratoLength.SelectedIndex = 1;

            comboAutoVibratoMinLength.Items.Clear();
            foreach ( AutoVibratoMinLength avml in Enum.GetValues( typeof( AutoVibratoMinLength ) ) ) {
                comboAutoVibratoMinLength.Items.Add( (int)avml + "" );
            }
            comboAutoVibratoMinLength.SelectedIndex = 0;

            comboAutoVibratoType.Items.Clear();
            foreach ( VibratoType vt in Enum.GetValues( typeof( VibratoType ) ) ) {
                comboAutoVibratoType.Items.Add( VibratoTypeUtil.ToString( vt ) );
            }
            comboAutoVibratoType.SelectedIndex = 0;

            comboDynamics.Items.Clear();
            comboAmplitude.Items.Clear();
            comboPeriod.Items.Clear();
            foreach ( ClockResolution cr in ClockResolution.GetEnumerator() ) {
                comboDynamics.Items.Add( cr.ToString() );
                comboAmplitude.Items.Add( cr.ToString() );
                comboPeriod.Items.Add( cr.ToString() );
            }
            comboDynamics.SelectedIndex = 0;
            comboAmplitude.SelectedIndex = 0;
            comboPeriod.SelectedIndex = 0;

            comboLanguage.Items.Clear();
            string[] list = Messaging.GetRegisteredLanguage();
            int index = 0;
            comboLanguage.Items.Add( "Default" );
            int count = 0;
            foreach ( string s in list ) {
                count++;
                comboLanguage.Items.Add( s );
                if ( s == Messaging.Language ) {
                    index = count;
                }
            }
            comboLanguage.SelectedIndex = index;

            Dictionary<int, SingerConfig> dict = VsqUtil.GetSingerConfigs();
            m_program_change = new List<int>();
            comboDefualtSinger.Items.Clear();
            foreach ( KeyValuePair<int, SingerConfig> kvp in dict ) {
                m_program_change.Add( kvp.Key );
                comboDefualtSinger.Items.Add( kvp.Value.VOICENAME );
            }
            comboDefualtSinger.Enabled = (comboDefualtSinger.Items.Count > 0);
            if ( comboDefualtSinger.Items.Count > 0 ) {
                comboDefualtSinger.SelectedIndex = 0;
            }

            comboPlatform.Items.Clear();
#if DEBUG
            Common.DebugWriteLine( "Preference.ctor()" );
            Common.DebugWriteLine( "    Environment.OSVersion.Platform=" + Environment.OSVersion.Platform );
#endif
            PlatformID platform = Environment.OSVersion.Platform;
            if ( platform == PlatformID.Win32NT ||
                 platform == PlatformID.Win32S ||
                 platform == PlatformID.Win32Windows ||
                 platform == PlatformID.WinCE ) {
                comboPlatform.Items.Add( Platform.Windows + "" );
                comboPlatform.Enabled = false;
                chkCommandKeyAsControl.Enabled = false;
            } else {
                foreach ( Platform p in Enum.GetValues( typeof( Platform ) ) ) {
                    comboPlatform.Items.Add( p + "" );
                }
            }
        }

        /// <summary>
        /// 現在表示されているタブページを取得、または設定します
        /// </summary>
        public PreferenceTab CurrentTab {
            get {
                return (PreferenceTab)tabPreference.SelectedTab.Tag;
            }
            set {
                foreach( TabPage tp in tabPreference.TabPages ){
                    if ( (PreferenceTab)tp.Tag == value ) {
                        tabPreference.SelectedTab = tp;
                        break;
                    }
                }
            }
        }


        public bool CurveVisibleVel {
            get {
                return chkVel.Checked;
            }
            set {
                chkVel.Checked = value;
            }
        }

        public bool CurveVisibleAccent {
            get {
                return chkAccent.Checked;
            }
            set {
                chkAccent.Checked = value;
            }
        }

        public bool CurveVisibleDecay {
            get {
                return chkDecay.Checked;
            }
            set {
                chkDecay.Checked = value;
            }
        }

        public bool CurveVisibleVibratoRate {
            get {
                return chkVibratoRate.Checked;
            }
            set {
                chkVibratoRate.Checked = value;
            }
        }

        public bool CurveVisibleVibratoDepth {
            get {
                return chkVibratoDepth.Checked;
            }
            set {
                chkVibratoDepth.Checked = value;
            }
        }

        public bool CurveVisibleDyn {
            get {
                return chkDyn.Checked;
            }
            set {
                chkDyn.Checked = value;
            }
        }

        public bool CurveVisibleBre {
            get {
                return chkBre.Checked;
            }
            set {
                chkBre.Checked = value;
            }
        }

        public bool CurveVisibleBri {
            get {
                return chkBri.Checked;
            }
            set {
                chkBri.Checked = value;
            }
        }

        public bool CurveVisibleCle {
            get {
                return chkCle.Checked;
            }
            set {
                chkCle.Checked = value;
            }
        }

        public bool CurveVisibleOpe {
            get {
                return chkOpe.Checked;
            }
            set {
                chkOpe.Checked = value;
            }
        }

        public bool CurveVisiblePor {
            get {
                return chkPor.Checked;
            }
            set {
                chkPor.Checked = value;
            }
        }

        public bool CurveVisibleGen {
            get {
                return chkGen.Checked;
            }
            set {
                chkGen.Checked = value;
            }
        }

        public bool CurveVisiblePit {
            get {
                return chkPit.Checked;
            }
            set {
                chkPit.Checked = value;
            }
        }

        public bool CurveVisiblePbs {
            get {
                return chkPbs.Checked;
            }
            set {
                chkPbs.Checked = value;
            }
        }

        public bool CurveSelectingQuantized {
            get {
                return chkCurveSelectingQuantized.Checked;
            }
            set {
                chkCurveSelectingQuantized.Checked = value;
            }
        }

        public bool PlayPreviewWhenRightClick {
            get {
                return chkPlayPreviewWhenRightClick.Checked;
            }
            set {
                chkPlayPreviewWhenRightClick.Checked = value;
            }
        }

        public int MouseHoverTime {
            get {
                return (int)numMouseHoverTime.Value;
            }
            set {
                numMouseHoverTime.Value = value;
            }
        }

        public int PxTrackHeight {
            get {
                return (int)numTrackHeight.Value;
            }
            set {
                numTrackHeight.Value = value;
            }
        }

        public bool KeepLyricInputMode {
            get {
                return chkKeepLyricInputMode.Checked;
            }
            set {
                chkKeepLyricInputMode.Checked = value;
            }
        }

        public Platform Platform {
            get {
                return m_platform;
            }
            set {
                m_platform = value;
                for ( int i = 0; i < comboPlatform.Items.Count; i++ ) {
                    string title = (string)comboPlatform.Items[i];
                    if ( title == m_platform + "" ) {
                        comboPlatform.SelectedIndex = i;
                        break;
                    }
                }
            }
        }

        public int MaximumFrameRate {
            get {
                return (int)numMaximumFrameRate.Value;
            }
            set {
                numMaximumFrameRate.Value = value;
            }
        }

        public bool CorrectPlacementOfTempoAndBeatDialog {
            get {
                return chkCorrectPlacementOfTempoAndBeatDialog.Checked;
            }
            set {
                chkCorrectPlacementOfTempoAndBeatDialog.Checked = value;
            }
        }

        public bool ScrollHorizontalOnWheel {
            get {
                return chkScrollHorizontal.Checked;
            }
            set {
                chkScrollHorizontal.Checked = value;
            }
        }

        public void ApplyLanguage() {
            Text = _( "Preference" );
            btnCancel.Text = _( "Cancel" );
            btnOK.Text = _( "OK" );
            #region tabSequence
            tabSequence.Text = _( "Sequence" );
            lblResolution.Text = _( "Resolution(VSTi)" );
            lblDynamics.Text = _( "Dynamics" ) + "(&D)";
            lblAmplitude.Text = _( "Vibrato Depth" ) + "(&R)";
            lblPeriod.Text = _( "Vibrato Rate" ) + "(&V)";
            lblVibratoConfig.Text = _( "Vibrato Settings" );
            lblVibratoLength.Text = _( "Default Vibrato Length" ) + "(&L)";
            groupAutoVibratoConfig.Text = _( "Auto Vibrato Settings" );
            chkEnableAutoVibrato.Text = _( "Enable Automatic Vibrato" ) + "(&E)";
            lblAutoVibratoMinLength.Text = _( "Minimum note length for Automatic Vibrato" ) + "(&M)";
            lblAutoVibratoType.Text = _( "Vibrato Type" ) + "(&T)";
            #endregion

            #region tabAudio
            //tabAudio.Text = _( "Audio Settings" );
            #endregion

            #region tabAnother
            tabAnother.Text = _( "Other Settings" );
            lblDefaultSinger.Text = _( "Default Singer" ) + "(&S)";
            lblPreSendTime.Text = _( "Pre-Send time" ) + "(&P)";
            lblWait.Text = _( "Waiting Time" ) + "(&W)";
            lblDefaultPremeasure.Text = _( "Default Pre-measure" ) + "(&M)";
            chkChasePastEvent.Text = _( "Chase Event" ) + "(&C)";
            lblSampleOutput.Text = _( "Playback Sample Sound" );
            chkEnableSampleOutput.Text = _( "Enable" ) + "(&E)";
            lblTiming.Text = _( "Timing" );
            lblPreSendTimeSample.Text = _( "Pre-Send Time for sample sound" ) + "(&G)";
            #endregion

            #region tabAppearance
            tabAppearance.Text = _( "Appearance" );
            groupFont.Text = _( "Font" );
            labelMenu.Text = _( "Menu / Lyrics" );
            labelScreen.Text = _( "Screen" );
            labelCounter.Text = _( "Counter" );
            lblLanguage.Text = _( "UI Language" );
            btnChangeCounterFont.Text = _( "Change" );
            btnChangeMenuFont.Text = _( "Change" );
            btnChangeScreenFont.Text = _( "Change" );
            lblTrackHeight.Text = _( "Track Height (pixel)" );
            groupVisibleCurve.Text = _( "Visible Control Curve" );
            #endregion

            #region tabOperation
            tabOperation.Text = _( "Operation" );
            labelWheelOrder.Text = _( "Mouse wheel Rate" );
            chkCursorFix.Text = _( "Fix Play Cursor to Center" );
            chkScrollHorizontal.Text = _( "Horizontal Scroll when Mouse wheel" );
            chkCorrectPlacementOfTempoAndBeatDialog.Text = _( "Place Tempo/Beat Dialog to appropriate Position" );
            lblMaximumFrameRate.Text = _( "Maximum Frame Rate" );
            chkKeepLyricInputMode.Text = _( "Keep Lyric Input Mode" );
            lblMouseHoverTime.Text = _( "Waiting Time for Preview" );
            lblMilliSecond.Text = _( "milli second" );
            chkPlayPreviewWhenRightClick.Text = _( "Play Preview On Right Click" );
            chkCurveSelectingQuantized.Text = _( "Enable Quantize for Curve Selecting" );
            #endregion

            #region tabPlatform
            tabPlatform.Text = _( "Platform" );
            groupPlatform.Text = _( "Platform" );
            lblPlatform.Text = _( "Current Platform" );
            chkCommandKeyAsControl.Text = _( "Use Command key as Control key" );
            #endregion
        }

        public static string _( string id ) {
            return Messaging.GetMessage( id );
        }

        public string Language {
            get {
                int index = comboLanguage.SelectedIndex;
                if ( 0 <= index && index < comboLanguage.Items.Count ) {
                    string title = (string)comboLanguage.Items[index];
                    if ( title == "Default" ) {
                        return "";
                    } else {
                        return title;
                    }
                } else {
                    return "";
                }
            }
        }

        public ClockResolution ControlCurveResolution {
            get {
                int count = -1;
                int index = comboDynamics.SelectedIndex;
                foreach ( ClockResolution vt in ClockResolution.GetEnumerator() ) {
                    count++;
                    if ( count == index ) {
                        return vt;
                    }
                }
                comboDynamics.SelectedIndex = 0;
                return ClockResolution.L30;
            }
            set {
                int count = -1;
                foreach ( ClockResolution vt in ClockResolution.GetEnumerator() ) {
                    count++;
                    if ( vt.Equals( value ) ) {
                        comboDynamics.SelectedIndex = count;
                        break;
                    }
                }
            }
        }

        public int PreSendTime {
            get {
                return (int)numPreSendTime.Value;
            }
            set {
                numPreSendTime.Value = value;
            }
        }

        public int PreMeasure {
            get {
                return comboDefaultPremeasure.SelectedIndex + 1;
            }
            set {
                comboDefaultPremeasure.SelectedIndex = value - 1;
            }
        }

        public bool EnableAutoVibrato {
            get {
                return chkEnableAutoVibrato.Checked;
            }
            set {
                chkEnableAutoVibrato.Checked = value;
            }
        }

        public VibratoType AutoVibratoType {
            get {
                int count = -1;
                int index = comboAutoVibratoType.SelectedIndex;
                foreach ( VibratoType vt in Enum.GetValues( typeof( VibratoType ) ) ) {
                    count++;
                    if ( count == index ) {
                        return vt;
                    }
                }
                comboAutoVibratoType.SelectedIndex = 0;
                return VibratoType.NormalType1;
            }
            set {
                int count = -1;
                foreach ( VibratoType vt in Enum.GetValues( typeof( VibratoType ) ) ) {
                    count++;
                    if ( vt == value ) {
                        comboAutoVibratoType.SelectedIndex = count;
                        break;
                    }
                }
            }
        }

        public AutoVibratoMinLength AutoVibratoMinimumLength {
            get {
                int count = -1;
                int index = comboAutoVibratoMinLength.SelectedIndex;
                foreach ( AutoVibratoMinLength avml in Enum.GetValues( typeof( AutoVibratoMinLength ) ) ) {
                    count++;
                    if ( count == index ) {
                        return avml;
                    }
                }
                comboAutoVibratoMinLength.SelectedIndex = 0;
                return AutoVibratoMinLength.L1;
            }
            set {
                int count = -1;
                foreach ( AutoVibratoMinLength avml in Enum.GetValues( typeof( AutoVibratoMinLength ) ) ) {
                    count++;
                    if ( avml == value ) {
                        comboAutoVibratoMinLength.SelectedIndex = count;
                        break;
                    }
                }
            }
        }

        public DefaultVibratoLength DefaultVibratoLength {
            get {
                int count = -1;
                int index = comboVibratoLength.SelectedIndex;
                foreach ( DefaultVibratoLength vt in Enum.GetValues( typeof( DefaultVibratoLength ) ) ) {
                    count++;
                    if ( index == count ) {
                        return vt;
                    }
                }
                comboVibratoLength.SelectedIndex = 1;
                return DefaultVibratoLength.L66;
            }
            set {
                int count = -1;
                foreach ( DefaultVibratoLength dvl in Enum.GetValues( typeof( DefaultVibratoLength ) ) ) {
                    count++;
                    if ( dvl == value ) {
                        comboVibratoLength.SelectedIndex = count;
                        break;
                    }
                }
            }
        }

        public bool CursorFixed {
            get {
                return chkCursorFix.Checked;
            }
            set {
                chkCursorFix.Checked = value;
            }
        }

        public int WheelOrder {
            get {
                return (int)numericUpDownEx1.Value;
            }
            set {
                if ( value < numericUpDownEx1.Minimum ) {
                    numericUpDownEx1.Value = numericUpDownEx1.Minimum;
                } else if ( numericUpDownEx1.Maximum < value ) {
                    numericUpDownEx1.Value = numericUpDownEx1.Maximum;
                } else {
                    numericUpDownEx1.Value = value;
                }
            }
        }

        public Font ScreenFont {
            get {
                return m_screen_font;
            }
            set {
                m_screen_font = value;
                labelScreenFontName.Text = m_screen_font.Name;
            }
        }

        public Font CounterFont {
            get {
                return m_counter_font;
            }
            set {
                m_counter_font = value;
                labelCounterFontName.Text = m_counter_font.Name;
            }
        }

        public Font BaseFont {
            get {
                return m_base_font;
            }
            set {
                m_base_font = value;
                labelMenuFontName.Text = m_base_font.Name;
                UpdateFonts( m_base_font.Name );
            }
        }

        public int DefaultSinger {
            get {
                if ( comboDefualtSinger.SelectedIndex >= 0 ) {
                    return m_program_change[comboDefualtSinger.SelectedIndex];
                } else {
                    return 0;
                }
            }
            set {
                int index = -1;
                for ( int i = 0; i < m_program_change.Count; i++ ) {
                    if ( m_program_change[i] == value ) {
                        index = i;
                        break;
                    }
                }
                if ( index >= 0 ) {
                    comboDefualtSinger.SelectedIndex = index;
                }
            }
        }

        private void btnChangeMenuFont_Click( object sender, EventArgs e ) {
            fontDialog.Font = BaseFont;
            if ( fontDialog.ShowDialog() == DialogResult.OK ) {
                BaseFont = (Font)fontDialog.Font.Clone();
            }
        }

        private void btnOK_Click( object sender, EventArgs e ) {
            this.DialogResult = DialogResult.OK;
        }

        private void btnChangeScreenFont_Click( object sender, EventArgs e ) {
            fontDialog.Font = ScreenFont;
            if ( fontDialog.ShowDialog() == DialogResult.OK ) {
                ScreenFont = (Font)fontDialog.Font.Clone();
            }
        }

        private void btnChangeCounterFont_Click( object sender, EventArgs e ) {
            fontDialog.Font = CounterFont;
            if ( fontDialog.ShowDialog() == DialogResult.OK ) {
                CounterFont = (Font)fontDialog.Font.Clone();
            }
        }

        void UpdateFonts( string font_name ) {
            if ( font_name == "" ) {
                return;
            }
            Font font = new Font( font_name, tabSequence.Font.Size, tabSequence.Font.Unit );
            foreach ( Control ctrl in this.Controls ) {
                ctrl.Font = font;
            }
        }

        private void comboPlatform_SelectedIndexChanged( object sender, EventArgs e ) {
            string title = (string)comboPlatform.SelectedItem;
            foreach ( Platform p in Enum.GetValues( typeof( Platform ) ) ) {
                if ( title == p + "" ) {
                    m_platform = p;
                    if ( p == Platform.Macintosh ) {
                        chkCommandKeyAsControl.Enabled = true;
                    } else {
                        chkCommandKeyAsControl.Enabled = false;
                    }
                    break;
                }
            }
        }

        public bool CommandKeyAsControl {
            get {
                return chkCommandKeyAsControl.Checked;
            }
            set {
                chkCommandKeyAsControl.Checked = value;
            }
        }
    }

}
