﻿/*
 * CurveType.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections;

namespace Boare.Cadencii {

    /// <summary>
    /// vsqファイルで編集可能なカーブ・プロパティの種類
    /// </summary>
    public struct CurveType : IEquatable<CurveType> {
        private string m_type;
        private bool m_is_scalar;
        private int m_minimum;
        private int m_maximum;
        private int m_default;
        private bool m_is_attach_note;
        private int m_index;

        private static readonly CurveType s_vel = new CurveType( "VEL", true, true, 0, 127, 64, -1 );
        private static readonly CurveType s_dyn = new CurveType( "DYN", false, false, 0, 127, 64, 0 );
        private static readonly CurveType s_bre = new CurveType( "BRE", false, false, 0, 127, 0, 1 );
        private static readonly CurveType s_bri = new CurveType( "BRI", false, false, 0, 127, 64, 2 );
        private static readonly CurveType s_cle = new CurveType( "CLE", false, false, 0, 127, 0, 3 );
        private static readonly CurveType s_ope = new CurveType( "OPE", false, false, 0, 127, 127, 4 );
        private static readonly CurveType s_gen = new CurveType( "GEN", false, false, 0, 127, 64, 5 );
        private static readonly CurveType s_por = new CurveType( "POR", false, false, 0, 127, 64, 6 );
        private static readonly CurveType s_pit = new CurveType( "PIT", false, false, -8192, 8191, 0, 7 );
        private static readonly CurveType s_pbs = new CurveType( "PBS", false, false, 0, 24, 2, 8 );
        private static readonly CurveType s_vibrato_rate = new CurveType( "V-Rate", false, true, 0, 127, 64, 9 );
        private static readonly CurveType s_vibrato_depth = new CurveType( "V-Depth", false, true, 0, 127, 50, 10 );
        private static readonly CurveType s_accent = new CurveType( "Accent", true, true, 0, 100, 50, -1 );
        private static readonly CurveType s_decay = new CurveType( "Decay", true, true, 0, 100, 50, -1 );

        private CurveType( string type, bool is_scalar, bool is_attach_note, int min, int max, int defalt_value, int index ) {
            m_type = type;
            m_is_scalar = is_scalar;
            m_minimum = min;
            m_maximum = max;
            m_default = defalt_value;
            m_is_attach_note = is_attach_note;
            m_index = index;
        }

        public override string ToString() {
            return m_type;
        }

        public int Index {
            get {
                return m_index;
            }
        }

        public string Name {
            get {
                return m_type;
            }
        }

        public bool IsAttachNote {
            get {
                return m_is_attach_note;
            }
        }

        public bool IsScalar {
            get {
                return m_is_scalar;
            }
        }

        public int Minimum {
            get {
                return m_minimum;
            }
        }

        public int Maximum {
            get {
                return m_maximum;
            }
        }

        public int Default {
            get {
                return m_default;
            }
        }

        public static IEnumerable GetEnumerator() {
            yield return s_vel;
            yield return s_dyn;
            yield return s_bre;
            yield return s_bri;
            yield return s_cle;
            yield return s_ope;
            yield return s_gen;
            yield return s_por;
            yield return s_pit;
            yield return s_pbs;
            yield return s_vibrato_rate;
            yield return s_vibrato_depth;
            yield return s_accent;
            yield return s_decay;
        }
        
        public static bool operator ==( CurveType a, CurveType b ) {
            return a.Equals( b );
        }

        public static bool operator !=( CurveType a, CurveType b ) {
            return !a.Equals( b );
        }

        public bool Equals( CurveType other ) {
            return (m_type == other.m_type && m_is_scalar == other.m_is_scalar);
        }

        /// <summary>
        /// VsqCurveTypeに変換できるかどうかを試します
        /// </summary>
        /// <param name="curve"></param>
        /// <param name="original"></param>
        /// <returns></returns>
        public static bool TryGetOriginal( CurveType curve, out string original ) {
            if ( curve == CurveType.BRE ||
                 curve == CurveType.BRI ||
                 curve == CurveType.CLE ||
                 curve == CurveType.DYN ||
                 curve == CurveType.GEN ||
                 curve == CurveType.OPE ||
                 curve == CurveType.PBS ||
                 curve == CurveType.PIT ||
                 curve == CurveType.POR ||
                 curve == CurveType.VEL ) {
                original = curve.Name;
                return true;
            } else {
                original = "";
                return false;
            }
        }

        public static int GetMaxValue( CurveType curve_type ) {
            if ( curve_type == CurveType.PIT ) {
                return 8192;
            } else if ( curve_type == CurveType.PBS ) {
                return 24;
            } else if ( curve_type == CurveType.Accent || curve_type == CurveType.Decay ) {
                return 100;
            } else {
                return 127;
            }
        }

        public static int GetMinValue( CurveType curve_type ) {
            if ( curve_type == CurveType.PIT ) {
                return -8192;
            } else {
                return 0;
            }
        }

        public static int GetDefaultValue( CurveType curve_type ) {
            if ( curve_type == CurveType.BRE ) {
                return 0;
            } else if ( curve_type == CurveType.BRI ) {
                return 64;
            } else if ( curve_type == CurveType.CLE ) {
                return 0;
            } else if ( curve_type == CurveType.DYN ) {
                return 64;
            } else if ( curve_type == CurveType.GEN ) {
                return 64;
            } else if ( curve_type == CurveType.OPE ) {
                return 127;
            } else if ( curve_type == CurveType.PBS ) {
                return 2;
            } else if ( curve_type == CurveType.PIT ) {
                return 0;
            } else if ( curve_type == CurveType.POR ) {
                return 64;
            } else if ( curve_type == CurveType.VEL ) {
                return 64;
            } else if ( curve_type == CurveType.VibratoRate ) {
                return 64;
            } else if ( curve_type == CurveType.VibratoDepth ) {
                return 50;
            } else if ( curve_type == CurveType.Accent ) {
                return 50;
            } else if ( curve_type == CurveType.Decay ) {
                return 50;
            } else {
                return 0;
            }
        }

        /// <summary>
        /// ベロシティ
        /// </summary>
        public static CurveType VEL {
            get {
                return s_vel;
            }
        }

        /// <summary>
        /// ダイナミクス　64
        /// </summary>
        public static CurveType DYN {
            get {
                return s_dyn;
            }
        }

        /// <summary>
        /// ブレシネス　0
        /// </summary>
        public static CurveType BRE {
            get {
                return s_bre;
            }
        }

        /// <summary>
        /// ブライトネス　64
        /// </summary>
        public static CurveType BRI {
            get {
                return s_bri;
            }
        }

        /// <summary>
        /// クリアネス　0
        /// </summary>
        public static CurveType CLE {
            get {
                return s_cle;
            }
        }

        /// <summary>
        /// オープニング　127
        /// </summary>
        public static CurveType OPE {
            get {
                return s_ope;
            }
        }

        /// <summary>
        /// ジェンダーファクター　64
        /// </summary>
        public static CurveType GEN {
            get {
                return s_gen;
            }
        }

        /// <summary>
        /// ポルタメントタイミング　64
        /// </summary>
        public static CurveType POR {
            get {
                return s_por;
            }
        }

        /// <summary>
        /// ピッチベンド　0
        /// </summary>
        public static CurveType PIT {
            get {
                return s_pit;
            }
        }

        /// <summary>
        /// ピッチベンドセンシティビティ　2
        /// </summary>
        public static CurveType PBS {
            get {
                return s_pbs;
            }
        }

        /// <summary>
        /// ビブラートの振動の速さ
        /// </summary>
        public static CurveType VibratoRate {
            get {
                return s_vibrato_rate;
            }
        }

        /// <summary>
        /// ビブラートの振幅の大きさ
        /// </summary>
        public static CurveType VibratoDepth {
            get {
                return s_vibrato_depth;
            }
        }

        public static CurveType Accent {
            get {
                return s_accent;
            }
        }

        public static CurveType Decay {
            get {
                return s_decay;
            }
        }
    }

}
