﻿/*
 * BezierPoint.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Drawing;
using System.Xml.Serialization;
using System.ComponentModel;

namespace Boare.Cadencii {

    [Serializable]
    public struct PointD {
        private double m_x;
        private double m_y;

        public PointD( double x, double y ){
            m_x = x;
            m_y = y;
        }

        public System.Drawing.Point ToPoint() {
            return new System.Drawing.Point( (int)m_x, (int)m_y );
        }

        public System.Drawing.PointF ToPointF() {
            return new System.Drawing.PointF( (float)m_x, (float)m_y );
        }

        public double X {
            get {
                return m_x;
            }
            set {
                m_x = value;
            }
        }

        public double Y {
            get {
                return m_y;
            }
            set {
                m_y = value;
            }
        }
    }

    /// <summary>
    /// ベジエ曲線を構成するデータ点。
    /// </summary>
    [Serializable]
    public class BezierPoint : IComparable<BezierPoint>, ICloneable {
        PointD m_base;
        internal PointD m_control_left;
        internal PointD m_control_right;
        BezierControlType m_type_left;
        BezierControlType m_type_right;
        ValueType m_type;
        [NonSerialized]
        int m_id;

        private enum ValueType {
            int_,
            float_,
            double_,
        }

        [XmlIgnore]
        public int ID {
            get {
                return m_id;
            }
            internal set {
                m_id = value;
            }
        }

        public override string ToString() {
            return "m_base=" + m_base.X + "," + m_base.Y + "\n" +
                "m_control_left=" + m_control_left.X + "," + m_control_left.Y + "\n" +
                "m_control_right=" + m_control_right.X + "," + m_control_right.Y + "\n" +
                "m_type_left=" + m_type_left + "\n" +
                "m_type_right=" + m_type_right + "\n";
        }

        private BezierPoint() {
        }

        public BezierPoint( PointD p1 ) : this( p1.X, p1.Y ) {
        }

        public BezierPoint( double x, double y ) {
            PointD p = new PointD( x, y );
            m_base = p;
            m_control_left = p;
            m_control_right = p;
            m_type_left = BezierControlType.None;
            m_type_right = BezierControlType.None;
        }

        public BezierPoint( PointD p1, PointD left, PointD right ) {
            m_base = p1;
            m_control_left = new PointD( left.X - m_base.X, left.Y - m_base.Y );
            m_control_right = new PointD( right.X - m_base.X, right.Y - m_base.Y );
            m_type_left = BezierControlType.None;
            m_type_right = BezierControlType.None;
        }

        public object Clone() {
            BezierPoint result = new BezierPoint( this.Base, this.ControlLeft, this.ControlRight );
            result.m_control_left = this.m_control_left;
            result.m_control_right = this.m_control_right;
            result.m_type_left = this.m_type_left;
            result.m_type_right = this.m_type_right;
            result.m_id = this.m_id;
            return result;
        }

        public int CompareTo( BezierPoint item ) {
            double thisx = this.Base.X;
            double itemx = item.Base.X;
            if ( thisx > itemx ) {
                return 1;
            } else if ( thisx < itemx ) {
                return -1;
            } else {
                if ( this.ID > item.ID ) {
                    return 1;
                } else if ( this.ID < item.ID ) {
                    return -1;
                } else {
                    return 0;
                }
            }
        }

        public PointD Base {
            get {
                return m_base;
            }
            set {
                m_base = value;
            }
        }

        public void SetPosition( BezierPickedSide picked_side, PointD new_position ) {
            if ( picked_side == BezierPickedSide.Base ) {
                this.Base = new_position;
            } else if ( picked_side == BezierPickedSide.Left ) {
                this.m_control_left = new PointD( new_position.X - this.Base.X, new_position.Y - this.Base.Y );
            } else {
                this.m_control_right = new PointD( new_position.X - this.Base.X, new_position.Y - this.Base.Y );
            }
        }

        public PointD GetPosition( BezierPickedSide picked_side ) {
            if ( picked_side == BezierPickedSide.Base ) {
                return this.Base;
            } else if ( picked_side == BezierPickedSide.Left ) {
                return this.ControlLeft;
            } else {
                return this.ControlRight;
            }
        }

        public BezierControlType GetControlType( BezierPickedSide picked_side ) {
            if ( picked_side == BezierPickedSide.Left ) {
                return this.ControlLeftType;
            } else if ( picked_side == BezierPickedSide.Right ) {
                return this.ControlRightType;
            } else {
                return BezierControlType.None;
            }
        }

        public PointD ControlLeft {
            get {
                if ( m_type_left != BezierControlType.None ) {
                    return new PointD( m_base.X + m_control_left.X, m_base.Y + m_control_left.Y );
                } else {
                    return m_base;
                }
            }
            set {
                m_control_left = new PointD( value.X - m_base.X, value.Y - m_base.Y );
            }
        }

        public PointD ControlRight {
            get {
                if ( m_type_right != BezierControlType.None ) {
                    return new PointD( m_base.X + m_control_right.X, m_base.Y + m_control_right.Y );
                } else {
                    return m_base;
                }
            }
            set {
                m_control_right = new PointD( value.X - m_base.X, value.Y - m_base.Y );
            }
        }

        public BezierControlType ControlLeftType {
            get {
                return m_type_left;
            }
            set {
                m_type_left = value;
                if ( m_type_left == BezierControlType.Master && m_type_right != BezierControlType.None ) {
                    m_type_right = BezierControlType.Master;
                }
            }
        }

        public BezierControlType ControlRightType {
            get {
                return m_type_right;
            }
            set {
                m_type_right = value;
                if ( m_type_right == BezierControlType.Master && m_type_left != BezierControlType.None ) {
                    m_type_left = BezierControlType.Master;
                }
            }
        }
    }

}
