﻿/*
 * VsqBPList.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace Boare.Lib.Vsq {

    /// <summary>
    /// BPListのデータ部分を取り扱うためのクラス。
    /// </summary>
    [Serializable]
    public class VsqBPList: ICloneable {
        private SortedList<int, int> m_list = new SortedList<int, int>();
        private int m_default = 0;
        private int m_maximum = 127;
        private int m_minimum = 0;

        /// <summary>
        /// このVsqBPListの同一コピーを作成します
        /// </summary>
        /// <returns></returns>
        public object Clone() {
            VsqBPList res = new VsqBPList( m_default, m_minimum, m_maximum );
            foreach( int key in m_list.Keys ){
                res.m_list.Add( key, m_list[key] );
            }
            return res;
        }
        
        /// <summary>
        /// コンストラクタ。デフォルト値はココで指定する。
        /// </summary>
        /// <param name="default_value"></param>
        public VsqBPList( int default_value, int minimum, int maximum ) {
            m_default = default_value;
            m_maximum = maximum;
            m_minimum = minimum;
        }

        /// <summary>
        /// このリストに設定された最大値を取得します。
        /// </summary>
        public int Maximum {
            get {
                return m_maximum;
            }
        }

        /// <summary>
        /// このリストに設定された最小値を取得します
        /// </summary>
        public int Minimum {
            get {
                return m_minimum;
            }
        }

        public IEnumerable<int> GetKeyClockEnumerator() {
            for ( int i = 0; i < m_list.Keys.Count; i++ ) {
                yield return m_list.Keys[i];
            }
        }

        public void Remove( int clock ) {
            if ( m_list.ContainsKey( clock ) ) {
                m_list.Remove( clock );
            }
        }

        public bool ContainsKey( int clock ) {
            return m_list.ContainsKey( clock );
        }

        public int Count {
            get {
                return m_list.Count;
            }
        }

        public int[] Keys {
            get {
                List<int> t = new List<int>( m_list.Keys );
                return t.ToArray();
            }
        }

        public void Clear() {
            m_list.Clear();
        }

        /// <summary>
        /// 新しいデータ点を追加します。
        /// </summary>
        /// <param name="clock"></param>
        /// <param name="value"></param>
        public void Add( int clock, int value ) {
            lock ( m_list ) {
                if ( m_list.ContainsKey( clock ) ) {
                    m_list[clock] = value;
                } else {
                    m_list.Add( clock, value );
                }
            }
        }

        public int this[int clock] {
            get {
                if ( m_list.Count == 0 ) {
                    return Default;
                } else {
                    if ( m_list.ContainsKey( clock ) ) {
                        return m_list[clock];
                    } else {
                        int index = 0;
                        int prev = 0;
                        foreach ( int key in m_list.Keys ) {
                            if ( clock < key ) {
                                index = prev;
                                break;
                            }
                            prev = key;
                        }
                        if ( m_list.ContainsKey( index ) ) {
                            return m_list[index];
                        } else {
                            return m_default;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// このBPListのデフォルト値を取得します
        /// </summary>
        public int Default {
            get {
                return m_default;
            }
        }

        /// <summary>
        /// このBPListの内容をテキストファイルに書き出します
        /// </summary>
        /// <param name="writer"></param>
        public void Print( StreamWriter writer ) {
            bool first = true;
            foreach ( int key in m_list.Keys ) {
                int val = m_list[key];
                if ( first ) {
                    writer.WriteLine( key + "=" + val );
                    first = false;
                } else {
                    writer.WriteLine( key + "=" + val );
                }
            }
        }

        /// <summary>
        /// このBPListの内容をテキストファイルに書き出します
        /// </summary>
        /// <param name="writer"></param>
        public void Print( TextMemoryStream writer, int start, string header ) {
            bool first = true;
            foreach ( int key in m_list.Keys ) {
                if ( start <= key ) {
                    if ( first ) {
                        writer.WriteLine( header );
                        first = false;
                    }
                    int val = m_list[key];
                    writer.WriteLine( key + "=" + val );
                }
            }
        }

        /// <summary>
        /// テキストファイルからデータ点を読込み、現在のリストに追加します
        /// </summary>
        /// <param name="reader"></param>
        /// <returns></returns>
        public string Read( TextMemoryStream reader ) {
            string last_line = reader.ReadLine();
            while ( !last_line.StartsWith( "[" ) ) {
                string[] spl = last_line.Split( new char[] { '=' } );
                int i1 = int.Parse( spl[0] );
                int i2 = int.Parse( spl[1] );
                m_list.Add( i1, i2 );
                if ( reader.Peek() < 0 ) {
                    break;
                } else {
                    last_line = reader.ReadLine();
                }
            }
            return last_line;
        }
    }

}
