﻿/*
 * SMF/MidiFile.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace Boare.Lib.Vsq {

    /*enum Mode {
        Read,
        Write,
    }*/

    class MidiFile {
        /* MIDI status commands most significant bit is 1 */
        const int note_off = 0x80;
        const int note_on = 0x90;
        const int poly_aftertouch = 0xa0;
        const int control_change = 0xb0;
        const int program_chng = 0xc0;
        const int channel_aftertouch = 0xd0;
        const int pitch_wheel = 0xe0;
        const int system_exclusive = 0xf0;
        const int delay_packet = 1111;

        /* 7 bit controllers */
        const int damper_pedal = 0x40;
        const int portamento = 0x41;
        const int sostenuto = 0x42;
        const int soft_pedal = 0x43;
        const int general_4 = 0x44;
        const int hold_2 = 0x45;
        const int general_5 = 0x50;
        const int general_6 = 0x51;
        const int general_7 = 0x52;
        const int general_8 = 0x53;
        const int tremolo_depth = 0x5c;
        const int chorus_depth = 0x5d;
        const int detune = 0x5e;
        const int phaser_depth = 0x5f;

        /* parameter values */
        const int data_inc = 0x60;
        const int data_dec = 0x61;

        /* parameter selection */
        const int non_reg_lsb = 0x62;
        const int non_reg_msb = 0x63;
        const int reg_lsb = 0x64;
        const int reg_msb = 0x65;

        /* Standard MIDI Files meta event definitions */
        const int meta_event = 0xFF;
        const int sequence_number = 0x00;
        const int text_event = 0x01;
        const int copyright_notice = 0x02;
        const int sequence_name = 0x03;
        const int instrument_name = 0x04;
        const int lyric = 0x05;
        const int marker = 0x06;
        const int cue_point = 0x07;
        const int channel_prefix = 0x20;
        const int end_of_track = 0x2f;
        const int set_tempo = 0x51;
        const int smpte_offset = 0x54;
        const int time_signature = 0x58;
        const int key_signature = 0x59;
        const int sequencer_specific = 0x7f;

        /* Manufacturer's ID number */
        const int Seq_Circuits = 0x01; /* Sequential Circuits Inc. */
        const int Big_Briar = 0x02; /* Big Briar Inc. */
        const int Octave = 0x03; /* Octave/Plateau */
        const int Moog = 0x04; /* Moog Music */
        const int Passport = 0x05; /* Passport Designs */
        const int Lexicon = 0x06; /* Lexicon */
        const int Tempi = 0x20; /* Bon Tempi */
        const int Siel = 0x21; /* S.I.E.L.  */
        const int Kawai = 0x41;
        const int Roland = 0x42;
        const int Korg = 0x42;
        const int Yamaha = 0x43;

        /* miscellaneous definitions */
        const long MThd = 0x4d546864L;
        const long MTrk = 0x4d54726bL;

        const int MTHD = 256;
        const int MTRK = 257;
        const int TRKEND = 258;

        const int ON = note_on;//#define ON note_on
        const int OFF = note_off;
        const int POPR = poly_aftertouch;
        const int PAR = control_change;
        const int PB = pitch_wheel;
        const int PRCH = program_chng;
        const int CHPR = channel_aftertouch;
        const int SYSEX = system_exclusive;

        const int ARB = 259;
        const int MINOR = 260;
        const int MAJOR = 261;

        const int CH = 262;
        const int NOTE = 263;
        const int VAL = 264;
        const int CON = 265;
        const int PROG = 266;

        const int INT = 267;
        const int STRING = 268;
        const int STRESC = 269;
        const int ERR = 270;
        const int NOTEVAL = 271;
        const int EOL = 272;

        const int META = 273;
        const int SEQSPEC = (META + 1 + sequencer_specific);
        const int TEXT = (META + 1 + text_event);
        const int COPYRIGHT = (META + 1 + copyright_notice);
        const int SEQNAME = (META + 1 + sequence_name);
        const int INSTRNAME = (META + 1 + instrument_name);
        const int LYRIC = (META + 1 + lyric);
        const int MARKER = (META + 1 + marker);
        const int CUE = (META + 1 + cue_point);
        const int SEQNR = (META + 1 + sequence_number);
        const int KEYSIG = (META + 1 + key_signature);
        const int TEMPO = (META + 1 + set_tempo);
        const int TIMESIG = (META + 1 + time_signature);
        const int SMPTE = (META + 1 + smpte_offset);

        private const int EOF = -1;
        private FileStream F = null;
        //private StreamWriter sw = null;
        private StringBuilder sb = null;
        private StreamReader yyin = null;

        private const int FALSE = 0;
        private const int TRUE = 1;

        static int err_cont = 0;

        private int fold = 0;		/* fold long lines */
        private int notes = 0;		/* print notes as a-g */
        private int times = 0;		/* print times as Measure/beat/click */
        private static int Measure, M0, Beat, Clicks;
        private int TrksToDo = 1;
        private int TrkNr;
        private static long T0;
        static char[] buffer = new char[] { };
        static int bufsiz = 0, buflen;

        private static String note = "n";
        private static String vol = "v";
        private static String con = "c";
        private static String val = "v";
        private static String prog = "p";

        private static String PolyPr = "PoPr";
        private static String Param = "Par";
        private static String ProgCh = "PrCh";
        private static String ChanPr = "ChPr";

        static int Format, Ntrks;

        private long Mf_currtime;
        private int Mf_nomerge;
        private long Mf_toberead;
        private string Msgbuff;
        private int Msgindex;

        wtrackDelegate Mf_wtrack = null;
        wtrackDelegate Mf_wtempotrack = null;
        putcDelegate Mf_putc = null;
        static long Mf_numbyteswritten = 0L;

        /*extern */
        static long yyval;
        /*extern */
        static int eol_seen;
        /*extern */
        static int lineno;
        /*extern */
        static int yyleng;
        /*extern */
        static string yytext;
        /*extern */
        static int do_hex;
        /*extern */
        static int Mf_RunStat;

        static int laststat;		/* last status code */
        static int lastmeta;		/* last meta event type */

        static char[] data = new char[5];//changed to static
        static int chan;//staticに変更

        private int m_nrpn_msb, m_nrpn_lsb;
#if DEBUG
        /*private StreamWriter m_sw_v1mean;
        private StreamWriter m_sw_d1mean;
        private StreamWriter m_sw_d1meanFirstNote;
        private StreamWriter m_sw_d2mean;
        private StreamWriter m_sw_d4mean;*/
#endif

        public string ReadToEnd() {
            return sb.ToString();
        }


        public MidiFile( string smf_path/*, string text_path, Mode mode*/ ) {
            //if ( mode == Mode.Read ) {
                F = new FileStream( smf_path, FileMode.Open, FileAccess.Read );
                //sw = new StreamWriter( text_path );
                sb = new StringBuilder();
                Mf_toberead = 0L;
                Mf_currtime = 0L;
                Mf_nomerge = 0;
                Mf_nomerge = 1;
                TrkNr = 0;
                Measure = 4;
                Beat = 0x60;
                Clicks = 0x60;
                T0 = 0L;
                M0 = 0;
                readheader();
                while ( readtrack() != 0 ) {
                }
                //sw.Close();
            /*} else if ( mode == Mode.Write ) {
                yyin = new StreamReader( text_path );
                F = new FileStream( smf_path, FileMode.Create, FileAccess.Write );
                Mf_putc = fileputc;
                Mf_wtrack = mywritetrack;
                TrkNr = 0;
                Measure = 4;
                Beat = 96;
                Clicks = 96;
                M0 = 0;
                T0 = 0;
                translate();
                F.Close();
                yyin.Close();
            }*/
        }

        public void Close() {
            if ( F != null ) {
                F.Close();
            }
#if DEBUG
            /*if ( m_sw_d1mean != null ) {
                m_sw_d1mean.Close();
            }
            if ( m_sw_d1meanFirstNote != null ) {
                m_sw_d1meanFirstNote.Close();
            }
            if ( m_sw_d2mean != null ) {
                m_sw_d2mean.Close();
            }
            if ( m_sw_d4mean != null ) {
                m_sw_d4mean.Close();
            }
            if ( m_sw_v1mean != null ) {
                m_sw_v1mean.Close();
            }*/
#endif
            //if ( sw != null ) {
            //    sw.Close();
            //}
        }

        #region t2mf

        int getbyte( string mess ) {
            //char ermesg[100];
            getint( mess );
            if ( yyval < 0 || yyval > 127 ) {
                string ermesg = "Wrong value (" + yyval + ") for " + mess;//	sprintf (ermesg, "Wrong value (%ld) for %s", yyval, mess);
                error( ermesg );
                yyval = 0;
            }
            return (int)yyval;
        }


        static void gethex() {
            int c;
            buflen = 0;
            do_hex = 1;
            c = yylex();
            if ( c == STRING ) {
                /* Note: yytext includes the trailing, but not the starting quote */
                int i = 0;
                if ( yyleng - 1 > bufsiz ) {
                    bufsiz = yyleng - 1;
                    if ( buffer != null ) {
                        buffer = realloc( buffer, bufsiz );
                    } else {
                        buffer = malloc( bufsiz );
                    }
                    /*if ( !buffer ){//buffer == TRUE => buffer != FALSE -> buffer != 0
                        error( "Out of memory" );
                    }さくじょしました。。*/
                }
                while ( i < yyleng - 1 ) {
                    c = yytext[i++];
                rescan:
                    if ( c == '\\' ) {
                        switch ( c = yytext[i++] ) {
                            case '0':
                                c = '\0';
                                break;
                            case 'n':
                                c = '\n';
                                break;
                            case 'r':
                                c = '\r';
                                break;
                            case 't':
                                c = '\t';
                                break;
                            case 'x':
                                //yytext[i]の位置の文字が16進数で使われる文字じゃないとき
                                try {
                                    Int32.Parse( "" + yytext[i] );
                                    c = yytext[i];
                                } catch {
                                    prs_error( "Illegal \\x in string" );
                                }
                                /*if ( sscanf( yytext + i, "%2x", &c ) != 1 )
                                    prs_error( "Illegal \\x in string" );*/
                                i += 2;
                                break;
                            case '\r':
                            case '\n':
                                while ( (c = yytext[i++]) == ' ' || c == '\t' || c == '\r' || c == '\n' )
                                    /* skip whitespace */
                                    ;
                                goto rescan; /* sorry EWD :=) */
                        }
                    }
                    buffer[buflen++] = (char)c;
                }
            } else if ( c == INT ) {
                do {
                    if ( buflen >= bufsiz ) {
                        bufsiz += 128;
                        if ( buffer != null ) {// buffer == TRUE -> buffer != FALSE => buffer != 0
                            buffer = realloc( buffer, bufsiz );
                        } else {
                            buffer = malloc( bufsiz );
                        }
                        /*if ( !buffer ) {
                            error( "Out of memory" );
                        }削除しました。。*/
                    }
                    /* This test not applicable for sysex
                            if (yyval < 0 || yyval > 127)
                                error ("Illegal hex value"); */
                    buffer[buflen++] = (char)yyval;
                    c = yylex();
                } while ( c == INT );
                if ( c != EOL )
                    prs_error( "Unknown hex input" );
            } else
                prs_error( "String or hex input expected" );
        }


        static int yylex() {
            // TODO: THIS IS DUMY!!
            return 0;
        }

        static void prs_error( string s ) {//static二変更j
            int c = 0;
            int count;
            int ln = (eol_seen != FALSE ? lineno - 1 : lineno);
            Console.WriteLine( ln + ": " + s );//        fprintf( stderr, "%d: %s\n", ln, s );
            if ( yyleng > 0 && yytext != "\n" ) {
                Console.WriteLine( "*** " + yyleng + " " + yytext + " ***" );//fprintf( stderr, "*** %*s ***\n", yyleng, yytext );
            }
            count = 0;
            while ( count < 100 && (c = yylex()) != EOL && c != EOF ) {
                count++/* skip rest of line */;
            }
            if ( c == EOF ) {
                Environment.Exit( 1 );// exit( 1 );
            }
            if ( err_cont != FALSE ) {
                //longjmp( erjump, 1 );
                throw new Exception( s );
            }
        }


        static char[] realloc( char[] source, int size ) {
            char[] result = new char[size];
            for ( int i = 0; i < size; i++ ) {
                if ( i < source.Length ) {
                    result[i] = source[i];
                } else {
                    break;
                }
            }
            return result;
        }


        static char[] malloc( int size ) {
            return new char[size];
        }


        static void checkeol() {
            if ( eol_seen != FALSE ) {
                return;
            }
            if ( yylex() != EOL ) {
                prs_error( "Garbage deleted" );
                while ( eol_seen == FALSE/*!eol_seen*/ ) {
                    yylex();	 /* skip rest of line */
                }
            }
        }

        private void translate() {
            if ( yylex() == MTHD ) {
                Format = getint( "MFile format" );
                Ntrks = getint( "MFile #tracks" );
                Clicks = getint( "MFile Clicks" );
                if ( Clicks < 0 ) {
                    Clicks = (Clicks & 0xff) << 8 | getint( "MFile SMPTE division" );
                }
                checkeol();
                mfwrite( Format, Ntrks, Clicks, F );
            } else {
                Console.WriteLine( "Missing MFile - can't continue\n" );
                Environment.Exit( 1 );
            }
        }


        private void mfwrite( int format, int ntracks, int division, FileStream fp ) {
            int i;
            //void mf_w_track_chunk(), mf_w_header_chunk();

            if ( Mf_putc == null/*NULLFUNC*/ ) {
                mferror( "mfmf_write() called without setting Mf_putc" );
            }

            if ( Mf_wtrack == null/*NULLFUNC*/ ) {
                mferror( "mfmf_write() called without setting Mf_mf_writetrack" );
            }

            /* every MIDI file starts with a header */
            mf_w_header_chunk( format, ntracks, division );

            /* In format 1 files, the first track is a tempo map */
            if ( format == 1 && (Mf_wtempotrack != null) ) {
                mf_w_track_chunk( -1, fp, Mf_wtempotrack );
                ntracks--;
            }

            /* The rest of the file is a series of tracks */
            for ( i = 0; i < ntracks; i++ ) {
                mf_w_track_chunk( i, fp, Mf_wtrack );
            }
        }


        void mf_w_header_chunk( int format, int ntracks, int division ) {
            ulong ident, length;
            //void write16bit(),write32bit();

            ident = MThd;           /* Head chunk identifier                    */
            length = 6;             /* Chunk length                             */

            /* individual bytes of the header must be written separately
               to preserve byte order across cpu types :-( */
            write32bit( ident );
            write32bit( length );
            write16bit( format );
            write16bit( ntracks );
            write16bit( division );
        } /* end gen_header_chunk() */


        delegate int wtrackDelegate();
        delegate int putcDelegate( int c );

        private void mf_w_track_chunk( int which_track, FileStream fp, wtrackDelegate wtrack /* int (*wtrack) wtrack*/) {
            ulong trkhdr, trklength;
            long offset, place_marker;
            //void write16bit(),write32bit();

            trkhdr = MTrk;
            trklength = 0;

            /* Remember where the length was written, because we don't
               know how long it will be until we've finished writing */
            offset = fp.Position;//ftell( fp );
#if DEBUG
            Console.WriteLine( "offset = " + offset );//printf( "offset = %d\n", (int)offset );
#endif
            /* Write the track chunk header */
            write32bit( trkhdr );
            write32bit( trklength );

            Mf_numbyteswritten = 0L; /* the header's length doesn't count */
            laststat = 0;

            /* Note: this calls Mf_writetempotrack with an unused parameter (-1)
               But this is innocent */

            wtrack();//引数無くていいんかな。。。   (*wtrack)( which_track );

            if ( laststat != meta_event || lastmeta != end_of_track ) {
                /* mf_write End of track meta event */
                eputc( 0 );
                eputc( meta_event );
                eputc( end_of_track );
                eputc( 0 );
            }

            laststat = 0;

            /* It's impossible to know how long the track chunk will be beforehand,
                   so the position of the track length data is kept so that it can
                   be written after the chunk has been generated */
            place_marker = fp.Position;// ftell( fp );

            /* This method turned out not to be portable because the
                   parameter returned from ftell is not guaranteed to be
                   in bytes on every machine */
            /* track.length = place_marker - offset - (long) sizeof(track); */

#if DEBUG
            Console.WriteLine( "length = " + trklength );//printf( "length = %d\n", (int)trklength );
#endif

            if ( fp.Seek( offset, SeekOrigin.Begin )/* fseek( fp, offset, 0 )*/ < 0 ) {
                mferror( "error seeking during final stage of write" );
            }

            trklength = (ulong)Mf_numbyteswritten;

            /* Re-mf_write the track chunk header with right length */
            write32bit( trkhdr );
            write32bit( trklength );

            fp.Seek( place_marker, SeekOrigin.Begin );//            fseek( fp, place_marker, 0 );
        }

        private void write32bit( ulong data ) {
            eputc( (int)((data >> 24) & 0xff) );
            eputc( (int)((data >> 16) & 0xff) );
            eputc( (int)((data >> 8) & 0xff) );
            eputc( (int)(data & 0xff) );
        }


        void write16bit( int data ) {
            eputc( (int)((data & 0xff00) >> 8) );
            eputc( (int)(data & 0xff) );
        }


        private int eputc( int c )/*unsigned char c*/{
            if ( Mf_putc == null )/* == NULLFUNC*/ {
                mferror( "Mf_putc undefined" );
                return (-1);
            }

            //return_val = (*Mf_putc)( c );
            int return_val = Mf_putc( c );

            if ( return_val == EOF ) {
                mferror( "error writing" );
            }

            Mf_numbyteswritten++;
            return (return_val);
        }


        static void checkchan() {
            if ( yylex() != CH || yylex() != INT ) {
                syntax();
            }
            if ( yyval < 1 || yyval > 16 ) {
                error( "Chan must be between 1 and 16" );
            }
            chan = (int)yyval - 1;
        }


        static void checknote() {
            int c = 0;
            if ( yylex() != NOTE || ((c = yylex()) != INT && c != NOTEVAL) ) {
                syntax();
            }
            if ( c == NOTEVAL ) {
                /*static */
                int[] notes = new int[]{
                9,		/* a */
                11,		/* b */
                0,		/* c */
                2,		/* d */
                4,		/* e */
                5,		/* f */
                7		/* g */
            };

                //char* p = yytext;
                string p = yytext;
                int index = 0;
                c = p[index];//*p++;
                if ( Char.IsUpper( (char)c )/* isupper( c )*/ ) {
                    string temp = c + "";
                    temp = temp.ToLower();
                    c = (int)temp[0];//tolower( c );
                }
                yyval = notes[c - 'a'];
                switch ( p[index]/**p*/ ) {
                    case '#':
                    case '+':
                        yyval++;
                        //p++;
                        index++;
                        break;
                    case 'b':
                    case 'B':
                    case '-':
                        yyval--;
                        //p++;
                        index++;
                        break;
                }
                yyval += 12 * int.Parse( p );//atoi( p );
            }
            if ( yyval < 0 || yyval > 127 )
                error( "Note must be between 0 and 127" );
            data[0] = (char)yyval;
        }


        static void syntax() {//staticに変更した
            prs_error( "Syntax error" );
        }


        static void checkval() {
            if ( yylex() != VAL || yylex() != INT )
                syntax();
            if ( yyval < 0 || yyval > 127 )
                error( "Value must be between 0 and 127" );
            data[1] = (char)yyval;
        }


        static void splitval() {
            if ( yylex() != VAL || yylex() != INT )
                syntax();
            if ( yyval < 0 || yyval > 16383 )
                error( "Value must be between 0 and 16383" );
            data[0] = (char)(yyval % 128);
            data[1] = (char)(yyval / 128);
        }


        static void get16val() {
            if ( yylex() != VAL || yylex() != INT )
                syntax();
            if ( yyval < 0 || yyval > 65535 )
                error( "Value must be between 0 and 65535" );
            data[0] = (char)((yyval >> 8) & 0xff);
            data[1] = (char)(yyval & 0xff);
        }


        static void checkcon() {
            if ( yylex() != CON || yylex() != INT )
                syntax();
            if ( yyval < 0 || yyval > 127 )
                error( "Controller must be between 0 and 127" );
            data[0] = (char)yyval;
        }


        void WriteVarLen( ulong value ) {
            ulong buffer;

            buffer = value & 0x7f;
            while ( (value >>= 7) > 0 ) {
                buffer <<= 8;
                buffer |= 0x80;
                buffer += (value & 0x7f);
            }
            while ( true/*1*/ ) {
                eputc( (int)(buffer & 0xff) );

                if ( (buffer & 0x80) != FALSE )
                    buffer >>= 8;
                else
                    return;
            }
        }/* end of WriteVarLen */


        int mf_w_midi_event( ulong delta_time, uint type, uint chan, char[] data, ulong size ) {
            //int i;
            /*unsigned */
            char c;

            WriteVarLen( delta_time );

            /* all MIDI events start with the type in the first four bits,
               and the channel in the lower four bits */
            c = (char)(type | chan);

            if ( chan > 15 ) {
                //perror( "error: MIDI channel greater than 16\n" );
                Console.Error.WriteLine( "error: MIDI channel greater than 16" );
            }

            if ( Mf_RunStat == FALSE || laststat != c )
                eputc( c );

            laststat = c;

            /* write out the data bytes */
            for ( ulong i = 0; i < size; i++ ) {
                eputc( (int)(data[i]) );
            }

            return (int)size;
        } /* end mf_write MIDI event */


        int mf_w_meta_event( ulong delta_time, uint type, char[] data, ulong size ) {
            //int i;

            WriteVarLen( delta_time );

            /* This marks the fact we're writing a meta-event */
            eputc( meta_event );
            laststat = meta_event;

            /* The type of meta event */
            eputc( (int)type );
            lastmeta = (int)type;

            /* The length of the data bytes to follow */
            WriteVarLen( size );

            for ( ulong i = 0; i < size; i++ ) {
                if ( eputc( data[i] ) != data[i] ) {
                    return (-1);
                }
            }
            return (int)size;
        } /* end mf_w_meta_event */


        static void checkprog() {
            if ( yylex() != PROG || yylex() != INT )
                syntax();
            if ( yyval < 0 || yyval > 127 )
                error( "Program number must be between 0 and 127" );
            data[0] = (char)yyval;
        }


        int mf_w_sysex_event( ulong delta_time, char[] data, ulong size ) {
            //int i;

            WriteVarLen( delta_time );

            /* The type of sysex event */
            eputc( data[0] );//eputc( *data );
            laststat = 0;

            /* The length of the data bytes to follow */
            WriteVarLen( size - 1 );

            for ( ulong i = 1; i < size; i++ ) {
                if ( eputc( data[i] ) != data[i] )
                    return (-1);
            }
            return (int)size;
        } /* end mf_w_sysex_event */


        void mf_w_tempo( ulong delta_time, ulong tempo ) {
            /* Write tempo */
            /* all tempos are written as 120 beats/minute, */
            /* expressed in microseconds/quarter note     */

            WriteVarLen( delta_time );

            eputc( meta_event );
            laststat = meta_event;
            eputc( set_tempo );

            eputc( 3 );
            eputc( (int)(0xff & (tempo >> 16)) );
            eputc( (int)(0xff & (tempo >> 8)) );
            eputc( (int)(0xff & tempo) );
        }


        int mywritetrack() {
            int opcode, c;
            long currtime = 0;
            long newtime, delta;
            int i, k;

            while ( (opcode = yylex()) == EOL )
                ;
            if ( opcode != MTRK )
                prs_error( "Missing MTrk" );
            checkeol();
            while ( true/*1*/ ) {
                err_cont = 1;
                //setjmp( erjump );
                try {
                    switch ( yylex() ) {
                        case MTRK:
                            prs_error( "Unexpected MTrk" );
                            continue;
                        case EOF:
                            err_cont = 0;
                            error( "Unexpected EOF" );
                            return -1;
                        case TRKEND:
                            err_cont = 0;
                            checkeol();
                            return 1;
                        case INT:
                            newtime = yyval;
                            if ( (opcode = yylex()) == '/' ) {
                                if ( yylex() != INT )
                                    prs_error( "Illegal time value" );
                                newtime = (newtime - M0) * Measure + yyval;
                                if ( yylex() != '/' || yylex() != INT )
                                    prs_error( "Illegal time value" );
                                newtime = T0 + newtime * Beat + yyval;
                                opcode = yylex();
                            }
                            delta = newtime - currtime;
                            #region switch( opcode )
                            switch ( opcode ) {
                                case ON:
                                case OFF:
                                case POPR:
                                    checkchan();
                                    checknote();
                                    checkval();
                                    mf_w_midi_event( (ulong)delta, (uint)opcode, (uint)chan, data, (char)2L );
                                    break;

                                case PAR:
                                    checkchan();
                                    checkcon();
                                    checkval();
                                    mf_w_midi_event( (ulong)delta, (uint)opcode, (uint)chan, data, (char)2L );
                                    break;

                                case PB:
                                    checkchan();
                                    splitval();
                                    mf_w_midi_event( (ulong)delta, (uint)opcode, (uint)chan, data, (char)2L );
                                    break;

                                case PRCH:
                                    checkchan();
                                    checkprog();
                                    mf_w_midi_event( (ulong)delta, (uint)opcode, (uint)chan, data, (char)1L );
                                    break;

                                case CHPR:
                                    checkchan();
                                    checkval();
                                    data[0] = data[1];
                                    mf_w_midi_event( (ulong)delta, (uint)opcode, (uint)chan, data, (char)1L );
                                    break;

                                case SYSEX:
                                case ARB:
                                    gethex();
                                    mf_w_sysex_event( (ulong)delta, buffer, (ulong)buflen );
                                    break;

                                case TEMPO:
                                    if ( yylex() != INT )
                                        syntax();
                                    mf_w_tempo( (ulong)delta, (ulong)yyval );
                                    break;

                                case TIMESIG: {
                                        int nn, denom, cc, bb;
                                        if ( yylex() != INT || yylex() != '/' )
                                            syntax();
                                        nn = (int)yyval;
                                        denom = getbyte( "Denom" );
                                        cc = getbyte( "clocks per click" );
                                        bb = getbyte( "32nd notes per 24 clocks" );
                                        for ( i = 0, k = 1; k < denom; i++, k <<= 1 )
                                            ;
                                        if ( k != denom )
                                            error( "Illegal TimeSig" );
                                        data[0] = (char)nn;
                                        data[1] = (char)i;
                                        data[2] = (char)cc;
                                        data[3] = (char)bb;
                                        M0 += (int)((newtime - T0) / (Beat * Measure));
                                        T0 = newtime;
                                        Measure = nn;
                                        Beat = 4 * Clicks / denom;
                                        mf_w_meta_event( (ulong)delta, time_signature, data, 4L );
                                    }
                                    break;

                                case SMPTE:
                                    for ( i = 0; i < 5; i++ ) {
                                        data[i] = (char)getbyte( "SMPTE" );
                                    }
                                    mf_w_meta_event( (ulong)delta, smpte_offset, data, 5L );
                                    break;

                                case KEYSIG:
                                    data[0] = (char)getint( "Keysig" );
                                    if ( data[0] < -7 || data[0] > 7 )
                                        error( "Key Sig must be between -7 and 7" );
                                    if ( (c = yylex()) != MINOR && c != MAJOR ) {
                                        syntax();
                                    }
                                    if ( c == MINOR ) {
                                        data[1] = (char)TRUE;//(c == MINOR);
                                    } else {
                                        data[1] = (char)FALSE;
                                    }
                                    mf_w_meta_event( (ulong)delta, key_signature, data, 2L );
                                    break;

                                case SEQNR:
                                    //get16val( "SeqNr" );
                                    get16val();//
                                    mf_w_meta_event( (ulong)delta, sequence_number, data, 2L );
                                    break;

                                case META: {
                                        int type = yylex();
                                        switch ( type ) {
                                            case TRKEND:
                                                type = end_of_track;
                                                break;
                                            case TEXT:
                                            case COPYRIGHT:
                                            case SEQNAME:
                                            case INSTRNAME:
                                            case LYRIC:
                                            case MARKER:
                                            case CUE:
                                                type -= (META + 1);
                                                break;
                                            case INT:
                                                type = (int)yyval;
                                                break;
                                            default:
                                                prs_error( "Illegal Meta type" );
                                                break;
                                        }
                                        if ( type == end_of_track )
                                            buflen = 0;
                                        else
                                            gethex();
                                        mf_w_meta_event( (ulong)delta, (uint)type, buffer, (ulong)buflen );
                                        break;
                                    }
                                case SEQSPEC:
                                    gethex();
                                    mf_w_meta_event( (ulong)delta, sequencer_specific, buffer, (ulong)buflen );
                                    break;
                                default:
                                    prs_error( "Unknown input" );
                                    break;
                            }
                            #endregion
                            currtime = newtime;
                            break;
                        case EOL:
                            break;
                        default:
                            prs_error( "Unknown input" );
                            break;
                    }
                } catch {
                    continue;
                }
                checkeol();
            }
        }


        static int getint( string mess ) {//changed to static
            //char[] ermesg = new char[100];
            if ( yylex() != INT ) {
                string ermesg = "Integer expected for " + mess;//sprintf( ermesg, "Integer expected for %s", mess );
                error( ermesg );
                yyval = 0;
            }
            return (int)yyval;
        }


        int fileputc( int c ) {
            //return  putc(c, F);
            F.WriteByte( (byte)c );
            return 0;
        }

        #endregion


        #region Functions to be called while processing the MIDI file.

        private int getc() {
            try {
                int res;
                res = F.ReadByte();
                return res;
            } catch ( Exception e ) {
                Console.Error.WriteLine( e.StackTrace );
                return EOF;
            }
        }

        private static void error( string s ) {//static二変更
            /*if ( TrksToDo <= 0 ) {
                Console.Error.WriteLine( "Error: Garbage at end" );
            } else {
                Console.Error.WriteLine( "Error: " + s );
            }*/
        }


        private void header( int format, int ntrks, int division ) {
            if ( (division & 0x8000) != FALSE ) {
                // SMPTE
                times = 0;   /* Can't do beats */
                sb.Append( "MFile " + format + " " + ntrks + " " + (-((-(division >> 8)) & 0xff)) + " " + (division & 0xff) + "\n" );
            } else {
                sb.Append( "MFile " + format + " " + ntrks + " " + division + "\n" );
            }
            //Console.Writef("MFile %d %d %d\n",format,ntrks,division);
            if ( format > 2 ) {
                //fprintf(stderr, "Can't deal with format %d files\n", format);
                Console.Error.WriteLine( "Can't deal with format " + format + " files" );
                //Console.Writef("Can't deal with format %d files\n", format);
                //System.Environment.Exit( 1 );
                return;
            }
            Beat = Clicks = division;
            TrksToDo = ntrks;
        }

        private void starttrack() {
            //sw.WriteLine( "MTrk" );
            sb.Append( "MTrk\n" );
            TrkNr++;
        }

        private void endtrack() {
            //sw.WriteLine( "TrkEnd" );
            sb.Append( "TrkEnd\n" );
            --TrksToDo;
        }

        private void on( int chan, int pitch, int v ) {
            prtime();
            //sw.WriteLine( "On ch=" + (chan + 1) + " " + note + "=" + mknote( pitch ) + " " + vol + "=" + v );
            sb.Append( "On ch=" + (chan + 1) + " " + note + "=" + mknote( pitch ) + " " + vol + "=" + v + "\n" );
        }

        private void off( int chan, int pitch, int v ) {
            prtime();
            //sw.WriteLine( "Off ch=" + (chan + 1) + " " + note + "=" + mknote( pitch ) + " " + vol + "=" + v );
            sb.Append( "Off ch=" + (chan + 1) + " " + note + "=" + mknote( pitch ) + " " + vol + "=" + v + "\n" );
        }

        private void pressure( int chan, int pitch, int press ) {
            prtime();
            //sw.WriteLine( PolyPr + " ch=" + (chan + 1) + " " + note + "=" + mknote( pitch ) + " " + val + "=" + press );
            sb.Append( PolyPr + " ch=" + (chan + 1) + " " + note + "=" + mknote( pitch ) + " " + val + "=" + press + "\n" );
        }

        struct Comment {
            public int Index;
            public string Msb;
            public string Lsb;
            public Comment( int index, string msb, string lsb ) {
                Index = index;
                Msb = msb;
                Lsb = lsb;
            }
        }
        
        private void parameter( int chan, int control, int value ) {
#if DEBUG
            Comment[] list = new Comment[]{
                new Comment( 0x5000, "NOTE Version number", "NOTE device number" ),
                new Comment( 0x5001, "NOTE Delay in millisec MSB", "NOTE Delay in millisec LSB" ),
                new Comment( 0x5002, "Note number", "" ),
                new Comment( 0x5003, "Velocity", "" ),
                new Comment( 0x5004, "Note Duration in millisec MSB", "Note Duration in millisec LSB" ),
                new Comment( 0x5005, "Note Location(00=Inside 01=Top 02=End 03=Top&End)", "" ),
                new Comment( 0x500c, "Index of Vibrato DB MSB", "Index of Vibrato DB LSB" ),
                new Comment( 0x500d, "Index of Vibrato Type", "Duration & Continuation parameter of vibrato" ),
                new Comment( 0x500e, "Vibrato Delay", "" ),
                new Comment( 0x5012, "Number of phonetic symbols in bytes", "" ),
                new Comment( 0x5013, "Phonetic symbol 1", "Consonant adjustment 1" ),
                new Comment( 0x5014, "Phonetic symbol 2", "Consonant adjustment 2" ),
                new Comment( 0x5015, "Phonetic symbol 3", "Consonant adjustment 3" ),
                new Comment( 0x5016, "Phonetic symbol 4", "Consonant adjustment 4" ),
                new Comment( 0x5017, "Phonetic symbol 5", "Consonant adjustment 5" ),
                new Comment( 0x5018, "Phonetic symbol 6", "Consonant adjustment 6" ),
                new Comment( 0x5019, "Phonetic symbol 7", "Consonant adjustment 7" ),
                new Comment( 0x501a, "Phonetic symbol 8", "Consonant adjustment 8" ),
                new Comment( 0x501b, "Phonetic symbol 9", "Consonant adjustment 9" ),
                new Comment( 0x501c, "Phonetic symbol 10", "Consonant adjustment 10" ),
                new Comment( 0x501d, "Phonetic symbol 11", "Consonant adjustment 11" ),
                new Comment( 0x501e, "Phonetic symbol 12", "Consonant adjustment 12" ),
                new Comment( 0x501f, "Phonetic symbol 13", "Consonant adjustment 13" ),
                new Comment( 0x5020, "Phonetic symbol 14", "Consonant adjustment 14" ),
                new Comment( 0x5021, "Phonetic symbol 15", "Consonant adjustment 15" ),
                new Comment( 0x5022, "Phonetic symbol 16", "Consonant adjustment 16" ),
                new Comment( 0x5023, "Phonetic symbol 17", "Consonant adjustment 17" ),
                new Comment( 0x5024, "Phonetic symbol 18", "Consonant adjustment 18" ),
                new Comment( 0x5025, "Phonetic symbol 19", "Consonant adjustment 19" ),
                new Comment( 0x5026, "Phonetic symbol 20", "Consonant adjustment 20" ),
                new Comment( 0x5027, "Phonetic symbol 21", "Consonant adjustment 21" ),
                new Comment( 0x5028, "Phonetic symbol 22", "Consonant adjustment 22" ),
                new Comment( 0x5029, "Phonetic symbol 23", "Consonant adjustment 23" ),
                new Comment( 0x502a, "Phonetic symbol 24", "Consonant adjustment 24" ),
                new Comment( 0x502b, "Phonetic symbol 25", "Consonant adjustment 25" ),
                new Comment( 0x502c, "Phonetic symbol 26", "Consonant adjustment 26" ),
                new Comment( 0x502d, "Phonetic symbol 27", "Consonant adjustment 27" ),
                new Comment( 0x502e, "Phonetic symbol 28", "Consonant adjustment 28" ),
                new Comment( 0x502f, "Phonetic symbol 29", "Consonant adjustment 29" ),
                new Comment( 0x5030, "Phonetic symbol 30", "Consonant adjustment 30" ),
                new Comment( 0x5031, "Phonetic symbol 31", "Consonant adjustment 31" ),
                new Comment( 0x5032, "Phonetic symbol 32", "Consonant adjustment 32" ),
                new Comment( 0x504f, "00=continue 7F=EndofPhoneticSymbols", "" ),
                new Comment( 0x5050, "v1mean in Cent/5", "" ),
                new Comment( 0x5051, "d1mean in millisec/5", "" ),
                new Comment( 0x5052, "d1meanFirstNote in millisec/5", "" ),
                new Comment( 0x5053, "d2mean in millisec/5", "" ),
                new Comment( 0x5054, "d4mean in millisec/5", "" ),
                new Comment( 0x5055, "pMeanOnsetFirstNote in Cent/5", "" ),
                new Comment( 0x5056, "vMeanNoteTransition in Cent/5", "" ),
                new Comment( 0x5057, "pMeanEndingNote in Cent/5", "" ),
                new Comment( 0x5058, "AddScoopToUpIntervals&AddPortamentoToDownIntervals", "" ),
                new Comment( 0x5059, "changeAfterPeak", "" ),
                new Comment( 0x505a, "Accent", "" ),
                new Comment( 0x507f, "00=continue 7F=EndofNoteMessage", "" ),
                /* VOCALOID2 Bank Select (MSB/LSB) */
                new Comment( 0x6000, "BNK Version number", "BNK device number" ),
                new Comment( 0x6001, "BNK Delay in millisec MSB", "BNK Delay in millisec LSB" ),
                new Comment( 0x6002, "Language type MSB", "Language type LSB" ),
                /* VOCALOID2 Channel Volume */
                new Comment( 0x6100, "VOL Version number", "VOL device number" ),
                new Comment( 0x6101, "VOL Delay in millisec MSB", "VOL Delay in millisec LSB" ),
                new Comment( 0x6102, "Volume value", "" ),
                /* VOCALOID2 Panpot */
                new Comment( 0x6200, "PAN Version number", "PAN device number" ),
                new Comment( 0x6201, "PAN Delay in millisec MSB", "PAN Delay in millisec LSB" ),
                new Comment( 0x6202, "Pan value", "" ),
                /* VOCALOID2 Expression */
                new Comment( 0x6300, "EXP Version number", "EXP device number" ),
                new Comment( 0x6301, "EXP Delay in millisec MSB", "EXP Delay in millisec LSB" ),
                new Comment( 0x6302, "Expression value", "" ),
                /* VOCALOID2 Vibrato Rate */
                new Comment( 0x6400, "VBR Version number", "VBR device number" ),
                new Comment( 0x6401, "VBR Delay in millisec MSB", "VBR Delay in millisec LSB" ),
                new Comment( 0x6402, "Vibrato Rate value", "" ),
                /* VOCALOID2 Vibrato Depth */
                new Comment( 0x6500, "VBD Version number", "VBD device number" ),
                new Comment( 0x6501, "VBD Delay in millisec MSB", "VBD Delay in millisec LSB" ),
                new Comment( 0x6502, "Vibrato Depth value", "" ),
                /* VOCALOID2 Pitch Bend Sensitivity */
                new Comment( 0x6700, "PBS Version number", "PBS device number" ),
                new Comment( 0x6701, "PBS Delay in millisec MSB", "PBS Delay in millisec LSB" ),
                new Comment( 0x6702, "Pitch Bend Sensitivity MSB", "Pitch Bend Sensitivity LSB" ),
                /* Begin of VOCALOID2 MIDI(BODM) */
                new Comment( 0x5100, "BODM Version number", "BODM device number" ),
                /* End of VOCALOID2 MIDI(EODM) */
                new Comment( 0x5200, "EODM Version number", "EODM device number" ),
                /* VOCALOID2 Program Change */
                new Comment( 0x5300, "PGM Version number", "PGM device number" ),
                new Comment( 0x5301, "PGM Delay in millisec MSB", "PGM Delay in millisec LSB" ),
                new Comment( 0x5302, "Voice type", "" ),
                /* VOCALOID2 Pitch Bend */
                new Comment( 0x5400, "PIT Version number", "PIT device number" ),
                new Comment( 0x5401, "PIT Delay in millisec MSB", "PIT Delay in millisec LSB" ),
                new Comment( 0x5402, "Pitch Bend value MSB", "Pitch Bend value LSB" ),
                /* VOCALOID2 Voice Change Parameters */
                new Comment( 0x5500, "VCE Version number", "VCE device number" ),
                new Comment( 0x5501, "VCE Delay in millisec MSB", "VCE Delay in millisec LSB" ),
                new Comment( 0x5502, "Voice Change Parameter ID", "" ),
                new Comment( 0x5503, "Voice Change Parameter value", "" ),
            };
            int nrpn;
            switch ( control ) {
                case 99:
                    m_nrpn_msb = value;
                    break;
                case 98:
                    m_nrpn_lsb = value;
                    break;
                case 6:
                    prtime();
                    nrpn = (m_nrpn_msb << 8) | m_nrpn_lsb;
                    sb.Append( "NRPN " + nrpn.ToString( "X4" ) + " " + value.ToString( "X2" ) + ".." );
                    /*if ( nrpn == 0x5050 ) {
                        if ( m_sw_v1mean == null ) {
                            m_sw_v1mean = new StreamWriter( "v1mean.txt", false );
                        }
                        m_sw_v1mean.WriteLine( Mf_currtime + "\t" + value );
                    } else if ( nrpn == 0x5051 ) {
                        if ( m_sw_d1mean == null ) {
                            m_sw_d1mean = new StreamWriter( "d1mean.txt", false );
                        }
                        m_sw_d1mean.WriteLine( Mf_currtime + "\t" + value );
                    } else if ( nrpn == 0x5052 ) {
                        if ( m_sw_d1meanFirstNote == null ) {
                            m_sw_d1meanFirstNote = new StreamWriter( "d1meanFirstNote.txt", false );
                        }
                        m_sw_d1meanFirstNote.WriteLine( Mf_currtime + "\t" + value );
                    } else if ( nrpn == 0x5053 ) {
                        if ( m_sw_d2mean == null ) {
                            m_sw_d2mean = new StreamWriter( "d2mean.txt", false );
                        }
                        m_sw_d2mean.WriteLine( Mf_currtime + "\t" + value );
                    } else if ( nrpn == 0x5054 ) {
                        if ( m_sw_d4mean == null ) {
                            m_sw_d4mean = new StreamWriter( "d4mean.txt", false );
                        }
                        m_sw_d4mean.WriteLine( Mf_currtime + "\t" + value );
                    }*/
                    for ( int i = 0; i < list.GetUpperBound( 0 ) + 1; i++ ) {
                        if ( list[i].Index == nrpn ) {
                            sb.Append( " " + list[i].Msb );
                            break;
                        }
                    }
                    sb.Append( "\n" );
                    break;
                case 38:
                    prtime();
                    nrpn = (m_nrpn_msb << 8) | m_nrpn_lsb;
                    sb.Append( "NRPN " + nrpn.ToString( "X4" ) + " .." + value.ToString( "X2" ) );
                    for ( int i = 0; i < list.GetUpperBound( 0 ) + 1; i++ ) {
                        if ( list[i].Index == nrpn ) {
                            sb.Append( " " + list[i].Lsb );
                            break;
                        }
                    }
                    sb.Append( "\n" );
                    break;
                default:
                    prtime();
                    sb.Append( "\n" );
                    break;
            }
#else
            prtime();
            //sw.WriteLine( Param + " ch=" + (chan + 1) + " " + con + "=" + control + " " + val + "=" + value );
            sb.Append( Param + " ch=" + (chan + 1) + " " + con + "=" + control + " " + val + "=" + value + "\n" );
#endif
        }

        private void pitchbend( int chan, int lsb, int msb ) {
            prtime();
            //sw.WriteLine( "Pb ch=" + (chan + 1) + " " + val + "=" + (128 * msb + lsb) );
            sb.Append( "Pb ch=" + (chan + 1) + " " + val + "=" + (128 * msb + lsb) + "\n" );
        }

        private void program( int chan, int program ) {
            prtime();
            //sw.WriteLine( ProgCh + " ch=" + (chan + 1) + " " + prog + "=" + program );
            sb.Append( ProgCh + " ch=" + (chan + 1) + " " + prog + "=" + program + "\n" );
        }

        private void chanpressure( int chan, int press ) {
            prtime();
            //sw.WriteLine( ChanPr + " ch=" + (chan + 1) + " " + val + "=" + press );
            sb.Append( ChanPr + " ch=" + (chan + 1) + " " + val + "=" + press + "\n" );
        }

        private void sysex() {
            sysex( msgleng(), msg() );
        }
        private void sysex( int leng, String mess ) {
            prtime();
            //sw.Write( "SysEx" );
            sb.Append( "SysEx" );
            prhex( mess, leng );
        }

        private void arbitrary( int leng, string mess ) {
            prtime();
            //sw.Write( "Arb" );
            sb.Append( "Arb" );
            prhex( mess, leng );
        }

        private void metamisc( int type, int leng, String mess ) {
            prtime();
            //sw.Write( "Meta 0x" + Convert.ToString( type, 16 ) );
            sb.Append( "Meta 0x" + Convert.ToString( type, 16 ) );
            prhex( mess, leng );
        }

        private void seqnum( int num ) {
            prtime();
            //sw.WriteLine( "SeqNr " + num );
            sb.Append( "SeqNr " + num + "\n" );
        }

        private void eot() {
            prtime();
            //sw.WriteLine( "Meta TrkEnd" );
            sb.Append( "Meta TrkEnd\n" );
        }

        private void smpte( int hr, int mn, int se, int fr, int ff ) {
            prtime();
            //sw.WriteLine( "SMPTE " + hr + " " + mn + " " + se + " " + fr + " " + ff );
            sb.Append( "SMPTE " + hr + " " + mn + " " + se + " " + fr + " " + ff + "\n" );
        }

        private void tempo( long tempo ) {
            prtime();
            //sw.WriteLine( "Tempo " + tempo );
            sb.Append( "Tempo " + tempo +"\n" );
        }

        private void timesig( int nn, int dd, int cc, int bb ) {
            int denom = 1;
            while ( dd-- > 0 ) {
                denom *= 2;
            }
            prtime();
            //sw.WriteLine( "TimeSig " + nn + "/" + denom + " " + cc + " " + bb );
            sb.Append( "TimeSig " + nn + "/" + denom + " " + cc + " " + bb + "\n" );
            M0 += (int)(Mf_currtime - T0) / (Beat * Measure);
            T0 = Mf_currtime;
            Measure = nn;
            Beat = 4 * Clicks / denom;
        }

        private void keysig( int sf, int mi ) {
            prtime();
            if ( mi != FALSE ) {
                //sw.WriteLine( "KeySig " + sf + " " + "minor" );
                sb.Append( "KeySig " + sf + " " + "minor\n" );
            } else {
                //sw.WriteLine( "KeySig " + sf + " " + "major" );
                sb.Append( "KeySig " + sf + " " + "major\n" );
            }
        }

        private void sqspecific( int leng, string mess ) {
            prtime();
            //sw.Write( "SeqSpec" );
            sb.Append( "SeqSpec" );
            prhex( mess, leng );
        }

        private void text( int type, int leng, string mess ) {
            string[] ttype = {
	            "",
	            "Text",		/* type=0x01 */
	            "Copyright",	/* type=0x02 */
	            "TrkName",
	            "InstrName",	/* ...       */
	            "Lyric",
	            "Marker",
	            "Cue",		/* type=0x07 */
	            "Unrec"
            };

            int unrecognized = ttype.Length - 1;

            prtime();
            if ( type < 1 || type > unrecognized ) {
                //sw.Write( "Meta 0x" + Convert.ToString( type, 16 ) );
                sb.Append( "Meta 0x" + Convert.ToString( type, 16 ) );
                prtext( mess, leng );
            } else if ( type == 3 && TrkNr == 1 ) {
                //sw.Write( "Meta SeqName " );
                sb.Append( "Meta SeqName " );
                prtext( mess, leng );
            } else {
                //sw.Write( "Meta " + ttype[type] + " " );
                sb.Append( "Meta " + ttype[type] + " " );
                prtext( mess, leng );
            }
        }

        #endregion

        #region support functions for mf2t

        private void prtime() {
            if ( times != FALSE ) {
                long m = (Mf_currtime - T0) / Beat;
                //sw.Write( "" + (m / Measure + M0) + ":" + (m % Measure) + ":" + ((Mf_currtime - T0) % Beat) + " " );
                sb.Append( "" + (m / Measure + M0) + ":" + (m % Measure) + ":" + ((Mf_currtime - T0) % Beat) + " " );
            } else {
                //sw.Write( "" + Mf_currtime + " " );
                sb.Append( "" + Mf_currtime + " " );
            }
        }

        private void prtext( string p, int leng ) {
            //Console.Error.WriteLine( "prtext" );
            int n;
            char c;
            int pos = 25;
            //int index = -1;
            //sw.Write( "\"" );
            sb.Append( "\"" );
            for ( n = 0; n < leng; n++ ) {
                //index++;
                c = char.Parse( p.Substring( n, 1 ) );
                if ( fold != FALSE && (pos >= fold) ) {
                    //sw.Write( "\\" + Environment.NewLine + "\t" );//"\\\n\t");
                    sb.Append( "\\" + Environment.NewLine + "\t" );
                    pos = 13;	/* tab + \xab + \ */
                    if ( c == ' ' || c == '\t' ) {
                        //sw.Write( "\\" );
                        sb.Append( "\\" );
                        ++pos;
                    }
                }
                switch ( c ) {
                    case '\\':
                    case '"':
                        //sw.Write( "\\" + c );
                        sb.Append( "\\" + c );
                        pos += 2;
                        break;
                    case '\r':
                        //sw.Write( "\\r" );
                        sb.Append( "\\r" );
                        pos += 2;
                        break;
                    case '\n':
                        //sw.Write( "\\n" );
                        sb.Append( "\\n" );
                        pos += 2;
                        break;
                    case '\0':
                        //sw.Write( "\\-" );
                        sb.Append( "\\-" );
                        pos += 2;
                        break;
                    default:
                        if ( isprint( c ) ) {
                            //sw.Write( c.ToString() );
                            sb.Append( c.ToString() );
                            ++pos;
                        } else {
                            //sw.Write( "\\x" + Convert.ToString( (int)c, 16 ) );
                            sb.Append( "\\x" + Convert.ToString( (int)c, 16 ) );
                            pos += 4;
                        }
                        break;
                }
                //Console.Error.WriteLine( "in for loop" );
            }
            //Console.Error.WriteLine( "buffer=" + buffer );
            //sw.WriteLine( "\"" );
            sb.Append( "\"\n" );
        }

        private string mknote( int pitch ) {
            string[] Notes = { "c", "c#", "d", "d#", "e", "f", "f#", "g", "g#", "a", "a#", "b" };
            if ( notes != FALSE ) {
                return Notes[pitch % 12] + pitch / 12;
            } else {
                return "" + pitch;
            }
        }

        private int msgleng() {
            return Msgindex;
        }


        private string msg() {
            return Msgbuff;
        }

        private void prhex( string p, int leng ) {
            int n;
            int pos = 25;

            int index = -1;
            for ( n = 0; n < leng; n++, index++ ) {
                if ( fold != FALSE && (pos) >= fold ) {
                    //sw.Write( "\\" + Environment.NewLine + "\t" + Convert.ToString( (int)char.Parse( p.Substring( index, 1 ) ), 16 ) );
                    sb.Append( "\\" + Environment.NewLine + "\t" + Convert.ToString( (int)char.Parse( p.Substring( index, 1 ) ), 16 ) );
                    pos = 14;	/* tab + ab + " ab" + \ */
                } else {
                    //sw.Write( " " + Convert.ToString( (int)char.Parse( p.Substring( index, 1 ) ), 16 ) );
                    sb.Append( " " + Convert.ToString( (int)char.Parse( p.Substring( index, 1 ) ), 16 ) );
                    pos += 3;
                }
            }
            //sw.WriteLine( "" );
            sb.Append( "\n" );
        }

        private static bool isprint( char ch ) {
            if ( 32 <= (int)ch && (int)ch <= 126 ) {
                return true;
            } else {
                return false;
            }
        }

        private void readheader() {
            if ( readmt( "MThd" ) != -1 ) {
                Mf_toberead = read32bit();
                int format = read16bit();
                int ntrks = read16bit();
                int division = read16bit();
                header( format, ntrks, division );
                while ( Mf_toberead > 0L ) {
                    egetc();
                }
            }
        }

        private int readtrack() {
            int[] ttype = { 0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 2, 2, 1, 1, 2, 0 };
            int num2 = 0;
            int num4 = 0;
            int num5 = 0;
            int status = 0;
            if ( readmt( "MTrk" ) == -1 ) {
                return 0;
            }
            Mf_toberead = read32bit();
            Mf_currtime = 0L;
            starttrack();
            while ( Mf_toberead > 0L ) {
                long unrecognized;
                Mf_currtime += readvarinum();
                int c = egetc();
                if ( (num4 != 0) && (c != 0xf7) ) {
                    mferror( "didn't find expected continuation of a sysex" );
                }
                if ( (c & 0x80) == 0 ) {
                    if ( status == 0 ) {
                        mferror( "unexpected running status" );
                    }
                    num5 = 1;
                    num2 = c;
                    c = status;
                } else if ( c < 240 ) {
                    status = c;
                    num5 = 0;
                }
                int num7 = ttype[(c >> 4) & 15];
                if ( num7 != 0 ) {
                    if ( num5 == 0 ) {
                        num2 = egetc();
                    }
                    chanmessage( status, num2, (num7 <= 1) ? 0 : egetc() );
                    continue;
                }
                switch ( c ) {
                    case 0xff: {
                            int type = egetc();
                            unrecognized = (Mf_toberead - readvarinum()) - 1L;
                            msginit();
                            while ( Mf_toberead > unrecognized ) {
                                msgadd( egetc() );
                            }
                            metaevent( type );
                            continue;
                        }
                    case 240: {
                            unrecognized = Mf_toberead - readvarinum();
                            msginit();
                            msgadd( 240 );
                            while ( Mf_toberead > unrecognized ) {
                                msgadd( c = egetc() );
                            }
                            if ( (c != 0xf7) && (Mf_nomerge != 0) ) {
                                break;
                            }
                            sysex();
                            continue;
                        }
                    case 0xf7: {
                            unrecognized = Mf_toberead - readvarinum();
                            if ( num4 == 0 ) {
                                msginit();
                            }
                            while ( Mf_toberead > unrecognized ) {
                                msgadd( c = egetc() );
                            }
                            if ( num4 == 0 ) {
                                arbitrary( msgleng(), msg() );
                            } else if ( c == 0xf7 ) {
                                sysex();
                                num4 = 0;
                            }
                            continue;
                        }
                    default:
                        goto Label_0260;
                }
                num4 = 1;
                continue;
            Label_0260:
                badbyte( c );
            }
            endtrack();
            //Console.Write( buffer );
            return 1;
        }

        private int readmt( string s ) {
            string res = s;
            int num2 = 4;
            int[] e = new int[num2];
            e[0] = getc();
            e[1] = getc();
            e[2] = getc();
            e[3] = getc();
            for ( int i = 0; i < 4; i++ ) {
                if ( e[i] != char.Parse( res.Substring( i, 1 ) ) ) {
                    mferror( "expecting " + s );
                }
            }
            return e[3];
        }

        private void mferror( string s ) {
            error( s );
            //System.Environment.Exit( 1 );
        }

        private int read16bit() {
            int num = egetc();
            int num2 = egetc();
            return to16bit( num, num2 );
        }


        private long read32bit() {
            int num = egetc();
            int num2 = egetc();
            int num3 = egetc();
            int num4 = egetc();
            return to32bit( num, num2, num3, num4 );
        }


        private int egetc() {
            int num = getc();
            if ( num == EOF ) {
                mferror( "premature EOF" );
            }
            Mf_toberead -= 1L;
            return num;
        }

        private long readvarinum() {
            int num2 = egetc();
            long num = num2;
            if ( (num2 & 0x80) != 0 ) {
                num &= 0x7fL;
                do {
                    num2 = egetc();
                    num = (num << 7) + (num2 & 0x7f);
                } while ( (num2 & 0x80) != 0 );
            }
            return num;
        }

        private void chanmessage( int status, int c1, int c2 ) {
            int chan = status & 15;
            switch ( (status & 240) ) {
                case 0x80:
                    off( chan, c1, c2 );
                    break;

                case 0x90:
                    on( chan, c1, c2 );
                    break;

                case 160:
                    pressure( chan, c1, c2 );
                    break;

                case 0xb0:
                    parameter( chan, c1, c2 );
                    break;

                case 0xe0:
                    pitchbend( chan, c1, c2 );
                    break;

                case 0xc0:
                    program( chan, c1 );
                    break;

                case 0xd0:
                    chanpressure( chan, c1 );
                    break;
            }
        }

        private void msginit() {
            Msgindex = 0;
            Msgbuff = "";
        }

        private void msgadd( int c ) {
            Msgbuff = Msgbuff + (char)c;
            Msgindex++;
        }

        private void metaevent( int type ) {
            int leng = msgleng();
            string m = msg();
            switch ( type ) {
                case 0:
                    seqnum( to16bit(
                        (int)char.Parse( m.Substring( 0, 1 ) ),
                        (int)char.Parse( m.Substring( 1, 1 ) ) )
                    );
                    break;
                case 1:
                case 2:
                case 3:
                case 4:
                case 5:
                case 6:
                case 7:
                case 8:
                case 9:
                case 10:
                case 11:
                case 12:
                case 13:
                case 14:
                case 15:
                    text( type, leng, m );
                    break;

                case 0x2f:
                    eot();
                    break;

                case 0x51:
                    tempo( to32bit(
                        0,
                        (int)char.Parse( m.Substring( 0, 1 ) ),
                        (int)char.Parse( m.Substring( 1, 1 ) ),
                        (int)char.Parse( m.Substring( 2, 1 ) ) )
                    );
                    break;

                case 0x54:
                    smpte(
                        (int)char.Parse( m.Substring( 0, 1 ) ),
                        (int)char.Parse( m.Substring( 1, 1 ) ),
                        (int)char.Parse( m.Substring( 2, 1 ) ),
                        (int)char.Parse( m.Substring( 3, 1 ) ),
                        (int)char.Parse( m.Substring( 4, 1 ) )
                    );
                    break;

                case 0x58:
                    timesig(
                        (int)char.Parse( m.Substring( 0, 1 ) ),
                        (int)char.Parse( m.Substring( 1, 1 ) ),
                        (int)char.Parse( m.Substring( 2, 1 ) ),
                        (int)char.Parse( m.Substring( 3, 1 ) )
                    );
                    break;

                case 0x59:
                    keysig(
                        (int)char.Parse( m.Substring( 0, 1 ) ),
                        (int)char.Parse( m.Substring( 1, 1 ) )
                    );
                    break;

                case 0x7f:
                    sqspecific( leng, m );
                    break;

                default:
                    metamisc( type, leng, m );
                    break;
            }
        }

        private void badbyte( int c ) {
            mferror( "unexpected byte: " + c );
        }

        private static int to16bit( int c1, int c2 ) {
            return (((c1 & 0xff) << 8) + (c2 & 0xff));
        }


        private static long to32bit( int c1, int c2, int c3, int c4 ) {
            long num = 0L;
            num = c1 & 0xff;
            num = (num << 8) + (c2 & 0xff);
            num = (num << 8) + (c3 & 0xff);
            return ((num << 8) + (c4 & 0xff));
        }

        #endregion

    }
}