﻿/*
 * FormBezierPointEdit.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using Boare.Lib.AppUtil;
using Boare.Lib.Vsq;

namespace Boare.Cadencii {
    
    partial class FormBezierPointEdit : Form {
        private BezierPoint m_point;
        private int m_min;
        private int m_max;
        private Point m_last_mouse_global_location;
        private TrackSelector m_parent;
        private bool m_btn_datapoint_downed = false;
        private double m_min_opacity = 0.4;
        private EditorManager m_manager;
        private CurveType m_curve_type;
        private int m_track;
        private int m_chain_id = -1;
        private int m_point_id = -1;
        private BezierPickedSide m_picked_side = BezierPickedSide.Base;

        public FormBezierPointEdit( TrackSelector parent, 
                                    EditorManager manager,
                                    CurveType curve_type,
                                    int selected_chain_id,
                                    int selected_point_id ) {
            InitializeComponent();
            ApplyLanguage();
            m_parent = parent;
            m_manager = manager;
            m_curve_type = curve_type;
            m_track = m_manager.Selected;
            m_chain_id = selected_chain_id;
            m_point_id = selected_point_id;
            bool found = false;
            List<BezierChain> chains = m_manager.VsqFile.AttachedCurves[m_track - 1][m_curve_type];
            for ( int i = 0; i < chains.Count; i++ ) {
                if ( chains[i].ID == m_chain_id ) {
                    found = true;
                    break;
                }
            }
            if ( !found ) {
                return;
            }
            bool smooth = false;
            foreach ( BezierPoint bp in m_manager.VsqFile.AttachedCurves[m_track - 1].GetBezierChain( m_curve_type, m_chain_id ).points ) {
                if ( bp.ID == m_point_id ) {
                    m_point = bp;
                    smooth = bp.ControlLeftType != BezierControlType.None || bp.ControlRightType != BezierControlType.None;
                    break;
                }
            }
            UpdateStatus();
            Misc.ApplyFontRecurse( this, AppManager.EditorConfig.BaseFont );
        }

        private void UpdateStatus() {
            txtDataPointClock.Text = m_point.Base.X.ToString();
            txtDataPointValue.Text = m_point.Base.Y.ToString();
            txtLeftClock.Text = ((int)(m_point.Base.X + m_point.m_control_left.X)).ToString();
            txtLeftValue.Text = ((int)(m_point.Base.Y + m_point.m_control_left.Y)).ToString();
            txtRightClock.Text = ((int)(m_point.Base.X + m_point.m_control_right.X)).ToString();
            txtRightValue.Text = ((int)(m_point.Base.Y + m_point.m_control_right.Y)).ToString();
            bool smooth = m_point.ControlLeftType != BezierControlType.None || m_point.ControlRightType != BezierControlType.None;
            chkEnableSmooth.Checked = smooth;
            btnLeft.Enabled = smooth;
            btnRight.Enabled = smooth;
            m_min = m_curve_type.Minimum;
            m_max = m_curve_type.Maximum;
        }

        private static string _( string message ) {
            return Messaging.GetMessage( message );
        }

        public void ApplyLanguage() {
            Text = _( "Edit Bezier Data Point" );
            
            groupDataPoint.Text = _( "Data Poin" );
            lblDataPointClock.Text = _( "Clock" );
            lblDataPointValue.Text = _( "Value" );

            groupLeft.Text = _( "Left Control Point" );
            lblLeftClock.Text = _( "Clock" );
            lblLeftValue.Text = _( "Value" );

            groupRight.Text = _( "Right Control Point" );
            lblRightClock.Text = _( "Clock" );
            lblRightValue.Text = _( "Value" );

            chkEnableSmooth.Text = _( "Smooth" );
        }
        
        private void btnOK_Click( object sender, EventArgs e ) {
            try {
                int x, y;
                x = int.Parse( txtDataPointClock.Text );
                y = int.Parse( txtDataPointValue.Text );
                if ( y < m_min || m_max < y ) {
                    MessageBox.Show( _( "Invalid value" ), _( "Error" ), MessageBoxButtons.OK, MessageBoxIcon.Error );
                    return;
                }
                if ( chkEnableSmooth.Checked ) {
                    x = int.Parse( txtLeftClock.Text );
                    y = int.Parse( txtLeftValue.Text );
                    x = int.Parse( txtRightClock.Text );
                    y = int.Parse( txtRightValue.Text );
                }
                DialogResult = DialogResult.OK;
            } catch {
                MessageBox.Show( _( "Integer format error" ), _( "Error" ), MessageBoxButtons.OK, MessageBoxIcon.Error );
            }
        }

        private void chkEnableSmooth_CheckedChanged( object sender, EventArgs e ) {
            bool value = chkEnableSmooth.Checked;
            txtLeftClock.Enabled = value;
            txtLeftValue.Enabled = value;
            btnLeft.Enabled = value;
            txtRightClock.Enabled = value;
            txtRightValue.Enabled = value;
            btnRight.Enabled = value;

            bool old = m_point.ControlLeftType != BezierControlType.None || m_point.ControlRightType != BezierControlType.None;
            if ( value ) {
                m_point.ControlLeftType = BezierControlType.Normal;
                m_point.ControlRightType = BezierControlType.Normal;
            } else {
                m_point.ControlLeftType = BezierControlType.None;
                m_point.ControlRightType = BezierControlType.None;
            }
            txtLeftClock.Text = ((int)(m_point.Base.X + m_point.m_control_left.X)).ToString();
            txtLeftValue.Text = ((int)(m_point.Base.Y + m_point.m_control_left.Y)).ToString();
            txtRightClock.Text = ((int)(m_point.Base.X + m_point.m_control_right.X)).ToString();
            txtRightValue.Text = ((int)(m_point.Base.Y + m_point.m_control_right.Y)).ToString();
            m_parent.Invalidate();
        }

        private void btnDataPoint_MouseDown( object sender, MouseEventArgs e ) {
            this.Opacity = m_min_opacity;
            m_last_mouse_global_location = Control.MousePosition;
            Point loc_on_trackselector = new Point( m_parent.XCoordFromClocks( (int)m_point.Base.X ), 
                                                    m_parent.YCoordFromValue( (int)m_point.Base.Y ) );
            Point loc_on_screen = m_parent.PointToScreen( loc_on_trackselector );
            Cursor.Position = loc_on_screen;
            MouseEventArgs event_arg = new MouseEventArgs( MouseButtons.Left, 0, loc_on_trackselector.X, loc_on_trackselector.Y, 0 );
            m_parent.TrackSelector_MouseDown( this, event_arg );
            m_picked_side = BezierPickedSide.Base;
            m_btn_datapoint_downed = true;
        }

        private void btnLeft_MouseDown( object sender, MouseEventArgs e ) {
            this.Opacity = m_min_opacity;
            m_last_mouse_global_location = Control.MousePosition;
            Point loc_on_trackselector = new Point( m_parent.XCoordFromClocks( (int)m_point.ControlLeft.X ), 
                                                    m_parent.YCoordFromValue( (int)m_point.ControlLeft.Y ) );
            Point loc_on_screen = m_parent.PointToScreen( loc_on_trackselector );
            Cursor.Position = loc_on_screen;
            MouseEventArgs event_arg = new MouseEventArgs( MouseButtons.Left, 0, loc_on_trackselector.X, loc_on_trackselector.Y, 0 );
            m_parent.TrackSelector_MouseDown( this, event_arg );
            m_picked_side = BezierPickedSide.Left;
            m_btn_datapoint_downed = true;
        }

        private void btnRight_MouseDown( object sender, MouseEventArgs e ) {
            this.Opacity = m_min_opacity;
            m_last_mouse_global_location = Control.MousePosition;
            Point loc_on_trackselector = new Point( m_parent.XCoordFromClocks( (int)m_point.ControlRight.X ), 
                                                    m_parent.YCoordFromValue( (int)m_point.ControlRight.Y ) );
            Point loc_on_screen = m_parent.PointToScreen( loc_on_trackselector );
            Cursor.Position = loc_on_screen;
            MouseEventArgs event_arg = new MouseEventArgs( MouseButtons.Left, 0, loc_on_trackselector.X, loc_on_trackselector.Y, 0 );
            m_parent.TrackSelector_MouseDown( this, event_arg );
            m_picked_side = BezierPickedSide.Right;
            m_btn_datapoint_downed = true;
        }

        private void common_MouseUp( object sender, MouseEventArgs e ) {
            m_btn_datapoint_downed = false;
            this.Opacity = 1.0;
            Point loc_on_screen = Control.MousePosition;
            Point loc_on_trackselector = m_parent.PointToClient( loc_on_screen );
            MouseEventArgs event_arg = new MouseEventArgs( MouseButtons.Left, 0, loc_on_trackselector.X, loc_on_trackselector.Y, 0 );
            m_parent.TrackSelector_MouseUp( this, event_arg );
            Cursor.Position = m_last_mouse_global_location;
            m_parent.Invalidate();
        }

        private void common_MouseMove( object sender, MouseEventArgs e ) {
            if ( m_btn_datapoint_downed ) {
                Point loc_on_screen = Control.MousePosition;
                Point loc_on_trackselector = m_parent.PointToClient( loc_on_screen );
                MouseEventArgs event_arg = new MouseEventArgs( MouseButtons.Left, 0, loc_on_trackselector.X, loc_on_trackselector.Y, 0 );
                BezierPoint ret = m_parent.HandleMouseMoveForBezierMove( event_arg, m_picked_side );

                txtDataPointClock.Text = ((int)ret.Base.X).ToString();
                txtDataPointValue.Text = ((int)ret.Base.Y).ToString();
                txtLeftClock.Text = ((int)ret.ControlLeft.X).ToString();
                txtLeftValue.Text = ((int)ret.ControlLeft.Y).ToString();
                txtRightClock.Text = ((int)ret.ControlRight.X).ToString();
                txtRightValue.Text = ((int)ret.ControlRight.Y).ToString();

                m_parent.Invalidate();
            }
        }

        private void btnBackward_Click( object sender, EventArgs e ) {
            BezierChain target = m_manager.VsqFile.AttachedCurves[m_track - 1].GetBezierChain( m_curve_type, m_chain_id );
            int index = -1;
            for ( int i = 0; i < target.Count; i++ ) {
                if ( target[i].ID == m_point_id ) {
                    index = i - 1;
                    break;
                }
            }
            if ( 0 <= index ) {
                m_point_id = target[index].ID;
                m_point = target[index];
                UpdateStatus();
                m_parent.EditingPointID = m_point_id;
                m_parent.Invalidate();
            }
        }

        private void btnForward_Click( object sender, EventArgs e ) {
            BezierChain target = m_manager.VsqFile.AttachedCurves[m_track - 1].GetBezierChain( m_curve_type, m_chain_id );
            int index = -2;
            for ( int i = 0; i < target.Count; i++ ) {
                if ( target[i].ID == m_point_id ) {
                    index = i + 1;
                    break;
                }
            }
            if ( 0 <= index && index < target.Count) {
                m_point_id = target[index].ID;
                m_point = target[index];
                UpdateStatus();
                m_parent.EditingPointID = m_point_id;
                m_parent.Invalidate();
            }
        }
    }

}
