﻿/*
 * AttachedCurve.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Runtime.Serialization;

using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    /// <summary>
    /// VsqTrackの各カーブに付属させるBezierChainのリスト
    /// </summary>
    [Serializable]
    public class AttachedCurve : ICloneable {
        /// <summary>
        /// カーブ本体
        /// </summary>
        private Dictionary<int, BezierChain>[] m_curves;

        public object Clone() {
            AttachedCurve ret = new AttachedCurve();
            for ( int i = 0; i < m_curves.Length; i++ ) {
                ret.m_curves[i] = new Dictionary<int, BezierChain>();
                foreach ( int key in m_curves[i].Keys ) {
                    ret.m_curves[i].Add( key, (BezierChain)m_curves[i][key].Clone() );
                }
            }
            return ret;
        }

        public AttachedCurve() {
            m_curves = new Dictionary<int, BezierChain>[11];
            for ( int i = 0; i < 11; i++ ) {
                m_curves[i] = new Dictionary<int, BezierChain>();
            }
        }

        public AttachedCurve( BezierCurves item ) {
            m_curves = new Dictionary<int, BezierChain>[11];
            for ( int i = 0; i < 11; i++ ) {
                m_curves[i] = new Dictionary<int, BezierChain>();
            }
            int track = -1;
            
            track++;
            for ( int i = 0; i < item.Dynamics.Length; i++ ) {
                BezierChain chain = (BezierChain)item.Dynamics[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.Brethiness.Length; i++ ) {
                BezierChain chain = (BezierChain)item.Brethiness[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.Brightness.Length; i++ ) {
                BezierChain chain = (BezierChain)item.Brightness[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.Clearness.Length; i++ ) {
                BezierChain chain = (BezierChain)item.Clearness[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.Opening.Length; i++ ) {
                BezierChain chain = (BezierChain)item.Opening[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.GenderFactor.Length; i++ ) {
                BezierChain chain = (BezierChain)item.GenderFactor[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.PortamentoTiming.Length; i++ ) {
                BezierChain chain = (BezierChain)item.PortamentoTiming[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.PitchBend.Length; i++ ) {
                BezierChain chain = (BezierChain)item.PitchBend[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.PitchBendSensitivity.Length; i++ ) {
                BezierChain chain = (BezierChain)item.PitchBendSensitivity[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.VibratoRate.Length; i++ ) {
                BezierChain chain = (BezierChain)item.VibratoRate[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }

            track++;
            for ( int i = 0; i < item.VibratoDepth.Length; i++ ) {
                BezierChain chain = (BezierChain)item.VibratoDepth[i].Clone();
                for ( int j = 0; j < chain.points.Count; j++ ) {
                    chain.points[j].ID = j;
                }
                m_curves[track].Add( i, chain );
            }
        }

        public int GetNextId( CurveType curve_type ) {
            int index = curve_type.Index;
            int ret = m_curves[index].Count;
            while ( m_curves[index].ContainsKey( ret ) ) {
                ret++;
            }
            return ret;
        }

        /// <summary>
        /// 指定したコントロールカーブにベジエ曲線を追加します。
        /// </summary>
        /// <param name="curve_type"></param>
        /// <param name="chain"></param>
        public void AddBezierChain( CurveType curve_type, BezierChain chain, int chain_id ) {
            int index = curve_type.Index;
            m_curves[index].Add( chain_id, (BezierChain)chain.Clone() );
        }

        /// <summary>
        /// 指定した種類のコントロールカーブにベジエ曲線を追加します。
        /// AddBezierChainとの違い、オーバーラップする部分があれば自動的に結合されます。
        /// chainには2個以上のデータ点が含まれている必要がある
        /// </summary>
        /// <param name="curve"></param>
        /// <param name="chain"></param>
        public void MergeBezierChain( CurveType curve, BezierChain chain ) {
            if ( chain.points.Count <= 1 ) {
                return;
            }
            int chain_start = (int)chain.Start;
            int chain_end = (int)chain.End;

            // まず、全削除する必要のあるBezierChainを検索
            List<int> delete_list = new List<int>();
            foreach ( int id in this[curve].Keys ) {
                BezierChain bc = this[curve][id];
                if ( bc.points.Count <= 0 ) {
                    continue;
                }
                int bc_start = (int)bc.Start;
                int bc_end = (int)bc.End;
                if ( chain_start <= bc_start && bc_end <= chain_end ) {
                    delete_list.Add( id );
                }
            }

            // 削除を実行
            foreach ( int id in delete_list ) {
                this[curve].Remove( id );
            }

            // マージする必要があるかどうかを検査。
            bool processed = true;
            while ( processed ) {
                processed = false;
                foreach ( int id in this[curve].Keys ) {
                    BezierChain bc = this[curve][id];
                    int start = (int)bc.Start;
                    int end = (int)bc.End;

                    // 被っている箇所が2箇所以上ある可能性があるので、ifでヒットしてもbreakしない
                    if ( start < chain_start && chain_start <= end && end < chain_end ) {
                        // bcのchain_start ~ endを削除し、chain_startで結合
                        BezierChain bc_edit = bc.ExtractPartialBezier( start, chain_start );
                        bc_edit.ID = bc.ID;
                        int last = bc_edit.Count - 1;

                        // 接合部分では、制御点無しでステップ変化する
                        bc_edit.points[last].ControlRightType = BezierControlType.None;
                        chain.points[0].ControlLeftType = BezierControlType.None;

                        int copy_start = 0;
                        if ( bc_edit.points[last].Base.Y == chain.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( bc_edit.points[last].ControlLeftType != BezierControlType.None ) {
                                bc_edit.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            bc_edit.points[last].ControlRight = chain.points[0].m_control_left;
                            if ( chain.points[0].ControlRightType != BezierControlType.None ) {
                                bc_edit.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            copy_start = 1;
                        }
                        for ( int i = copy_start; i < chain.points.Count; i++ ) {
                            chain.points[i].ID = bc_edit.GetNextId();
                            bc_edit.Add( chain.points[i] );
                        }
                        this[curve].Remove( id );
                        chain = bc_edit;
                        chain_start = (int)chain.Start;
                        chain_end = (int)chain.End;
                        processed = true;
                        break;
                    } else if ( chain_start < start && start <= chain_end && chain_end < end ) {
                        // bcのstart ~ chain_endを削除し、chain_endで結合
                        BezierChain bc_edit = bc.ExtractPartialBezier( chain_end, end );
                        bc_edit.ID = bc.ID;
                        int last = chain.Count - 1;

                        // 接合部分では、制御点無しでステップ変化する
                        bc_edit.points[0].ControlLeftType = BezierControlType.None;
                        chain.points[last].ControlRightType = BezierControlType.None;

                        int copy_end = last;
                        if ( chain.points[last].Base.Y == bc_edit.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( chain[last].ControlLeftType != BezierControlType.None ) {
                                chain.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            chain.points[last].ControlRight = bc_edit.points[0].m_control_left;
                            if ( bc_edit.points[0].ControlRightType != BezierControlType.None ) {
                                chain.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            copy_end = last - 1;
                        }
                        for ( int i = 0; i <= copy_end; i++ ) {
                            chain.points[i].ID = bc_edit.GetNextId();
                            bc_edit.Add( chain.points[i] );
                        }
                        this[curve].Remove( id );
                        chain = bc_edit;
                        chain_start = (int)chain.Start;
                        chain_end = (int)chain.End;
                        processed = true;
                        break;
                    } else if ( start < chain_start && chain_end < end ) {
                        // bcのchain_start ~ chain_endをchainで置き換え
                        // left + chain + right
                        BezierChain left = bc.ExtractPartialBezier( start, chain_start );
                        BezierChain right = bc.ExtractPartialBezier( chain_end, end );
                        left.ID = bc.ID;

                        // 接合部ではステップ変化
                        left.points[left.Count - 1].ControlRightType = BezierControlType.None;
                        chain.points[0].ControlLeftType = BezierControlType.None;
                        chain.points[chain.Count - 1].ControlRightType = BezierControlType.None;
                        right.points[0].ControlLeftType = BezierControlType.None;

                        int copy_start = 0;
                        int copy_end = chain.Count - 1;

                        if ( left.points[left.Count - 1].Base.Y == chain.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( left.points[left.Count - 1].ControlLeftType != BezierControlType.None ) {
                                left.points[left.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            left.points[left.Count - 1].ControlRight = chain.points[0].m_control_left;
                            if ( chain.points[0].ControlRightType != BezierControlType.None ) {
                                left.points[left.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            copy_start = 1;
                        }

                        if ( chain.points[chain.Count - 1].Base.Y == right.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( chain.points[chain.Count - 1].ControlLeftType != BezierControlType.None ) {
                                chain.points[chain.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            chain.points[chain.Count - 1].ControlRight = right.points[0].m_control_left;
                            if ( right.points[0].ControlRightType != BezierControlType.None ) {
                                chain.points[chain.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            copy_end = chain.Count - 2;
                        }

                        // 追加
                        for ( int i = copy_start; i < chain.points.Count; i++ ) {
                            chain.points[i].ID = left.GetNextId();
                            left.Add( chain.points[i] );
                        }
                        for ( int i = 0; i <= copy_end; i++ ) {
                            right.points[i].ID = left.GetNextId();
                            left.Add( right.points[i] );
                        }
                        this[curve].Remove( id );
                        chain = left;
                        chain_start = (int)chain.Start;
                        chain_end = (int)chain.End;
                        processed = true;
                        break;
                    }
                }
            }

            if ( !processed ) {
                chain.ID = this.GetNextId( curve );
            }
            this[curve].Add( chain.ID, chain );
        }

        public Dictionary<int, BezierChain> this[CurveType vct] {
            get {
                int index = vct.Index;
                if ( 0 <= index && index < m_curves.Length ) {
                    return m_curves[index];
                } else {
                    return null;
                }
            }
        }

        public bool DeleteBeziers(
            List<CurveType> target_curve,
            int clock_start,
            int clock_end
        ) {
            bool edited = false;
            foreach ( CurveType curve in target_curve ) {
                if ( curve.IsScalar || curve.IsAttachNote ) {
                    continue;
                }
                List<BezierChain> tmp = new List<BezierChain>();
                foreach ( BezierChain bc in this[curve].Values ) {
                    int len = bc.points.Count;
                    if ( len < 1 ) {
                        continue;
                    }
                    int chain_start = (int)bc.points[0].Base.X;
                    int chain_end;
                    if ( len < 2 ) {
                        chain_end = chain_start;
                    } else {
                        chain_end = (int)bc.points[len - 1].Base.X;
                    }
                    if ( clock_end < chain_start && chain_start < clock_end && clock_end < chain_end ) {
                        // end ~ chain_endを残す
                        BezierChain chain = bc.ExtractPartialBezier( clock_end, chain_end );
                        chain.ID = bc.ID;
                        tmp.Add( chain );
                        edited = true;
                    } else if ( chain_start <= clock_start && clock_end <= chain_end ) {
                        // chain_start ~ startとend ~ chain_endを残す
                        BezierChain chain1 = bc.ExtractPartialBezier( chain_start, clock_start );
                        chain1.ID = bc.ID;
                        BezierChain chain2 = bc.ExtractPartialBezier( clock_end, chain_end );
                        chain2.ID = -1;  // 後で番号をつける
                        tmp.Add( chain1 );
                        tmp.Add( chain2 );
                        edited = true;
                    } else if ( chain_start < clock_start && clock_start < chain_end && clock_end <= chain_end ) {
                        // chain_start ~ startを残す
                        BezierChain chain = bc.ExtractPartialBezier( chain_start, clock_start );
                        chain.ID = bc.ID;
                        tmp.Add( chain );
                        edited = true;
                    } else if ( clock_start <= chain_start && chain_end <= clock_end ) {
                        // 全体を削除
                        edited = true;
                    } else {
                        // 全体を残す
                        tmp.Add( (BezierChain)bc.Clone() );
                    }
                }
                this[curve].Clear();
                foreach ( BezierChain bc in tmp ) {
                    if ( bc.ID >= 0 ) {
                        this[curve].Add( bc.ID, bc );
                    }
                }
                foreach ( BezierChain bc in tmp ) {
                    if ( bc.ID < 0 ) {
                        bc.ID = this.GetNextId( curve );
                        this[curve].Add( bc.ID, bc );
                    }
                }
            }
            return edited;
        }
    }

}
