/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is bbs2chreader.
 *
 * The Initial Developer of the Original Code is
 * flyson.
 * Portions created by the Initial Developer are Copyright (C) 2008
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *    flyson <flyson at users.sourceforge.jp>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */


function b2rStorageBoard(){
	this._init();
}

b2rStorageBoard.prototype = {

	_init: function(){
		var database = gStorageService.database;
		database.beginTransaction();
		try{
			if(!database.tableExists("board_subject")){
				let sql = <>
					CREATE TABLE board_subject(
					    board_id   TEXT NOT NULL,
					    dat_id     TEXT NOT NULL,
					    title      TEXT,
					    title_n    TEXT,
					    line_count INTEGER DEFAULT 0,
					    ordinal    INTEGER DEFAULT 0
					);
				</>.toString().replace(/\t/g, "");
				database.executeSimpleSQL(sql);
				database.executeSimpleSQL("CREATE INDEX IF NOT EXISTS board_subject_board_id_index ON board_subject(board_id);");
			}

			if(!database.tableExists("board_data")){
				let sql = <>
					CREATE TABLE board_data(
					    board_id       TEXT NOT NULL,
					    url            TEXT NOT NULL,
					    title          TEXT,
					    title_n        TEXT,
					    type           INTEGER DEFAULT 0,
					    last_modified  INTEGER DEFAULT 0,
					    subscribed     INTEGER DEFAULT 0,
					    post_name      TEXT,
					    post_mail      TEXT
					);
				</>.toString().replace(/\t/g, "");
				database.executeSimpleSQL(sql);
				database.executeSimpleSQL("CREATE INDEX IF NOT EXISTS board_data_board_id_index ON board_data(board_id);");
			}
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			database.commitTransaction();
		}

		this._statements = new Array();
		this._statements["boardSubjectUpdate_insert"] = database.createStatement(
				"INSERT INTO board_subject(board_id, dat_id, title, title_n, line_count, ordinal) VALUES(?1,?2,?3,?4,?5,?6);");

		this._statements["_setBoardData_select"] = database.createStatement(
				"SELECT _rowid_ FROM board_data WHERE board_id=?1;");
		this._statements["_setBoardData_update"] = database.createStatement(
				"UPDATE board_data SET url=?1, type=?2, last_modified=?3 WHERE _rowid_=?4;");
		this._statements["_setBoardData_insert"] = database.createStatement(
				"INSERT OR REPLACE INTO board_data(board_id, url, type, last_modified) VALUES(?1,?2,?3,?4);");

		this._statements["getBoardData_select"] = database.createStatement(
				"SELECT board_id, url, type, last_modified, subscribed, post_name, post_mail FROM board_data WHERE board_id=?1");
	},


	boardSubjectUpdate: function(aBoardURL, aBoardType, aSubjectFile){
		var b2rService = XPC.getService("@bbs2ch.sourceforge.jp/b2r-global-service;1", "b2rIGlobalService");
		var boardID = b2rService.threadUtils.getBoardID(aBoardURL);
		var database = gStorageService.database;


			// 行の解析に使う正規表現
		var regLine;
		switch(aBoardType){
			case b2rService.BOARD_TYPE_2CH:
			case b2rService.BOARD_TYPE_BE2CH:
				regLine = /^(\d{9,10})\.dat<>(.+) ?\((\d{1,4})\)/;
				break;
			case b2rService.BOARD_TYPE_JBBS:
			case b2rService.BOARD_TYPE_MACHI:
				regLine = /^(\d{9,10})\.cgi,(.+) ?\((\d{1,4})\)/;
				break;
		}

		var charset;
		switch(aBoardType){
			case b2rService.BOARD_TYPE_2CH:
			case b2rService.BOARD_TYPE_MACHI:
				charset = "Shift_JIS";
				break;
			case b2rService.BOARD_TYPE_BE2CH:
			case b2rService.BOARD_TYPE_JBBS:
				charset = "euc-jp";
				break;
		}

		var fileStream = b2rService.io.getFileInputStream(aSubjectFile, charset)
								.QueryInterface(Ci.nsIUnicharLineInputStream);

		database.beginTransaction();
		try{
			database.executeSimpleSQL("DELETE FROM board_subject WHERE board_id='" + boardID + "';");
			var statement = this._statements["boardSubjectUpdate_insert"];
			let(line={}, c=0, ordinal=0){
				do {
					c = fileStream.readLine(line);
					if(!regLine.test(line.value)) continue;
					let datID = RegExp.$1;
					let title = gStorageService._html2Text(RegExp.$2);
					let count = Number(RegExp.$3);

					statement.bindStringParameter(0, boardID);
					statement.bindStringParameter(1, datID);
					statement.bindStringParameter(2, title);
					statement.bindStringParameter(3, gStorageService._getNormarizedString(title));
					statement.bindInt32Parameter(4, count);
					statement.bindInt32Parameter(5, ordinal);
					statement.execute();
					ordinal++;
				}while(c);
			}
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			database.commitTransaction();
			fileStream.close();
		}
		this._setBoardData(aBoardURL, aBoardType, aSubjectFile.lastModifiedTime);
	},


	_setBoardData: function(aBoardURL, aType, aLastModified){
		var b2rService = XPC.getService("@bbs2ch.sourceforge.jp/b2r-global-service;1", "b2rIGlobalService");
		var boardID = b2rService.threadUtils.getBoardID(aBoardURL);

		var database = gStorageService.database;
		database.beginTransaction();
		try{
			var statement;
			statement = this._statements["_setBoardData_select"];
			statement.bindStringParameter(0, boardID);
			var boardRowID = 0;
			if(statement.executeStep()){
				boardRowID = statement.getInt64(0);
			}
			statement.reset();

			if(boardRowID){
				statement = this._statements["_setBoardData_update"];
				statement.bindStringParameter(0, aBoardURL.spec);
				statement.bindInt32Parameter(1, aType);
				statement.bindInt64Parameter(2, aLastModified);
				statement.bindInt64Parameter(3, boardRowID);
			}else{
				statement = this._statements["_setBoardData_insert"];
 				statement.bindStringParameter(0, boardID);
				statement.bindStringParameter(1, aBoardURL.spec);
				statement.bindInt32Parameter(2, aType);
				statement.bindInt64Parameter(3, aLastModified);
				statement.execute();
			}
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			database.commitTransaction();
		}
	},

	getBoardData: function(aBoardURL){
		var b2rService = XPC.getService("@bbs2ch.sourceforge.jp/b2r-global-service;1", "b2rIGlobalService");
		var boardID = b2rService.threadUtils.getBoardID(aBoardURL);
		var statement = this._statements["getBoardData_select"];
		statement.bindStringParameter(0, boardID);

		var result = null;
		gStorageService.database.beginTransaction();
		try{
			if(statement.executeStep()){
				result = new b2rBoardData(
					statement.getString(0), // boardID
					statement.getString(1), // url
					statement.getInt32(2),  // type
					statement.getInt64(3),  // last_modified
					statement.getInt32(4),  // subscribed
					statement.getString(5), // post_name
					statement.getString(6)  // post_mail
				);
			}
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			statement.reset();
			gStorageService.database.commitTransaction();
		}

		return result;
	}

};


function b2rBoardData(aBoardID, aURL, aType, aLastModified, aSubscribed, aPostName, aPostMail){
	this._boardID = aBoardID;
	this._url = aURL;
	this._type = aType;
	this._lastModified = aLastModified;
	this._subscribed = Boolean(aSubscribed==1);
	this._postName = aPostName;
	this._postMail = aPostMail;
}

b2rBoardData.prototype = {
	get boardID() {
		return this._boardID;
	},
	get url() {
		if(this._url){
			var ioService = Components.classes["@mozilla.org/network/io-service;1"]
					.getService(Components.interfaces.nsIIOService);
			return ioService.newURI(this._url, null, null).QueryInterface(Components.interfaces.nsIURL);
		}
		return null;
	},
	get type() {
		return this._type;
	},
	get lastModified() {
		return this._lastModified;
	},
	get subscribed() {
		return this._subscribed;
	},
	get postName() {
		return this._postName;
	},
	get postMail() {
		return this._postMail;
	}
};