//
//  BSTGrepClientWindowController.m
//  BathyScaphe
//
//  Created by Tsutomu Sawada on 10/09/20.
//  Copyright 2010 BathyScaphe Project. All rights reserved.
//  encoding="UTF-8"
//

#import "BSTGrepClientWindowController.h"
#import "CocoMonar_Prefix.h"
#import <CocoaOniguruma/OnigRegexp.h>
#import "BSTGrepResult.h"
#import "AppDefaults.h"
#import <SGAppKit/SGAppKit.h>
#import "BSQuickLookPanelController.h"
#import "BSQuickLookObject.h"
#import "CMRThreadSignature.h"
#import "CMRDocumentController.h"
#import "missing.h"


@interface NSObject(BSTGrepCLientArrayControllerStub)
- (void)quickLook:(NSIndexSet *)indexes parent:(NSWindow *)parentWindow keepLook:(BOOL)flag;
@end


@interface BSTGrepClientWindowController(Private)
- (NSString *)createDestinationFolder;
- (NSArray *)parseHTMLSource:(NSString *)source;
- (void)cleanupDownloadTask;
- (void)openThreadsWithNewWindow:(NSArray *)threadSignatures;
- (void)searchFinished:(NSArray *)searchResult;
- (NSArray *)targetThreadsForActionSender:(id)sender;
@end


@implementation BSTGrepClientWindowController

@synthesize lastQuery = m_lastQuery;

APP_SINGLETON_FACTORY_METHOD_IMPLEMENTATION(sharedInstance)

- (id)init
{
    if (self = [super initWithWindowNibName:@"BSTGrepClientWindow"]) {
        ;
    }
    return self;
}

- (void)dealloc
{
    [m_cacheIndex release];
    [m_lastQuery release];
    [m_download release];
    // nib top-level object
    [m_searchResultsController setContent:nil];
    [m_searchResultsController release];

    [super dealloc];
}

- (void)windowDidLoad
{
    [[self window] setAutorecalculatesContentBorderThickness:YES forEdge:NSMinYEdge];
    [[self window] setContentBorderThickness:22 forEdge:NSMinYEdge];
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(applicationWillTerminate:) 
                                                 name:NSApplicationWillTerminateNotification 
                                               object:NSApp];
    [[self searchOptionButton] selectItemWithTag:[CMRPref tGrepSearchOption]];
    [[self searchOptionButton] synchronizeTitleAndSelectedItem];
    [m_tableView setDoubleAction:@selector(tableViewDoubleClick:)];
    [m_tableView setVerticalMotionCanBeginDrag:NO];
    [[m_infoField cell] setBackgroundStyle:NSBackgroundStyleRaised];
}

- (IBAction)showWindow:(id)sender
{
    [super showWindow:sender];
    [[self window] makeFirstResponder:[self searchField]];
}

- (NSArrayController *)searchResultsController
{
    return m_searchResultsController;
}

- (NSSearchField *)searchField
{
    return m_searchField;
}

- (NSPopUpButton *)searchOptionButton
{
    return m_searchOptionButton;
}

- (NSProgressIndicator *)progressIndicator
{
    return m_progressIndicator;
}

- (NSMutableArray *)cacheIndex
{
    if (!m_cacheIndex) {
        m_cacheIndex = [[NSMutableArray alloc] init];
    }
    return m_cacheIndex;
}

- (void)bsURLDownloadDidFinish:(BSURLDownload *)aDownload
{
    NSString *path = [aDownload downloadedFilePath];
    NSString *downloadedContents = [NSString stringWithContentsOfFile:path encoding:NSUTF8StringEncoding error:NULL];
    NSArray *searchResult = [self parseHTMLSource:downloadedContents];
    [self searchFinished:searchResult];
    [[self cacheIndex] addObject:[NSDictionary dictionaryWithObjectsAndKeys:self.lastQuery, @"Query",
                                  path, @"Path", [NSDate dateWithTimeIntervalSinceNow:3600], @"ExpiredDate", NULL]];
    [self cleanupDownloadTask];
}

- (void)bsURLDownload:(BSURLDownload *)aDownload didFailWithError:(NSError *)aError
{    
    NSAlert *alert = [[[NSAlert alloc] init] autorelease];
    [alert setAlertStyle:NSWarningAlertStyle];
    [alert setMessageText:NSLocalizedString(@"BSTGrepClientWindowController Error", @"")];
    [alert setInformativeText:[aError localizedDescription]];

    [self cleanupDownloadTask];
    [alert beginSheetModalForWindow:[self window] modalDelegate:self didEndSelector:@selector(alertDidEnd:returnCode:contextInfo:) contextInfo:NULL];
}

#pragma mark IBActions
- (IBAction)chooseSearchOption:(id)sender
{
    [CMRPref setTGrepSearchOption:[[sender selectedItem] tag]];
}

- (IBAction)startTGrep:(id)sender
{
    if (m_download) {
        NSBeep();
        return;
    }
    NSString *encodedString = [[sender stringValue] stringByURLEncodingUsingEncoding:NSUTF8StringEncoding];
    if (!encodedString || [encodedString isEqualToString:@""]) {
        [[self searchResultsController] setContent:nil];
        return;
    }
    NSString *option = ([[[self searchOptionButton] selectedItem] tag] == BSTGrepSearchByNew) ? @"new" : @"fast";
    NSString *queryString = [NSString stringWithFormat:@"http://page2.xrea.jp/tgrep/search?q=%@&o=%@&n=250&v=2", encodedString, option];

    [[self progressIndicator] startAnimation:self];

    self.lastQuery = [NSURL URLWithString:queryString];
    // Search cache
    NSArray *queries = [[self cacheIndex] valueForKey:@"Query"];
    NSUInteger index = [queries indexOfObject:self.lastQuery];
    if (index != NSNotFound) {
        NSDictionary *cache = [[self cacheIndex] objectAtIndex:index];
        NSDate *expiredDate = [cache objectForKey:@"ExpiredDate"];
        if ([expiredDate compare:[NSDate date]] != NSOrderedAscending) {
            NSString *cachedHTMLPath = [cache objectForKey:@"Path"];
            BOOL isDir;
            if ([[NSFileManager defaultManager] fileExistsAtPath:cachedHTMLPath isDirectory:&isDir] && !isDir) {
                // Use cache.
                NSString *cachedContents = [NSString stringWithContentsOfFile:cachedHTMLPath encoding:NSUTF8StringEncoding error:NULL];
                NSArray *searchResult = [self parseHTMLSource:cachedContents];
                [self searchFinished:searchResult];
                [[self progressIndicator] stopAnimation:self];
                return;
            } else {
                // Invalid cache.
                NSString *dirPath = [cachedHTMLPath stringByDeletingLastPathComponent];
                [[NSFileManager defaultManager] removeItemAtPath:dirPath error:NULL];
                [[self cacheIndex] removeObjectAtIndex:index];
            }
        } else {
            // Expired cache.
            NSString *dirPath = [[cache objectForKey:@"Path"] stringByDeletingLastPathComponent];
            [[NSFileManager defaultManager] removeItemAtPath:dirPath error:NULL];
            [[self cacheIndex] removeObjectAtIndex:index];
        }
    }

    m_download = [[BSURLDownload alloc] initWithURL:self.lastQuery
                                           delegate:self
                                        destination:[self createDestinationFolder]];
}

- (IBAction)cancelCurrentTask:(id)sender
{
    if (m_download) {
        [m_download cancel];
        [self cleanupDownloadTask];
    }
}

- (IBAction)openSelectedThreads:(id)sender
{
    [self openThreadsWithNewWindow:[[self targetThreadsForActionSender:sender] valueForKey:@"threadSignature"]];
}

- (IBAction)openInBrowser:(id)sender
{
    NSArray *urls = [[self targetThreadsForActionSender:sender] valueForKey:@"threadURL"];
    [[NSWorkspace sharedWorkspace] openURLs:urls inBackground:[CMRPref openInBg]];
}

- (IBAction)quickLook:(id)sender
{
    NSIndexSet *indexes;
    if ([sender tag] == 700) {
        NSInteger clickedRow = [m_tableView clickedRow];
        if (clickedRow != -1) {
            if ([m_tableView isRowSelected:clickedRow] && ([m_tableView numberOfSelectedRows] > 1)) {
                indexes = [[self searchResultsController] selectionIndexes];
            } else {
                indexes = [NSIndexSet indexSetWithIndex:clickedRow];
            }
        } else {
            indexes = [[self searchResultsController] selectionIndexes];
        }
    } else {
        indexes = [[self searchResultsController] selectionIndexes];
    }

    [[self searchResultsController] quickLook:indexes parent:[self window] keepLook:NO];
}

- (IBAction)showMainBrowser:(id)sender
{
    id object = [[self targetThreadsForActionSender:sender] objectAtIndex:0];
    CMRThreadSignature *signature = [object valueForKey:@"threadSignature"];
    if (!signature) {
        NSBeep();
        return;
    }
    [[NSApp delegate] showThreadsListForBoard:[signature boardName] selectThread:[signature threadDocumentPath] addToListIfNeeded:YES];
}

- (IBAction)openBBSInBrowser:(id)sender
{
    [[NSWorkspace sharedWorkspace] openURL:self.lastQuery inBackground:[CMRPref openInBg]];
}

- (BOOL)validateUserInterfaceItem:(id<NSValidatedUserInterfaceItem>)anItem
{
    SEL action = [anItem action];
    if (action == @selector(cancelCurrentTask:)) {
        return (m_download != nil);
    }

    if (action == @selector(openSelectedThreads:) || 
        action == @selector(openInBrowser:) || 
        action == @selector(quickLook:) || 
        action == @selector(showMainBrowser:)) {
        if ([anItem tag] == 700) { // Contexual Menu Item
            NSInteger clickedRow = [m_tableView clickedRow];
            if (clickedRow != -1) {
                return YES;
            }
        }
        return ([[[self searchResultsController] selectionIndexes] count] > 0);
    }

    if (action == @selector(openBBSInBrowser:)) {
        if ([(id)anItem isKindOfClass:[NSMenuItem class]]) {
            [(NSMenuItem *)anItem setTitle:NSLocalizedString(@"Open Search Results in Browser", @"")];
        }
        return (m_lastQuery != nil);
    }
    return YES;
}

- (void)applicationWillTerminate:(NSNotification *)aNotification
{       
    NSString *cacheDir = [[[CMRFileManager defaultManager] supportDirectoryWithName:@"tGrep cache"] filepath];
    [[NSFileManager defaultManager] removeItemAtPath:cacheDir error:NULL];
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

#pragma mark Table View Delegate
- (BOOL)tableView:(NSTableView *)aTableView shouldPerformKeyEquivalent:(NSEvent *)theEvent
{
    if ([aTableView selectedRow] == -1) {
        return NO;
    }

    NSString *whichKey = [theEvent charactersIgnoringModifiers];

    if ([whichKey isEqualToString:@" "]) { // space key
        [self quickLook:aTableView];
        return YES;
    }
    
    if ([whichKey isEqualToString:[NSString stringWithCharacter:NSCarriageReturnCharacter]]) { // return key
        [self openSelectedThreads:aTableView];
        return YES;
    }
    return NO;
}

- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
    BSQuickLookPanelController *qlc = [BSQuickLookPanelController sharedInstance];
    if ([qlc isLooking]) {
        [[self searchResultsController] quickLook:[[self searchResultsController] selectionIndexes] parent:[self window] keepLook:YES];
    }
}
@end


@implementation BSTGrepClientWindowController(Private)
- (NSString *)createDestinationFolder
{
    NSString *path_;

    NSString *cacheDir = [[[CMRFileManager defaultManager] supportDirectoryWithName:@"tGrep cache"] filepath];
    NSString *tmpDir = [cacheDir stringByAppendingPathComponent:@"rskXXXXXX"];

    char *cTmpDir = strdup([tmpDir fileSystemRepresentation]);
    
    mkdtemp(cTmpDir);
    path_ = [[NSFileManager defaultManager] stringWithFileSystemRepresentation:cTmpDir length:strlen(cTmpDir)];
    
    free(cTmpDir);
    
    return path_;
}

- (NSArray *)parseHTMLSource:(NSString *)source
{
    static OnigRegexp *re = nil;
    if (!re) {
        re = [[OnigRegexp compile:@"<a id=\"tt([0-9]*)\" href=\"(.*)\" target=\"_blank\">(.*)</a>" ignorecase:NO multiline:NO] retain];
    }
    NSUInteger length = [source length];
    NSRange searchRange = NSMakeRange(0, length);
    NSMutableArray *foo = [NSMutableArray arrayWithCapacity:250];
    OnigResult *result;
    BSTGrepResult *obj;
    NSRange foundRange;
    while (searchRange.length > 0) {
        result = [re search:source range:searchRange];
        if (!result) {
            break;
        }
        obj = [[BSTGrepResult alloc] initWithOrder:[result stringAt:1] URL:[result stringAt:2] title:[result stringAt:3]];
        [foo addObject:obj];
        [obj release];
        foundRange = [result bodyRange];
        searchRange.location = NSMaxRange(foundRange);
        searchRange.length = length - searchRange.location;
    }

    return foo;
}

- (void)alertDidEnd:(NSAlert *)alert returnCode:(NSInteger)code contextInfo:(void *)contextInfo
{
    ;
}

- (void)cleanupDownloadTask
{
    [[self progressIndicator] stopAnimation:nil];
    [m_download release];
    m_download = nil;
}

- (IBAction)fromQuickLook:(id)sender
{
    NSObjectController *oc = [(BSQuickLookPanelController *)[sender windowController] objectController];
    id obj = [oc valueForKeyPath:@"selection.threadSignature"];
    if (!obj) {
        return;
    }
    
    [self openThreadsWithNewWindow:[NSArray arrayWithObject:obj]];
}

- (IBAction)tableViewDoubleClick:(id)sender
{
    if ([sender clickedRow] == -1) {
        return;
    }
    [self openSelectedThreads:sender];
}

- (void)openThreadsWithNewWindow:(NSArray *)threadSignatures
{
    NSString *path;
    NSDictionary *boardInfo;
    for (id thread in threadSignatures) {
        if (![thread isKindOfClass:[CMRThreadSignature class]]) {
            continue;
        }
        path = [thread threadDocumentPath];
        boardInfo = [NSDictionary dictionaryWithObjectsAndKeys:[thread boardName], ThreadPlistBoardNameKey,
                     [thread identifier], ThreadPlistIdentifierKey,
                     NULL];
        [[CMRDocumentController sharedDocumentController] showDocumentWithContentOfFile:[NSURL fileURLWithPath:path] boardInfo:boardInfo];
    }
}

- (void)searchFinished:(NSArray *)searchResult
{
    [[self searchResultsController] setContent:searchResult];
    if ([searchResult count] > 0) {
        [[self window] makeFirstResponder:[m_tableView enclosingScrollView]];
    }
}

- (NSArray *)targetThreadsForActionSender:(id)sender
{
    if ([sender tag] == 700) {
        NSInteger clickedRow = [m_tableView clickedRow];
        if (clickedRow != -1) {
            if ([m_tableView isRowSelected:clickedRow] && ([m_tableView numberOfSelectedRows] > 1)) {
                return [[self searchResultsController] selectedObjects];
            } else {
                return [NSArray arrayWithObject:[[[self searchResultsController] arrangedObjects] objectAtIndex:clickedRow]];
            }
        }        
    }
    return [[self searchResultsController] selectedObjects];
}    
@end
