//
//  CMRKeychainManager.m
//  BathyScaphe
//
//  Updated by Tsutomu Sawada on 11/01/13.
//  Copyright 2005-2011 BathyScaphe Project. All rights reserved.
//  encoding="UTF-8"
//

#import "CMRKeychainManager.h"
#import "CocoMonar_Prefix.h"
#import "AppDefaults.h"
#import <AppKit/NSApplication.h>
#import <Security/Security.h>

@implementation CMRKeychainManager
APP_SINGLETON_FACTORY_METHOD_IMPLEMENTATION(defaultManager);

#pragma mark Accessors
- (BOOL)shouldCheckHasAccountInKeychain
{
    return m_shouldCheckHasAccountInKeychain;
}
- (void)setShouldCheckHasAccountInKeychain:(BOOL)flag
{
    m_shouldCheckHasAccountInKeychain = flag;
}

- (NSURL *)x2chAuthenticationRequestURL
{
    return [CMRPref x2chAuthenticationRequestURL];
}

- (NSURL *)be2chAuthenticationRequestURL
{
    return [CMRPref be2chAuthenticationRequestURL];
}

- (NSString *)x2chUserAccount
{
    return [CMRPref x2chUserAccount];
}

- (NSString *)be2chAccountMailAddress
{
    return [CMRPref be2chAccountMailAddress];
}

#pragma mark Public Methods
- (id)init
{
    if (self = [super init]) {
        [self setShouldCheckHasAccountInKeychain:YES];
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(applicationDidBecomeActive:)
                                                     name:NSApplicationDidBecomeActiveNotification
                                                   object:NSApp];
    }
    return self;
}

- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [super dealloc];
}

- (void)checkHasAccountInKeychainImpl:(BSKeychainAccountType)type
{
    BOOL result_ = NO;
    NSString *account_;
    NSURL *url_;
    SecProtocolType protocolType;
    if (type == BSKeychainAccountX2chAuth) {
        account_ = [self x2chUserAccount];
        url_ = [self x2chAuthenticationRequestURL];
        protocolType = kSecProtocolTypeHTTPS;
    } else if (type == BSKeychainAccountBe2chAuth) {
        account_ = [self be2chAccountMailAddress];
        url_ = [self be2chAuthenticationRequestURL];
        protocolType = kSecProtocolTypeHTTP;
    }

    if (account_) {
        OSStatus err;
        SecKeychainItemRef item = nil;        
        NSString    *host_ = [url_ host];
        NSString    *path_ = [url_ path];

        const char  *accountUTF8 = [account_ UTF8String];
        const char  *hostUTF8 = [host_ UTF8String];
        const char  *pathUTF8 = [path_ UTF8String];

        err = SecKeychainFindInternetPassword(NULL,
                                              strlen(hostUTF8),
                                              hostUTF8,
                                              0,
                                              NULL,
                                              strlen(accountUTF8),
                                              accountUTF8,
                                              strlen(pathUTF8),
                                              pathUTF8,
                                              0,
                                              protocolType,
                                              kSecAuthenticationTypeDefault,
                                              NULL,
                                              NULL,
                                              &item);

        if ((err == noErr) && item) {
            result_ = YES;
        }
    }

    [CMRPref setHasAccountInKeychain:result_ forType:type];
}    

- (void)checkHasAccountInKeychainIfNeeded
{
    if ([self shouldCheckHasAccountInKeychain]) {
        [self checkHasAccountInKeychainImpl:BSKeychainAccountX2chAuth];
        [self checkHasAccountInKeychainImpl:BSKeychainAccountBe2chAuth];
        [self setShouldCheckHasAccountInKeychain:NO];
    }
}

/*- (BOOL)deleteAccountCompletely
{
    NSString        *account_ = [self x2chUserAccount];

    if (!account_) return YES;

    OSStatus err;
    SecKeychainItemRef item = nil;

    NSURL           *url_ = [self x2chAuthenticationRequestURL];
    NSString        *host_ = [url_ host];
    NSString        *path_ = [url_ path];
    
    const char      *accountUTF8 = [account_ UTF8String];
    const char      *hostUTF8 = [host_ UTF8String];
    const char      *pathUTF8 = [path_ UTF8String];

    err = SecKeychainFindInternetPassword(NULL,
                                          strlen(hostUTF8),
                                          hostUTF8,
                                          0,
                                          NULL,
                                          strlen(accountUTF8),
                                          accountUTF8,
                                          strlen(pathUTF8),
                                          pathUTF8,
                                          0,
                                          kSecProtocolTypeHTTPS,
                                          kSecAuthenticationTypeDefault,
                                          NULL,
                                          NULL,
                                          &item);

    if ((noErr == err) && item) {
        err = SecKeychainItemDelete(item);
        if (err == noErr) {
            CFRelease(item);
            return YES;
        } else {
            return NO;
        }
    } else {
        return YES;
    }
}*/

- (NSString *)passwordFromKeychain:(BSKeychainAccountType)type
{
    NSString *account_;
    NSURL *url_;
    SecProtocolType protocolType;
    if (type == BSKeychainAccountX2chAuth) {
        account_ = [self x2chUserAccount];
        url_ = [self x2chAuthenticationRequestURL];
        protocolType = kSecProtocolTypeHTTPS;
    } else if (type == BSKeychainAccountBe2chAuth) {
        account_ = [self be2chAccountMailAddress];
        url_ = [self be2chAuthenticationRequestURL];
        protocolType = kSecProtocolTypeHTTP;
    }
    if (!account_) {
        return nil;
    }
    OSStatus err;
    SecKeychainItemRef item = nil;

    NSString        *host_ = [url_ host];
    NSString        *path_ = [url_ path];

    const char      *accountUTF8 = [account_ UTF8String];
    const char      *hostUTF8 = [host_ UTF8String];
    const char      *pathUTF8 = [path_ UTF8String];
    char *passwordData;
    UInt32 passwordLength;

    err = SecKeychainFindInternetPassword(NULL,
                                          strlen(hostUTF8),
                                          hostUTF8,
                                          0,
                                          NULL,
                                          strlen(accountUTF8),
                                          accountUTF8,
                                          strlen(pathUTF8),
                                          pathUTF8,
                                          0,
                                          protocolType,
                                          kSecAuthenticationTypeDefault,
                                          &passwordLength,
                                          (void **)&passwordData,
                                          &item);

    if ((err == noErr) && item) {
        NSString *result_ = [[NSString alloc] initWithBytesNoCopy:passwordData
                                                           length:passwordLength
                                                         encoding:NSUTF8StringEncoding
                                                     freeWhenDone:YES];
        return [result_ autorelease];
    }

    return nil;
}

- (BOOL)createKeychainWithPassword:(NSString *)password forType:(BSKeychainAccountType)type
{
    NSString *account_;
    NSURL *url_;
    SecProtocolType protocolType;
    if (type == BSKeychainAccountX2chAuth) {
        account_ = [self x2chUserAccount];
        url_ = [self x2chAuthenticationRequestURL];
        protocolType = kSecProtocolTypeHTTPS;
    } else if (type == BSKeychainAccountBe2chAuth) {
        account_ = [self be2chAccountMailAddress];
        url_ = [self be2chAuthenticationRequestURL];
        protocolType = kSecProtocolTypeHTTP;
    }

    if (!account_) {
        return NO;
    }
    OSStatus err;

    NSString        *host_ = [url_ host];
    NSString        *path_ = [url_ path];

    const char      *accountUTF8 = [account_ UTF8String];
    const char      *hostUTF8 = [host_ UTF8String];
    const char      *pathUTF8 = [path_ UTF8String];
    const char      *passwordUTF8 = [password UTF8String];

    err = SecKeychainAddInternetPassword(NULL,
                                         strlen(hostUTF8),
                                         hostUTF8,
                                         0,
                                         NULL,
                                         strlen(accountUTF8),
                                         accountUTF8,
                                         strlen(pathUTF8),
                                         pathUTF8,
                                         0,
                                         protocolType,
                                         kSecAuthenticationTypeDefault,
                                         strlen(passwordUTF8),
                                         passwordUTF8,
                                         NULL);
 
    if(err == errSecDuplicateItem) {
//      NSLog(@"Keychain item already exists.");
        SecKeychainItemRef item = nil;
        err = SecKeychainFindInternetPassword(NULL,
                                              strlen(hostUTF8),
                                              hostUTF8,
                                              0,
                                              NULL,
                                              strlen(accountUTF8),
                                              accountUTF8,
                                              strlen(pathUTF8),
                                              pathUTF8,
                                              0,
                                              protocolType,
                                              kSecAuthenticationTypeDefault,
                                              NULL,
                                              NULL,
                                              &item);

        if(item) {
            err = SecKeychainItemModifyContent(item, NULL, strlen(passwordUTF8), passwordUTF8);
            if(err == noErr) {
//              NSLog(@"Replacing keychain item's data successfully finished.");
                CFRelease(item);
                [CMRPref setHasAccountInKeychain:YES forType:type];
                return YES;
            }
        } else {
//          NSLog(@"Some Error Occurred while modifying content.");
            return NO;
        }
    } else if (err == noErr) {
//      NSLog(@"Keychain item was successfully created.");
        [CMRPref setHasAccountInKeychain:YES forType:type];
        return YES;
    } else {
//      NSLog(@"Some error occurred while creating keychain item");
        return NO;
    }
    return NO;
}

#pragma mark Notifications
- (void)applicationDidBecomeActive:(NSNotification *)theNotification
{
    UTILAssertNotificationName(theNotification, NSApplicationDidBecomeActiveNotification);
    UTILAssertNotificationObject(theNotification, NSApp);

    [self setShouldCheckHasAccountInKeychain:YES];
}
@end
